#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sgtty.h>

#define SHIFT { --argc; ++argv; }
#define LINESIZE 512
#define BLANK ' '
#define UNDER '-'
#define COLON ':'
#define SHORT 15
#define LONG  75
#define MAXWORDS     96
#define MAXLONGWORDS 24

#define min(a, b) ((a < b)? a: b)
#define wordchar(c) (isalnum(c) || c == '-' || c == '&')

int pid, rpid, status;
int multfile, coi, british, nodict;
int nedneeded, nedwrote, wordsrespelled;
char mspellnedF[30], tmpa[30], tmpb[30];
char *sowarning =
	": WARNING - there may be misspelled words in the above file";
char *doublechars = "{}[]^`";
char dict[80];
FILE *df;

int nwords, nlongwords, nmspwords;
char *word[ MAXWORDS + MAXLONGWORDS + 1 ];
char displayword[ MAXWORDS + 1 ][ SHORT + 1];
char longdisplayword[ MAXLONGWORDS + 1 ][ LONG + 1 ];
char code[ MAXWORDS + MAXLONGWORDS + 1 ][3];
int mspword[ MAXWORDS + MAXLONGWORDS + 1 ];

int index(), thecursoriswithinaword(), catch();
char *fgets(), *strsave(), *malloc();

main(argc, argv)
int argc;
char **argv;
{
	char *d;
	char dir[80];

	initialize();
	SHIFT;
	/*
	 * what dictionary should we use?
	 * look for the -b option and report on invalid options.
	 */
	*dict = 0;
	nodict = 0;
	british = 0;
	while (**argv == '-') {
		if (strcmp(*argv, "-d") == 0) {
			SHIFT;
			if (!argc)
				error("no file after -d\n");
			strcpy(dict, *argv);
			if (access(dict, 6) < 0)
	                        error("cannot access %s\n", dict);
		} else if (strcmp(*argv, "-n") == 0)
			nodict = 1;
		else if (strcmp(*argv, "-b") == 0)
			british = 1;
		else
			error("invalid option: %s\n", *argv);
		SHIFT;
	}
	if (!*dict) {
		strcpy(dict, "dict");
	        if (dictaccess(dict) < 0) {
		        sprintf(dict, "%s/.../dict", getenv("HOME"));
		        if (dictaccess(dict) < 0) {
				*dict = 0;
			        if (access(".", 2) < 0) {
                                        fprintf(stderr, "mspell:\n");
                                        fprintf(stderr, "would not be able to create a dictionary\n");
                                        fprintf(stderr, "in the current directory if you ever put\n");
                                        fprintf(stderr, "a 'd' by a word.\n");
                                        exit(1);
				}
			}
	        }
	}
	/*
	 * now mspell each of the files
	 */
	multfile = argc > 1;
	while (argc) {
		dname(strcpy(dir, *argv));
		if (access(*argv, 6) < 0)
			fprintf(stderr,
                                "mspell: cannot access %s\n", *argv);
		else if (access(dir, 2) < 0)
                        fprintf(stderr,
                                "mspell: cannot write in %s\n", dir);
			/*
			 * we need to creat a tmp file
			 * in the same directory.
			 */
		else {
			if (multfile)
				fprintf(stderr, "%s:\n", *argv);
		        mspell(*argv);
		}
		SHIFT;
	}
	unlink(tmpa);
	unlink(tmpb);
	exit(0);
}

initialize()
{
	struct sgttyb buf;

	setbuf(stdout, NULL);
	signal(SIGINT, catch);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGTERM, SIG_IGN);
	strcpy(mspellnedF, getenv("HOME"));
	strcat(mspellnedF, "/.../mspell");
	if (access(mspellnedF, 4) < 0)
		strcpy(mspellnedF, "/usr/lib/mspell");
	strcpy(tmpa, "/tmp/mspellXXXXXX");
	mktemp(tmpa);
	gtty(open("/dev/tty", 0), &buf);
        coi = (buf.sg_flags & ODBL);
}

int
dictaccess(d)
char *d;
{
	if (access(d, 0) < 0)
		return(-1);
	else if (access(d, 6) < 0)
		error("cannot access %s\n", d);
}

catch(sig)
int sig;
{
	signal(sig, catch);
	unlink(tmpa);
	unlink(tmpb);
	exit(1);
}

mspell(source)
char *source;
{
	int i;
	FILE *f;

        spell(source, tmpa);
	df = 0;
	if ((f = fopen(tmpa, "r")) == NULL)
		error("cannot open %s\n", tmpa);
	maketmpb(source);
	while (!feof(f)) {
                nwords = 0;
                nlongwords = 0;
                nmspwords = 0;
	        nedneeded = 0;
	        readwords(f);
		if (nwords > 0) {
			showshortwords();
			checkcodes(1, nwords);
		}
		if (nlongwords > 0) {
			showlongwords();
			checkcodes(MAXWORDS + 1, MAXWORDS + nlongwords);
		}
	        if (nmspwords != 0) {
			nedwrote = wordsrespelled = 0;
	                indexmerge(source, tmpb);
		        if (nedneeded) {
				signal(SIGINT, SIG_IGN);
	                        ned(tmpb);
				signal(SIGINT, catch);
			}
			if (nedwrote || wordsrespelled)
	                        filter(tmpb, source);
		}
	        /*
	         * free the space allocated when saving the words.
	         */
	        for (i = 1; i <= nwords; ++i)
		        free(word[i]);
	}
	fclose(f);
	unlink(tmpb);
	/*
	 * Close the tube in case we didn't exec ned at the end
	 * and there is more than one file to mspell.
	 * The "fname:" would need the tube.
	 */
	qsclose();
	/*
	 * if some words were put in the dictionary, 
	 * either by show*words or by filter, 
	 * sort it.
	 */
	if (df) {
		fclose(df);
		sort(dict);
	}
}

/*
 * make a new tmp file name that looks like the source file.
 * try these: "source", "source ", "source.A", ..., "source.Z".
 *                             *
 */

maketmpb(source)
char *source;
{
	char append;
	char *p;

	p = sname(source);
	sprintf(tmpb, "/tmp/%.14s", p);
	if (access(tmpb, 4) < 0)
	        return;
	sprintf(tmpb, "/tmp/%.13s ", p);
	if (access(tmpb, 4) < 0)
	        return;
	append = 'A';
	for (append = 'A'; append <= 'Z'; append++) {
		sprintf(tmpb, "/tmp/%.12s.%c", p, append);
		if (access(tmpb, 4) < 0)
			return;
	}
	error("cannot create tmpb\n");
}

spell(source, misspellings)
char *source, *misspellings;
{
	char s[100];

        if (nodict || !*dict)
                sprintf(s, "/bin/spell %s %s", 
                          (british)? "-b": "", source);
        else
                sprintf(s, "/bin/spell %s %s | /bin/comm -f -13 %s -", 
                          (british)? "-b": "", source, dict);
	if ((pid = fork()) < 0)
		error("cannot fork\n");
	if (pid == 0) {
                close(1);                  /* redirect stdout */
                creat(misspellings, 0600);  /* to mispellings  */
		execl("/bin/sh", "sh", "-c", s, 0);
		error("cannot exec /bin/sh\n");
	}
	while ((rpid = wait(&status)) != pid && rpid >= 0)
		;
	if (rpid < 0)
		error("no child to wait for\n");
}

readwords(wf)
FILE *wf;
{
	int i;
	char *p;
	char s[ LINESIZE ];

	while (nwords < MAXWORDS &&
               nlongwords < MAXLONGWORDS &&
               fgets(s, LINESIZE, wf) != NULL) {
		p = strsave(s);
		if (strlen(p) <= SHORT) {
			++nwords;
			word[nwords] = p;
			strcpy(displayword[nwords], p);
		} else {
			++nlongwords;
			word[nlongwords + MAXWORDS] = p;
			strncpy(longdisplayword[nlongwords], p, LONG);
		}
	}
	for (i = 1; i <= MAXWORDS + MAXLONGWORDS; ++i)
		*code[i] = 0;
}

showshortwords()
{
	int i, curpos, pos;
	char prefix[ SHORT + 1 ];

	curpos = 101;
	qskp = CLEAR;
	while (qskp != ENTER) {
	        panel (cursor = curpos) {
C               for (i = 1; i <= nwords; ++i) {
C                        pos = 100*((i-1)%24+1)+         /* row    */
C                               20*((i-1)/24  )+1;       /* column */
	                 #@pos##IN, code[i], 2# #IN, displayword[i], SHORT#
C               }
	        }

		curpos = qscrow*100 + qsccol;

		if (qskp == CLEAR || qskp == TESTREQ)
			;
		else if (qskp == PF1)
                        help();
		else if (qskp == PF2)
			shell();
                else if (qskp == PF10 &&
                         thecursoriswithinaword(prefix))
			look(prefix);
		else if (qskp == PF12) {
			for (i = 1; i <= nwords; ++i)
				if (strcmp(word[i], displayword[i]) == 0 &&
                                    !*code[i])
					*code[i] = 'd';
		}
	}
}

showlongwords()
{
	int i, curpos, pos;
	char prefix[ LONG + 1 ];

	curpos = 101;
	qskp = CLEAR;
	while (qskp != ENTER) {
	        panel (cursor = curpos) {
C               for (i = 1; i <= nlongwords; ++i) {
C                        pos = 100*i+1;
	                 #@pos##IN, code[i+MAXWORDS], 2#
                         #@pos+3##IN, longdisplayword[i], LONG#
C               }
	        }

		curpos = qscrow*100 + qsccol;

		if (qskp == CLEAR || qskp == TESTREQ)
			;
		else if (qskp == PF1)
                        help();
		else if (qskp == PF2)
			shell();
                else if (qskp == PF10 &&
                         longthecursoriswithinaword(prefix))
			look(prefix);
		else if (qskp == PF12) {
		       for (i = MAXWORDS+1; i <= MAXWORDS+nlongwords; ++i)
				if (strcmp(word[i], 
                                      longdisplayword[i-MAXWORDS]) == 0 &&
                                    !*code[i])
					*code[i] = 'd';
		}
	}
}

checkcodes(a, b)
int a, b;
{
	int i;

	for (i = a; i <= b; ++i)
		if (*code[i] == 'd') {
			if (!df)
		                getdict();
			fputs(word[i], df);
		} else if (*code[i] == 's') {
			mspword[++nmspwords] = i;
			nedneeded = 1;
                } else if ((i <= MAXWORDS &&
                            strcmp(word[i], displayword[i]) != 0) ||
		           (i > MAXWORDS &&
                            strcmp(word[i], longdisplayword[i-MAXWORDS]) != 0)
                          ) {
			        mspword[++nmspwords] = i;
				*code[i] = 'r';
		} else
			*code[i] = 0;         /* to be sure */
}

help()
{
	at 410;
	panel {
        Words can be respelled in the document
        by typing over them on the screen.

	If the cursor is within a word, pressing PF10 will
	do a 'look' to find all words in the dictionary that
	begin with that prefix.  The word you are trying to
        spell may be among them.

        Codes to the left of words:
                 d     Place the word in the dictionary.
                 s     Select the word to be underlined within ned.

        PF Keys:
		 ENTER  Go on to ned (or another screen of words).
                 PF1    Help
                 PF2    Exec a shell
                 PF10   Do a 'look' on the prefix.
                 PF12   Put d's everywhere else.
	}
	qskp = PF1;
	/*
         * exiting the help screen with ENTER
         * will not terminate the other screen
	 */
}

shell()
{
	int pid, rpid, status;

        qsclose();
        if ((pid = fork()) < 0)
                error("cannot fork\n");
        if (pid == 0) {
                execl("/bin/sh", "sh", 0);
                error("cannot exec /bin/sh\n");
        }
        while ((rpid = wait(&status)) != pid && rpid >= 0)
                ;
	if (rpid < 0)
		error("no child to wait for\n");
}

int
thecursoriswithinaword(prefix)
char *prefix;
{
	int nchars, i;
	char *p;

	nchars = qsccol%20 - 4;
	if (nchars < 1 || 15 < nchars)
		return(0);
	i = 24*(qsccol/20) + qscrow;
	if (i > nwords)
		return(0);
	p = displayword[i];
	while (nchars--)
		*prefix++ = *p++;
	*prefix = 0;
	return(1);
}

int
longthecursoriswithinaword(prefix)
char *prefix;
{
	int nchars, i;
	char *p;

	nchars = qsccol - 4;
	if (nchars < 1)
		return(0);
	i = qscrow;
	if (i > nlongwords)
		return(0);
	p = longdisplayword[i];
	while (nchars--)
		*prefix++ = *p++;
	*prefix = 0;
	return(1);
}

look(prefix)
char *prefix;
{
	int i;

	qsclose();
        fprintf(stderr, "!look %s\n", prefix);
	if ((pid = fork()) < 0)
		error("cannot fork\n");
	if (pid == 0) {
		execl("/bin/look", "look", prefix, 0);
		error("cannot execl /bin/look\n");
	}
	while ((rpid = wait(&status)) != pid && rpid >= 0)
		;
	if (rpid < 0)
		error("no child to wait for\n");
	putc('!\n', stderr);
	for (i = 1; i <= 23; ++i)
		putc('\n', stderr);
}

sort(file)
char *file;
{
	if ((pid = fork()) < 0)
		error("cannot fork\n");
	if (pid == 0) {
		execl("/bin/sort", "sort", 
                      "-u", "+0f", "+0", file, "-o", file, 0);
		error("cannot execl /bin/sort\n");
	}
	while ((rpid = wait(&status)) != pid && rpid >= 0)
		;
	if (rpid < 0)
		error("no child to wait for\n");
}

int offset, len, perfect;
char bold, underscore;
char *pw, *ps, *pb, *pl, *pm;
char markline[LINESIZE];
char srcline[LINESIZE];

indexmerge(source, mergedfile)
char *source, *mergedfile;
{
	int i, j, n;
	FILE *sf, *mf;
	char *p;

	if ((sf = fopen(source, "r")) == NULL)
		error("cannot open %s\n", source);
	if ((mf = fopen(mergedfile, "w")) == NULL)
		error("cannot open %s\n", mergedfile);

	bold = 0;
	underscore = 0;
	while (fgets(srcline, LINESIZE, sf) != NULL) {
		if (*srcline == COLON) {
			qsclose();
			fprintf(stderr, "%s\nThe above line from the file %s\n",
                                        srcline, source);
			fprintf(stderr, "begins with a colon which has a special meaning\n");
                        fprintf(stderr, "to mspell.  You must fix this before mspell can continue.\n");
			nedneeded = 0;
			break;
		} else if (strncmp(srcline, ".so", 3) == 0) {
		        fputs(srcline, mf);
	                fputs(sowarning, mf);
			nedneeded = 1;
			continue;
		} else if (strncmp(srcline, ".fo", 3) == 0) {
		        fputs(srcline, mf);
			p = srcline+3;
			while (*p == BLANK)
				++p;
			bold = *p++;
			if (*p)
				underscore = *p;
			continue;
		}
		/*
		 * The function index is special and is defined below.
		 * It returns true if the word was found otherwise false.
		 * It communicates through these global variables:
		 *
		 * pw       Points to the word to search for.
		 *          It is unchanged by index.
		 * ps       Points to the srcline.  This moves along with
		 *          successive calls to index so that the same
		 *          word misspelled twice in the same line can
		 *          be found.
		 * pb       Points to the beginning of the word
		 *          that index found in the srcline.
		 * len      Is the length of the word it found. It might
		 *          be greater than the length of the word if
		 *          there were embedded font changes.
		 * perfect  Is true if the found word was surrounded by
		 *          word terminators and had no embedded font
		 *          changes. Words can be automatically respelled
		 *          only if the index is perfect.
		 *
		 * The above variables are also used by the functions
		 * underline and respell.
		 *
		 * Underline also uses:
		 *
		 * pm       Points to the end of the marked line.
                 * pl       Points to the last character of the srcline
                 *          that was "taken care of" in the markline.
		 */
		pm = markline;
		pl = srcline;
		for (i = 1; i <= nmspwords; ++i) {
			j = mspword[i];
                        pw = word[j];
			ps = srcline;
			while (index())
				if (*code[j] == 'r' && perfect)
				        respell((j <= MAXWORDS)?
                                                  displayword[j]:
                                             longdisplayword[j-MAXWORDS]);
				else {
					underline();
					nedneeded = 1;
				}
		}
		fputs(srcline, mf);
		if (pm > markline) {
			*pm = 0;
			*markline = COLON;
			fputs(markline, mf);
		}
	}
	fclose(sf);
	fclose(mf);
}

/*
 * Is there an occurence of the word in the source line?
 * For illustration we are looking for the word "hello".
 */

int
index()
{
	int i;
	char *ops, *opw, *p;

	ops = ps;
	opw = pw;
	while (*ps)
		if (*ps != *pw)                              /* goodbye */
			++ps;
		else {                                      /* good how */
                        perfect = 1;
                        if (ps-1 >= ops && isalnum(*(ps-1))) {
				/*
				 * the character before could be part
				 * of a word. we need to determine if
				 * there is a construction like:
                                 *       \fIhello\fR.
				 */
				if (ps-1 == ops) {             /* |show */
                                        ++ps;
                                        continue;
                                }
                                i = 2;
                                p = ps-2;
                                while (p >= ops && *p != BLANK &&
                                       *p != '\\' && i <= 5      ) {
                                        --p;
                                        ++i;
                                }
                                if (*p != '\\') {          /* good show */
                                        ++ps;
                                        continue;
                                }
				/*
				 * yes, there is such a construction.
				 */
                                perfect = 0;
                        }
                        pb = ps;
                        ++ps;
                        ++pw;
                        while (*pw)
                                if (*ps == *pw) {
                                        ++ps;
                                        ++pw;
                                } else if (*ps == bold ||    /* hel_l_o */
                                           *ps == underscore) {
                                        ++ps;
                                        perfect = 0;
                                } else
                                        break;
			if (!*pw) {                       /* it matched */
	                        len = ps-pb;
	                        pw = opw;
	                        while (*ps &&            /* _hello_pp   */
                                       (*ps == bold || *ps == underscore))
	                                ++ps;
	                        if (!isalnum(*ps))            /* hello. */
	                                return(1);
	                        else
	                                ps = pb+1;       /* hellonearth */
		        } else
	                        pw = opw;
                }
	return(0);
}

underline()
{
	int i;
	char *p, *q;

	if (pl <= pb) {
	        while (pl < pb) {
	                if (*pl == '\t')
	                        for (i = 1; i <= 7; ++i)
	                                *pm++ = BLANK;
	                else if (coi && any(*pl, doublechars))
	                        *pm++ = BLANK;
	                *pm++ = BLANK;
	                ++pl;
	        }
	        pl += len;
	        while (len--)
	                *pm++ = UNDER;
	} else {
		p = srcline;
		q = markline;
		while (p < pb) {
			if (*p == '\t')
				q += 8;
			else if (coi && any(*p, doublechars))
				q += 2;
			else
				q += 1;
			++p;
		}
	        while (len--)
	                *q++ = UNDER;
	}
}

respell(newspelling)
char *newspelling;
{
	int newlen, d;
	char *p, *q, *last;

	/*
	 * if the new spelling is longer or shorter than the old
	 * then we must adjust srcline to accomodate it.
	 */
	d = strlen(newspelling)-len;
	if (d == 0)
		;
	else if (d < 0)
		strcpy(pb+len+d, pb+len);
	else if (d > 0) {
		p = strend(ps);
		last = pb+len;
		while (p >= last) {
			*(p+d) = *p;
			--p;
		}
	}
	/*
	 * now respell the word.
	 */
	p = pb;
	q = newspelling;
	while (*q)
		*p++ = *q++;
	/*
	 * and set ps after the new word.
	 */
	ps = p;
	/*
	 * if there were underlines below the srcline and we adjusted
	 * srcline then we must move any underlines that appear to the
	 * right of the new word.
	 */
	p = markline+(pb-srcline)+offset;
	if (pm > p && d != 0) {
                if (d < 0) {
                        strcpy(p, p-d);
			pm += d;
                } else if (d > 0) {
                        q = pm;
                        while (q >= p) {
                                *(q+d) = *q;
                                --q;
                        }
			pm += d;
                        /*
                         * fill the hole with spaces.
                         */
                        while (d--)
                                *q-- = BLANK;
                }
	}
	wordsrespelled = 1;
}

ned(file)
char *file;
{
	qsclose();
	if ((pid = fork()) < 0)
		error("cannot fork\n");
	if (pid == 0) {
		execl("/bin/ned", "ned", "-F", mspellnedF, file, 0);
		error("cannot execl /bin/ned\n");
	}
	while ((rpid = wait(&status)) != pid && rpid >= 0)
		;
	if (rpid < 0)
		error("no child to wait for\n");
	status >>= 8;
	nedwrote = (status == 1);
}

/*
 * copy lines from the "in" file to the "out" file
 * filtering those with a ':' in the first character.
 *
 * if a colon line has a 'd' in it, look at the word "above" it.
 * find the word by scanning left and right to a word terminator.
 * if it was selected, place it in the dictionary; otherwise give
 * an error message.  use double buffering to keep the colon line
 * and the line above it.
 *
 * make sure the "out" file is not lost if the system crashes
 * by writing into another file (tmpout) and linking "out" to it
 * in a careful way.  the file "tmpout" is created same directory as out.
 *
 * also be sure that you don't change the mode of the out file.
 */

filter(in, out)
char *in, *out;
{
	int i;
	struct stat statbuf;
	FILE *inf, *tmpoutf;
	char *p;
	char tmpout[40];
	char line[2][LINESIZE];

	maketmpout(out, tmpout);
	if ((inf = fopen(in, "r")) == NULL)
		error("cannot open %s\n", in);
	if ((tmpoutf = fopen(tmpout, "w")) == NULL)
		error("cannot open %s\n", tmpout);

	i = 0;
	while (fgets(line[i], LINESIZE, inf) != NULL) {
		if (*line[i] != COLON)
		        fputs(line[i], tmpoutf);
		else {
			p = line[i]+1;
			while (*p) {
				if (*p == 'd')
					putindict(line[!i], p-line[i]);
				++p;
			}
		}
		i = !i;
	}
	fclose(inf);
	fclose(tmpoutf);
	if (stat(out, &statbuf) < 0)
		error("cannot stat %s\n", out);
	if (chmod(tmpout, statbuf.st_mode) < 0)
		error("cannot chmod %s\n", tmpout);

	sync();
	unlink(out);
	if (link(tmpout, out) < 0)
		error("cannot link %s to %s\n", tmpout, out);
	sync();
	unlink(tmpout);
}

/*
 * make a filename in t from s.
 * take up to 12 chars from the last component of s and then append X's.
 * then call mktemp to make sure it is unique.
 */

maketmpout(s, t)
char *s, *t;
{
	int nc, nX;
	char *p, *q;

	p = sname(s);
	nc = min(12, strlen(p));
	nX = min(6, 14-nc);
	q = t;
	while (s < p)
		*q++ = *s++;
	while (nc--)
		*q++ = *p++;
	while (nX--)
		*q++ = 'X';
	*q = 0;
	mktemp(t);
}

getdict()
{
	if (!*dict)
		/*
		 * we must create a dictionary in the current directory.
		 */
		strcpy(dict, "dict");
	if ((df = fopen(dict, "a")) == NULL)
		error("cannot open %s\n", dict);
}

putindict(s, i)
char *s;
int i;
{
	int j, k;
	char *p, *q;

	/*
	 * we must find the word in s that is i characters
	 * from the beginning. first worry about tabs and
	 * double characters that would throw the count off.
	 *
	 * then worry about the definition of a word:
	 *    "a series of letters, digits, &, or -.
	 *     the & and - must be in the middle of the word."
	 */
	p = s;
	while (i) {
		if (*p == '\t')
			i -= 8;
		else if (coi && any(*p, doublechars))
			i -= 2;
		else
			i -= 1;
		++p;
	}
	if (!wordchar(*p)) {
		fprintf(stderr, "'d' placed below '%c'?\n", *p);
		return;
	}
	q = p+1;
        while (wordchar(*q))
                ++q;
	while (*(q-1) == '&' || *(q-1) == '-')
		--q;
        *q = 0;       /* we found the end of the word */
        --p;
        while (p >= s && wordchar(*p))
                --p;
	++p;
	while (*p == '&' || *p == '-')
		++p;  /* and we found the beginning */
	for (j = 1; j <= nmspwords; ++j) {
		k = mspword[j];
		if (*code[k] == 's' && strcmp(p, word[k]) == 0) {
			if (!df)
		                getdict();
			fputs(word[k], df);
			return;
		}
	}
	fprintf(stderr, "cannot find \"%s\" among selected words\n", p);
}

char *
strsave(s)
char *s;
{
	char *p;

	if ((p = malloc(strlen(s)+1)) == NULL)
		error("malloc unable to get space\n");
	strcpy(p, s);
	return(p);
}


char *
fgets(s, n, f)
char *s;
int n;
FILE *f;
{
        char *os;
        int c, i;

        os = s;
	i = 0;
        while (i <= n && (c = getc(f)) != '\n' && c != EOF) {
                *s++ = c;
                ++i;
        }
        if (i > n) {
		*s = 0;
		fprintf(stderr,
                        "mspell: no room for chars in fgets\n%s\n", os);
                exit(1);
        } else if (c == EOF)
                return(NULL);
        else {
                *s = 0;
                return(os);
        }
}

fputs(s, f)
char *s;
FILE *f;
{
	while (*s)
		putc(*s++, f);
	putc('\n', f);
}

error(arglist)
int *arglist;
{
        fprintf(stderr, "mspell: %r", &arglist);
        exit(1);
}
