{ SWIVEL: Simple World Interactive Viewing Enhanced Library
     (It's not really very enhanced, but the word fit in the
     acronym so it stands.  Everybody knows that programmers
     think up names first, and then write their programs
     around them.)
  Written by Ken Van Camp
  May 1988
  Released into the public domain by the author.

  This program will plot The World Digitized, the data for which
  is Copyright (C) John B. Alison.  I highly recommend that
  anyone who uses this program (and therefore Mr. Alison's data)
  send Mr. Alison the $20 donation he requests (see the READ.ME
  file for his address).
}

program SWIVEL;

const UP    = 242; DOWN = 250;  LEFT = 245;
      RIGHT = 247; ESC  = 27;   SPACE= 32;
      RET   = 13;
      BLACK = 0;   BLUE = 1;    GREEN = 2;
      CYAN = 3;    RED = 4;     MAGENTA = 5;
      BROWN = 6;   LTGREY = 7;  DKGREY = 8;
      LTBLUE = 9;  LTGRN = 10;  LTCYAN = 11;
      LTRED = 12;  LTMGNT = 13; YELLOW = 14;
      WHITE = 15;
      { SWIVEL commands: }
      VIEW = 1;
      COLOR = 2;
      RESCALE = 3;
      MERCATOR = 4;
      DIRECTORY = 5;
      ERASE = 6;
      ALL = 7;
      QUIT = 8;
      MAXCMMD = 8;       { total # of SWIVEL cmmds currently available }
const COMMANDS: array[1..MAXCMMD] of char = 'VCRMDEAQ';
type text80 = string[80];
var  xmin, xmax, ymin, ymax: real;      { plotting limits }
     c: integer;                        { user input char }
     p: integer;                        { parameter number }
     mercate: boolean;                  { use mercator projection? }
     col: integer;                      { user color number }
     directry: text80;                  { directory for file search }
     erase_cmmds: boolean;              { user want command prompt erased? }

{$I GX2DE.PAS}
{$I GXZOOM.PAS}
{$I GXGIN.PAS}

procedure initgraphic;
begin
  mercate := TRUE;
  xmin := -246;
  ymin := -4.833;
  xmax := 246;
  ymax := 4.833;
  col := RED;
  directry := '';
  erase_cmmds := FALSE;

  clipon2d;
  graphicsopen;
  zoomcolour (12);
  gxborderindex := WHITE;
  window (xmin, ymin, xmax, ymax);
  viewport (1, 1, 639, 300);
  lineindex (WHITE);
  border (WHITE);
end; { initgraphic }

procedure grferr (messg: text80);   { error in graphics mode }
begin
  graphicsclose;
  writeln (messg);
  halt(1);
end; { grferr }

procedure grid;         { Draw a grid at 20-degree increments }
var x, y, ydelta: real;
    deg90: real;
begin
  x := trunc(xmin/20.0) * 20.0;
  while (x <= xmax) do begin
    if (x < 0.1) and (x > -0.1) then
      lineindex (YELLOW)
    else
      lineindex (WHITE);
    clip2d (x, ymin, x, ymax);
    x := x + 20;
  end;
  if (mercate) then
    ydelta := 20.0 * 2.9 / 85.0
  else
    ydelta := 20.0;
  y := trunc(ymin/ydelta) * ydelta;
  while (y <= ymax) do begin
    if (y < 0.1) and (y > -0.1) then
      lineindex (YELLOW)
    else
      lineindex (WHITE);
    clip2d (xmin, y, xmax, y);
    y := y + ydelta;
  end;

  { Draw an outline around the usable space: +-180 degrees longitude,
    +-90 degrees latitude }
  lineindex (YELLOW);
  if (mercate) then
    deg90 := 90.0 * 2.9 / 85.0
  else
    deg90 := 90.0;
  clip2d (-180.0, -deg90, 180.0, -deg90);
  clip2d (180.0, -deg90, 180.0, deg90);
  clip2d (180.0, deg90, -180.0, deg90);
  clip2d (-180.0, deg90, -180.0, -deg90);
end; { grid }

{ in2real: Read 2 real numbers from a line, and ignore anything after it.
  Return TRUE if successful, or FALSE if the input line is blank.
}
function in2real (var filin: text; var a1, a2: real): boolean;
var Line: string[127];            { line of input }
    p1, p2: integer;              { positions of spaces within line }
    Retcode: integer;             { return code from function }

begin
  Line[1] := ' ';
  readln (Filin, Line);
  if (length(Line) = 0) or (Line[1] = ' ') then begin
    in2real := FALSE;
  end else begin
    Line := Line + ' ';
    p1 := pos (' ', Line);
    if (p1 = 0) then
      grferr ('Error 1 reading file');
    val (copy (Line, 1, p1-1), a1, Retcode);
    if (Retcode <> 0) then
      grferr ('Error 2 reading file');
    p2 := pos (' ', copy (Line, p1+1, 100)) + p1;
    if (p2 = 0) then
      grferr ('Error 3 reading file');
    val (copy (Line, p1+1, p2-p1-1), a2, Retcode);
    if (Retcode <> 0) then
      grferr ('Error 4 reading file');
    in2real := TRUE;
  end;
end; { function in2real }

function tan (angle: real): real;
begin
  tan := sin(angle) / cos(angle);
end;

{ mercat: compute the Mercator projection of the latitude }
function mercat (latitude: real): real;
begin
  if (abs (latitude-90) < 1) or (abs (latitude-270) < 1) then
    { too close to pole: don't project }
    mercat := latitude
  else
    mercat := ln (tan ((45.0 + 0.5 * latitude) * 0.01745));
end; { mercat }

procedure dispfile (filename: text80);    { Read & display a type-1 map file }
var filin: text;
    lat, long: real;               { latitude & longitude of pt. }
    lastlat, lastlong: real;       { last pt. }
    lastconn: boolean;             { is last pt connected to next one? }
    line: text80;                  { a line of input from text file }
begin
  assign (filin, filename);
  {$I-}
  reset (filin);
  {$I+}
  if (ioresult <> 0) then begin
    gotoxy (1,2);
    write ('File :', filename,
      ': does not exist.  Use D command to set directory');
    delay (3000);
  end else begin
    gotoxy (1,2);
    writeln('Plotting file ', filename,' ...                                ');
    lastconn := FALSE;
    lineindex (col);
    repeat
      if (in2real (filin, lat, long)) then begin
        { a non-blank line was read }
        if (mercate) then
          lat := mercat (lat);
        if (lastconn) then
          clip2d (lastlong, lastlat, long, lat)
        else
          lastconn := TRUE;
        lastlong := long;
        lastlat := lat;
      end else begin
        { blank line read: break vector connection }
        lastconn := FALSE;
      end;
    until (eof (filin));
    close (filin);
  end;
end; { dispfile }

{ fileprompt: Prompt the user for a file name }
procedure fileprompt;
begin
  if      (pos ('AFRICA', directry) <> 0) or
          (pos ('africa', directry) <> 0) then
    write ('File (AF0,AF1,AF2, or AF3): ')
  else if (pos ('ANTARCTI', directry) <> 0) or
          (pos ('antarcti', directry) <> 0) then
    write ('File (AN0 or AN1): ')
  else if (pos ('ASIA', directry) <> 0) or
          (pos ('asia', directry) <> 0) then
    write ('File (AS0,AS1,AS2, or AS3): ')
  else if (pos ('AUSTRALI', directry) <> 0) or
          (pos ('australi', directry) <> 0) then
    write ('File (AU0,AU1, or AU2): ')
  else if (pos ('EUROPE', directry) <> 0) or
          (pos ('europe', directry) <> 0) then
    write ('File (E0,E1,E2 or E3): ')
  else if (pos ('NORTHAME', directry) <> 0) or
          (pos ('northame', directry) <> 0) then
    write ('File (NA0,NA1,NA2,NA3,USA0,USA1,GR0,GR1, or PA1): ')
  else if (pos ('SOUTHAME', directry) <> 0) or
          (pos ('southame', directry) <> 0) then
    write ('File (SA0,SA1,SA2, or SA3): ')
  else
    write ('Enter file code (e.g., NA0): ');
end; { fileprompt }

{ checkkey: return TRUE if Escape key pressed, FALSE if no key or any other
  key pressed.
}
function checkkey: boolean;
var c: integer;
begin
  if (keypressed) then begin
    c := getch;
    if (c = ESC) then
      checkkey := TRUE
    else
      checkkey := FALSE;
  end else
    checkkey := FALSE;
end; { checkkey }

{ all_file_view: Plot all files in succession, checking between each file
  for the Escape key which aborts it
}
procedure all_file_view;
var ctrlfil: text;
    filecol: integer;              { color to plot this file }
    filename: text80;              { file name }
begin
  assign (ctrlfil, 'ALLFILES.DAT');
  {$I-}
  reset (ctrlfil);
  {$I+}
  if (ioresult <> 0) then begin
    write ('ERROR: File ALLFILES.DAT does not exist.');
    delay (3000);
  end else begin
    gotoxy (1,1);
    writeln ('To exit after current file, press Escape.                     ',
             '        ');
    repeat
      readln (ctrlfil, filecol, filename);
      if (filecol <> 0) then begin
        { strip the leading blank off the file name }
        filename := copy (filename, 2, 60);
        col := filecol;
        dispfile (filename);
        if (checkkey) then begin
          write ('STOP THE WORLD, I WANNA GET OFF!');
          delay (3000);
          filecol := 0;
        end;
      end;
    until (filecol = 0) or (eof (ctrlfil));
    close (ctrlfil);
  end;
end; { all_file_view }

{ cmmd_eval: Evaluate interactive commands }
procedure cmmd_eval;
var c: integer;
    cmmd: integer;
    filename: text80;
    oxmin, oxmax, oymin, oymax: real;      { temps for plot limits }
    xmean, ymean: real;                    { center screen coords }
begin
  repeat
    gotoxy (1,1);
    writeln ('                                                              ',
             '          ');
    writeln ('                                                              ',
             '          ');
    write   ('                                                              ',
             '          ');
    gotoxy (1,1);
    if (NOT erase_cmmds) then
      write ('Command (All/View/Color/Rescale/Mercator/Dir/Erase/Quit): ');
    repeat
      c := getch;
      cmmd := pos (upcase(chr(c)), COMMANDS);
    until (cmmd <> 0);
    writeln (chr(c));
    case cmmd of
      ALL:
        all_file_view;
      VIEW: begin
        fileprompt;
        readln (filename);
        dispfile (concat (directry, '\', filename, '.MP1'));
      end;
      COLOR: begin
        gotoxy (1,1);
        writeln ('1=Blue,2=Green,3=Cyan,4=Red,5=Magenta,6=Brown,7=LtGrey,',
                 '8=DkGrey');
        writeln ('9=LtBlue,10=LtGrn,11=LtCyan,12=LtRed,13=LtMagenta,',
                 '14=Yellow,15=White');
        write ('Enter color number (1-15): ');
        readln (col);
      end;
      RESCALE: begin
        if (mercate) then
          writeln ('Old values are ', xmin:5:2,' ',(ymin*85.0/2.9):5:2,' ',
            xmax:5:2,' ',(ymax*85.0/2.9):5:2)
        else
          writeln ('Old values are ',xmin:5:2,' ',ymin:5:2,' ',xmax:5:2,' ',
            ymax:5:2);
        write ('Enter degrees W-long, S-lat, E-long, N-lat: ');
        oxmin := xmin;
        oxmax := xmax;
        oymin := ymin;
        oymax := ymax;
        readln (xmin, ymin, xmax, ymax);
        if (xmin >= xmax) or (ymin >= ymax) then begin
          write ('ILLEGAL SCALE');
          delay (5000);
          xmin := oxmin;
          xmax := oxmax;
          ymin := oymin;
          ymax := oymax;
        end else begin
          { keep the latitude-longitude scaling true for typical screen }
          if ((xmax-xmin) / (ymax-ymin) > (246.0/141.66)) then begin
            { X scale larger than Y: increase Y }
            ymean := (ymin + ymax) / 2.0;
            ymin := ymean - (xmax - xmin) * 0.5 * 141.66 / 246.0;
            ymax := ymean + (xmax - xmin) * 0.5 * 141.66 / 246.0;
          end else begin
            { Y scale larger than X: increase X }
            xmean := (xmin + xmax) / 2.0;
            xmin := xmean - (ymax - ymin) * 0.5 * 246.0 / 141.66;
            xmax := xmean + (ymax - ymin) * 0.5 * 246.0 / 141.66;
          end;
          if (mercate) then begin
            { scale Y plotting limits to Mercator displacement units }
            ymin := ymin * 2.9 / 85.0;
            ymax := ymax * 2.9 / 85.0;
          end;
          window (xmin, ymin, xmax, ymax);
          graphics (0, col);     { clear the screen }
          grid;
        end; { if xmin >= xmax ... }
      end;
      MERCATOR: begin
        if (mercate) then begin
          writeln ('MERCATOR option is now OFF.');
          mercate := FALSE;
          { scale Y plotting limits to degrees }
          ymin := ymin * 85.0 / 2.9;
          ymax := ymax * 85.0 / 2.9;
        end else begin
          writeln ('MERCATOR option is now ON.');
          mercate := TRUE;
          { scale Y plotting limits to Mercator displacement units }
          ymin := ymin * 2.9 / 85.0;
          ymax := ymax * 2.9 / 85.0;
        end;
        window (xmin, ymin, xmax, ymax);
        delay (3000);
      end;
      DIRECTORY: begin
        writeln ('Select AFRICA, ANTARCTI, ASIA, AUSTRALI, EUROPE, NORTHAME,',
          'or SOUTHAME.');
        write ('Enter default directory: ');
        readln (directry);
      end;
      ERASE:
        erase_cmmds := NOT erase_cmmds;
    end; { case }
  until (cmmd = QUIT);
end; { cmmd_eval }

begin { main }
  if (paramcount > 0) then begin
    writeln ('  SWIVEL: Simple World Interactive Viewing Enhanced Library');
    writeln ('Version 1.0 by Ken Van Camp  May 1988');
    writeln ('SWIVEL is in the public domain, and may not be distributed');
    writeln ('for profit.');
    writeln ('  usage: SWIVEL');
    writeln ('Any parameters brings up this help screen.');
    writeln ('Available commands from within SWIVEL are:');
    writeln ('  A - View all files (specified in ALLFILES.DAT)');
    writeln ('  V - Specify another file to view');
    writeln ('  C - Specify new color # to use for plotting');
    writeln ('  R - Rescale the plot to new limits (specify in degrees)');
    writeln ('  M - Toggle use of Mercator projection (default is ON)');
    writeln ('  D - Change directory to search for file specified with V');
    writeln ('  E - Erase the command prompt from screen (for printing)');
    writeln ('  Q - Quit from SWIVEL');
    writeln ('NOTES:');
    writeln ('To specify a file to read, first set the default directory for');
    writeln ('searching with the D command.  The use V to specify the file ',
             'name');
    writeln ('Do NOT include the .MP1 file name extension.');
    writeln ('    When rescaling, just separate the four numbers by a space');
    writeln ('(no commas allowed).  Grid lines are always drawn in 20-degree');
    write   ('increments.');
    halt(1);
  end;
  initgraphic;
  grid;
  gotoxy (15,1);
  write ('SWIVEL: Simple World Interactive Viewing Enhanced Library');
  delay (2000);
  gotoxy (15,1);
  write ('        Version 1.0  by Ken Van Camp   (May 1988)        ');
  delay (2000);
  gotoxy (15,1);
  write ('            LET YOUR FINGERS DO THE WALKING!             ');
  delay (2000);
  cmmd_eval;
  graphicsclose;
end. {main}
