#define Version "DOSCVT V2.7"

/*

  doscvt [-d | -u ] [-v] [-z] [filename ...]

  -d    convert to MSDOS format.
  -u    convert to UNIX.
  -t	perform character set translation.
  -v    display the program version id.
  -z    don't append a ^Z to the end of a DOS file

  defaults to -d on MSDOS systems and -u on UNIX systems.

  Convert MSDOS text file format to UNIX text file format
  Namely, convert <CR><LF> pairs to just <LF> and remove the
  trailing ^Z EOF marker.
  Also converts Unix format to MS-DOS format by adding <CR>
  before <LF> and, optionally, adding ^Z at the end.
  Works by converting the specified file into a temp file and,
  when finished, copying it back.
  The accessed and modified times of the original are preserved.
  Multiple filenames (but no wildcards) may be specified and each
  will be converted in turn.  If no filename is specified, stdin
  will be converted and written on stdout (for use as a filter).

  Author:
  Norm Brake
  N.E.B. Computer Services

  Internet:  norm@modcomp.com
  Snailmail: 6521 N.W. 1 St.
             Margate, FL 33063-5101

  Modifications by Jari Kokko <jkokko@snakemail.hut.fi>:

  1) Fixed the tempfile stuff, now you can "doscvt a:\blah.txt"
     from drive c: on MSDOS.

  2) Added character translation option -t, now also ISO-Latin.

  3) Changed adding the ^Z to DOS files to being the default.

  4) Added filename globbing (handles wildcards on MSDOS) - on UNIX
     this is done by your shell.  This code is for Turbo C.
     With Microsoft C, this is handled at program startup before the
     user code gets control via the optional "setargv.o" which is
     linked in from /lib.
     
*/

#include <stdio.h>
#ifdef STDLIB
#include <stdlib.h>
#endif

/* include the stuff to preserve the time stamp. This stuff is not ANSI,
 * but I'm not proud */

#include <sys/types.h>
#include <sys/stat.h>
#ifdef MSC
#include <sys/utime.h>
#else
#include <utime.h>
#endif

#ifdef __MSDOS__                /* for Turbo C */
#define MSDOS
#endif

#ifdef MSDOS
#define SUCCESS (1)             /* DOS success and failure exit codes */
#define FAILURE (0)

/* include the stuff we need under DOS to change stdin and stdout
 * to binary mode */
#include <fcntl.h>
#include <io.h>

#else
#define SUCCESS (0)             /* UNIX success and failure exit codes */
#define FAILURE (1)

#endif

#define EPUTCHAR(c, f, n) if (fputc(c, f) == EOF) file_error(n);

struct stat hstat;
struct utimbuf utb;

FILE *f;                        /* file to be converted */
FILE *t;                        /* temp file */

int fn = 1;                     /* argv entry currently being processed */
char ts[255];                   /* temp string for errors and system calls */
char *tn;                       /* temp file name, gets results of tmpnam() */

typedef enum format {
    dos, unx
} format_t;

#ifdef MSDOS
/* under dos, we want the default to convert unix to dos */
format_t fmt = dos;

#else
/* under unix, we want the default to convert dos to unix */
format_t fmt = unx;

#endif

int vflag = 0;                  /* if true, -v specified on command line */
int zflag = 1;                  /* if true, append a ^Z to the file */
int tflag = 0;                  /* if true, do character translation */

char *cmd;                      /* pointer to argv[0] */
char *filename;                 /* name of file being processed */

/* display usage instructions */
void usage(int vflag)
{
    fprintf(stderr, "%s\n", Version);
    if (!vflag) {
        fprintf(stderr, "usage: %s [-u | -d ] [-v] [-z] [-t] [ <filename> ]\n", cmd);
        fprintf(stderr, "        -u   :  convert <filename> to unix format\n");
        fprintf(stderr, "        -d   :  convert <filename> to DOS format\n");
        fprintf(stderr, "        -v   :  display the program version and exit\n");
        fprintf(stderr, "        -z   :  don't append a CTRL-Z\n");
        fprintf(stderr, "        -t   :  do character translation\n");
        exit(FAILURE);
    } else {
        exit(SUCCESS);
    }
}

/* issue a FATAL warning concerning a filename */
void file_error(char *fname)
{
    sprintf(ts, "%s - '%s'", cmd, filename);
    perror(ts);
    exit(FAILURE);
}

/* issue a FATAL warning message */
void fatal(char *msg)
{
    fprintf(stderr, "%s: error: '%s' - %s\n", cmd, filename, msg);
    remove(tn);
    exit(FAILURE);
}

/* Issue a non-fatal warning message */
void warning(char *msg)
{
    fprintf(stderr, "%s: warning: '%s' - %s\n", cmd, filename, msg);
}

/* translate between character sets */
int trchar(int c)
{

    if (fmt == unx) {
        switch (c) {

            /* MSDOS cp 437 */
        case 134:           c = '\175';            break;
        case 132:           c = '\173';            break;
        case 148:           c = '\174';            break;
        case 142:           c = '\133';            break;
        case 143:           c = '\135';            break;
        case 153:           c = '\134';            break;

            /* ANSI (mswin) */
        case 229:           c = '\175';            break;
        case 228:           c = '\173';            break;
        case 246:           c = '\174';            break;
        case 197:           c = '\133';            break;
        case 196:           c = '\135';            break;
        case 214:           c = '\134';            break;

	   /* ISO-Latin */
        case '\344':           c = '\175';            break;
        case '\345':           c = '\173';            break;
        case '\366':           c = '\174';            break;
        case '\304':           c = '\133';            break;
        case '\305':           c = '\135';            break;
        case '\326':           c = '\134';            break;

        default:            break;

        }

    } else if (fmt == dos) {
        switch (c) {

            /* UNIX (local?) convension */
        case 125:           c = '\206';            break;
        case 123:           c = '\204';            break;
        case 124:           c = '\224';            break;
        case  93:           c = '\217';            break;
        case  91:           c = '\216';            break;
        case  92:           c = '\231';            break;

           /* ANSI (mswin) */
        case 229:           c = '\206';            break;
        case 228:           c = '\204';            break;
        case 246:           c = '\224';            break;
        case 197:           c = '\217';            break;
        case 196:           c = '\216';            break;
        case 214:           c = '\231';            break;

	   /* ISO-Latin */
        case '\344':           c = '\206';            break;
        case '\345':           c = '\204';            break;
        case '\366':           c = '\224';            break;
        case '\304':           c = '\217';            break;
        case '\305':           c = '\216';            break;
        case '\326':           c = '\231';            break;

        default:            break;
        }
    }
    return c;
}

#ifdef MSDOS
int glob(int *argc, char **argv[], int attr);

#endif

/* MAIN Program */
int main(int argc, char *argv[])
{
    int c1, c2;

#ifdef MSDOS
    cmd = "doscvt";             /* program name for reporting errors */

#ifndef MSC
    /* change to proper filenames */
    if (!glob(&argc, &argv, 0)) {       
        fprintf(stderr, "%s: unable to parse command line.\n", cmd);
        return -1;
    }
#endif
#else
    cmd = argv[0];              /* program name for reporting errors */
#endif

    tn = tempnam(".", "dc");    /* temp file for conversion */

    if (argc > 1) {             /* if we had some command args */
      while (argv[fn][0] == '-') {    /* process any option switches */
	switch (c1 = argv[fn][1]) {
	case 'd':                fmt = dos;                break;
	case 'u':                fmt = unx;                break;
	case 'z':                zflag = 0;                break;
	case 't':                tflag = 1;                break;
	case 'v':                vflag = 1;
	default:                 usage(vflag);
	}
	fn++;
	argc--;
	if (fn > argc)
	  break;
      }
    }
    /* We have processed all of the option args, now do the files */
    do {
        if (argc > 1) {                         /* if a file was specified */
            filename = argv[fn];                /* remember the file name  */
            if (stat(filename, &hstat) != 0)    /* get the file modified time */
                file_error(filename);           /* if stat failed, it won't read */
            utb.actime = hstat.st_atime;        /* copy the timestamp */
            utb.modtime = hstat.st_mtime;       /* both accessed and modified */
#if defined(__m88k__) && defined(realix)
            utb.acusec = hstat.st_ausec;        /* if this is realix and moto
                                                 * 88k, */
            utb.modusec = hstat.st_musec;       /* time does down to micro
                                                 * seconds */
#endif
            if ((f = freopen(filename, "rb", stdin)) == NULL)   /* open in as stdin */
                file_error(filename);
            if ((f = freopen(tn, "wb", stdout)) == NULL)        /* open tn as stdout */
                file_error(tn);
        }
#ifdef MSDOS
        else {
            /* Under MSDOS, stdin and stdout are text mode files by default.
             * If there were no files on the command files, we are filtering
             * stdin to stdout.  Under MSDOS, we need to change their modes
             * to binary for the following algorithm to work. The code to do
             * this is MSDOS (maybe MicroSoft C) specific.     */

            if (setmode(fileno(stdin), O_BINARY) == -1)
                file_error("<STDIN>");
            if (setmode(fileno(stdout), O_BINARY) == -1)
                file_error("<STDOUT>");
        }
#endif

        /* copy and convert loop: stop on end-of-file or a CTRL-Z (if
         * converting to UNIX). If we see a CR and the next char is LF and we
         * are converting to UNIX, delete the CR. If we see a LF and
         * converting to DOS, add the CR. Check errors on the output since if
         * we got a write error, we REALLY do not want to replace the input
         * file with the output! */

        while ((c1 = getchar()) != EOF) {
            if ((c1 == '\032') && (fmt == unx))
                break;
            if (c1 == '\r') {
                if ((c2 = getchar()) == '\n') {
                    if (fmt == unx) {
                        EPUTCHAR(c2, stdout, tn);
                        continue;
                    } else if (fmt == dos) {
                        EPUTCHAR(c1, stdout, tn);
                        EPUTCHAR(c2, stdout, tn);
                        continue;
                    }
                } else
                    ungetc(c2, stdin);
            } else if (c1 == '\n') {
                if (fmt == dos)
                    EPUTCHAR('\r', stdout, tn);
            }
            if (tflag)
                c1 = trchar(c1);/* call char translator */
            EPUTCHAR(c1, stdout, tn);
        }

        /* if converting to DOS and the user wants it, put out a CTRL-Z */
        if ((fmt == dos) && (zflag))
            EPUTCHAR('\032', stdout, tn);

        /* if we opened a file name, copy the tempfile over the original file,
         * and set it's timestamp to match that of the original file. */
        if (argc > 1) {
            
            fclose(stdin), fclose(stdout);

            if ((f = freopen(tn, "rb", stdin)) == NULL)
                file_error(tn);
            if ((f = freopen(filename, "wb", stdout)) == NULL)
                file_error(filename);

            /* copy the whole file back */
            while ((c1 = getchar()) != EOF)
                EPUTCHAR(c1, stdout, filename);

            fclose(stdin), fclose(stdout);

            remove(tn);

            if (utime(filename, &utb) != 0)
                warning("unable to copy file's timestamp");
        }
        fn++;                   /* bump the argv index */
        argc--;                 /* count down until no more files to convert */
    } while (argc > 1);

    return SUCCESS;
}

#ifdef MSDOS
#ifndef MSC
/* ACKNOWLEDGEMENTS:
 *
 * Version 1.0 - Jeff Dragovich (09/21/92). This routine is modeled after, and
 * very similar to GLOB.C, v1.06, written by Bob Kamins (11/28/87). Ported to
 * Borlandc by Jari Kokko, Sun Aug 08 16:15:52 1993. Also, I fixed a bug
 * in zglob.c of not adding the full path to globbered args. E.g.
 * "c:\junk\*.doc" expanded into "FOO.DOC BLAH.DOC ..." and not into
 * "c:\junk\FOO.DOC c:\junk\BLAH.DOC ..." as is nicer. */

#include <dir.h>
#include <string.h>

static int insert(char *);
static int expand(char *, int);
static char memerr[] = "memory allocation error: %s\n";

typedef struct node Node;       /* linked list node */
struct node {
    char *s;
    Node *pn;
};

static Node *phead;             /* pointer to list head */
static Node *ptail;             /* pointer to list tail */
static int newc;                /* new number of arguments */

int glob(int *argc, char **argv[], int attr)
{
    int i;
    char **pnew;
    Node *pcurr, *pnext;

    newc = 0;
    phead = NULL;

    for (i = 0; i < *argc; i++) {       /* expand all arguments */
        if (!expand((*argv)[i], attr))
            return (0);
    }

    pnew = (char **) malloc(newc * sizeof(char *));     /* new argv */

    if (pnew == NULL) {
        fprintf(stderr, memerr, "pnew");
        return (0);
    }
    for (i = 0, pcurr = phead; i < newc; i++) { /* write over to new argv */
        pnew[i] = pcurr->s;
        pnext = pcurr->pn;
        free(pcurr);            /* free up node */
        pcurr = pnext;
    }

    *argv = pnew;

    *argc = newc;
    return (1);
}

static int expand(char *argv, int attr)
{
    int ret, nomatch, flags;
    struct ffblk pfind;
    char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILE], ext[MAXEXT], pathname[MAXPATH];

    if (strpbrk(argv, "*?") == NULL)    /* is there a wildcard */
        return (insert(argv));  /* no, but add to list */

    nomatch = findfirst(argv, &pfind, attr);    /* attempt to expand */
    if (nomatch) {              /* must be a "no match" */
        ret = insert(argv);     /* put wildcarded name in list */
        if (!ret)
            return (ret);       /* must be a malloc error in insert() */
    }
    while (!nomatch) {          /* loop until no more matches */

        pathname[0] = '\0';

        flags = fnsplit(argv, drive, dir, fname, ext);

        if (flags & DRIVE)
            strcat(pathname, drive);
        if (flags & DIRECTORY)
            strcat(pathname, dir);
        strcat(pathname, pfind.ff_name);

        ret = insert(pathname); /* put expanded name in list */
        if (!ret)
            return (ret);       /* must be a malloc error in insert() */
        nomatch = findnext(&pfind);     /* get next match */
    }
    return (1);                 /* everything ok */
}

static int insert(char *str)
{
    char *p;
    Node *nptr;

    p = (char *) malloc(strlen(str) + 1);       /* new argv[i] */
    nptr = (Node *) malloc(sizeof(Node));       /* get node */
    if ((p == NULL) || (nptr == NULL)) {        /* no memory */
        fprintf(stderr, memerr, "insert");
        return (0);
    }
    strcpy(p, str);             /* copy over to heap */
    newc++;                     /* another argument */

    /* update linked list */
    if (phead == NULL)
        phead = nptr;
    else
        ptail->pn = nptr;

    ptail = nptr;
    ptail->pn = NULL;
    ptail->s = p;
    return (1);
}
# endif /* MSC */
#endif /* MSDOS */

