/*-------------------------------------------------------------------------*/
/* Program:    CC      .C                                                  */
/* Purpose:    Processes ASA carriage control characters in a file.        */
/* Notes:      Compiles under Borland C++, v3.1. Should work on any        */
/*                machine running DOS, v2.xx or higher.                    */
/* Status:     Source released into the public domain. If you find this    */
/*                program useful, I'd appreciate a postcard.               */
/* Updates:    02-Mar-89, GAT                                              */
/*                - initial version.                                       */
/*             02-May-89, GAT                                              */
/*                - eliminated some start-up code.                         */
/*             03-Dec-89, GAT                                              */
/*                - Improved efficiency of help message display.           */
/*                - Renamed ErrMsg() to write_ErrMsg().                    */
/*                - Now write_ErrMsg adds a period and CR/LF combination.  */
/*             03-Jan-90, GAT                                              */
/*                - Rewrote principal functions.                           */
/*                - Replaced most #defines with enumerations.              */
/*                - Used AT&T's getopt() for program args.                 */
/*                - Separated usage message into its own function.         */
/*                - Tweaked variable storage.                              */
/*                - Eliminated -t option for processing tab characters.    */
/*                - Added headers in case of multiple files.               */
/*             02-Aug-90, GAT                                              */
/*                - Added a CR to output stream before each formfeed.      */
/*                - Used return in main() rather than exit().              */
/*             17-Feb-91, GAT                                              */
/*                - Added copyright message.                               */
/*                - Made use of new TifaWARE library functions.            */
/*             26-Jan-92, GAT, v2.1c                                       */
/*                - A carriage return instead of a linefeed now is output  */
/*                   before a formfeed.                                    */
/*             05-Jul-93, GAT, v2.1d                                       */
/*                - compiled with BCC 3.1.                                 */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/* Author:     George A. Theall                                            */
/* SnailMail:  TifaWARE                                                    */
/*             610 South 48th St                                           */
/*             Philadelphia, PA.  19143                                    */
/*             U.S.A.                                                      */
/* E-Mail:     george@tifaware.com                                         */
/*             theall@popmail.tju.edu                                      */
/*             theall@mcneil.sas.upenn.edu                                 */
/*             george.theall@satalink.com                                  */
/*-------------------------------------------------------------------------*/

/* Useful type definitions. */
typedef int BOOLEAN;

typedef enum {                               /* error classes           */
   err_open,                                 /*    can't open a file    */
   err_read,                                 /*    can't read from file */
   err_write                                 /*    can't write to file  */
} ERR_TYPE;

typedef enum {                               /* return codes */
   rc_ok    = 0,
   rc_help  = 1,
   rc_open  = 10,
   rc_read  = 15,
   rc_write = 20
} RC_TYPE;

#define FALSE        0
#define TRUE         1

#include <assert.h>
#include <ctype.h>                           /* for isdigit() */
#include <dir.h>                             /* for fnsplit(), MAXFILE, etc */
#include <stdarg.h>                          /* for va_arg, etc.. */
#include <stdio.h>
#include <stdlib.h>                          /* for exit() */
#include <string.h>                          /* for strlwr() */
#include "tifa.h"                            /* TifaWARE library routines */

char ProgName[MAXFILE];                      /* space for filename */
char *ifn, *ofn;                             /* input/output file names */
static BOOLEAN                               /* flags for various options */
   hFlag = FALSE;                            /*    needs help?            */

/* Define the program's error messages. */
/*
 * NB: getopt() itself is responsible for generating the following
 * error messages, which do not appear in the structure below:
 *    ": illegal option -- %c"
 *    ": option requires an argument -- %c"
 */
const static struct {
   ERR_TYPE Type;
   char *Msg;
} Error[] ={
   {err_open,  ": can't open %s"},
   {err_read,  ": can't read from %s; processing halted at line %lu"},
   {err_write, ": can't write to %s; processing halted at line %lu of %s"}
};

void _setenvp(void) {};                      /* drop some start-up code */


/*---  main  --------------------------------------------------------------+
|  Purpose:    Main body of program.                                       |
|  Notes:      none                                                        |
|  Entry:      argc = argument count,                                      |
|              argv = array of argument variables.                         |
|  Exit:       Return code as enumerated by RC_TYPE.                       |
+-------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
   char ch;
   BOOLEAN hdrs;                             /* need headers? */
   RC_TYPE rc = rc_ok;                       /* program return code */
   FILE *ifp, *ofp;                          /* input/output file pointers */
   void write_usage(void);
   void cc(FILE *, FILE *);

   /*
    * Isolate program name to keep error messages neat. This kludge
    * is necessary for DOS v2.xx when argv[0] is NULL.
    */
   fnsplit((*argv == NULL) ? __FILE__ : *argv,  /* TURBO C extension! */   
      NULL, NULL, ProgName, NULL);
   *argv = strlwr(ProgName);                    /* TURBO C extension! */

   /* All options must appear before any filenames. */
   while ((ch = getopt(argc, argv, "?")) != EOF)
      switch (ch)
      {
         case '?': hFlag = TRUE; break;      /* help needed or requested */
         default:  ; /* EMPTY */             /* Unreached */
      }
   do
   {
      --argc;
      ++argv;
   } while (--optind);                       /* nb: optind >= 1 in getopt() */

   if (hFlag == TRUE)
   {
      write_usage();
      exit(rc_help);
   }

   /* Loop thru each file named on commandline. */
   assert(argc >= 0);
   ofp = stdout;                             /* might be changed later */
   ofn = "stdout";
   if (argc == 0)
   {
      ifn = "stdin";
      cc(stdin, ofp);
   }
   else
   {
      hdrs = (argc > 1) ? TRUE : FALSE;      /* headers needed if > 1 file */
      do
      {
         ifn = *argv;
         if ((ifp = fopen(ifn, "r")) == NULL)
         {
            write_errmsg(Error[err_open].Msg, ifn);
            rc = rc_open;
            continue;                        /* skip file; don't abort */
         }
         if (hdrs == TRUE)
            fprintf(ofp, "==> %s <==\n", ifn);
         cc(ifp, ofp);
         fclose(ifp);
      } while (*++argv);
   }
   return rc;
}

/*---  write_usage  -------------------------------------------------------+
|  Purpose:    Provides a brief message about program usage.               |
|  Notes:      none                                                        |
|  Entry:      n/a                                                         |
|  Exit:       n/a                                                         |
+-------------------------------------------------------------------------*/
void write_usage(void)
{
   fprintf(stderr, 
      "TifaWARE CC, v" VERS_STR
         ", processes ASA carriage control characters.\n"
      "Copyright (c) 1991-1993 by TifaWARE/George A. Theall.\n"
      "\n"
      "usage:  %s [options] [file(s)]\n"
      "\n"
      "Options:\n"
      "  -? = provide this help message\n"
      "\n"
      "Stdin is used if no files are specified. Wildcards are not allowed.\n",
      ProgName);
}


/*---  cc  ----------------------------------------------------------------+
|  Purpose:    Processes carriage controls in a given file.                |
|  Notes:      An error here will cause program to abort via exit().       |
|              Global variables ifn, ofn are used but not changed.         |
|  Entry:      ifp = pointer to input file,                                |
|              ofp = pointer to output file.                               |
|  Exit:       n/a                                                         |
+-------------------------------------------------------------------------*/
void cc(FILE *ifp, FILE *ofp)
{
   int ch;
   unsigned long line = 0L;                  /* up to ~4 billion lines! */
   BOOLEAN BadCC;                            /* an invalid CC? */

   /*
    * Process a line at a time. Read each character separately to avoid
    * reserving space for a line and arbitary limits on line length.
    */
   do
   {
      /*
       * As described in IBM's _CMS Command and Macro Reference_, p. 568, 
       * ASA carriage control characters have the following meanings:
       * 
       *    space   advance 1 line before printing,
       *    0       advance 2 lines before printing,
       *    -       advance 3 lines before printing,
       *    +       surpress line advance before printing,
       *    1       advance to new page.
       *
       * An invalid control character causes the line advance to be
       * surpressed before printing but added afterwards. 
       */
      /*
       * NB: an empty line is treated as if it had a sole blank. This
       * behaviour is not spelled out anywhere, but then again records
       * in files under CMS must contain at least 1 character.
       */
      ++line;
      BadCC = FALSE;
      switch (ch = fgetc(ifp))
      {
         case '-':  fputc('\n', ofp);           /* 3 line feeds */
                    /* FALL THROUGH */
         case '0':  fputc('\n', ofp);           /* 2 line feeds */
                    /* FALL THROUGH */
         case ' ':  fputc('\n', ofp); break;    /* 1 line feed */
         case '+':  fputc('\r', ofp); break;    /* carriage return */
         case '1':  fputc('\n', ofp);           /* top of form */
                    fputc('\f', ofp); break;
         case '\n': fputc('\n', ofp); continue; /* line is empty */
         case EOF:  continue;                   /* nothing more */
         default:                               /* invalid character */
            fputc('\r', ofp);
            BadCC = TRUE;
            break;
      }

      /* Print rest of line. NB: EOF = fgetc() if eof or error arises. */
      while ((ch = fgetc(ifp)) != '\n' && ch != EOF)
         fputc(ch, ofp);

      /* Handle any bad carriage controls. */
      if (BadCC == TRUE)
         fputc('\n', ofp);
   } while (ch != EOF && !ferror(ofp));      /* EOF => eof or error wrt ifp */

   /* Test for errors. */
   if (ferror(ifp))
   {
      write_errmsg(Error[err_read].Msg, ifn, line);
      exit(rc_read);
   }
   fputc('\n', ofp);                         /* for good measure */
   if (ferror(ofp))
   {
      write_errmsg(Error[err_write].Msg, ofn, line, ifn);
      exit(rc_write);
   }
}
