
(*************************************************

  Procedure: Corner right bottom with shadow.
    This procedure draws the right bottom corner
    of the frame on the screen.

*************************************************)

    Procedure Corner_Right_Bottom_Shadow( Var Frame: Frame_Type; Right, Bottom: Byte );
      Begin
        Corner_Right_Bottom( Frame, Right, Bottom );
        Make_Shadow( Bottom, Succ( Right ), 1 );
        Make_Shadow( Succ( Bottom ), Right, 2 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Corner right top.
    This procedure draws the right top corner of
    the frame on the screen.

*************************************************)

    Procedure Corner_Right_Top( Var Frame: Frame_Type; Right, Top: Byte );
      Begin
        Put_Character_On_Screen( Right, Top, Frame.Data[ 3 ], Frame.Attribute );
        Put_Character_On_Screen( Right, Succ( Top ), Frame.Data[ 6 ], Frame.Attribute );
        Put_Character_On_Screen( Pred( Right ), Top, Frame.Data[ 2 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Corner right top with shadow.
    This procedure draws the right top corner of
    the frame on the screen.

*************************************************)

    Procedure Corner_Right_Top_Shadow( Var Frame: Frame_Type; Right, Top: Byte );
      Begin
        Corner_Right_Top( Frame, Right, Top );
        Make_Shadow( Succ( Top ), Succ( Right ), 1 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw shadow window frame.
    This procedure draws a shadowed window frame
    on the screen at the given location.

*************************************************)

    Procedure Draw_Shadow_Window_Frame( Var Frame: Frame_Type; Left, Top, Right, Bottom: Byte );
      Var
        Hold,
        Count: Byte;
      Begin
        FillChar( Work_Area, SizeOf( Work_Area ), Frame.Attribute );
        If ( ( Top < Bottom ) and ( Left < Right ) )
          then
            Begin
              If ( Left <  Right )
                then
                  Begin
                    Hold := Succ( Right - Left );
                    Work_Area[ 1 ].Character := Frame.Data[ 1 ];
                    For Count := 2 to Pred( Hold ) do
                      Work_Area[ Count ].Character := Frame.Data[ 2 ];
                    Work_Area[ Hold ].Character := Frame.Data[ 3 ];
                   {$IFNDEF OS2}
                    Write_Data( Top, Left, Work_Area, Hold );
                   {$ELSE}
                    Write_Data( Top, Left, Addr( Work_Area ), Hold );
                   {$ENDIF}
                    Make_Shadow( Succ( Top ), Succ( Right ), 1 );
                    Work_Area[ 1 ].Character := Frame.Data[ 7 ];
                    For Count := 2 to Pred( Hold ) do
                      Work_Area[ Count ].Character := Frame.Data[ 8 ];
                    Work_Area[ Hold ].Character := Frame.Data[ 9 ];
                   {$IFNDEF OS2}
                    Write_Data( Bottom, Left, Work_Area, Hold );
                   {$ELSE}
                    Write_Data( Bottom, Left, Addr( Work_Area ), Hold );
                   {$ENDIF}
                    Make_Shadow( Succ( Bottom ), Succ( Left ), Hold );
                  End;
              If ( Top < Bottom )
                then
                  For Count := Succ( Top ) to Pred( Bottom ) do
                    Begin
                      Put_Character_On_Screen( Left, Count, Frame.Data[ 4 ], Frame.Attribute );
                      Put_Character_On_Screen( Right, Count, Frame.Data[ 6 ], Frame.Attribute );
                      Make_Shadow( Succ( Count ), Succ( Right ), 1 );
                    End;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set up expand.
    This procedure sets up the expand record to
    the given parameters for the window animation.
    It also draws the initial animation window.

*************************************************)

    Procedure Set_Up_Expand( Flag: Boolean; Left, Top, Right, Bottom, Current_Left, Current_Top, Current_Right,
                             Current_Bottom: Byte; Var Data: Expand_Data_Type; Var Frame: Frame_Type; Shadow: Boolean );
      Var
        Row,
        Column: Byte;
      Begin
        If Flag
          then
            For Row := Current_Top to Current_Bottom do
              For Column := Current_Left to Current_Right do
                Put_Character_On_Screen( Column, Row, Frame.Data[ 5 ], Frame.Attribute );
        If ( Left < Current_Left )
          then
            Dec( Current_Left );
        If ( Right > Current_Right )
          then
            Inc( Current_Right );
        If ( Top < Current_Top )
          then
            Dec( Current_Top );
        If ( Bottom > Current_Bottom )
          then
            Inc( Current_Bottom );
        Data.Expand_Left := ( Left < Current_Left );
        Data.Expand_Right := ( Right > Current_Right );
        Data.Expand_Up := ( Top < Current_Top );
        Data.Expand_Down := ( Bottom > Current_Bottom );
        Data.Current_Top := Current_Top;
        Data.Current_Bottom := Current_Bottom;
        Data.Current_Left := Current_Left;
        Data.Current_Right := Current_Right;
        Data.Top := Top;
        Data.Bottom:= Bottom;
        Data.Left := Left;
        Data.Right := Right;
        Data.Frame := Frame;
        If Shadow
          then
            Draw_Shadow_Window_Frame( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom )
          else
            Draw_Window_Frame( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set up.
    This procedure sets up the expand record for
    the window animation using the corresponding
    open method.

*************************************************)

    Procedure Set_Up( Var Frame: Frame_Type; Open_Method, Left, Top, Right, Bottom: Byte; Var Data: Expand_Data_Type;
                      Var Continue, Shadow: Boolean );
      Var
        Mid_1,
        Mid_2: Byte;
      Begin
        Mid_1 := ( ( Bottom + Top ) Div 2 );
        Mid_2 := ( ( Left + Right ) Div 2 );
        Case Open_Method of
          Window_Pop: Set_Up_Expand( False, Left, Top, Right, Bottom, Left, Top, Right, Bottom, Data, Frame, Shadow );
          Window_Down: Set_Up_Expand( True, Left, Top, Right, Bottom, Left, Top, Right, Top, Data, Frame, Shadow );
          Window_Up: Set_Up_Expand( True, Left, Top, Right, Bottom, Left, Bottom, Right, Bottom, Data, Frame, Shadow );
          Window_Right: Set_Up_Expand( True, Left, Top, Right, Bottom, Left, Top, Left, Bottom, Data, Frame, Shadow );
          Window_Left: Set_Up_Expand( True, Left, Top, Right, Bottom, Right, Top, Right, Bottom, Data, Frame, Shadow );
          Window_Vertical: Set_Up_Expand( True, Left, Top, Right, Bottom, Left, Mid_1, Right, Mid_1, Data, Frame, Shadow );
          Window_Horizontal: Set_Up_Expand( True, Left, Top, Right, Bottom, Mid_2, Top, Mid_2, Bottom, Data, Frame, Shadow );
          Window_Explode: Set_Up_Expand( True, Left, Top, Right, Bottom, Mid_2, Mid_1, Mid_2, Mid_1, Data, Frame, Shadow );
          Window_Down_Right: Set_Up_Expand( True, Left, Top, Right, Bottom, Left, Top, Left, Top, Data, Frame, Shadow );
          Window_Down_Left: Set_Up_Expand( True, Left, Top, Right, Bottom, Right, Top, Right, Top, Data, Frame, Shadow );
          Window_Up_Right: Set_Up_Expand( True, Left, Top, Right, Bottom, Left, Bottom, Left, Bottom, Data, Frame, Shadow );
          Window_Up_Left: Set_Up_Expand( True, Left, Top, Right, Bottom, Right, Bottom, Right, Bottom, Data, Frame, Shadow );
          Window_Horizontal_Right:
            Set_Up_Expand( True, Left, Top, Right, Bottom, Left, Mid_1, Left, Mid_1, Data, Frame, Shadow );
          Window_Horizontal_Left:
            Set_Up_Expand( True, Left, Top, Right, Bottom, Right, Mid_1, Right, Mid_1, Data, Frame, Shadow );
          Window_Vertical_Up:
            Set_Up_Expand( True, Left, Top, Right, Bottom, Mid_2, Bottom, Mid_2, Bottom, Data, Frame, Shadow );
          Window_Vertical_Down: Set_Up_Expand( True, Left, Top, Right, Bottom, Mid_2, Top, Mid_2, Top, Data, Frame, Shadow );
        End; {Case}
        Continue := ( Data.Expand_Left or Data.Expand_Right or Data.Expand_Up or Data.Expand_Down );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw moment window.
    This procedure draws the window at the given
    moment, then sets everything up for the next
    moment.

*************************************************)

    Procedure Draw_Moment_Window( Var Data: Expand_Data_Type; Var Continue: Boolean );
      Begin
        With Data do
          Begin
            If Expand_Left
              then
                Dec( Current_Left );
            If Expand_Right
              then
                Inc( Current_Right );
            If Expand_Up
              then
                Dec( Current_Top );
            If Expand_Down
              then
                Inc( Current_Bottom );
            If ( Expand_Left )
              then
                Draw_Left( Data.Frame, Current_Top, Current_Left, Current_Bottom );
            If ( Expand_Right )
              then
                Draw_Right( Data.Frame, Current_Top, Current_Right, Current_Bottom );
            If Expand_Up
              then
                Begin
                  Draw_Top( Data.Frame, Current_Left, Current_Top, Current_Right );
                  If ( Not Expand_Left )
                    then
                      Corner_Left_Top( Data.Frame, Current_Left, Current_Top );
                  If ( Not Expand_Right )
                    then
                      Corner_Right_Top( Data.Frame, Current_Right, Current_Top );
                End;
            If Expand_Down
              then
                Begin
                  Draw_Bottom( Data.Frame, Current_Left, Current_Bottom, Current_Right );
                  If ( Not Expand_Left )
                    then
                      Corner_Left_Bottom( Data.Frame, Current_Left, Current_Bottom );
                  If ( Not Expand_Right )
                    then
                      Corner_Right_Bottom( Data.Frame, Current_Right, Current_Bottom );
                End;
            If Expand_Up
              then
                Expand_Up := ( Top < Current_Top );
            If Expand_Left
              then
                Expand_Left := ( Left < Current_Left );
            If Expand_Right
              then
                Expand_Right := ( Right > Current_Right );
            If Expand_Down
              then
                Expand_Down := ( Bottom > Current_Bottom );
            Continue := ( Expand_Left or Expand_Right or Expand_Up or Expand_Down );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw moment window with shadow.
    This procedure draws the window at the given
    moment, then sets everything up for the next
    moment.

*************************************************)

    Procedure Draw_Moment_Window_Shadow( Var Data: Expand_Data_Type; Var Continue: Boolean );
      Begin
        With Data do
          Begin
            If Expand_Left
              then
                Dec( Current_Left );
            If Expand_Right
              then
                Inc( Current_Right );
            If Expand_Up
              then
                Dec( Current_Top );
            If Expand_Down
              then
                Inc( Current_Bottom );
            If ( Expand_Left )
              then
                Draw_Left_Shadow( Data.Frame, Current_Top, Current_Left, Current_Bottom );
            If ( Expand_Right )
              then
                Draw_Right_Shadow( Data.Frame, Current_Top, Current_Right, Current_Bottom );
            If Expand_Up
              then
                Begin
                  Draw_Top_Shadow( Data.Frame, Current_Left, Current_Top, Current_Right );
                  If ( Not Expand_Left )
                    then
                      Corner_Left_Top( Data.Frame, Current_Left, Current_Top );
                  If ( Not Expand_Right )
                    then
                      Corner_Right_Top_Shadow( Data.Frame, Current_Right, Current_Top );
                End;
            If Expand_Down
              then
                Begin
                  Draw_Bottom_Shadow( Data.Frame, Current_Left, Current_Bottom, Current_Right );
                  If ( Not Expand_Left )
                    then
                      Corner_Left_Bottom_Shadow( Data.Frame, Current_Left, Current_Bottom );
                  If ( Not Expand_Right )
                    then
                      Corner_Right_Bottom_Shadow( Data.Frame, Current_Right, Current_Bottom );
                End;
            If Expand_Up
              then
                Expand_Up := ( Top < Current_Top );
            If Expand_Left
              then
                Expand_Left := ( Left < Current_Left );
            If Expand_Right
              then
                Expand_Right := ( Right > Current_Right );
            If Expand_Down
              then
                Expand_Down := ( Bottom > Current_Bottom );
            Continue := ( Expand_Left or Expand_Right or Expand_Up or Expand_Down );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw window.
    This procedure animates the opening of a
    window on the screen.

*************************************************)

    Procedure Draw_Window( Var Frame: Frame_Type; Method, Left, Top, Right, Bottom: Byte; Shadow: Boolean );
      Var
        Continue: Boolean;
        Data: Expand_Data_Type;
        Count: Byte;
      Begin
        If ( ( ( Bottom - Top ) >= 3 ) and ( ( Right - Left ) >= 3 ) )
          then
            If Shadow
              then
                Begin
                  Set_Up( Frame, Method, Left, Top, Pred( Right ), Pred( Bottom ), Data, Continue, Shadow );
                  While Continue do
                    Begin
                      Delay( Delay_Amount );
                      Draw_Moment_Window_Shadow( Data, Continue );
                    End;
                End
              else
                Begin
                  Set_Up( Frame, Method, Left, Top, Right, Bottom, Data, Continue, Shadow );
                  While Continue do
                    Begin
                      Delay( Delay_Amount );
                      Draw_Moment_Window( Data, Continue );
                    End;
                End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw small corners.
    This procedure draws all four corners of a
    frame only on the screen.

*************************************************)

    Procedure Draw_Small_Corners( Var Frame: Frame_Type; Left, Top, Right, Bottom: Byte );
      Begin
        Put_Character_On_Screen( Left, Top, Frame.Data[ 1 ], Frame.Attribute );
        Put_Character_On_Screen( Right, Top, Frame.Data[ 3 ], Frame.Attribute );
        Put_Character_On_Screen( Left, Bottom, Frame.Data[ 7 ], Frame.Attribute );
        Put_Character_On_Screen( Right, Bottom, Frame.Data[ 9 ], Frame.Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore it.
    This procedure restores the original data on
    the screen using a specified animation manner.

*************************************************)

    Procedure Restore_It( Var Frame: Frame_Type; Left, Top, Right, Bottom: Byte;
                          Do_Left, Do_Top, Do_Right, Do_Bottom: Boolean );
      Var
        Current_Top,
        Current_Left,
        Current_Right,
        Current_Bottom: Byte;
      Begin
        Current_Left := Left;
        Current_Top := Top;
        Current_Right := Right;
        Current_Bottom := Bottom;
        If Current_Window^.Shadow
          then
            Begin
              Restore_Column( Right, Top, Bottom, Left );
              Restore_Row( Bottom, Left, Right, Top );
              Dec( Current_Bottom );
              Dec( Current_Right );
            End;
        While ( ( Current_Left <= Current_Right ) and ( Current_Top <= Current_Bottom ) ) do
          Begin
            Draw_Small_Corners( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom );
            Delay( Delay_Amount );
            If Do_Left
              then
                Begin
                  Restore_Column( Current_Left, Top, Current_Bottom, Left );
                  Inc( Current_Left );
                  Draw_Small_Left( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom );
                  Do_Left := ( Current_Left <= Current_Right );
                End;
            If Do_Top
              then
                Begin
                  Restore_Row( Current_Top, Left, Current_Right, Top );
                  Inc( Current_Top );
                  Draw_Small_Top( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom );
                  Do_Top := ( Current_Top <= Current_Bottom );
                End;
            If Do_Right
              then
                Begin
                  Restore_Column( Current_Right, Top, Current_Bottom, Left );
                  Dec( Current_Right );
                  Draw_Small_Right( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom );
                  Do_Right := ( Current_Right >= Current_Left );
                End;
            If Do_Bottom
              then
                Begin
                  Restore_Row( Current_Bottom, Left, Current_Right, Top );
                  Dec( Current_Bottom );
                  Draw_Small_Bottom( Frame, Current_Left, Current_Top, Current_Right, Current_Bottom );
                  Do_Bottom := ( Current_Bottom >= Current_Top );
                End;
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Quick restore.
    This procedure restores the original data on
    the screen without using animation.

*************************************************)

    Procedure Quick_Restore( Left, Top, Right, Bottom: Byte );
      Var
        Row: Byte;
        Count,
        Value: Word;
      Begin
        Value := ( Top + Bottom );
        For Row := Top to Succ( Value Div 2 ) do
          Begin
            Count := ( Value - Row );
            Restore_Row( Row, Left, Right, Top );
            Restore_Row( Count, Left, Right, Top );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Restore window.
    This procedure restores the window using the
    appropriate animation method.

*************************************************)

    Procedure Restore_Window( Var Frame: Frame_Type; Method, Top, Bottom, Left, Right: Byte );
      Begin
        If ( Current_Window <> Nil )
          then
            Begin
              Case Method of
                Window_Down: Restore_It( Frame, Left, Top, Right, Bottom, False, False, False, True );
                Window_Up: Restore_It( Frame, Left, Top, Right, Bottom, False, True, False, False );
                Window_Right: Restore_It( Frame, Left, Top, Right, Bottom, False, False, True, False );
                Window_Left: Restore_It( Frame, Left, Top, Right, Bottom, True, False, False, False );
                Window_Vertical: Restore_It( Frame, Left, Top, Right, Bottom, True, False, True, False );
                Window_Horizontal: Restore_It( Frame, Left, Top, Right, Bottom, False, True, False, True );
                Window_Explode: Restore_It( Frame, Left, Top, Right, Bottom, True, True, True, True );
                Window_Horizontal_Right: Restore_It( Frame, Left, Top, Right, Bottom, False, True, True, True );
                Window_Horizontal_Left: Restore_It( Frame, Left, Top, Right, Bottom, True, True, False, True );
                Window_Vertical_Up: Restore_It( Frame, Left, Top, Right, Bottom, True, True, True, False );
                Window_Vertical_Down: Restore_It( Frame, Left, Top, Right, Bottom, True, False, True, True );
                Window_Down_Right: Restore_It( Frame, Left, Top, Right, Bottom, False, False, True, True );
                Window_Down_Left: Restore_It( Frame, Left, Top, Right, Bottom, True, False, False, True );
                Window_Up_Right: Restore_It( Frame, Left, Top, Right, Bottom, False, True, True, False );
                Window_Up_Left: Restore_It( Frame, Left, Top, Right, Bottom, True, True, False, False );
                else Quick_Restore( Left, Top, Right, Bottom );
              End; {Case}
              Deallocate_Storage( Current_Window^.Storage );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get screen line.
    This procedure gets the line of data from the
    screen and puts it in the work area.

*************************************************)

    Procedure Get_Screen_Line( Left, Right, Row: Byte );
      Var
        Where: Pointer;
      Begin
        Where := Addr( Work_Area[ 1 ] );
       {$IFNDEF OS2}
        Read_Data( Row, Left, Where^, Succ( Right - Left ) );
       {$ELSE}
        Read_Data( Row, Left, Where, Succ( Right - Left ) );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put temp line.
    This procedure gets the data in the work area
    and puts it into storage in the specified
    location.

*************************************************)

    Procedure Put_Temp_Line( Left, Right, Row: Byte );
      Var
        Where1,
        Where2: Pointer;
      Begin
        Where1 := Addr( Work_Area[ 1 ] );
        Where2 := Address_Storage( Current_Window^.Storage, Row, 1 );
        Move( Where1^, Where2^, ( 2 * Succ( Right - Left ) ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get screen column.
    This procedure reads the data from the screen
    in a specified column and puts it in the temp
    buffer.

*************************************************)

    Procedure Get_Screen_Column( Top, Bottom, Column: Byte );
      Var
        Counter: Byte;
      Begin
        For Counter := Top to Bottom do
          With Work_Area[ Counter ] do
            Get_Character_From_Screen( Column, Counter, Character, Attribute );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Put temp column.
    This procedure copies the data in the temp
    buffer into storage.

*************************************************)

    Procedure Put_Temp_Column( Top, Bottom, Column: Byte );
      Var
        Placer,
        Counter: Byte;
      Begin
        Placer := 1;
        For Counter := Top to Bottom do
          With Work_Area[ Counter ] do
            Begin
              Put_to_Storage( Current_Window^.Storage, Placer, Column, Character, Attribute );
              Inc( Placer );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Get row and column.
    This procedure gets the current cursor row
    column and stores them in the given variables.

*************************************************)

    Procedure Get_Row_And_Column( Var Row, Column: Byte );
      Begin
        Row := WhereY;
        Column := WhereX;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Open window.
    As previously described.

*************************************************)

    Function Open_Window( New_Frame_Style, Open_Method, New_Left, New_Top, New_Right, New_Bottom, Attribute: Byte ): Boolean;
      Var
        Temp_Point: Window_Pointer_Type;
        Okay: Boolean;
      Begin
        Get_The_Mode;
        If ( Current_Window <> Nil ) and ( Old_Mode <> The_Mode )
          then
            Begin
              TextMode( Old_Mode );
              Close_all_Windows;
              Write_Error( 201, 'Open_Window: Video modes can not be changed between windows!' );
            End
          else
            Old_Mode := The_Mode;
        If ( New_Left < 1 )
          then
            New_Left := 1;
        If ( New_Top < 1 )
          then
            New_Top := 1;
        If ( New_Top > New_Bottom )
          then
            New_Bottom := Succ( New_Top );
        If ( New_Left > New_Right )
          then
            New_Right := Succ( New_Left );
        If ( New_Right > Screen_Column_Limit )
          then
            New_Right := Screen_Column_Limit;
        If ( New_Bottom > Screen_Row_Limit )
          then
            New_Bottom := Screen_Row_Limit;
        New( Temp_Point );
        Okay := ( Temp_Point <> Nil );
        If Okay
          then
            Begin
              Temp_Point^.Next := Current_Window;
              Current_Window := Temp_Point;
              With Current_Window^ do
                Begin
                  Old_Attribute := TextAttr;
                  Old_Where_Column := WhereX;
                  Old_Where_Row := WhereY;
                  Old_Left := Left_Of_Window^;
                  Old_Top := Top_Of_Window^;
                  Old_Right := Right_Of_Window^;
                  Old_Bottom := Bottom_Of_Window^;
                  Safety_Left := New_Left;
                  Safety_Right := New_Right;
                  Safety_Top := New_Top;
                  Safety_Bottom := New_Bottom;
                  Shadow := ( New_Frame_Style in [ Frame_s1 .. Frame_s24 ] );
                  Frame_Style := ( New_Frame_Style and $0F );
                  New_Attribute := Attribute;
                  Calculate_Storage_Amount( Storage, Succ( New_Bottom - New_Top ), Succ( New_Right - New_Left ) );
                  Storage.Location := Nil;
                  Virtual_Window := Nil;
                 {$IFNDEF VER40}
                  Reposition := Procedure_Default;
                 {$ENDIF}
                End;
              Okay := Save_Window( New_Top, New_Bottom, New_Left, New_Right );
              If Okay
                then
                  Begin
                    If ( New_Frame_Style <> No_Frame )
                      then
                        Begin
                          With Current_Window^ do
                            Begin
                              Set_Window_Frame( Frame, New_Attribute, Frame_Style );
                              Draw_Window( Frame, Open_Method, New_Left, New_Top, New_Right, New_Bottom, Shadow );
                              If Shadow
                                then
                                  Window( Succ( New_Left ), Succ( New_Top ), Pred( Pred( New_Right ) ),
                                          Pred( Pred( New_Bottom ) ) )
                                else
                                  Window( Succ( New_Left ), Succ( New_Top ), Pred( New_Right ), Pred( New_Bottom ) );
                            End;
                          TextAttr := Attribute;
                         {$IFDEF OS2}
                          TextBackground( Attribute shr 4 );
                         {$ENDIF}
                          ClrScr;
                        End
                      else
                        Window( New_Left, New_Top, New_Right, New_Bottom );
                    TextAttr := Attribute;
                   {$IFDEF OS2}
                    TextBackground( Attribute shr 4 );
                   {$ENDIF}
                  End
                else
                  Begin
                    Temp_Point := Current_Window;
                    Current_Window := Current_Window^.Next;
                    Dispose( Temp_Point );
                  End;
            End;
        Open_Window := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Close window.
    As previously defined.

*************************************************)

    Procedure Close_Window( Method: Byte );
      Var
        Temp_Point: Window_Pointer_Type;
        Where: Virtual_Window_Pointer_Type;
      Begin
        Get_The_Mode;
        If ( Current_Window <> Nil )
          then
            Begin
              Allow_Control := False;
              If ( Current_Window^.Virtual_Window <> Nil )
                then
                  Begin
                    Where := Current_Window^.Virtual_Window;
                    Deallocate_Storage( Where^.Storage );
                    Dispose( Current_Window^.Virtual_Window );
                  End;
              If ( Current_Window^.Frame_Style = No_Frame )
                then
                  Method := Window_Pop;
              With Current_Window^ do
                Restore_Window( Frame, Method, Safety_Top, Safety_Bottom, Safety_Left, Safety_Right );
              Temp_Point := Current_Window;
              With Temp_Point^ do
                Begin
                  Window( Succ( Old_Left ), Succ( Old_Top ), Succ( Old_Right ), Succ( Old_Bottom ) );
                  GotoXY( Old_Where_Column, Old_Where_Row );
                  TextAttr := Old_Attribute;
                 {$IFDEF OS2}
                  TextBackground( Old_Attribute shr 4 );
                 {$ENDIF}
                End;
              Current_Window := Temp_Point^.Next;
              Dispose( Temp_Point );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Close all windows.
    As previously defined.

*************************************************)

    Procedure Close_All_Windows;
      Begin
        While ( Current_Window <> Nil ) do
          Close_Window( Window_Pop );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Label window.
    As previously defined.

*************************************************)

    Procedure Label_Window( Attribute: Byte; Name: String );
      Var
        Count,
        Start,
        Middle,
        Old_Row,
        Old_Column,
        The_Length: Byte;
      Begin
        If ( ( Current_Window <> Nil ) and ( Current_Window^.Frame_Style <> No_Frame ) )
          then
            Begin
              The_Length := Length( Name );
              If ( The_Length > ( Current_Window^.Safety_Right - Current_Window^.Safety_Left ) )
                then
                  The_Length := ( Current_Window^.Safety_Right - Current_Window^.Safety_Left );
              Get_The_Mode;
              Old_Column := WhereX;
              Old_Row := WhereY;
              Middle := ( Current_Window^.Safety_Right + Current_Window^.Safety_Left ) div 2;
              Start := Succ( Middle - ( The_Length div 2 ) );
              For Count := 1 to The_Length do
                Begin
                  Put_Character_On_Screen( Start, Current_Window^.Safety_Top, Name[ Count ], Attribute );
                  Inc( Start );
                End;
              GotoXY( Old_Column, Old_Row );
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Internal move window up.
    This function tries to move the window up on
    the screen.

*************************************************)

    Function Internal_Move_Window_Up( Var Data: Window_Type ): Boolean;
      Var
        Old_Row,
        New_Top,
        New_Bottom,
        Old_Column: Byte;
      Begin
        If ( Data.Safety_Top > 1 )
          then
            With Data do
              Begin
                Get_The_Mode;
                Get_Row_And_Column( Old_Row, Old_Column );
                New_Top := Pred( Safety_Top );
                New_Bottom := Pred( Safety_Bottom );
                Get_Screen_Line( Safety_Left, Safety_Right, New_Top );
                If Shadow
                  then
                    Scroll_Region_Up( Safety_Left, New_Top, Pred( Safety_Right ), Pred( Safety_Bottom ) )
                  else
                    Scroll_Region_Up( Safety_Left, New_Top, Safety_Right, Safety_Bottom );
                Restore_Row( Safety_Bottom, Safety_Left, Safety_Right, Safety_Top );
                Scroll_Storage_Down( Storage );
                Put_Temp_Line( Safety_Left, Safety_Right, 1 );
                If Shadow
                  then
                    Begin
                      Make_Shadow( Succ( New_Top ), Safety_Right, 1 );
                      Restore_Row_Shadow( New_Bottom, Safety_Left, Safety_Right, New_Top );
                    End;
                Safety_Top := New_Top;
                Safety_Bottom := New_Bottom;
                Reset_Window;
                GotoXY( Old_Column, Old_Row );
                Internal_Move_Window_Up := True;
               {$IFNDEF VER40}
                Reposition;
               {$ELSE}
                VW_Reposition;
               {$ENDIF}
                Delay( Delay_Amount );
              End
          else
            Internal_Move_Window_Up := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************


  Function: Internal move window down.
    This function tries to move the window down
    on the screen.

*************************************************)

    Function Internal_Move_Window_Down( Var Data: Window_Type ): Boolean;
      Var
        New_Top,
        Old_Row,
        Old_Column,
        New_Bottom: Byte;
      Begin
        Get_The_Mode;
        If ( Data.Safety_Bottom < Screen_Row_Limit )
          then
            With Data do
              Begin
                Get_Row_And_Column( Old_Row, Old_Column );
                New_Top := Succ( Safety_Top );
                New_Bottom := Succ( Safety_Bottom );
                Get_Screen_Line( Safety_Left, Safety_Right, New_Bottom );
                If Shadow
                  then
                    Scroll_Region_Down( Safety_Left, Safety_Top, Pred( Safety_Right ), Pred( New_Bottom ) )
                  else
                    Scroll_Region_Down( Safety_Left, Safety_Top, Safety_Right, New_Bottom );
                Restore_Row( Safety_Top, Safety_Left, Safety_Right, Safety_Top );
                Scroll_Storage_Up( Storage );
                Put_Temp_Line( Safety_Left, Safety_Right, Storage.Row_Length );
                If Shadow
                  then
                    Begin
                      Restore_Column( Safety_Right, New_Top, New_Top, Safety_Left );
                      Make_Shadow( New_Bottom, Succ( Safety_Left ), ( Safety_Right - Safety_Left ) );
                    End;
                Safety_Top := New_Top;
                Safety_Bottom := New_Bottom;
                Reset_Window;
                GotoXY( Old_Column, Old_Row );
                Internal_Move_Window_Down := True;
               {$IFNDEF VER40}
                Reposition;
               {$ELSE}
                VW_Reposition;
               {$ENDIF}
                Delay( Delay_Amount );
              End
          else
            Internal_Move_Window_Down := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Internal move window left.
    This function tries to move the window left
    on the screen.

*************************************************)

    Function Internal_Move_Window_Left( Var Data: Window_Type ): Boolean;
      Var
        Old_Row,
        New_Left,
        New_Right,
        Old_Column: Byte;
      Begin
        If ( Data.Safety_Left > 1 )
          then
            With Data do
              Begin
                Get_The_Mode;
                Get_Row_And_Column( Old_Row, Old_Column );
                New_Left := Pred( Safety_Left );
                New_Right := Pred( Safety_Right );
                Get_Screen_Column( Safety_Top, Safety_Bottom, New_Left );
                If Shadow
                  then
                    Scroll_Region_Left( New_Left, Safety_Top, Pred( Safety_Right ), Pred( Safety_Bottom ) )
                  else
                    Scroll_Region_Left( New_Left, Safety_Top, Safety_Right, Safety_Bottom );
                Restore_Column( Safety_Right, Safety_Top, Safety_Bottom, Safety_Left );
                Scroll_Storage_Right( Storage );
                Put_Temp_Column( Safety_Top, Safety_Bottom, 1 );
                If Shadow
                  then
                    Begin
                      Restore_Column_Shadow( New_Right, Safety_Top, Safety_Bottom, New_Left );
                      Make_Shadow( Safety_Bottom, Succ( New_Left ), 1 );
                    End;
                Safety_Left := New_Left;
                Safety_Right := New_Right;
                Reset_Window;
                GotoXY( Old_Column, Old_Row );
                Internal_Move_Window_Left := True;
               {$IFNDEF VER40}
                Reposition;
               {$ELSE}
                VW_Reposition;
               {$ENDIF}
                Delay( Delay_Amount );
              End
          else
            Internal_Move_Window_Left := False;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Internal move window right.
    This function tries to move the window right
    on the screen.

*************************************************)

    Function Internal_Move_Window_Right( Var Data: Window_Type ): Boolean;
      Var
        Old_Row,
        New_Left,
        New_Right,
        Old_Column: Byte;
      Begin
        Get_The_Mode;
        If ( Data.Safety_Right < Screen_Column_Limit )
          then
            With Data do
              Begin
                Get_Row_And_Column( Old_Row, Old_Column );
                New_Left := Succ( Safety_Left );
                New_Right := Succ( Safety_Right );
                Get_Screen_Column( Safety_Top, Safety_Bottom, New_Right );
                If Shadow
                  then
                    Scroll_Region_Right( Safety_Left, Safety_Top, Pred( New_Right ), Pred( Safety_Bottom ) )
                  else
                    Scroll_Region_Right( Safety_Left, Safety_Top, New_Right, Safety_Bottom );
                Restore_Column( Safety_Left, Safety_Top, Safety_Bottom, Safety_Left );
                Scroll_Storage_Left( Storage );
                Put_Temp_Column( Safety_Top, Safety_Bottom, Storage.Column_Length );
                If Shadow
                  then
                    Begin
                      Restore_Column_Shadow( New_Right, Safety_Top, Safety_Bottom, New_Left );
                      Restore_Row( Safety_Bottom, New_Left, New_Left, Safety_Top );
                    End;
                Safety_Left := New_Left;
                Safety_Right := New_Right;
                Reset_Window;
                GotoXY( Old_Column, Old_Row );
                Internal_Move_Window_Right := True;
               {$IFNDEF VER40}
                Reposition;
               {$ELSE}
                VW_Reposition;
               {$ENDIF}
                Delay( Delay_Amount );
              End
          else
            Internal_Move_Window_Right := False;
      End;

