{$F+} { Compiler Directive: Generate far procedure calls: On }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  TextEdit
    Version 1.02

  Purpose:
    To generate a full scale, easy to use, text processing editor engine.

  Features:
    Handles very large files.
      Up to 2 million lines of up to 60 thousand characters each.
    Uses disk space for storing the data. ( Optionally )
    Uses standard keyboard command combinations.
    Performs background idle time screen updating. ( Ripple effect )
    Performs background idle time text searching.
    Allows overriding of default message system for custom user interfacing.
    Conforms to current screen sizes. ( Tested at 128x43 and 80x50 )
    Use of global flags to alter the performance parameters.
    Allows use of block sizes exceeding most limits.
    Beeper warning on line buffer overload instead of system lockup.
    Recovers gracefully from memory errors with proper HeapError routine.
    Error checking for file input and output.
    Status indicators for all block operations and file reads.
    Keyboard monitoring during status updating.
    Command Cancel overrides. ( Allows interruptions of block commands by
      escape key )
    Option toggles that effect Search and Search & Replace operation.
    Status indicators during Search and Search & Replace.
    Escape during a search / replace operation.
    Automatic tab expansion during file reading.
    Paragraph reforming commands.
    Automatic paragraph reform option.
    Faster file save than before.
    Function key interlinks ( also allows for Alternate keys ).
    Wildcards allowed for a search.

  Optional features:
    Word spelling check ( Automatically linked with the Correct unit. )
    Extra block functions ( Automatically linked with the Extras unit. )
    Pointer system support ( Automatically links with the Pointer unit. )
      Single click button 1 - Moves the cursor toward that location.
      Double click button 1 - Highlights the word immediately under the
        pointer.
      Click and drag button 1 - Selects the text. (Downward movement only)

  Limitations:
    Block operations on large blocks take long processing times.
    Reading and writing text from a large file may take a long time.

  Versions:
    1.01 - Updated to work with updated pointer unit, repaired minor bugs.
    1.02 - Repaired several bugs including: centering bug, search failure.
           Updated search to allow escaping during operation.

  CopyRight 1994, All rights reserved.
    By Paul Renaud.

  Compiler:
    Turbo Pascal versions xxx to 6.0
   { Should compile under version 5.0, but won't. }

  System:
    MS-DOS, MDOS

*****************************************************************************)

Unit TextEdit;

  Interface

   { When defined as memory, an alternative unit is included
     which uses memory instead of disk space to store the text data. }

    {$DEFINE NoMemory }

    Uses
      CRT,
      Core,
      KeyBoard,
      TextLine,
     {$IFDEF Memory}
      TextMem,
     {$ELSE}
      TextFile,
     {$ENDIF}
      String_Utilities;

(***********************************************************

  These attributes modify how the default routines display
    on the screen.

***********************************************************)

    Const
     { The normal text display attribute. }
      Normal: Byte = 7;

     { The blocked text display attribute. }
      HighLight: Byte = 112;

     { The normal message display attribute. }
      Message_Normal: Byte = 3;

     { The highlight message display attribute. }
      Message_HighLight: Byte = 15;

     { The Marked text display attribute. }
      Search_HighLight: Byte = 240;

(***********************************************************

  These variable constants modify the editor's behavior.

***********************************************************)

     { Default tab spacing. }
      Tab_Amount: Byte = 8;

     { Allows the tab key to insert spaces. }
      Insert_Tab: Boolean = True;

     { Allows entered characters to be inserted as opposed to overwritten. }
      Insert_Mode: Boolean = True;

     { Allows the right arrow to add spaces instead of hitting the end of line. }
      Space_Right: Boolean = True;

     { Allows marked text to appear on screen. }
      Display_Block: Boolean = True;

     { Allows extra characters to be added to the end of the line. }
      Insert_At_EoLn: Boolean = True;

     { Allows cursor keys to move past the ends of the line. }
      Automatic_Return: Boolean = True;

     { Allows the enter key to split a line. }
      Return_Split_Line: Boolean = True;

     { Allows the delete key to combine two lines. }
      Delete_Combine_Line: Boolean = True;

     { Allows the backspace key to combine two lines. }
      Backspace_Combine_Line: Boolean = True;

     { Alters the way the screen is updated. }
      Quick_Update_Screen = False;

(***********************************************************

  These variable constants modify the search routines
    behavior

***********************************************************)

     { Adds special code to use idle time for searching text. }
      Allow_BackGround_Searching = True;

     { When activated, allows multiple Search & Replace. }
      Search_Change_All: Boolean = True;

     { When activated, allows only the marked block to be Searched. }
      Search_Selected_Text: Boolean = False;

     { When activated, allows search to start at cursor, otherwise it starts at the top. }
      Start_Search_At_Cursor: Boolean = True;

     { When activated, prompts before making the change. }
      Search_Prompt_On_Replace: Boolean = True;

     { When activated, only identifies whole words. }
      Search_Only_Whole_Words: Boolean = False;

     { When deactivated, treats both capitals and lowercase letters alike. }
      Search_Case_Sensitive: Boolean = True;

(***********************************************************

  This variable adds special code that displays the status
  during long operations.

***********************************************************)

      Show_Status = True;

(***********************************************************

  When activated, this allows automatic paragraph reforming.
    Automatic paragraph reforming means that words are
    automatically shifted to the next line when the line
    gets to long so that the lines will always fit into the
    current margins.

***********************************************************)

      Automatic_Paragraph_Reform: Boolean = False;

(***********************************************************

  Data structures.

***********************************************************)

    Type
     { This structure holds information for the text searching engines. }
      Search_Data_Type = Record
                           Find_String,
                           Replace_String: String;
                           Find_Position: Point_Type;
                           Last_Operation: ( Find, Replace );
                         End;
     { This structure holds information for handling the cursor. }
      Data_Type = Record
                    Cursor: Point_Type;
                    Screen_Row: Byte;
                  End;
     { This structure holds the information to handle the text and the cursor }
      All_Type = Record
                   Text: Text_Type;
                   Data: Data_Type;
                 End;

(***********************************************************

  These variable procedures are available for replacement by
  the main program.  They are all initialized to the default
  procedures which allow maximum versatility, but can still
  be altered to make a more consistent and elegant user
  interface.  Study the code thoroughly before attempting to
  replace it, because it may also perform a operation that
  isn't so obvious at first.

    To replace one of these routines, define a replacement
    routine that mimics the default code in functions.

      For example...

        Procedure Replacement( parameters ); Far;
          Begin
            do something here.
          End;

    Then, somewhere in the initialization section of the
    main program, substitute your new routine for the old
    one.

      For example...

        Old_Routine := Replacement;

   That's all there is to it.  Now the new routine will
   be called in place of the old one.

***********************************************************)

    Var
     { This procedure is called when the help key is pressed. }
      Help,

     { This procedure is called before the block operation starts. }
      Write_Wait,

     { This procedure is called when the block operation is completed. }
      Write_Complete,

     { This procedure is called when a Search or Search & Replace operation fails. }
      Write_Search_Failure: Procedure;

     { This procedure is called when an error occurs. }
      Write_Error: Procedure( Code: Word );

     { This procedure returns the name of the file to read. }
      Get_Read_File_Name: Procedure( Var Data: String );

     { This procedure returns the name of the file to write to. }
      Get_Write_File_Name: Procedure( Var Data: String );

     { This procedure returns the data necessary for a text search.  It can
       also allow the user to alter the search variable constants. }
      Get_Search_Data: Procedure( Var Data: String );

     { This procedure returns the data necessary for a text Search & Replace
       operation.  It can also allow the user to alter the search variable
       constants. }
      Get_Search_And_Replace_Data: Procedure( Var Find, Replace: String );

     { This function is called to confirm a change.
       It's result should be either Y, N or E. }
      Get_Confirmation_Of_Change: Function: Char;

     { This procedure is called after a Search & Replace operation to display
       the results of the operation. }
      Confirm_Changes: Procedure( Amount: Word );

     { This procedure is called to display the status during a read operation. }
      Display_Read_Status,

     { This procedure is called to display the location status during the Search operations. }
      Display_Where_Status: Procedure( Where: LongInt );

     { This procedure is called to display the status during a write operation. }
      Display_Write_Status: Procedure( Where, Limit: LongInt );

     { This procedure is called to display information on the text file. }
      Display_Information: Procedure( Var All: All_Type );

     { This procedure is called to get the new line information.  New_Line
       should be between 1 and Limit. }
      Get_New_Line: Procedure( Var New_Line: LongInt; Limit: LongInt );

     { This procedure is called when Control P is pressed to get the
       extended character.  It displays a menu from which to choose from. }
      Character_Menu: Function( Var New_Character: Char ): Boolean;

     { This procedure is called when Escape is pressed during an operation.
       If desired, it can allow the user to verify the exit.  To cause the
       operation to exit, change Continue to False. }
      Escape_Operation: Procedure( Var Continue: Boolean );

(***********************************************************

  These variable procedure are available for interlinking by
  the main program.  They are initialized to the default
  routines that should be called when the function of the
  first ones haven't been evoked.  Make sure that the new
  function codes don't interfere with the old ones.
  Finally, if something goes wrong, make it a point to try
  to follow the code and find out where things are getting
  lost.

    To interlink into the code, first define a global
    procedure variable to hold the old variable.

      For example...

        Var
          Old_Function: Procedure( parameters );

    Then define a linking routine that extends the function
    of the default code.  Make sure to call the old routine
    somewhere in the routine so that backward functionality
    won't be lost.

      For example...

        Procedure New_Function( parameters ); Far;
          Begin
            Case Command of
              Perform substitutions here.
              else
                Old_Function( parameters );
            End; { Case }
          End;

    Finally, somewhere in the initialization section of the
    main program, initialize your old routine global
    variable and substitute your new routine for the old
    one.

      For example...

        Old_Function := Function;
        Function := New_Function;

   That's all there is to it.  Now your new procedure will
   be called to handle your own special functions before the
   old one is.

   { Check out Extras and Correct to see how it's done! }

***********************************************************)

      { This procedure is called to allow substituting for function keys. }
      Function_Key: Procedure( Var Command: Byte; Var Character: Char );

      { These procedures are called to allow adding special functions to the editor. }
      Control_J_Link,
      Control_K_Link,
      Control_O_Link,
      Control_Q_Link: Procedure( Var All: All_Type; Character: Char; Var Start, Finish: Point_Type; Var Done: Boolean );

(***********************************************************

  These variables are available for the external functions
  to use.  Take care that one function doesn't use the same
  variable internally.

***********************************************************)

      { Pointer to the other buffer. }
      Other_Buffer: Line_Pointer_Type absolute Work_Space;

      { Pointer to the current working buffer. }
      Working_Buffer: Line_Pointer_Type;

      { These procedures allows the other units to link to the system. }
      Get_Line,
      Look_Line: Procedure( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type );
      Put_Line: Function( Var Text: Text_Type; Row: LongInt; Var Line: Line_Type; Reduce: Boolean ): Boolean;
      Swap_Line: Function( Var Text: Text_Type; Row1, Row2: LongInt ): Boolean;

(***********************************************************

  Procedure: Edit the text.

    This procedure allows the text from the text file to be
    edited using the standard commands.

      Left - ^S, <-                 Right - ^D, ->
      Up - ^E, up                   Down - ^X, down
      Word Left - ^A, ^<-           Word Right - ^F, ^->
      Start of line - ^QS, home     End of line - ^QD, end
      Tab left - shift tab          Tab right - ^I, tab
      Scroll up - ^W                Scroll down - ^Z
      Page up - ^R, pg up           Page down - ^C, pg dn
      Top of screen - ^QE, ^home    Bottom of screen - ^QX, ^end
      Top of file - ^QR, ^pg up     Bottom of file - ^QC, ^pg dn
      Toggle Insert - ^V, ins       Insert line - ^N
      Delete character - ^G, del    Delete line - ^Y
      Backspace - <-backspace, ^H   Delete to end - ^QY
      Delete word right - ^T        Restore line - ^JL
      Mark block start - ^KB        Mark block finish - ^KK
      Top of block - ^QB            Bottom of block - ^QK
      Read block - ^KR              Write block - ^KW
      Delete block - ^KY, ^del      Print block - ^KP
      Mark single word - ^KT        Hide block - ^KH
      Copy block - ^KC              Move block - ^KV
      Find string - ^QF             Find and replace - ^QA
      Set Left margin - ^OL         Set Right margin - ^OR
      Repeat search - ^L            Center text - ^OC
      Reformat paragraph - ^B, ^JO  Reform entire text - ^JQ
      Reform block - ^JK            Toggle auto reform - ^OW
      Display info - ^J[space]      Goto line number - ^JN
      Help - ^JH, F1                Call character menu - ^P
      Print text - ^JP
      Escape from editor - [esc]

    If Changed is set, the text was altered.

***********************************************************)

    Procedure Edit_Text( Var All: All_Type; Var Changed: Boolean );

(***********************************************************

  Procedure: Initialize the text.

    This procedure initializes everything for the text
    editor.

***********************************************************)

   Procedure Initialize_The_Text( Var All: All_Type );

(***********************************************************

  Procedure: Dispose of text.

    This function releases all the memory used up by holding
    the text.

***********************************************************)

    Procedure Dispose_Of_Text( Var All: All_Type );

(***********************************************************

  Function: Read text file.

    This function reads text from a file into memory for the
    editor.  The InFile is considered ready for input.

***********************************************************)

    Function Read_Text_File( Var InFile: Text; Var All: All_Type ): Boolean;

(***********************************************************

  Function: Write text file.

    This function writes the text in memory into a file.

***********************************************************)

    Function Write_Text_File( FileName: String; Var All: All_Type ): Boolean;

(***********************************************************

  Procedure: Draw screen.

    This procedure is designed to draw the editing window on
    the screen for a static moment.  It is intended mainly
    for a static screen moment such as during search or
    replace functions.

***********************************************************)

    Procedure Draw_Screen( Var Text: Text_Type; Cursor: Point_Type; Row, Adjustment, Max_Window_Row: Byte );

(***********************************************************

  Function: Valid block operation.

    This function returns true if the block defined by Start
    and Finished is valid and if position is a location that
    is not inside the block.

***********************************************************)

    Function Valid_Block_Operation( Start, Finish, Position: Point_Type ): Boolean;

(***********************************************************

  Function: Copy the block.

    This function attempts to make a copy of the block
    marked off in Text by Start and Finish and returns that
    copy in the ClipBoard.  It returns true only when it's
    successful.  In the event this function fails, part of
    the block may still have been copied.  Text and
    ClipBoard must both be initialized.

***********************************************************)

    Function Copy_Block( Var Text, ClipBoard: Text_Type; Start, Finish: Point_Type ): Boolean;

(***********************************************************

  Function: Cut the block.

    This function attempts to remove the block marked off in
    Text by Start and Finish and returns the clipping in the
    ClipBoard.  It returns true only when it's successful.
    In the event this function fails, part of the block may
    still have been removed.  Text and ClipBoard must both
    be initialized.

***********************************************************)

    Function Cut_Block( Var Text, ClipBoard: Text_Type; Var Buffer: Line_Type; Var Start, Finish,
                        Cursor: Point_Type ): Boolean;

(***********************************************************

  Function: Paste the block.

    This function attempts to paste the block in ClipBoard
    into the Text at the cursor point.  Buffer should not
    hold any necessary data.  Start and Finish are
    automatically adjusted to the new locations of the
    block.  This function returns true only if it's
    successful.  In the event this function fails, part of
    the block may still have been inserted.  Text and Clip-
    Board must both be initialized.

***********************************************************)

    Function Paste_Block( Var Text, ClipBoard: Text_Type; Var Buffer: Line_Type; Var Start, Finish, Cursor: Point_Type; Point,
                          Limit: LongInt ): Boolean;

(***********************************************************

  These default procedures and functions are provided so
    that the main program may use them if necessary.  They
    are the corresponding routines that are assigned to the
    variable procedures above.

***********************************************************)

    Procedure Help_Default;
    Procedure Write_Wait_Default;
    Procedure Write_Error_Default( Code: Word );
    Procedure Write_Complete_Default;
    Procedure Write_Search_Failure_Default;
    Procedure Get_File_Name_Default( Var Data: String );
    Procedure Get_Search_Data_Default( Var Data: String );
    Procedure Get_Search_And_Replace_Data_Default( Var Data1, Data2: String );
    Function Get_Confirmation_Of_Change_Default: Char;
    Procedure Confirm_Changes_Default( Amount: Word );
    Procedure Display_Read_Status_Default( Where: LongInt );
    Procedure Display_Where_Status_Default( Where: LongInt );
    Procedure Display_Write_Status_Default( Where, Limit: LongInt );
    Procedure Display_Information_Default( Var All: All_Type );
    Procedure Get_New_Line_Default( Var New_Line: LongInt; Limit: LongInt );
    Function Character_Menu_Default( Var Character: Char ): Boolean;
    Procedure Escape_Operation_Default( Var Continue: Boolean );

(***********************************************************

  Error codes passed to Write_Error.

    400 - The new line wasn't accepted.
    401 - Couldn't delete the line.
    402 - Couldn't add a new line.
    403 - Couldn't remove the return.
    404 - Read block failure.
    405 - Invalid read block name.
    406 - Write block failure.
    407 - Copy block failure.
    408 - Delete block failure.
    409 - Move block failure.
    410 - Editor couldn't accept new line.
    411 - Printer write error.
    412 - Invalid write text name.

***********************************************************)

{----------------------------------------------------------------------------}

  Implementation

    {$Define Quick }   { This option allows alternate code to be compiled. }

    Uses
      TextLink;

    Const
     { During searches, this value in the search string is considered equal to any other value in the line. }
      WildCard: Char = #0;
     { Holds the left margin default number. }
      Left_Margin: Word = 1;
     { Holds the right margin default number. }
      Right_Margin: Word = 70;
     { Command which signals time to reform the paragraph. }
      Press_Automatic_Reform = 255;
     { These are the legitimate values that are regarded as word separators. }
      Word_Separators = [ ' ', '.', ':', ',', '(', ')', '{', '}', '[', ']',
                          '<', '>', '?', '/', ';', '!', '@', '^', '&', '*',
                          '-', '+', '\', '~', '`', '=', '|', '''' ];
      Punctuation = [ '.', ':', '?', ';', '!' ];
      Split_Points = [ ' ', '-' ];
     { These are the values which allow the Function_Key routine to be called. }
      Function_Keys = [ Press_F1, Press_F2, Press_F3, Press_F4, Press_F5, Press_F6, Press_F7, Press_F8, Press_F9, Press_F10,
                        Press_F11, Press_F12, Press_Shift_F1, Press_Shift_F2, Press_Shift_F3, Press_Shift_F4, Press_Shift_F5,
                        Press_Shift_F6, Press_Shift_F7, Press_Shift_F8, Press_Shift_F9, Press_Shift_F10, Press_Shift_F11,
                        Press_Shift_F12, Press_Control_F1, Press_Control_F2, Press_Control_F3, Press_Control_F4,
                        Press_Control_F5, Press_Control_F6, Press_Control_F7, Press_Control_F8, Press_Control_F9,
                        Press_Control_F10, Press_Control_F11, Press_Control_F12, Press_Alternate_F1, Press_Alternate_F2,
                        Press_Alternate_F3, Press_Alternate_F4, Press_Alternate_F5, Press_Alternate_F6, Press_Alternate_F7,
                        Press_Alternate_F8, Press_Alternate_F9, Press_Alternate_F10, Press_Alternate_F11,
                        Press_Alternate_F12, Press_Alternate_A .. Press_Alternate_Z ];

    Type
     { This structure holds information for editing of the data line. }
      Cluster_Type = Record
                       Where: Point_Type;
                       Start,
                       Finish,
                       String_Length: Word;
                       Character: Char;
                       Line_Altered: Boolean;
                     End;
     { This structure holds information for displaying the screen. }
      Ripple_Cluster = Record
                         Start,
                         Finish: Word;
                         Screen_Up,
                         Screen_Down,
                         Window_Bottom: Byte;
                         Where,
                         Ripple_Up,
                         Ripple_Down: LongInt;
                         Buffer: ^Text_Type;
                         Block_Start,
                         Block_Finish: Point_Type;
                       End;

    Var
     { Defines when rippling is in effect. }
      Rippling,
     { Eliminates showing the screen during a Search & Replace operation. }
      Search_Show_Line,
     { Defines when background searching is in effect. }
      BackGround_Searching: Boolean;
     { Holds the last command entered. }
      Command: Byte;
     { Holds the last character entered. }
      Character: Char;
     { Holds the information necessary for the ripple. }
      Wave: Ripple_Cluster;
     { Holds the previous keyboard wait routine. }
      Hold_Routine: Procedure;
     { Pointer to the current text type.  Necessary to eliminate parameters. }
      BackGround_Text: ^Text_Type;
     { Data necessary for the background search routine. }
      BackGround_Info: Search_Data_Type;
     { Points to the data row during the background search routine. }
      BackGround_Row: LongInt;

{----------------------------------------------------------------------------}

  {$I TextEdi2.Pas }
  {$I TextEdi3.Pas }
  {$I TextEdi4.Pas }
  {$I TextEdi5.Pas }

{----------------------------------------------------------------------------}

(*************************************************

  Function: Initialize the text.
    As previously defined.

*************************************************)

    Procedure Initialize_The_Text( Var All: All_Type );
      Begin
        Read_Status := Display_Read_Status;
        Write_Status := Display_Write_Status;
        Initialize_Text( All.Text );
        All.Data.Cursor.Row := 1;
        All.Data.Cursor.Column := 1;
        All.Data.Screen_Row := 1;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Read text file.
    As previously defined.

*************************************************)

    Function Read_Text_File( Var InFile: Text; Var All: All_Type ): Boolean;
      Begin
        Read_Text_File := Read_Text( InFile, All.Text );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Write text file.
    As previously defined.

*************************************************)

    Function Write_Text_File( FileName: String; Var All: All_Type ): Boolean;
      Var
        OutFile: File;
        Error_Code: Word;
      Begin
        If Valid( FileName, False )
          then
            Begin
              Assign( OutFile, FileName );
             {$I-}
              Rewrite( OutFile, 1 );
              Error_Code := IoResult;
             {$I+}
              If ( Error_Code = 0 )
                then
                  Begin
                    Write_Text_File := Write_Text( OutFile, All.Text );
                    Close( OutFile );
                  End
                else
                  Write_Error( Error_Code );
            End
          else
            Write_Error( 412 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Dispose of text.
    As previously defined.

*************************************************)

    Procedure Dispose_Of_Text( Var All: All_Type );
      Begin
        Dispose_Text( All.Text );
        Erase_Text( All.Text );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Edit text.
    As previously defined.

*************************************************)

    Procedure Edit_Text( Var All: All_Type; Var Changed: Boolean );
      Var
        Done,
        Okay,
        Line_Changed: Boolean;
        Search: Search_Data_Type;
        Start,
        Finish: LongInt;
      Begin
        Changed := False;
        If Check_Allocation( All.Text )
          then
            Begin
              Search.Find_String := '';
              Hold_Routine := Keyboard.Wait_Routine;
              Keyboard.Wait_Routine := Ripple;
              Wave.Window_Bottom := Succ( Bottom_Of_Window^ - Top_Of_Window^ );
              Wave.Buffer := @All.Text;
              Wave.Block_Start := All.Data.Cursor;
              Wave.Block_Finish := All.Data.Cursor;
              Done := False;
              Start := 1;
              Finish := ( Right_Of_Window^ - Left_Of_Window^ );
              Repeat
                Wave.Where := All.Data.Cursor.Row;
                Wave.Ripple_Up := Pred( All.Data.Cursor.Row );
                Wave.Screen_Up := Pred( All.Data.Screen_Row );
                Wave.Ripple_Down := Succ( All.Data.Cursor.Row );
                Wave.Screen_Down := Succ( All.Data.Screen_Row );
                Repeat
                  Get_Text_Line( All.Text, All.Data.Cursor.Row, Working_Buffer^ );
                  Line_Changed := Edit_Row( All.Data.Screen_Row, Working_Buffer^, All.Data.Cursor, Start, Finish );
                Until ( Command <> Press_Restore_Line );
                If Line_Changed
                  then
                    Begin
                      Changed := True;
                      If not Put_Text_Line( All.Text, All.Data.Cursor.Row, Working_Buffer^, True )
                        then
                          Write_Error( 410 );
                    End;
                If Quick_Update_Screen
                  then
                    Finish_Screen;
                Case Command of
                  Press_Automatic_Reform:
                    Automatic_Reform( All, Working_Buffer^, Work_Space^, Changed );
                  Press_Enter:
                    Move_Return( All, Working_Buffer^, Changed );
                  Press_Down_Arrow:
                    Move_Down( All );
                  Press_Page_Down:
                    Move_Page_Down( All );
                  Press_Up_Arrow:
                    Move_Up( All.Data );
                  Press_Page_Up:
                    Move_Page_Up( All.Data );
                  Press_Escape:
                    Done := True;
                  Press_Control_Page_Up:
                    Move_Where( All.Data, 1 );
                  Pointer_Up:
                    Move_Where_Short( All.Data, All.Data.Cursor.Row - Adjust_Amount, All.Data.Screen_Row - Adjust_Amount );
                  Pointer_Down:
                    Move_Where_Short( All.Data, All.Data.Cursor.Row + Adjust_Amount, All.Data.Screen_Row + Adjust_Amount );
                  Press_Control_Page_Down:
                    Move_Where( All.Data, All.Text.FileSize );
                  Press_Scroll_Up:
                    Scroll_Up( All.Data );
                  Press_Scroll_Down:
                    Scroll_Down( All );
                  Press_Control_Home:
                    Move_Top_of_Window( All.Data );
                  Press_Control_End:
                    Move_Bottom_of_Window( All );
                  Press_Control_N:
                    Insert_Control_N( All, Working_Buffer^, Changed );
                  Press_Delete_Line:
                    Delete_Line( All, Changed );
                  Press_Delete:
                    Combine( All, Working_Buffer^, Changed );
                  Press_Delete_Arrow:
                    Back_Combine( All, Working_Buffer^, Changed );
                  Pointer_Button1_Down,
                  Press_Mark_Block_Top:
                    Wave.Block_Start := All.Data.Cursor;
                  Pointer_Button1_Up,
                  Press_Mark_Block_End:
                    Wave.Block_Finish := All.Data.Cursor;
                  Press_Move_Block_Top:
                    Move_To_Block( All.Data, Wave.Block_Start );
                  Press_Move_Block_End:
                    Move_To_Block( All.Data, Wave.Block_Finish );
                  Press_Copy_Block:
                    Copy_The_Block( All.Text, Working_Buffer^, Wave.Block_Start, Wave.Block_Finish, All.Data.Cursor, Changed );
                  Press_Delete_Block:
                    Delete_The_Block( All.Text, Working_Buffer^, Wave.Block_Start, Wave.Block_Finish, All.Data.Cursor,
                                      Changed );
                  Press_Move_Block:
                    Move_The_Block( All.Text, Working_Buffer^, Wave.Block_Start, Wave.Block_Finish, All.Data.Cursor, Changed );
                  Press_Read_Block:
                    Read_The_Block( All.Text, Working_Buffer^, Wave.Block_Start, Wave.Block_Finish, All.Data.Cursor, Changed );
                  Press_Write_Block:
                    Write_The_Block( All.Text, Wave.Block_Start, Wave.Block_Finish );
                  Press_Print_Block:
                    Print_The_Block( All.Text, Wave.Block_Start, Wave.Block_Finish );
                  Press_Hide_Block:
                    Display_Block := ( not Display_Block );
                  Press_Search:
                    Search_Text( All, Search );
                  Press_Control_L:
                    Continue_Search( All, Search, Changed );
                  Press_Replace:
                    Search_and_Replace_Text( All, Search, Changed );
                  Press_Control_B:
                    Reform_Paragraph( All, Working_Buffer^, Work_Space^, Changed );
                  Pressed_Lock:
                    Lock_Routine;
                  Pressed_Move_Window_Up:
                    Okay := Up_Routine;
                  Pressed_Move_Window_Down:
                    Okay := Down_Routine;
                  Pressed_Move_Window_Left:
                    Okay := Left_Routine;
                  Pressed_Move_Window_Right:
                    Okay := Right_Routine;
                  Pressed_Expand_Window_Width:
                    If Expand_Width_Routine
                      then
                        Inc( Finish );
                  Pressed_Reduce_Window_Width:
                    If Reduce_Width_Routine
                      then
                        Dec( Finish );
                  Pressed_Expand_Window_Height:
                    If Expand_Height_Routine
                      then
                        Inc( Wave.Window_Bottom );
                  Pressed_Reduce_Window_Height:
                    If Reduce_Height_Routine
                      then
                        Dec( Wave.Window_Bottom );
                  Press_Help:
                    Help;
                  Press_Control_J:
                    Case Character of
                      'K': Reform_Block( All, Working_Buffer^, Work_Space^, Changed );
                      'N': Move_Line_To( All );
                      'O': Reform_Paragraph( All, Working_Buffer^, Work_Space^, Changed );
                      'P': Print_Text( All.Text );
                      'Q': Reform_Text( All, Working_Buffer^, Work_Space^, Changed );
                      ' ': Show_Where( All );
                      else
                        Begin
                          Finish_Screen;
                          Control_J_Link( All, Character, Wave.Block_Start, Wave.Block_Finish, Done );
                        End;
                    End; { Case }
                  Press_Control_K:
                    Begin
                      Finish_Screen;
                      Control_K_Link( All, Character, Wave.Block_Start, Wave.Block_Finish, Done );
                    End;
                  Press_Control_O:
                    Case Character of
                      'W': Automatic_Paragraph_Reform := ( not Automatic_Paragraph_Reform );
                      else
                        Begin
                          Finish_Screen;
                          Control_O_Link( All, Character, Wave.Block_Start, Wave.Block_Finish, Done );
                        End;
                    End; { Case }
                  Press_Control_Q:
                    Begin
                      Finish_Screen;
                      Control_Q_Link( All, Character, Wave.Block_Start, Wave.Block_Finish, Done );
                    End;
                End; { Case }
              Until Done;
              Keyboard.Wait_Routine := Hold_Routine;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Control link default.
    This procedure is a empty procedure that the
    main program can replace to add special
    features.

*************************************************)

    {$F+}
    Procedure Control_Link_Default( Var All: All_Type; Character: Char; Var Start, Finish: Point_Type; Var Done: Boolean );
      Begin
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Allocates a working buffer.
    Initializes all the variable procedures.

*************************************************)

  Begin
    New( Working_Buffer );
    If ( Working_Buffer = Nil )
      then
        RunError( 203 );
    Help := Help_Default;
    Get_Read_File_Name := Get_File_Name_Default;
    Get_Write_File_Name := Get_File_Name_Default;
    Get_Search_Data := Get_Search_Data_Default;
    Write_Search_Failure := Write_Search_Failure_Default;
    Get_Search_And_Replace_Data := Get_Search_And_Replace_Data_Default;
    Get_Confirmation_Of_Change := Get_Confirmation_Of_Change_Default;
    Write_Wait := Write_Wait_Default;
    Write_Complete := Write_Complete_Default;
    Confirm_Changes := Confirm_Changes_Default;
    Write_Error := Write_Error_Default;
    Character_Menu := Character_Menu_Default;
    Display_Read_Status := Display_Read_Status_Default;
    Display_Write_Status := Display_Write_Status_Default;
    Display_Where_Status := Display_Where_Status_Default;
    Display_Information := Display_Information_Default;
    Get_New_Line := Get_New_Line_Default;
    Function_Key := Function_Key_Default;
    Control_J_Link := Control_Link_Default;
    Control_K_Link := Control_Link_Default;
    Control_O_Link := Control_Link_Default;
    Control_Q_Link := Control_Link_Default;
    Put_Line := Put_Text_Line;
    Look_Line := Peek_Text_Line;
    Get_Line := Get_Text_Line;
    Swap_Line := Swap_Lines;
    Escape_Operation := Escape_Operation_Default;
  End.

