(*****************************************************************************

  Over.
    Version 1.01

  This unit was designed to make it easy to overlay units with
    implementation sections.

  Purpose:
    To gather all the overlay code into a single unit.

  How it works:
    Simply place it in the uses clause before any of the overlaid units
    and it will initialize the overlay manager before the overlaid units
    are called.

  Features:
    This unit will try to load the overlays into Expanded Memory if
      available, otherwise it will use the largest block of regular
      memory necessary or the default maximum size, whichever is larger.
    This unit writes the status of the overlay manager on the screen.
    The default overlay file is expected to be included at the end of
      the execute file, otherwise it is expected to have the same name.

  Limitations:
    Can't use an overlay file of a different name than the program.

  Compilers:
    Turbo Pascal versions 5.0 to 6.0
   { Overlays are not supported in Turbo Pascal version 4.0 }
   { Overlays not supported in Speed Pascal/2 version 1.5 }

 Systems:
   MS-DOS, MDOS, OS/2 {Merely empty in OS/2}

*****************************************************************************)

Unit Over;

  Interface

   {$IFNDEF OS2}
    Uses
      Overlay;
   {$ENDIF}

    Const
     { This constant holds the minimum memory for which to increase buffer size. }
      Minimum_Memory_Allow = 100000;
     { This constant holds the default large overlay buffer increase amount. }
      Maximum_Overlay_Size = 50000;

{----------------------------------------------------------------------------}

(***********************************************************

  No procedures or functions are necessary because this
    unit is expected to perform entirely in it's
    initialization section.  That way, the overlaid units
    can call their initialization sections before the
    program starts.

***********************************************************)

  Implementation

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Expand the buffer.
    This procedure tries to put the expand the
    normal memory buffer if there is enough memory
    to handle it.

*************************************************)

    Procedure Expand_Buffer;
      Var
        Extra_Memory: LongInt;
      Begin
       {$IFNDEF OS2}
        If ( MemAvail > Minimum_Memory_Allow )
          then
            Begin
              Extra_Memory := ( MemAvail - Minimum_Memory_Allow );
              If ( Extra_Memory > Maximum_OverLay_Size )
                then
                  OvrSetBuf( OvrGetBuf + Maximum_OverLay_Size )
                else
                  OvrSetBuf( OvrGetBuf + ( Extra_Memory div 4 ) );
            End;
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Try extended memory system.
    This procedure tries to put the overlays into
    the extended memory.  If it fails, it tries
    to expand the conventional memory buffer size
    to increase the program running speed.

*************************************************)

    Procedure Try_EMS;
      Begin
       {$IFNDEF OS2}
       { We'll try to use the expanded memory system. }
        OvrInitEMS;
       { Check the results.}
        Case OvrResult of
          OvrOk:
            WriteLn( 'Using EMS memory for overlays.' );
          OvrIOError:
            WriteLn( 'Input/Output Error in Overlay File.' );
          OvrNoEMSDriver:
            Begin
              WriteLn( 'EMSDriver not installed.' );
              WriteLn( 'Using conventional memory for overlays.' );
              Expand_Buffer;
            End;
          OvrNoEMSMemory:
            Begin
              WriteLn( 'EMS memory is unavailable.' );
              WriteLn( 'Using conventional memory for overlays.' );
              Expand_Buffer;
            End;
        End; { Case }
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Over try again.
    This procedure is called if the first overlay
    file name fails.  This procedure will try and
    modify the first name, hoping to get a valid
    overlay file with the same name as that of
    the program.
    If it fails, the program is halted.

*************************************************)

    Procedure Over_Try_Again( Var Current_File_Name: String );
      Var
        Point: Integer;
      Begin
       {$IFNDEF OS2}
       { Next we'll check in the overlay file of the same name }
        Point := Pos( '.', Current_File_Name );
        If ( Point <> 0 )
          then
            Delete( Current_File_Name, Point, Length( Current_File_Name ) );
        Current_File_Name := Current_File_Name + '.Ovr';
       { Initialize the overlay file. }
        OvrInit( Current_File_Name );
       { Check the result. }
        If ( OvrResult = OvrOk )
          then
            Try_EMS
          else
            Begin
              WriteLn( 'Error:  Can''t find the overlay file!' );
              RunError( 2 );
            End;
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set things up.
    Placing the main initialization section into a
    procedure allows the default variables to be
    allocated on the stack instead of in the data
    section of the program and frees up more
    memory for the program.

*************************************************)

    Procedure Set_Things_Up;
      Var
        Overlay_File_Name: String;
      Begin
       {$IFNDEF OS2}
       { First we'll look in the back of the current program for the overlay. }
        Overlay_File_Name := ParamStr( 0 );
       { Initialize the Overlay Manager. }
        OvrInit( Overlay_File_Name );
       { Next we'll check to see what happened. }
        Case OvrResult of
          OvrOk:
           { Good! Everything is okay. }
            Try_EMS;
          OvrError,
          OvrNotFound:
           { We didn't find the overlay file }
            Over_Try_Again( Overlay_File_Name );
          OvrNoMemory:
            Begin
             { There wasn't enough memory for the overlay buffer }
              WriteLn( 'Error in main program:  Not enough memory for overlay buffer management!' );
              WriteLn( 'Please try to free up more memory and restart.' );
              RunError( 203 );
            End;
          else
            Begin
             { Something else went Wrong! }
              WriteLn( 'Error in overlay initialize sequence: ', OVRResult );
              RunError( OvrResult );
            End;
        End; { Case }
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.

*************************************************)

    Begin
      Set_Things_Up;
    End.

