(*****************************************************************************

  Unit JoyStick
    Version 1.1

    This unit contains an interface for the use of a joystick.  The standard
    joystick interface is accessed through the computer's BIOS.

    Purpose:
      To make adding joystick to a program easy.

    How it works:
      The system handles the joystick reading or the program can call the
        procedures directly.

    Features:
      When a program uses the keyboard unit, Joystick will link up with it
        automatically. ( Only with Pascal versions greater than 4.0 )

    Limitations:
      The BIOS service is not supported on the IBM-PC prior to 1-10-86 or the
        original PC/XT BIOS prior to 11-8-82.  In the event it is unsupported,
        this unit will not work.
      While both Pointer and JoyStick can work nicely together with Keyboard,
        it isn't recommended.

  Compiler:
    Turbo Pascal version 4.0 to 6.0

  System:
    MS-DOS, MDOS

*****************************************************************************)

Unit JoyStick;

  Interface

    Uses
      DOS, 
      CRT
     {$IFNDEF VER40},
      KeyBoard
     {$ENDIF};

    Const
     { This reduces the joystick's possible values and eliminates drift. }
      Adjust: Byte = 10;
     { Used to simulate the delay value of the keyboard. }
      Delay_Amount = 100;

    Type
     { This data structure returns the data. }
      Stick_Type = Record
                     X,
                     Y: Integer;
                     A,
                     B: Boolean;
                   End;

(***********************************************************

  Function: Set the joy stick.

    This function sets up the adjustment values initially.
    That way, since the values tend to float a little, they
    can be adjusted.  The value at the center can hover any-
    where between 50 and 150.  Thus we can get a negative
    value by subtracting our initial values from the new
    ones.
    It is important that this function be called with the
    joystick in the exact center.
    This function returns false if it doesn't find any
    JoySticks.

***********************************************************)

    Function Set_JoyStick: Boolean;

(***********************************************************

  Procedure: Read the joy stick.

    This function reads the status of the joystick and
    returns the results in the supplied data structures.
    It will automatically adjust the X, Y values that it
    reads in to make them integers.  A negative X value
    indicates that it was offset to the left while a
    positive value indicates it was offset to the right.
    Likewise, a negative Y values signifies up, while a
    positive Y value reflects down.
    Depending on Adjust, the values will take on a certain
    number of ranges having zero at the exact center.

***********************************************************)

    Procedure Read_JoyStick( Var Stick1, Stick2: Stick_Type );

{----------------------------------------------------------------------------}

  Implementation

    Var
     {$IFNDEF VER40}
      Old_Interface: Function: Byte;
     {$ENDIF}
     { Holds the initial values used to adjust the readings. }
      Adjustment1,
      Adjustment2: Record
                     X,
                     Y: Word;
                   End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read joystick buttons.
    This procedure only reads the joystick's
    buttons and returns them in the sticks
    variable.

*************************************************)

    Procedure Read_JoyStick_Buttons( Var Stick1, Stick2: Stick_type );
      Var
        The_Registers: Registers;
      Begin
        The_Registers.Ah := $84;
        The_Registers.Dx := 0;
        Intr( $15, The_Registers );
        Stick1.A := ( ( The_Registers.Al and $10 ) = 0 );
        Stick1.B := ( ( The_Registers.Al and $20 ) = 0 );
        Stick2.A := ( ( The_Registers.Al and $40 ) = 0 );
        Stick2.B := ( ( The_Registers.Al and $80 ) = 0 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read joystick resistors.
    This procedure only reads the joystick's
    resistors which is the joystick itself.

*************************************************)

    Procedure Read_JoyStick_Resistors( Var Stick1, Stick2: Stick_type );
      Var
        The_Registers: Registers;
      Begin
        The_Registers.Ah := $84;
        The_Registers.Dx := 1;
        Intr( $15, The_Registers );
        Stick1.X := The_Registers.Ax;
        Stick1.Y := The_Registers.Bx;
        Stick2.X := The_Registers.Cx;
        Stick2.Y := The_Registers.Dx;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Set joystick.
    As previously defined.

*************************************************)

    Function Set_JoyStick: Boolean;
      Var
        The_Registers: Registers;
      Begin
        The_Registers.Ah := $84;
        The_Registers.Dx := 1;
        Intr( $15, The_Registers );
        Adjustment1.X := The_Registers.Ax;
        Adjustment1.Y := The_Registers.Bx;
        Adjustment2.X := The_Registers.Cx;
        Adjustment2.Y := The_Registers.Dx;
        Set_JoyStick := ( Adjustment1.X <> 0 ) or ( Adjustment1.Y <> 0 ) or ( Adjustment2.X <> 0 ) or ( Adjustment2.Y <> 0 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read joystick.
    As previously defined.

*************************************************)

    Procedure Read_JoyStick( Var Stick1, Stick2: Stick_type );
      Begin
        Read_JoyStick_Buttons( Stick1, Stick2 );
        Read_JoyStick_Resistors( Stick1, Stick2 );
        With Stick1 do
          Begin
            X := ( ( X - Adjustment1.X ) div Adjust );
            Y := ( ( Y - Adjustment1.Y ) div Adjust );
          End;
        With Stick2 do
          Begin
            X := ( ( X - Adjustment2.X ) div Adjust );
            Y := ( ( Y - Adjustment2.Y ) div Adjust );
          End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: New interface.
    This function provides a new interface for
    the keyboard unit if it is used.

*************************************************)

   {$IFNDEF VER40}
   {$F+} {Generate far procedure calls: On } { Do not change! }
    Function New_Interface: Byte;
      Var
        Data: Byte;
        Stick1,
        Stick2: Stick_Type;
      Begin
        Data := Old_Interface;
        If ( Data = 0 )
          then
            Begin
              Read_JoyStick( Stick1, Stick2 );
              If ( Stick1.X < 0 )
                then
                  Begin
                    Data := Press_Left_Arrow;
                    Delay( Delay_Amount );
                  End
                else
                  If ( Stick1.X > 0 )
                    then
                      Begin
                        Data := Press_Right_Arrow;
                        Delay( Delay_Amount );
                      End
                    else
                      If ( Stick1.Y < 0 )
                        then
                          Begin
                            Data := Press_Up_Arrow;
                            Delay( Delay_Amount );
                          End
                        else
                          If ( Stick1.Y > 0 )
                            then
                              Begin
                                Data := Press_Down_Arrow;
                                Delay( Delay_Amount );
                              End
                            else
                              If Stick1.A
                                then
                                  Begin
                                    Data := Press_Enter;
                                    Delay( Delay_Amount );
                                  End
                                else
                                  If Stick1.B
                                    then
                                      Begin
                                        Data := Press_Escape;
                                        Delay( Delay_Amount );
                                      End;
            End;
        New_Interface := Data;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Sets up the joystick in case the keyboard
    unit is going to be used.

*************************************************)

    Begin
      If Set_JoyStick
        then
          Begin
            Old_Interface := Alternative_Input;
            Alternative_Input := New_Interface;
          End;
     {$ENDIF}
    End.
