{ STPLIB.PAS -- String handling routine library

  Title   : STPLIB
  Version : 5.2
  Language: Borland Turbo Pascal 4.0 through 7.0 (all targets)
            Borland Delphi 1.0 for Windows
  Date    : Dec 10,1997
  Author  : J.R. Ferguson
  Usage   : Unit

  Remarks : Declarations for MaxStp, StpTyp, StpPtr and StpInd can be found
            in DEFLIB.PAS.
            Character positions are numbered starting with 1.

}

{$B- : short-circuit Boolean expression generation }

{$DEFINE ASMLIB}

UNIT StpLib;

INTERFACE
Uses Deflib;

function StpAlloc(s: StpTyp): StpPtr;
{ Allocates memory storage for StpLen(s)+1 characters and copies the
  contents of string s to this new allocated block of memory.
  Returns a pointer to this allocated string.

  See also: <StpFree>, <StpReAlloc>.
}

procedure StpAfter(var dst: StpTyp; src,pat: StpTyp);
{ Extracts into dst the part from string src that comes after pattern pat.
  If pat is not a part of src, an empty string is returned in dst.

  See also: <StpBefore>, <StpRight>, <StpSub>
}

procedure StpBefore(var dst: StpTyp; src,pat: StpTyp);
{ Extracts into dst the part from string src that comes before pattern pat.
  If pat is not a part of src, string src is copied to dst without change.

  See also: <StpAfter>, <StpRight>, <StpSub>
}

procedure StpCat(var dst: StpTyp; src: StpTyp);
{ Appends string src to string dst.
  src and dst may be the same string variable.

  See also: <StpNCat>, <StpcCat>
}

procedure StpCenter(var s: StpTyp; n: StpInd);
{ Centers string s to be printed on a line of n characters wide by
  inserting spaces in front of it. }

function StpCmp(s1, s2: StpTyp): integer;
{ Compares strings s1 and s2, based on the ordinal value of their
  characters.
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also: <StpLexCmp>, <StpUppCmp>, <StpNCmp>, <StpLexNCmp>, <StpUppNCmp>,
            <StpSoundAlike>, <StpSoundEx>
}

procedure StpCpy(var dst: StpTyp; src: StpTyp);
{ Copies string src to dst.

  See also: <StpNCpy>, <StpcCpy>, <StpStzCpy>
}

procedure StpCreate(var s: StpTyp);
{ Creates an empty string s.

  See also: <StpMake>, <StpFill>
}

procedure StpcCat(var s: StpTyp; c: char);
{ Appends character c to string s.

  See also: <StpCat>, <StpNCat>
}

procedure StpcCpy(var s: StpTyp; c: char);
{ Returns into s a string consisting of 1 character c.

  See also: <StpCpy>, <StpNCpy>, <StpStzCpy>
}

function StpcGet(var s: StpTyp): char;
{ Returns the first character from string s and removes that character from
  string s.
  If s is an empty string, a character with the ordinal value 0 is returned.

  See also: <StpcRet>, <StpGtw>
}

procedure StpcIns(var s:StpTyp; c:char; i:StpInd);
{ Inserts character c into string s at position i.
  If i=0 the character will be inserted at the front.
  If i>StpLen(s) the character is appended to string s.

  See also: <StpIns>, <StpNIns>
}

function StpcPos(s: StpTyp; c: char): StpInd;
{ Returns the first position of character c in string s, or 0 if c can not
  be found in s.

  See also: <StpcRPos>, <StpcUppPos>, <StpcUppRPos>
            <StpPos>, <StpRPos>, <StpUppPos>, <StpUppRPos>
}

function StpcRet(s: StpTyp; i: StpInd): char;
{ Returns the character at position i in string s. String s will be left
  unchanged.
  If i=0, the first character of s is returned.
  If i>StpLen(s) a character with the ordinal value 0 is returned.

  See also: <StpcGet>, <StpGtw>
}

function StpcRPos(s: StpTyp; c: char): StpInd;
{ Returns the last position of character c in string s, or 0 if c can not
  be found in s.

  See also: <StpcPos>, <StpcUppPos>, <StpcUppRPos>
            <StpPos>, <StpRPos>, <StpUppPos>, <StpUppRPos>
}

procedure StpcUpd(var s: StpTyp; c: char; i: StpInd);
{ Replaces the character at position i in string s with character c.
  If i=0 or i>StpLen(s), string s is left unchanged.

  See also: <StpRepl>, <StpNRepl>
}

function StpcUppPos(s: StpTyp; c: char): StpInd;
{ Returns the first position of character c in string s, or 0 if c can not
  be found in s.
  No distinction is made between upper case and lower case letters.

  See also: <StpcPos>, <StpcRPos>, <StpcUppRPos>
            <StpPos>, <StpRPos>, <StpUppPos>, <StpUppRPos>
}

function StpcUppRPos(s: StpTyp; c: char): StpInd;
{ Returns the last position of character c in string s, or 0 if c can not
  be found in s.
  No distinction is made between upper case and lower case letters.

  See also: <StpcPos>, <StpcRPos>, <StpcUppPos>,
            <StpPos>, <StpRPos>, <StpUppPos>, <StpUppRPos>
}

procedure StpDel(var s: StpTyp; i,n : StpInd);
{ Deletes n characters from string s, starting at position i.
  If i=0, the result is the same as if i=1. }

procedure StpDetab(var dst: StpTyp; src: StpTyp; n: StpInd);
{ Expands tabs in string src into space groups, using n as the tab field
  width. Result in dst.
  This function recognizes the following control characters:
    HT : Horizontal tab : expand to spaces.
    BS : Back space     : decrement column position by 1.
    CR : Carriage return: reset column position to 1.
  Other characters with an ordinal value in the range 0..31 are considered
  as non-printable. They are copied without change, but don't alter the
  current column position.
  Characters with an ordinal value in the range 128..255 are considered to
  be printable, so they are copied and increment the column position by 1.
  Remarks:
  - The column positioning may be at fault when string src containts BS
    characters immediately following HT or other control characters.
  - If n=0 string src is copied to dst without change.
  - src and dst must be separate string variables.

  See also: <StpEntab> }

function StpEmpty(s: StpTyp): boolean;
{ Tests if string s is empty. }

procedure StpEntab(var dst: StpTyp; src: StpTyp; n: StpInd);
{ Replaces space groups in src by horizontal tab characters, using
  multiples of n as tab columns. Single spaces are never replaced by tabs.
  Result in dst.
  This function recognizes the following control characters:
    HT : Horizontal tab : expand to spaces.
    BS : Back space     : decrement column position by 1.
    CR : Carriage return: reset column position to 1.
  Other characters with an ordinal value in the range 0..31 are considered
  as non-printable. They are copied without change, but don't alter the
  current column position.
  Characters with an ordinal value in the range 128..255 are considered to
  be printable, so they are copied and increment the column position by 1.
  Remarks:
  - The column positioning may be at fault when string src containts BS
    characters immediately following a space group, a HT or another control
    character.
  - If n=0 string src is copied to dst without change.
  - src and dst may be the same string variables.

  See also: <StpDetab> }

procedure StpFill(var s: StpTyp; c: char; n: StpInd);
{ Fills (lengthens) string s to a length of n, by appending characters c.
  If n < StpLen(s) string s is left unchanged.

  See also: <StpCreate>, <StpMake>
}

procedure StpFree(var p: StpPtr);
{ Frees the memory block used for string p^, previously allocated with
  StpAlloc or StpRealloc, disregarding the current length of p^, and
  sets p to nil.

  See also: <StpAlloc>, <StpReAlloc>.
}

procedure StpGtw(var w,s: StpTyp);
{ Returns the first word from string s into w and removes this word from s.
  If string s does not contain a word, w and s are both made empty.

  The folowing ASCII characters are seen as word separators:
  carriage return, line feed, form feed, horizontal tab, vertical tab
  and space.

  w and s must be separate string variables.

  See also: <StpcGet>, <StpcRet>
}

procedure StpIns(var dst: StpTyp; src: StpTyp; i: StpInd);
{ Inserts string src at position i into string dst.
  If i=0 string src will be inserted in front.
  If i>StpLen(src), string src will be appended to dst.

  src and dst must be separate string variables.

  See also: <StpNIns>, <StpcIns>
}

function StpLen(s: StpTyp): StpInd;
{ Returns the number of characters in string s. }

function StpLexCmp(s1, s2: StpTyp): integer;
{ Compares strings s1 and s2, based on the ChrLib.Lexorder character order.
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also: <StpCmp>, <StpUppCmp>, <StpNCmp>, <StpLexNCmp>, <StpUppNCmp>
            <StpSoundAlike>, <StpSoundEx>
}

function StpLexNCmp(s1, s2: StpTyp; n: StpInd): integer;
{ Compares a maximum of n characters of strings s1 and s2, based on the
  ChrLib.Lexorder character order.
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also: <StpCmp>, <StpLexCmp>, <StpUppCmp>, <StpNCmp>, <StpUppNCmp>
            <StpSoundAlike>, <StpSoundEx>
}

procedure StpLexSrt(var dst: StpTyp; src: StpTyp; n: StpInd);
{ Like StpSrt, using the ChrLib.LexOrder character order.

  See also: <StpSrt>, <StpUppSrt>
}

procedure StpLow(var s: StpTyp);
{ Converts all upper case letters in string s to lower case.

  See also: <StpUpp>
}

procedure StpMake(var s: StpTyp; c: char; n: StpInd);
{ Returns into s a string consisting of n chacarters c.

  See also: <StpCreate>, <StpFill>
}

procedure StpNCat(var dst: StpTyp; src: StpTyp; n: StpInd);
{ Appends a maximum of n characters from string src to string dst.
  src and dst may be the same string variable.

  See also: <StpCat>, <StpcCat>
}

function StpNCmp(s1, s2: StpTyp; n: StpInd): integer;
{ Compares a maximum of n characters of strings s1 and s2, based on the
  ordinal values of their characters.
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also: <StpCmp>, <StpLexCmp>, <StpUppCmp>, <StpLexNCmp>, <StpUppNCmp>
            <StpSoundAlike>, <StpSoundEx>
}

procedure StpNCpy(var dst: StpTyp; src: StpTyp; n: StpInd);
{ Copies a maximum of n characters from string src into dst.

  See also: <StpCpy>, <StpcCpy>, <StpStzCpy>
}

procedure StpNIns(var dst: StpTyp; src: StpTyp; i,n: StpInd);
{ Inserts a maximum of n characters from string src at position i of string
  dst.
  If i=0 the characters are inserted in front.
  If i>StpLen(src) the characters are appended to dst.

  src and dst must be separate string variables.

  See also: <StpIns>, <StpcIns>
}

procedure StpNRepl(var dst: StpTyp; src: StpTyp; i,n: StpInd);
{ Like StfRepl, replacing a maximum of n characters of string dst.

  src and dst must be separate string variables.

  See also: <StpRepl>, <StpcUpd>
}

function StpPos(src, pat: StpTyp): StpInd;
{ Returns the first position of string pat in string src, or 0 if pat
  is not a part of src.

  See also: <StpRPos>, <StpUppPos>, <StpUppRPos>
            <StpcPos>, <StpcRPos>, <StpcUppPos>, <StpcUppRPos>
}

procedure StpRAS(var s: StpTyp);
{ Remove All Spaces: Removes all carriage return, line feed, form feed,
  horizontal tab, vertical tab and space characters from string s.

  See also: <StpRLS>, <StpRTS>
}

function StpReAlloc(var p: StpPtr; s: StpTyp): StpPtr;
{ Frees the memory block used for string p^, previously allocated with
  StpAlloc or StpRealloc, disregarding the current length of p^; allocates
  new memory storage for StpLen(s)+1 characters and copies the contents of
  string s to this new allocated block of memory.
  Returns a pointer to this newly allocated string and also sets p to
  this pointer value.

  See also: <StpAlloc>, <StpFree>.
}

procedure StpRepl(var dst: StpTyp; src: StpTyp; i: StpInd);
{ Replaces characters from dst by those of string src, starting at position
  i in dst. The resulting string dst may be longer than its original value.
  If i=0 the result is the same as if i=1.
  If i>StpLen(dst), spaces are added to dst until its length is i, and then
  string src is appended to it.

  src and dst must be separate string variables.

  See also: <StpNRepl>, <StpcUpd>
}

procedure StpRev(var s: StpTyp);
{ Reverses the character order of string s. }

procedure StpRight(var dst: StpTyp; src: StpTyp; n: StpInd);
{ Copies the n rightmost characters of string src into dst.
  If n=0 dst will be made empty.

  src and dst must be separate string variables.

  See also: <StpAfter>, <StpBefore>, <StpSub>
}

procedure StpRLS(var s: StpTyp);
{ Removes Leading Spaces: Remove all leading carriage return, line feed,
  form feed, horizontal tab, vertical tab and space characters from
  string s.

  See also: <StpRAS>, <StpRTS>
}

function StpRPos(src, pat: StpTyp): StpInd;
{ Returns the last position of string pat in string src, or 0 if pat
  is not a part of src.

  See also: <StpPos>, <StpUppPos>, <StpUppRPos>
            <StpcPos>, <StpcRPos>, <StpcUppPos>, <StpcUppRPos>
}

procedure StpRTS(var s: StpTyp);
{ Removes Trailing Spaces: Remove all trailing carriage return, line feed,
  form feed, horizontal tab, vertical tab and space characters from
  string s.

  See also: <StpRAS>, <StpRLS>
}

function StpSoundAlike(s1, s2: StpTyp; i: StpInd): boolean;
{ Tests if StpSoundEx(d1,s1,i) and StpSoundEx(d2,s2,i) yield the same
  result.

  See also: <StpSoundEx>, <StpCmp>, <StpLexCmp>, <StpUppCmp>, <StpNCmp>,
            <StpLexNCmp>, <StpUppNCmp>
}

procedure StpSoundEx(var dst: StpTyp; src: StpTyp; i: StpInd);
{ Sound Expression: Returns into dst a string that is derived from string
  src by copying the first i characters, translating lower case letters to
  upper case, and then adding the 'sound' of the remaining characters. Two
  strings that yield the same StfSoundEx result will probably sound alike
  in English. This function can therefore be used when searching a name or
  another string value in a database where the correct spelling is not
  certain.

  The exact algorithm for deriving the function result is as follows:
  1. Translate the first i characters to upper case.
  2. Translate the remaining characters to upper case, then code them
     as follows:
       'B','F','P','V'                 become '1'
       'C','G','J','K','Q','S','X','Z' become '2'
       'D','T'                         become '3'
       'L'                             become '4'
       'M','N'                         become '5'
       'R'                             become '6'
       alle other characters           are skipped.
     Moreover, never append the same code digit twice in this coding
     process.

  See also: <StpSoundAlike>, <StpCmp>, <StpLexCmp>, <StpUppCmp>, <StpNCmp>,
            <StpLexNCmp>, <StpUppNCmp>
}

procedure StpSrt(var dst: StpTyp; src: StpTyp; n: StpInd);
{ Sorts substring fields of string src, having a fixed field length n,
  and puts the result in dst.
  If n=0, string src is copied to dst without change.

  Example:  src = 'IF    THEN  BEGIN END   WHILE  REPEATDO    ', n=6
            dst = 'BEGIN DO    END   IF    REPEAT THEN  WHILE '

  src and dst must be separate string variables.

  See also: <StpLexSrt>, <StpUppSrt>
}

procedure StpStzCpy(var dst: StpTyp; const src: StzPtr);
{ Converts the type StzTyp string pointed to by src into type StpTyp string
  dst.

  See also: <StpCpy>, <StpNCpy>, <StpcCpy>
}

procedure StpSub(var dst: StpTyp; src: StpTyp; i,n: StpInd);
{ Returns into dst a substring from s, consisting of a maximum of
  n characters starting at position i.
  If i=0 the result is the same as if i=1.

  src and dst must be separate string variables.

  See also: <StpAfter>, <StpBefore>, <StpRight>
}

procedure StpTrunc(var s: StpTyp; i: StpInd);
{ Removes all characters in string s after position i.
  If n=0 an empty string is returned.
  If n>StpLen(s) string s is returned unaltered.
}

procedure StpUpp(var s: StpTyp);
{ Converts all upper case letters in string s to lower case.

  See also: <StpLow>
}

function StpUppCmp(s1, s2: StpTyp): integer;
{ Compares strings s1 and s2, based on the ChrLib.UppOrder character order.
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also: <StpCmp>, <StpLexCmp>, <StpNCmp>, <StpLexNCmp>, <StpUppNCmp>
            <StpSoundAlike>, <StpSoundEx>
}

function StpUppNCmp(s1, s2: StpTyp; n: StpInd): integer;
{ Compares a maximum of n characters from strings s1 and s2, based on the
  ChrLib.UppOrder character order.
  The result is negative if s1<s2, zero if s1=s2 and positive if s1>s2.

  See also: <StpCmp>, <StpLexCmp>, <StpUppCmp>, <StpNCmp>, <StpLexNCmp>
            <StpSoundAlike>, <StpSoundEx>
}

function StpUppPos(src, pat: StpTyp): StpInd;
{ Returns the first position of string pat in string src, or 0 if pat
  is not a part of src.
  No distinction is made between upper case and lower case letters.

  See also: <StpPos>, <StpRPos>, <StpUppRPos>
            <StpcPos>, <StpcRPos>, <StpcUppPos>, <StpcUppRPos>
}

function StpUppRPos(src, pat: StpTyp): StpInd;
{ Returns the last position of string pat in string src, or 0 if pat
  is not a part of src.
  No distinction is made between upper case and lower case letters.

  See also: <StpPos>, <StpRPos>, <StpUppPos>,
            <StpcPos>, <StpcRPos>, <StpcUppPos>, <StpcUppRPos>
}

procedure StpUppSrt(var dst: StpTyp; src: StpTyp; n: StpInd);
{ Like StpSrt, based on the ChrLib.UppOrder character order.

  See also: <StpSrt>, <StpLexSrt>
}



IMPLEMENTATION
Uses ChrLib;

{$IFDEF ASMLIB}
{$L STPLIB1}
{$ENDIF}

const
  NUL = chr(AsciiNUL); { #000 }
  BS  = chr(AsciiBS) ; { #008 }
  HT  = chr(AsciiHT) ; { #009 }
  CR  = chr(AsciiCR) ; { #013 }

  SoundChr: record case boolean of
             false: (AA,AB,AC,AD: array[ $00.. $3F] of char);
             true : (A          : array[#000..#255] of char);
            end
      =  {0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF}
{0-3}(AA:'                                                                ';
{4-7} AB:'  123 12  22455 12623 1 2 2       123 12  22455 12623 1 2 2     ';
{8-B} AC:'                                                                ';
{C-F} AD:'                                                                ');

function StpAlloc(s: StpTyp): StpPtr;
var p: StpPtr;
begin
  GetMem(p,Length(s)+1);
  if p<>nil then p^:= s;
  StpAlloc:= p;
end;

procedure StpAfter(var dst: StpTyp; src,pat: StpTyp);
var i : StpInd;
begin
  i:= Pos(pat,src);
  if i=0 then dst:= '' else StpSub(dst,src,i+Length(pat),Length(src));
end;

procedure StpBefore(var dst: StpTyp; src,pat: StpTyp);
var i: StpInd;
begin
  i:= Pos(pat,src);
  if i=0 then dst:= src else StpSub(dst,src,1,i-1);
end;

procedure StpCat(var dst: StpTyp; src: StpTyp);
begin dst:=dst+src; end;

procedure StpCenter(var s: StpTyp; n: StpInd);
var tmp: StpTyp;
begin
  StpRLS(s); StpRTS(s);
  if n>Length(s) then begin
    StpMake(tmp,' ',(n-Length(s)) div 2);
    StpIns(s,tmp,1);
  end;
end;

function StpCmp(s1, s2: StpTyp): integer;
{$IFDEF ASMLIB}
external;
{$ELSE}
begin
  if s1>s2 then StpCmp:=1 else if s1=s2 then StpCmp:=0 else StpCmp:=-1;
end;
{$ENDIF}

procedure StpCpy(var dst: StpTyp; src: StpTyp);
begin dst:=src; end;

procedure StpCreate(var s: StpTyp);
begin s[0]:= NUL; end;

procedure StpcCat(var s: StpTyp; c: char);
{$IFDEF ASMLIB}
external;
{$ELSE}
begin s:=s+c; end;
{$ENDIF}

procedure StpcCpy(var s: StpTyp; c: char);
begin s:= c; end;

function StpcGet(var s: StpTyp): char;
{$IFDEF ASMLIB}
external;
{$ELSE}
var c: char;
begin c:=StpcRet(s,1); StpDel(s,1,1); StpcGet:=c; end;
{$ENDIF}

procedure StpcIns(var s:StpTyp; c:char; i:StpInd);
begin StpIns(s,c,i); end;

function StpcPos(s: StpTyp; c: char): StpInd;
{$IFDEF ASMLIB}
external;
{$ELSE}
begin Stpcpos:=Pos(c,s); end;
{$ENDIF}

function StpcRet(s: StpTyp; i: StpInd): char;
{$IFDEF ASMLIB}
external;
{$ELSE}
begin
 if i=0 then i:=1;
 if i>Length(s) then StpcRet:=NUL else StpcRet:=s[i];
end;
{$ENDIF}

function StpcRPos(s: StpTyp; c: char): StpInd;
{$IFDEF ASMLIB}
external;
{$ELSE}
var i: StpInd;
 function niet: boolean;
 begin if i=0 then niet:=false else niet:=s[i]<>c; end;
 begin i:=Length(s); while niet do i:=pred(i); StpcRPos:=i; end;
{$ENDIF}

procedure StpcUpd(var s: StpTyp; c: char; i: StpInd);
{$IFDEF ASMLIB}
external;
{$ELSE}
begin if (i>0) and (i<=Length(s)) then s[i]:=c; end;
{$ENDIF}

function StpcUppPos(s: StpTyp; c: char): StpInd;
{$IFDEF ASMLIB}
external;
{$ELSE}
begin
  StpUpp(s);
  StpcUppPos:=Pos(ToUpper(c),s);
end;
{$ENDIF}

function StpcUppRPos(s: StpTyp; c: char): StpInd;
begin
  StpUpp(s);
  StpcUppRPos:= StpcRPos(s,ToUpper(c));
end;

procedure StpDel(var s: StpTyp; i,n : StpInd);
begin if i=0 then i:=1; Delete(s,i,n); end;

procedure StpDetab(var dst: StpTyp; src: StpTyp; n: StpInd);
var i : StpInd;   { src index               }
    k : integer;  { current column position }
    c : char;     { current character       }
begin if n = 0 then dst:= src else begin
  i:= 0; k:= 0; dst:= '';
  while i < Length(src) do begin
    inc(i); c:= src[i];
    case c of
      HT : repeat dst:= dst + ' '; inc(k); until (k mod n) = 0;
      CR : begin  dst:= dst + CR ; k:= 0; end;
      BS : begin  dst:= dst + BS ; if k>0 then dec(k); end;
      else begin  dst:= dst + c  ; if not IsCntrl(c) then inc(k); end;
    end;
  end;
end end;

function StpEmpty(s: StpTyp): boolean;
begin StpEmpty:= s[0]=NUL; end;

procedure StpEntab(var dst: StpTyp; src: StpTyp; n: StpInd);
var
  tmp: StpTyp;
  i  : StpInd;     { tmp index                                        }
  k0 : integer;    { starting column position of the last space group }
  k1 : integer;    { current column position                          }
  c  : char;       { current character                                }
begin if n = 0 then dst:= src else begin
  { pass 1, src -> tmp : replace tabs by space groups }
  StpDetab(tmp,src,n);
  { pass 2, tmp -> dst : replace space groups by tabs }
  k0:= 0; k1:= 0; i:= 0; dst:= '';
  while i < Length(tmp) do begin
    inc(i); c:= tmp[i];
    case c of
      ' ' : begin
              inc(k1);
              if (k1 mod n = 0) then begin
                if k1 - k0 > 1 then c:= HT;
                dst:= dst + c; k0:= k1;
              end;
            end;
      BS  : begin
              dst:= dst + BS;
              if k1 > 0  then dec(k1);
              if k0 > k1 then k0:= k1;
            end;
      CR  : begin
              dst:= dst + CR;
              k0:= 0; k1:= 0;
            end;
      else  begin
              while k0 < k1 do begin dst:= dst + ' '; inc(k0); end;
              dst:= dst + c;
              if not IsCntrl(c) then inc(k0);
              k1:= k0;
            end;
    end;
  end;
  while k0 < k1 do begin dst:= dst + ' '; inc(k0); end;
end end;

procedure StpFill(var s: StpTyp; c: char; n: StpInd);
{$IFDEF ASMLIB}
external;
{$ELSE}
var i: StpInd;
begin
  i:= Length(s);
  while i<n do begin inc(i); s[i]:= c; end;
  s[0]:= chr(i);
end;
{$ENDIF}

procedure StpFree(var p: StpPtr);
begin if p<>nil then begin FreeMem(p,Length(p^)+1); p:= nil; end; end;

procedure StpGtw(var w,s: StpTyp);
{$IFDEF ASMLIB}
external;
{$ELSE}
var c: char;
begin
 repeat c:=StpcGet(s); until not IsSpace(c); w:='';
 while (c<>NUL) and not IsSpace(c) do begin w:=w+c; c:=StpcGet(s); end;
 if c<>NUL then s:=c+s;
end;
{$ENDIF}

procedure StpIns(var dst: StpTyp; src: StpTyp; i: StpInd);
begin if i=0 then i:=1; Insert(src,dst,i); end;

function StpLen(s: StpTyp): StpInd;
begin StpLen:= ord(s[0]); end;

function StpLexCmp(s1, s2: StpTyp): integer;
{$IFDEF ASMLIB}
external;
{$ELSE}
var i,n: StpInd; gelijk: boolean;
begin
 if Length(s1)<Length(s2) then n:=Length(s1) else n:=Length(s2);
 i:=0; gelijk:=true;
 while gelijk and (i<n) do begin
  i:=succ(i); gelijk:=UpCase(ToAscii(s1[i]))=UpCase(ToAscii(s2[i]));
 end;
 if gelijk then StpLexCmp:=Length(s1)-Length(s2)
           else StpLexCmp:=LexOrder(s1[i],s2[i]);
end;
{$ENDIF}

function StpLexNCmp(s1, s2: StpTyp; n: StpInd): integer;
{$IFDEF ASMLIB}
external;
{$ELSE}
begin StpLexNCmp:=StpLexCmp(Copy(s1,1,n),Copy(s2,1,n)); end;
{$ENDIF}

procedure StpLexSrt(var dst: StpTyp; src: StpTyp; n: StpInd);
var i,k: StpInd;
begin if (n=0) or (n>=Length(src)) then dst:=src else begin
 dst:='';
 while src<>'' do begin
  k:=1; i:=n+1;
  while i<=Length(src) do begin
   if StpLexCmp(Copy(src,i,n),Copy(src,k,n))<0 then k:=i; i:=i+n;
  end;
  dst:=dst+Copy(src,k,n); Delete(src,k,n);
 end
end end;

procedure StpLow(var s: StpTyp);
{$IFDEF ASMLIB}
external;
{$ELSE}
var i: StpInd;
begin for i:=1 to Length(s) do s[i]:=ToLower(s[i]); end;
{$ENDIF}

procedure StpMake(var s: StpTyp; c: char; n: StpInd);
begin s:= ''; StpFill(s,c,n); end;

procedure StpNCat(var dst: StpTyp; src: StpTyp; n: StpInd);
{$IFDEF ASMLIB}
external;
{$ELSE}
begin dst:=dst+Copy(src,1,n); end;
{$ENDIF}

function StpNCmp(s1, s2: StpTyp; n: StpInd): integer;
{$IFDEF ASMLIB}
external;
{$ELSE}
begin StpNCmp:=StpCmp(Copy(s1,1,n),Copy(s2,1,n)); end;
{$ENDIF}

procedure StpNCpy(var dst: StpTyp; src: StpTyp; n: StpInd);
begin StpSub(dst,src,1,n); end;

procedure StpNIns(var dst: StpTyp; src: StpTyp; i,n: StpInd);
begin StpIns(dst,Copy(src,1,n),i); end;

procedure StpNRepl(var dst: StpTyp; src: StpTyp; i,n: StpInd);
var w: integer;
begin
  if n>Length(src) then n:= Length(src);
  w:= i+n-1; if w<0 then w:= 0 else if w>MaxStp then w:= MaxStp;
  StpFill(dst,' ',w);
  StpDel(dst,i,n);
  StpNIns(dst,src,i,n);
end;

function StpPos(src, pat: StpTyp): StpInd;
begin if src[0]=NUL then StpPos:=0 else StpPos:=Pos(pat,src); end;

procedure StpRAS(var s: StpTyp);
{$IFDEF ASMLIB}
external;
{$ELSE}
var i: StpInd;
begin
 i:=Length(s);
 while i>0 do begin if IsSpace(s[i]) then Delete(s,i,1); i:=i-1; end
end;
{$ENDIF}

function StpReAlloc(var p: StpPtr; s: StpTyp): StpPtr;
begin StpFree(p); p:= StpAlloc(s); StpReAlloc:= p; end;

procedure StpRepl(var dst: StpTyp; src: StpTyp; i: StpInd);
begin StpNRepl(dst,src,i,Length(src)); end;

procedure StpRev(var s: StpTyp);
{$IFDEF ASMLIB}
external;
{$ELSE}
var l,r: StpInd; c: char;
begin
 l:=1; r:=Length(s);
 while l<r do begin
   c:=s[l]; s[l]:=s[r]; s[r]:=c;
   l:=succ(l); r:=pred(r);
 end
end;
{$ENDIF}

procedure StpRight(var dst: StpTyp; src: StpTyp; n: StpInd);
var i: StpInd;
begin
  if n>=Length(src) then i:=1 else i:=Length(src)-n+1;
  StpSub(dst,src,i,n);
end;

procedure StpRLS(var s: StpTyp);
{$IFDEF ASMLIB}
external;
{$ELSE}
var i: StpInd;
begin
  i:= 1; while (i <= Length(s)) and IsSpace(s[i]) do inc(i);
  StpDel(s,1,i-1);
end;
{$ENDIF}

function StpRPos(src, pat: StpTyp): StpInd;
{$IFDEF ASMLIB}
external;
{$ELSE}
{ VERY brute-force}
var i0,i: StpInd; tmp: StpTyp;
begin
  i0:= 0; i:= StpPos(src,pat);
  while i<>i0 do begin
    i0:= i; StpSub(tmp,src,i+1,MaxStp);
    i:= i0+StpPos(tmp,pat);
  end;
  StpRPos:= i0;
end;
{$ENDIF}

procedure StpRTS(var s: StpTyp);
{$IFDEF ASMLIB}
external;
{$ELSE}
begin
  while (ord(s[0])>0) and IsSpace(s[ord(s[0])]) do Dec(s[0]);
end;
{$ENDIF}

function StpSoundAlike(s1, s2: StpTyp; i: StpInd): boolean;
var d1,d2: StpTyp;
begin
  StpSoundEx(d1,s1,i);
  StpSoundEx(d2,s2,i);
  StpSoundAlike:= d1=d2;
end;

procedure StpSoundEx(var dst: StpTyp; src: StpTyp; i: StpInd);
var i0,i1: StpInd; c: char;
begin
  if i > StpInd(src[0]) then i:= StpInd(src[0]);
  dst[0]:= #0; for i0:= 1 to i do dst[i0]:= ToUpper(src[i0]);
  i0:= i; i1:= i;
  while i0 < StpInd(src[0]) do begin
    Inc(i0);
    c:= SoundChr.A[src[i0]];
    if (c<>' ') and (c<>dst[i1]) then begin
      Inc(i1); dst[i1]:= c;
    end;
  end;
  dst[0]:= chr(i1);
end;

procedure StpSrt(var dst: StpTyp; src: StpTyp; n: StpInd);
var i,k: StpInd;
begin if (n=0) or (n>=Length(src)) then dst:=src else begin
 dst:='';
 while src<>'' do begin
  k:=1; i:=n+1;
  while i<=Length(src) do begin
    if Copy(src,i,n) < Copy(src,k,n) then k:=i;
    i:=i+n;
  end;
  dst:=dst+Copy(src,k,n); Delete(src,k,n);
 end
end end;

procedure StpStzCpy(var dst: StpTyp; const src: StzPtr);
{$IFDEF ASMLIB}
external;
{$ELSE}
var p: PChar; n: StpInd;
begin
  n:= 0;
  if src <> nil then begin
    p:= src;
    while (p^ <> NUL) and (n < MaxStp) do begin
      Inc(n); dst[n]:= p^; Inc(p);
    end;
  end;
  dst[0]:= chr(n);
end;
{$ENDIF}

procedure StpSub(var dst: StpTyp; src: StpTyp; i,n: StpInd);
{$IFDEF ASMLIB}
external;
{$ELSE}
begin if i=0 then i:=1; dst:=Copy(src,i,n); end;
{$ENDIF}

procedure StpTrunc(var s: StpTyp; i: StpInd);
begin if i<Length(s) then s[0]:= chr(i); end;

procedure StpUpp(var s: StpTyp);
{$IFDEF ASMLIB}
external;
{$ELSE}
var i: StpInd;
begin for i:=1 to Length(s) do s[i]:=UpCase(s[i]); end;
{$ENDIF}

function StpUppCmp(s1, s2: StpTyp): integer;
{$IFDEF ASMLIB}
external;
{$ELSE}
var i,n: StpInd; gelijk: boolean;
begin
 if Length(s1)<Length(s2) then n:=Length(s1) else n:=Length(s2);
 i:=0; gelijk:=true;
 while gelijk and (i<n) do begin
  i:=succ(i); gelijk:=UpCase(s1[i])=UpCase(s2[i]);
 end;
 if gelijk then StpUppCmp:=Length(s1)-Length(s2)
           else StpUppCmp:=UppOrder(s1[i],s2[i]);
end;
{$ENDIF}

function StpUppNCmp(s1, s2: StpTyp; n: StpInd): integer;
{$IFDEF ASMLIB}
external;
{$ELSE}
begin StpUppNCmp:=StpUppCmp(Copy(s1,1,n),Copy(s2,1,n)); end;
{$ENDIF}

function StpUppPos(src, pat: StpTyp): StpInd;
begin
  StpUpp(src); StpUpp(pat);
  StpUppPos:= StpPos(src,pat);
end;

function StpUppRPos(src, pat: StpTyp): StpInd;
begin
  StpUpp(src); StpUpp(pat);
  StpUppRPos:= StpRPos(src,pat);
end;

procedure StpUppSrt(var dst: StpTyp; src: StpTyp; n: StpInd);
var i,k: StpInd;
begin if (n=0) or (n>=Length(src)) then dst:=src else begin
 dst:='';
 while src<>'' do begin
  k:=1; i:=n+1;
  while i<=Length(src) do begin
   if StpUppCmp(Copy(src,i,n),Copy(src,k,n))<0 then k:=i; i:=i+n;
  end;
  dst:=dst+Copy(src,k,n); Delete(src,k,n);
 end
end end;

END.
