unit GVisible;

{ Unit GVisible, Graphics Vision Visibility Detection & Drawing Support,
  Copyright 1994,1996 Matthias Kppe.
}

{$G+,X+}

interface

uses Objects, GvViews;

type

{ Visibility abstract data structure
}
  PVis = pointer;

{ Operations on visibility structures
}
function NewVis(var Bounds: TRect): PVis;
procedure SubtractRegion(Vis: PVis; var Bounds: TRect);
procedure IndirectRegion(Vis: PVis; var Bounds: TRect; Transparent: PGView);
procedure Unindirect(Vis: PVis; GOwner: PGView);
function Unraw(Vis: PVis): PVis;
procedure ClipVis(Vis: PVis; var Clip: TRect);
procedure SubtractVis(Vis, SubVis: PVis);
function IntersectVis(Vis1, Vis2: PVis): PVis;
function IntersectDelta(Vis: PVis; Delta: TPoint): PVis;
procedure MoveVis(Vis: PVis; Delta: TPoint);
function SortVis(Vis: PVis; Delta: TPoint): PVis;
function CopyVis(Vis: PVis): PVis;
procedure FreeVis(Vis: PVis);
procedure SubtractHiddenSources(Vis: PVis; Delta: TPoint);

{ Visibility drawing support
}
procedure DrawVis(Vis: PVis; Origin: TPoint; Self: PGView);
procedure DrawVisLocal(Vis: PVis; Origin: TPoint;
  Self: PGView; Local: pointer; Frame: Word);
procedure DrawFirstPass(P: PGView; Self: PGView);

{ Drawing operation frame
}
function BeginDraw(Vis: PVis; Origin: TPoint;
  Self: PGView; Modifying: Boolean): Boolean;
function EndDraw(Vis: PVis; Self: PGView): Boolean;

{ Scroll view proc
}
type
  PRect = ^TRect;

procedure CritUnionDelta(Delta: TPoint);
procedure ScrollView(Self: PGView; Delta: TPoint; SubRect: PRect);
procedure ScrollGroup(Self: PGGroup; Delta: TPoint);

{ Utility macros
}
function PrevBP: Word;
inline(
  $8B/ $46/ $00 { mov ax, [bp] }
);

function CurBP: Word;
inline(
  $8B/ $C5 { mov ax, bp }
);

const
  GlobalOptions: Word = $FFFF {and not (ofBuffer + ofMetafile)};

implementation

{$ifdef Windows}
uses WinTypes, WinProcs, WinGr, VgaMem;
{$else}
uses Gr, MyMouse, VgaMem, MetaGr;
{$endif}

{ Primitives 
}

function Min(x, y: Integer): Integer; assembler;
asm
	MOV	AX, X
	CMP	AX, Y
	JLE	@@1
	MOV	AX, Y
@@1:
end;

function Max(x, y: Integer): Integer; assembler;
asm
	MOV	AX, X
	CMP 	AX, Y
	JGE	@@1
	MOV	AX, Y
@@1:
end;

function Intersect(var Dest, Source: Objects.TRect): Boolean;
{ Returns true iff Dest and Source intersect.
  Dest is the intersection rectangle iff there is one.
  & faster implementation &
}
begin
  Dest.Intersect(Source);
  Intersect := not Dest.Empty
end;

function DoesIntersect(Dest: Objects.TRect; var Source: Objects.TRect): Boolean;
{ Returns true iff Dest and Source intersect.
  & faster implementation &
}
begin
  DoesIntersect := Intersect(Dest, Source)
end;

{ TViewCollection object 
}

type
  PViewCollection = ^TViewCollection;
  TViewCollection = object(TCollection)
    constructor CopyOf(C: PCollection);
    constructor Join(C1, C2: PCollection);
    procedure FreeItem(Item: pointer); virtual;
  end;

constructor TViewCollection.CopyOf(C: PCollection);
begin
  TObject.Init;
  If C = nil
  then begin
    Fail;
    Exit
  end;
  Count := C^.Count;
  Limit := C^.Limit;
  Delta := C^.Delta;
  GetMem(Items, Limit * SizeOf(pointer));
  Move(C^.Items^, Items^, Limit * SizeOf(pointer))
end;

constructor TViewCollection.Join(C1, C2: PCollection);

	procedure Ins(C2Item: pointer); far;

		function Matches(C1Item: pointer): Boolean; far;
		begin
		  Matches := C1Item = C2Item
		end;

	begin
	  If C1^.FirstThat(@Matches) = nil
	  then Insert(C2Item)
	end;

begin
  If (C1 = nil) and (C2 = nil) then Fail else
  if C1 = nil then CopyOf(C2) else
  if C2 = nil then CopyOf(C1)
  else begin
    CopyOf(C1);
    C2^.ForEach(@Ins)
  end
end;

procedure TViewCollection.FreeItem(Item: pointer);
begin
end;

{ Data structure
}
{$ifdef Windows}
type
  PRectVis = ^TRectVis;
  TRectVis = record
    Region: HRgn;
    CurTrans: Integer;
    Trans: PCollection;
    DrawMode: Word;
  end;
{$else}
const
  maxRect = 16;

type
  PPVRect = ^PVRect;
  PVRect = ^TVRect;
  TVRect = record
    Count: Integer;
    Next: PVRect;
    R: array[0..maxRect-1] of TRect;
  end;

type
  PRectVis = ^TRectVis;
  TRectVis = record
    TheBounds: TRect;
    List: PVRect;
    CurTrans: Integer;
    Trans: PCollection;
    case DrawMode: Word of
      1: (VgaBuf: TVgaBuf);
      2: (MetaBuf: pointer);
      3: ();
  end;
{$endif}

function RawCopyVis(Vis: PVis): PVis; forward;

function CopyVis(Vis: PVis): PVis;
var
  Copy: PRectVis;
begin
  CopyVis := nil;
  If Vis = nil then Exit;
  Copy := RawCopyVis(Vis);
  Copy^.Trans := New(PViewCollection, CopyOf(PRectVis(Vis)^.Trans));
  Copy^.CurTrans := PRectVis(Vis)^.CurTrans;
  CopyVis := Copy
end;

{ Transparency support 
}

procedure IndirectRegion(Vis: PVis; var Bounds: Objects.TRect; Transparent: PGView);
begin
  If (Vis = nil) {$ifndef Windows}
     or not DoesIntersect(PRectVis(Vis)^.TheBounds, Bounds) {$endif}
  then Exit;
  with PRectVis(Vis)^ do
  begin
    If CurTrans < 0 then Exit;          { Raw mode }
    If Transparent = nil
    then begin
      If Trans <> nil
      then CurTrans := Trans^.Count
    end
    else begin
      If Trans = nil
      then begin
	Trans := New(PViewCollection, Init(2, 8));
	CurTrans := 0
      end;
      Trans^.AtInsert(CurTrans, Transparent)
    end
  end
end;

procedure Unindirect(Vis: PVis; GOwner: PGView);
var
  i: Integer;
begin
  If Vis <> nil then
  with PRectVis(Vis)^ do
  begin
    If Trans <> nil then
    with Trans^ do
    begin
      For i := Count - 1 downto 0 do
	if PGView(PGView(At(i))^.GOwner) = GOwner
	then AtFree(i);
      If CurTrans > Count then CurTrans := Count;
    end
  end
end;

function Unraw(Vis: PVis): PVis;
begin
  If Vis <> nil then
  with PRectVis(Vis)^ do
  begin
    DrawMode := $FFFF;
  end;
  Unraw := Vis
end;

function HasTrans(Vis: PVis): Boolean;

	function IsTrans(View: PGView): Boolean; far;
	begin
	  IsTrans := View^.State and sfTransparent <> 0
	end;

begin
  HasTrans := (Vis <> nil) and (PRectVis(Vis)^.Trans <> nil) and
    (PRectVis(Vis)^.Trans^.FirstThat(@IsTrans) <> nil)
end;

function CondRawCopyVis(Vis: PVis): PVis;
var
  Copy: PVis;
begin
  Copy := RawCopyVis(Vis);
  If PRectVis(Vis)^.DrawMode = $FFFF
  then PRectVis(Copy)^.CurTrans := 0;
  CondRawCopyVis := Copy
end;

procedure HideTrans(Vis: PVis; Self: PGView);
var
  i: Integer;
  P: PGView;
  SaveState: Word;
begin
  If Vis = nil then Exit;
  with PRectVis(Vis)^ do
  begin
    If Trans <> nil then
      with Trans^ do
	For i := Count - 1 downto 0 do
	begin
	  P := PGView(At(i));
	  with P^ do
	    If State and sfTransparent <> 0
	    then HideDrawClipped(CondRawCopyVis(Vis), Self)
	end;
  end;
end;

procedure ShowTrans(Vis: PVis; Self: PGView);
var
  i: Integer;
  P: PGView;
begin
  If Vis = nil then Exit;
  with PRectVis(Vis)^ do
  begin
    If Trans <> nil then
      with Trans^ do
	For i := 0 to Count - 1 do
	begin
	  P := PGView(At(i));
	  with P^ do
	  begin
	    If State and sfTransparent <> 0
	    then DrawClipped(RawCopyVis(Vis), P);
	    FreeBack
	  end
	end
  end;
end;

procedure FreeBacks(Vis: PVis; Self: PGView);
var
  i: Integer;
  P: PGView;
begin
  If Vis = nil then Exit;
  with PRectVis(Vis)^ do
  begin
    If Trans <> nil then
      with Trans^ do
	For i := 0 to Count - 1 do
	begin
	  P := PGView(At(i));
	  with P^ do
	    FreeBack;
	end
  end;
end;

procedure DrawTransBack(Vis: PVis; Self: PGView);
{ Draw Trans' background
}
	procedure SetVis(Trans: PCollection; Enable: Boolean);
        var
          i: Integer;
	begin
	  with Trans^ do
	    For i := 0 to Count - 1 do
              with PGView(At(i))^ do
		If State and sfTransparent <> 0 then
                if Enable
                then State := State or sfVisible
                else State := State and not sfVisible
	end;

begin
  If Vis = nil then Exit;
  with PRectVis(Vis)^ do
    If Trans <> nil
    then begin
      SetVis(Trans, false);
      DrawFirstPass(Self^.GOwner, Self);
      SetVis(Trans, true)
    end
end;

procedure DrawTrans(Vis: PVis; Self: PGView);
{ Draw Trans by first-pass
}
var
  i: Integer;
  P: PGView;
begin
  If Vis = nil then Exit;
  with PRectVis(Vis)^ do
  begin
    If Trans <> nil then
      with Trans^ do
	For i := 0 to Count - 1 do
	begin
	  P := PGView(At(i));
	  with P^ do
	  begin
	    If State and sfTransparent <> 0
	    then DrawFirstPass(P, Self);
	    FreeBack
	  end
	end
  end;
end;

{$ifdef Windows}
{$i gviswin.pas}
{$else}

{$ifdef gvisible_v1}
{$i gvisshap.pas}
{$else}

{ Rect-list handling, adapted from GvViews 
}

function NextAvailable(var List: PVRect): PRect;
var
  P: PVRect;
Begin
  If (List = nil) or (List^.Count >= maxRect) then Begin
    New(P);
    with P^ do Begin
      Count := 0;
      Next := List
    End;
    List := P
  End;
  with List^ do
    NextAvailable := @R[Count]
End;

procedure InsNewRect(var List: PVRect; var Rect: TRect);
Begin
  NextAvailable(List)^ := Rect;
  Inc(List^.Count)
End;

procedure DoSubtractRect(var List: PVRect; Rect: PRect;
  R1: TRect);
var
  NewR, R2: TRect;
  atNewR: PRect;
  IsNewRect: Boolean;
Begin
  R2 := Rect^;
  If (R1.B.X <= R2.A.X) or (R1.B.Y <= R2.A.Y) or
     (R1.A.X >= R2.B.X) or (R1.A.Y >= R2.B.Y) or (R2.A.X = R2.B.X) or
     (R2.A.Y = R2.B.Y) then Exit;

  Rect^.Assign(0, 0, 0, 0);
  IsNewRect := false;
  atNewR := @NewR;

  If R1.A.X > R2.A.X then Begin
    asm
	les	di, atNewR
	cld

	mov	ax, R2.A.X
	stosw

	mov	ax, R1.A.Y
	mov	dx, R2.A.Y
	cmp	ax, dx
	jg	@@1
	mov	ax, dx
@@1:	stosw

	mov	ax, R1.A.X
	stosw

	mov	ax, R1.B.Y
	mov	dx, R2.B.Y
	cmp	ax, dx
	jl	@@2
	mov	ax, dx
@@2:	stosw
    end;
    Rect^ := NewR;
    Rect := NextAvailable(List);
    IsNewRect := true;
  End;
  If R1.A.Y > R2.A.Y then Begin
    asm
	les	di, atNewR
	cld

	mov	ax, R2.A.X
	stosw
	mov	ax, R2.A.Y
	stosw
	mov	ax, R2.B.X
	stosw
	mov	ax, R1.A.Y
	stosw
    end;
    Rect^ := NewR;
    If IsNewRect then Inc(List^.Count);
    Rect := NextAvailable(List);
    IsNewRect := true
  End;
  If R1.B.X < R2.B.X then Begin
    asm
	les	di, atNewR
	cld

	mov	ax, R1.B.X
	stosw

	mov	ax, R1.A.Y
	mov	dx, R2.A.Y
	cmp	ax, dx
	jg	@@3
	mov	ax, dx
@@3:	stosw

	mov	ax, R2.B.X
	stosw

	mov	ax, R1.B.Y
	mov	dx, R2.B.Y
	cmp	ax, dx
	jl	@@4
	mov	ax, dx
@@4:	stosw
    end;
    Rect^ := NewR;
    If IsNewRect then Inc(List^.Count);
    Rect := NextAvailable(List);
    IsNewRect := true
  End;
  If R1.B.Y < R2.B.Y then Begin
    asm
	les	di, atNewR
	cld

	mov	ax, R2.A.X
	stosw
	mov	ax, R1.B.Y
	stosw
	mov	ax, R2.B.X
	stosw
	mov	ax, R2.B.Y
	stosw
    end;
    Rect^ := NewR;
    If IsNewRect then Inc(List^.Count)
  End;
End;

procedure SubtractRect(var List: PVRect; var Rect: TRect);
var
  P: PVRect;
  i: Integer;
Begin
  P := List;
  while P <> nil do Begin
    For i := 0 to P^.Count - 1 do
      DoSubtractRect(List, @P^.R[i], Rect);
    P := P^.Next
  End
End;

{ A ForEachRect implementation *MUST* not call the action proc with an
  empty rectangle.
  The action proc must be near-encoded.
}
procedure ForEachRect(List: PVRect; Action: pointer); near; external;

{$L gvisible.obj   (gvisible.asm) }

procedure ClipList(List: PVRect; var Clip: TRect);

 procedure Intersect(var Rect: TRect);
 Begin
   Rect.Intersect(Clip)
 End;

Begin
  ForEachRect(List, @Intersect)
End;

function IntersectLists(List1, List2: PVRect): PVRect;
var
  List: PVRect;

 procedure IntersectList1(var Rect1: TRect);

  procedure IntersectList2(var Rect2: TRect);
  var
    R: TRect;
  Begin
    R := Rect1;
    R.Intersect(Rect2);
    If not R.Empty then InsNewRect(List, R)
  End;

 Begin
   ForEachRect(List2, @IntersectList2)
 End;

Begin
  List := nil;
  ForEachRect(List1, @IntersectList1);
  IntersectLists := List
End;

function IntersectListDelta(List: PVRect; Delta: TPoint): PVRect;
var
  IList: PVRect;

 procedure DoIntersectDelta(var Rect: TRect);
 var
   R: TRect;
 Begin
   R := Rect;
   R.Move(Delta.x, Delta.y);
   R.Intersect(Rect);
   If not R.Empty then InsNewRect(IList, R)
 End;

Begin
  IList := nil;
  ForEachRect(List, @DoIntersectDelta);
  IntersectListDelta := IList
End;

procedure MoveList(List: PVRect; Delta: TPoint);

 procedure DoMove(var Rect: TRect);
 Begin
   Rect.Move(Delta.X, Delta.Y)
 End;

Begin
  ForEachRect(List, @DoMove)
End;

{$endif}

procedure FreeRectList(List: PVRect);
var
  Next: PVRect;
Begin
  while List <> nil do Begin
    Next := List^.Next;
    Dispose(List);
    List := Next
  End
End;

{ Operations using ForEachRect 
}

{ UniteList stores the bounding rectangle into Bounds, and returns the
  rectangle count.
}
function UniteList(List: PVRect; var Bounds: TRect): Integer;
var
  Cnt: Integer;

	procedure DoUnite(var Rect: TRect);
	begin
	  Inc(Cnt);
	  If Bounds.Empty
	  then Bounds := Rect
	  else Bounds.Union(Rect)
	end;

Begin
  Bounds.Assign(0, 0, 0, 0);
  Cnt := 0;
  ForEachRect(List, @DoUnite);
  UniteList := Cnt
End;

procedure SubtractList(var List: PVRect; Sub: PVRect);

 procedure Subtract(var Rect: TRect);
 Begin
   SubtractRect(List, Rect)
 End;

Begin
  ForEachRect(Sub, @Subtract)
End;

{ Visibility detection by rectangle-oriented operations 

  These routines use the abstract operations defined precedingly, such that
  they need not be modified if there is a more efficient rectangle-oriented
  data structure available.
}

function NewVis(var Bounds: TRect): PVis;
var
  Vis: PRectVis;
Begin
  New(Vis);
  with Vis^ do Begin
    TheBounds := Bounds;
    List := nil;
    Trans := nil;
    CurTrans := 0;
    DrawMode := 0;
    InsNewRect(List, Bounds)
  End;
  NewVis := PVis(Vis)
End;

procedure SubtractRegion(Vis: PVis; var Bounds: TRect);
Begin
  If (Vis <> nil) and DoesIntersect(PRectVis(Vis)^.TheBounds, Bounds)
  then SubtractRect(PRectVis(Vis)^.List, Bounds)
End;

procedure ClipVis(Vis: PVis; var Clip: TRect);
Begin
  If Vis <> nil then
  with PRectVis(Vis)^ do
  begin
    TheBounds.Intersect(Clip);
    ClipList(List, Clip)
  end
End;

function IntersectVis(Vis1, Vis2: PVis): PVis;
var
  Vis: PRectVis;
Begin
  IntersectVis := nil;
  If (Vis1 = nil) or (Vis2 = nil) then Exit;
  If not DoesIntersect(PRectVis(Vis1)^.TheBounds, PRectVis(Vis2)^.TheBounds)
  then Exit;
  New(Vis);
  with Vis^ do Begin
    TheBounds := PRectVis(Vis1)^.TheBounds;
    TheBounds.Intersect(PRectVis(Vis2)^.TheBounds);
    List := IntersectLists(PRectVis(Vis1)^.List, PRectVis(Vis2)^.List);
    Trans := New(PViewCollection, Join(PRectVis(Vis1)^.Trans, PRectVis(Vis2)^.Trans));
    CurTrans := 0;
    DrawMode := 0
  End;
  IntersectVis := PVis(Vis)
End;

function IntersectDelta(Vis: PVis; Delta: TPoint): PVis;
var
  aVis: PRectVis;
Begin
  IntersectDelta := nil;
  If Vis = nil then Exit;
  New(aVis);
  with aVis^ do Begin
    TheBounds := PRectVis(Vis)^.TheBounds;
    List := IntersectListDelta(PRectVis(Vis)^.List, Delta);
    Trans := New(PViewCollection, CopyOf(PRectVis(Vis)^.Trans));
    CurTrans := PRectVis(Vis)^.CurTrans;
    DrawMode := 0
  End;
  IntersectDelta := PVis(aVis)
End;

procedure SubtractVis(Vis, SubVis: PVis);
Begin
  If (Vis = nil) or (SubVis = nil) then Exit;
  SubtractList(PRectVis(Vis)^.List, PRectVis(SubVis)^.List)
End;

procedure MoveVis(Vis: PVis; Delta: TPoint);
Begin
  If Vis = nil then Exit;
  with PRectVis(Vis)^ do Begin
    TheBounds.Move(Delta.x, Delta.y);
    MoveList(List, Delta)
  End
End;

function RawCopyVis(Vis: PVis): PVis;
var
  Copy: PRectVis;
  Cur: PPVRect;
  Src: PVRect;
begin
  RawCopyVis := nil;
  If Vis = nil then Exit;
  New(Copy);
  Copy^ := PRectVis(Vis)^;
  Copy^.Trans := nil;
  Copy^.CurTrans := -1;
  Cur := @Copy^.List;
  while Cur^ <> nil do
  begin
    Src := Cur^;
    New(Cur^);
    Cur^^ := Src^;
    Cur := @Src^.Next
  end;
  RawCopyVis := Copy
end;

procedure FreeVis(Vis: PVis);
Begin
  If Vis = nil then Exit;
  with PRectVis(Vis)^ do
  begin
    FreeRectList(List);
    If Trans <> nil
    then Dispose(Trans, Done)
  end;
  Dispose(PRectVis(Vis))
End;

{ Sorting 
}

procedure SortRect(var Dst: PVRect; Src: PVRect; Delta: TPoint);
var
  clear: Boolean;

	procedure Test(var Rect: TRect);
	var
	  Copy: TRect;
	  Moved: TRect;
	  Result: Boolean;

		procedure DoesOverlap(var Obj: TRect);
		begin
		  Result := Result or DoesIntersect(Moved, Obj)
		end;

	begin
	  clear := false;
	  Copy := Rect;
	  FillChar(Rect, SizeOf(Rect), 0);
	  Moved := Copy;
	  Moved.Move(Delta.x, Delta.y);
	  Result := false;
	  ForEachRect(Src, @DoesOverlap);
	  If not Result
	  then InsNewRect(Dst, Copy)
	  else Rect := Copy
	end;

begin
  repeat
    clear := true;
    ForEachRect(Src, @Test)
  until clear;
end;

function SortVis(Vis: PVis; Delta: TPoint): PVis;
var
  sorted: PRectVis;
begin
  {$ifdef gvisible_v1}
  SortVis := Vis
  {$else}
  New(sorted);
  with sorted^ do
  begin
    TheBounds := PRectVis(Vis)^.TheBounds;
    List := nil;
    DrawMode := 0;
    Trans := PRectVis(Vis)^.Trans;
    CurTrans := PRectVis(Vis)^.CurTrans;
    PRectVis(Vis)^.Trans := nil
  end;
  SortRect(sorted^.List, PRectVis(Vis)^.List, Delta);
  FreeVis(Vis);
  SortVis := sorted
  {$endif}
end;

{ 
}

function BeginDraw(Vis: PVis; Origin: TPoint;
  Self: PGView; Modifying: Boolean): Boolean;
var
  Bounds: TRect;
  Options: Word;

 function PrepMeta(Size: Word): Boolean;
 Begin
   with PRectVis(Vis)^ do Begin
     MetaBuf := GetBuffer(Size);
     PrepMeta := MetaBuf <> nil
   End
 End;

 function PrepVga: Boolean;
 Begin
   if Modifying or (Self^.State and sfTransparent <> 0)
   then begin
     with Bounds do
       SetCriticalArea(A.x, A.y, B.x, B.y);
     HideMouse;
     PrepVga := PrepBuf(Bounds, pbCopy, PRectVis(Vis)^.VgaBuf);
     ShowMouse;
     SetCriticalArea(0, 0, 0, 0)
   end
   else
     PrepVga := PrepBuf(Bounds, pbNone, PRectVis(Vis)^.VgaBuf);
 End;

Begin
  BeginDraw := true;
  Inc(Self^.ViewLock);
  If Self^.ViewLock <> 1 then Exit;
  SourcePage := 2;
  DestPage := 2;
  MetaError := 0;

  If Vis = nil
  then begin
    SetClipRect(0, 0, 0, 0);
    MetaClipRect := ClipRect;
    Exit
  end;

  Options := Self^.Options and GlobalOptions;

  with PRectVis(Vis)^ do Begin
    case UniteList(List, Bounds) of
      0:
	Begin
	  SetClipRect(0, 0, 0, 0);
	  MetaClipRect := ClipRect;
	  DrawMode := 0;
	  Exit
	End;
      1:
	begin
	  {Dec(Self^.ViewLock);
	  If Modifying then HideTrans(Vis, Self);
	  Inc(Self^.ViewLock);}
	  If (Modifying or (Self^.State and sfTransparent = 0)) and
	     (Options and ofBuffer <> 0) and
	     ((Options and ofBufferOne <> 0) or HasTrans(Vis)) and
	     PrepVga
	  then Begin
	    SetDrawOrigin(Origin.x + DrawOrigin.x, Origin.y + DrawOrigin.y);
	    MetaClipRect := ClipRect;
            DrawTransBack(Vis, Self);
	    DrawMode := 1;
	    Exit
	  End
	  else Begin
	    Dec(Self^.ViewLock);
	    If Modifying or (Self^.State and sfTransparent <> 0)
	    then HideTrans(Vis, Self);
	    Inc(Self^.ViewLock);
	    SetDrawOriginP(Origin);
	    SetClipRectR(Bounds);
	    MetaClipRect := ClipRect;
	    with Bounds do SetCriticalArea(A.x, A.y, B.x, B.y);
	    HideMouse;
	    DrawMode := 3;
	    Exit
	  End;
	End
    End;

    {Dec(Self^.ViewLock);
    If Modifying then HideTrans(Vis, Self);
    Inc(Self^.ViewLock);}

    If (Modifying or (Self^.State and sfTransparent = 0)) and
       (Options and ofBuffer <> 0) and PrepVga
    then Begin
      SetDrawOrigin(Origin.x + DrawOrigin.x, Origin.y + DrawOrigin.y);
      MetaClipRect := ClipRect;
      DrawTransBack(Vis, Self);
      DrawMode := 1;
      Exit
    End;

    Dec(Self^.ViewLock);
    If Modifying or (Self^.State and sfTransparent <> 0)
    then HideTrans(Vis, Self);
    Inc(Self^.ViewLock);

    SetDrawOriginP(Origin);

    If (Options and ofMetaFile <> 0) and PrepMeta($4000)
    then Begin
      SetBuffer(MetaBuf);
      SetMetaState(ms_Record);
      SetClipRectR(Bounds);
      MetaClipRect := ClipRect;
      SetCriticalArea(0, 0, 0, 0);
      DrawMode := 2;
      Exit
    End
  End;
  BeginDraw := false;
  Self^.ViewLock := 0;
End;

function EndDraw(Vis: PVis; Self: PGView): Boolean;

 procedure PasteDraw(var Rect: TRect);
 Begin
   with Rect do SetCriticalArea(A.x, A.y, B.x, B.y);
   with PRectVis(Vis)^ do
     PasteRect(Rect, VgaBuf);
 End;

 procedure MetaDraw(var Rect: TRect);
 Begin
   MetaClipRect := Rect;
   with Rect do SetCriticalArea(A.x, A.y, B.x, B.y);
   with PRectVis(Vis)^ do
     DrawBuffer(MetaBuf)
 End;

Begin
  EndDraw := false;
  If Self^.ViewLock = 1 then
  If Vis <> nil then
  with PRectVis(Vis)^ do
  begin
    case DrawMode of
      1:
	Begin
	  DrawTrans(Vis, Self);
	  EndBufDraw;
	  SetCriticalArea(0, 0, 0, 0);
	  HideMouse;
	  ForEachRect(List, @PasteDraw);
	  ShowMouse;
	  ReleaseBuf(VgaBuf)
	End;
      2:
	Begin
	  SetMetaState(ms_Draw + ms_Execute + ms_BGI);
	  If not MetaFunctions or (MetaError <> 0)
	  then Begin
	    FreeBuffer(MetaBuf);
	    Self^.Options := Self^.Options and not ofMetaFile;
	    Exit
	  End;
	  LongInt(MetaOrigin) := 0;
	  SetCriticalArea(0, 0, 0, 0);
	  HideMouse;
	  ForEachRect(List, @MetaDraw);
	  ShowMouse;
	  SetMetaState(ms_Draw + ms_BGI);
	  FreeBuffer(MetaBuf)
	End;
      3:
	ShowMouse;
      0:
	FreeBacks(Vis, Self);
    end;
    If DrawMode >= 2 then ShowTrans(Vis, Self);
    DrawMode := 0
  end;
  Dec(Self^.ViewLock);
  If Self^.ViewLock < 0 then Self^.ViewLock := 0;
  EndDraw := true
End;

procedure SubtractHiddenSources(Vis: PVis; Delta: TPoint);
{ nothing to subtract }
begin
end;

{$endif !Windows}

{ 
}

procedure DrawVisLocal(Vis: PVis; Origin: Objects.TPoint; Self: PGView;
  Local: pointer; Frame: Word);
var
  P: TPoint;

 procedure CallDraw; near;
 Begin
   If Local = nil
     then Self^.Draw
     else asm
	push	bp
	mov	bp, [bp+4]		{ DrawVisLocal's frame }
{$IFDEF Windows}
	MOV	AX,[Frame]
	AND	AL,0FEH
	PUSH	AX
{$ELSE}
	push    [Frame]
{$ENDIF}
	push	cs			{ push return address }
	push	OFFSET @@1
	push	WORD PTR Local[2]	{ push local proc address }
	push	WORD PTR Local[0]
	retf				{ jump to local proc }
@@1:	pop	bp
    end
 End;

 procedure SimpleDraw(var Rect: Objects.TRect);
 Begin
   SetClipRectR(Rect);
   MetaClipRect := ClipRect;
   with Rect do SetCriticalArea(A.x, A.y, B.x, B.y);
   CallDraw
 End;

Begin
  If Vis = nil then Exit;
  Self^.ViewLock := 0;
  If BeginDraw(Vis, Origin, Self, Local <> nil)
  then Begin
    CallDraw;
    Self^.ViewLock := 1;
    If EndDraw(Vis, Self) then Exit
  End;
{$ifndef Windows}
  SetDrawOriginP(Origin);
  Self^.ViewLock := 1;
  HideMouse;
  ForEachRect(PRectVis(Vis)^.List, @SimpleDraw);
  ShowMouse;
  ShowTrans(Vis, Self);
  Self^.ViewLock := 0
{$endif}
End;

procedure DrawVis(Vis: PVis; Origin: Objects.TPoint; Self: PGView);
Begin
  DrawVisLocal(Vis, Origin, Self, nil, 0)
End;

procedure DrawFirstPass(P: PGView; Self: PGView);
var
  SaveDrawOrigin: Objects.TPoint;
  SaveClipRect: Objects.TRect;
  SaveMetaClipRect: Objects.TRect;
  ViewBounds: Objects.TRect;
  Delta: Objects.TPoint;
const
  theSelf: PGView = nil;
  GroupDrawOrigin: Objects.TPoint = ();
  GroupMetaClipRect: Objects.TRect = ();
begin
  SaveDrawOrigin := DrawOrigin;
  SaveMetaClipRect := MetaClipRect;
  SaveClipRect := ClipRect;
  If Self = nil
  then begin
    Self := theSelf;
    DrawOrigin := GroupDrawOrigin;
    MetaClipRect := GroupMetaClipRect
  end
  else begin
    theSelf := Self;
    GroupDrawOrigin := DrawOrigin;
    GroupMetaClipRect := MetaClipRect;
  end;
  P^.GetBounds(ViewBounds);
  If PGView(P^.GOwner) <> Self
  then begin
    LongInt(Delta) := 0;
    Self^.MakeGlobal(Delta, Delta);
    Delta.X := -Delta.X;
    Delta.Y := -Delta.Y;
    If P^.GOwner <> nil
    then P^.GOwner^.MakeGlobal(Delta, Delta);
    ViewBounds.Move(Delta.X, Delta.Y)
  end;
  If Self^.SetSubRect(ViewBounds)
  then begin
    ViewBounds.Move(GroupDrawOrigin.X, GroupDrawOrigin.Y);
    SetMetaClipRectR(ClipRect {ViewBounds});
    SetDrawOriginP(ViewBounds.A);
    with P^ do
    begin
      Inc(ViewLock); { disable old-style SetViewPort calls }
      Draw;
      Dec(ViewLock);
    end;
    SetDrawOriginP(SaveDrawOrigin);
    SetMetaClipRectR(SaveMetaClipRect);
    SetClipRectR(SaveClipRect)
  end
end;

procedure CritUnionDelta(Delta: Objects.TPoint);
var
  R: Objects.TRect;
Begin
  R := CriticalArea;
  If Delta.x > 0 then Dec(R.A.x, Delta.x) else Dec(R.B.x, Delta.x);
  If Delta.y > 0 then Dec(R.A.y, Delta.y) else Dec(R.B.y, Delta.y);
  SetCriticalArea(R.A.x, R.A.y, R.B.x, R.B.y);
End;

procedure ScrollView(Self: PGView; Delta: Objects.TPoint; SubRect: PRect);
var
  P: Objects.TPoint;
  avis, ivis: PVis;
  R: Objects.TRect;
  Opt: Word;

 procedure DoCopy; far;
 Begin
   CritUnionDelta(Delta);
   CopyScreen(DrawOrigin.X, DrawOrigin.Y,
	      Self^.Size.X + DrawOrigin.X, Self^.Size.Y + DrawOrigin.Y,
	      Delta.X + DrawOrigin.X, Delta.Y + DrawOrigin.Y)
 End;

Begin
  LongInt(P) := 0;
  Self^.MakeGlobal(P, P);
  avis := Self^.GetVisibility;
  If GetVgaMemCaps and vmcCopy <> 0
  then begin
    If SubRect <> nil then Begin
      R := SubRect^;
      R.Move(P.x, P.y);
      ClipVis(avis, R)
    End;
    ivis := IntersectDelta(avis, Delta);
    SubtractHiddenSources(ivis, Delta);
    Opt := Self^.Options;
    Self^.Options := Opt and not (ofBuffer + ofMetafile); { CopyScreen is not bufferable }
    DrawVisLocal(ivis, P, Self, @DoCopy, CurBP);	{ copy region }
    Self^.Options := Opt;
    SubtractVis(avis, ivis);
    FreeVis(ivis);
  end;
  DrawVis(avis, P, Self);			{ draw region }
  FreeVis(avis)
End;

procedure ScrollGroup(Self: PGGroup; Delta: Objects.TPoint);
var
  P: Objects.TPoint;
  avis, ivis: PVis;
  Opt: Word;

 procedure DoCopy; far;
 Begin
   CritUnionDelta(Delta);
   CopyScreen(DrawOrigin.X, DrawOrigin.Y,
	      Self^.Size.X + DrawOrigin.X, Self^.Size.Y + DrawOrigin.Y,
	      Delta.X + DrawOrigin.X, Delta.Y + DrawOrigin.Y)
 End;

Begin
  LongInt(P) := 0;
  Self^.MakeGlobal(P, P);
  avis := Self^.GetVisibility;
  If GetVgaMemCaps and vmcCopy <> 0
  then begin
    ivis := IntersectDelta(avis, Delta);
    SubtractHiddenSources(ivis, Delta);
    Opt := Self^.Options;
    Self^.Options := Opt and not (ofBuffer + ofMetafile); { CopyScreen is not bufferable }
    DrawVisLocal(ivis, P, Self, @DoCopy, CurBP);	{ copy region }
    Self^.Options := Opt;
    SubtractVis(avis, ivis);
    FreeVis(ivis);
  end;
  Self^.DrawClipped(avis, Self); { draw region }
End;

END.
