/*-------------------------------------------------------------------------

  mview.c - Browser for garbo.uwasa.fi:/pub/pc/pd2/moder??.zip moder.lst

  Under Unix, the MODERLST macro may be defined to the pathname of a
  global copy of the "moder.lst" file to use by default.  Under MS-DOS,
  the files "moder.lst" or "moder.zip" in the current directory or the
  same directory as the executable are used by default.

  Copyright (c) 1993 Rhys Weatherley

  Permission to use, copy, modify, and distribute this material
  for any purpose and without fee is hereby granted, provided
  that the above copyright notice and this permission notice
  appear in all copies, and that the name of Rhys Weatherley not be
  used in advertising or publicity pertaining to this
  material without specific, prior written permission.
  RHYS WEATHERLEY MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR
  SUITABILITY OF THIS MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED
  "AS IS", WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.

  Revision History:
  ================

   Version  DD/MM/YY  By  Description
   -------  --------  --  --------------------------------------
     1.0    06/09/93  RW  Original Version of mview.c

-------------------------------------------------------------------------*/

#ifdef	__TURBOC__
#ifndef	MSDOS
#define	MSDOS
#endif
  #pragma warn -pro
  #pragma warn -par
#endif

#include <stdio.h>
#include <ctype.h>
#include "display.h"

/*
 * Define some system-specific things.
 */
#ifdef	MSDOS

#include <malloc.h>
#include <string.h>
#include <io.h>
#define	PTR		huge
#define	MALLOC(x)	(farmalloc((x)))
#define	FREE(x)		(farfree((x)))
#define	OPENMODE	"rb"
#define	MEMCPY		_fmemcpy
#define	STRNCMP		_fmemicmp

#else

char	*malloc ();
#define	PTR
#define	MALLOC(x)	(malloc((int)((x))))
#define	FREE(x)		(free((x)))
#define	OPENMODE	"r"
#define	MEMCPY		our_memcpy
#define	STRNCMP		strncmp

static	void	our_memcpy (dest, src, len)
char	*dest, *src;
int	len;
{
  while (len-- > 0)
    *dest++ = *src++;
} /* our_memcpy */

#endif

/*
 * Define a buffer for storing moder.lst in memory.
 */
static	char PTR *moderlst;
static	long	  moderlen;

/*
 * Define the information needed for a contents list entry.
 */
typedef	struct _ModerContentEntry
		{
		  int	    kind;	/* Kind of entry */
		  char PTR *name;	/* Name of the entry */
		  int	    namelen;	/* Length of the name */
		  char PTR *location;	/* Location of the site */
		  int	    loclen;	/* Length of the location name */
		  char PTR *entry;	/* Pointer to the entry start */
		  long	    entrylen;	/* Length of the entire entry */
		  struct _ModerContentEntry *next;
		} ModerContentEntry;
#define	MODER_MAIN		0	/* Get main contents */
#define	MODER_TEXTUAL		1	/* Textual data only */
#define	MODER_SITELIST		2	/* List of sites */
#define	MODER_MIRRORS		3	/* Mirror cross-references */
#define	MODER_SITE		4	/* Individual site information */
#define	MODER_MIRRORLINKS	5	/* List of mirror site links */
#define	MODER_SITELINK		6	/* Link to any site in the file */
#define	MODER_DOMAINLIST	7	/* List of top-level domains */
#define	MODER_DOMAIN		8	/* Entry for a top-level domain */

extern	char	*FindDomain ();	/* in domains.c */
extern	char	*UncModerList (); /* in uncomp.c */
extern	void	RemoveUncFiles (); /* in uncomp.c */
extern	ModerContentEntry *ExtractContents ();
extern	ModerContentEntry *ExtractSite ();
extern	void	DestroyContents ();
extern	void	ShowMainContents ();

static	char	buffer[BUFSIZ];
static	char	buffer2[BUFSIZ];
static	char	moderdate[80];		/* Date list was released */
static	char	moderversion[80];	/* Version number on the list */
static	ModerContentEntry *linktofind;	/* For site link searches */
static	int	outofmem;

int	main (argc, argv)
int	argc;
char	**argv;
{
  ModerContentEntry *contents;
  char *filename;
#ifdef	MSDOS
  if (argc < 2)
    {
      if (access ("moder.lst", 0) == 0)
        filename = "moder.lst";	/* Use "moder.lst" in the current directory */
       else if (access ("moder.zip", 0) == 0)
        filename = "moder.zip";	/* Use "moder.zip" in the current directory */
       else
        {
          /* Determine the default filename from the program name */
	  int len;
          static char name[BUFSIZ];
          strcpy (name, argv[0]);
	  len = strlen (name);
	  while (len > 0 && name[len - 1] != '\\' && name[len - 1] != '/')
	    --len;
	  strcpy (name + len, "moder.lst");
	  if (access (name, 0) != 0)
	    {
	      /* path\moder.lst doesn't exist: try moder.zip */
	      strcpy (name + len, "moder.zip");
	      if (access (name, 0) != 0)
	        {
		  fprintf (stderr, "Could not find a default moder.lst or moder.zip file to view.\n");
		  fprintf (stderr, "Usage: %s [path]moder.lst\n", argv[0]);
		  return (1);
		} /* if */
	    } /* if */
	  filename = name;
	} /* else */
    } /* then */
#else	/* !MSDOS */
#ifdef	MODERLST
  if (argc < 2)
    filename = MODERLST;	/* Use the default compiled-in filename */
#else	/* !MODERLST */
  if (argc < 2)
    {
      fprintf (stderr, "Usage: %s [path]moder.lst\n", argv[0]);
      return (1);
    } /* then */
#endif	/* !MODERLST */
#endif	/* !MSDOS */
   else
    filename = argv[1];		/* Use the command-line supplied filename */
  printf ("MVIEW Version 1.0  Copyright 1993 Rhys Weatherley.  All rights reserved.\n\n");
  if (LoadModerList (filename))
    return (1);
  contents = ExtractContents (moderlst, moderlen, MODER_MAIN);
  InitDisplay ();
  sprintf (buffer, "Version %s - %s (? for help)", moderversion, moderdate);
  SetBottomLine (buffer);
  ShowMainContents (contents, (DrawPtr)"Main Contents", 13);
  TermDisplay ();
  DestroyContents (contents);
  FREE (moderlst);
  return (0);
} /* main */

/*
 * Load the moder.lst file into memory.
 */
int	LoadModerList (filename)
char	*filename;
{
  FILE *file;
  char PTR *output;
  int len;
  char *newname = UncModerList (filename);
  if (newname == NULL)
    {
      fprintf (stderr, "Could not uncompress %s\n", filename);
      return (-1);
    } /* if */
  if ((file = fopen (newname, OPENMODE)) == NULL)
    {
      perror (newname);
      RemoveUncFiles ();
      return (-1);
    } /* if */
  fseek (file, 0L, 2);
  moderlen = ftell (file);
  rewind (file);
  if ((moderlst = (char PTR *)MALLOC (moderlen)) == NULL)
    {
      fprintf (stderr, "Not enough memory to load %s\n", filename);
      fclose (file);
      RemoveUncFiles ();
      return (-1);
    } /* if */
  printf ("Loading %s ...\n", newname);
  output = moderlst;
  while ((len = fread (buffer, 1, BUFSIZ, file)) > 0)
    {
      MEMCPY (output, buffer, len);
      output += len;
    } /* while */
  fclose (file);
  RemoveUncFiles ();
  return (0);
} /* LoadModerList */

/*
 * Get a line from the moder.lst file.  Returns
 * zero if there are no more lines.
 */
int	GetModerLine (moderlst, moderlen)
char	* PTR *moderlst;
long	*moderlen;
{
  int posn = 0;
  char ch;
  while (*moderlen > 0 && (ch = *(*moderlst)) != '\n')
    {
      if (ch != '\r')
        buffer[posn++] = ch;
      --(*moderlen);
      ++(*moderlst);
    } /* while */
  buffer[posn] = '\0';
  if (*moderlen > 0)
    {
      --(*moderlen);
      ++(*moderlst);
      return (1);
    } /* then */
   else
    return (posn > 0);
} /* GetModerLine */

/*
 * Report an out of memory error.
 */
void	OutOfMemory ()
{
  if (!outofmem)
    {
      outofmem = 1;	/* Only report the message once */
      ReportError ("Out of memory", (DrawPtr)NULL, 0);
    } /* if */
} /* OutOfMemory */

/*
 * Add a new entry to a contents list.
 */
int	AddEntry (list, last, kind, name, namelen, entry)
ModerContentEntry **list, **last;
int	kind, namelen;
char	PTR *name, PTR *entry;
{
  ModerContentEntry *newentry;
  newentry = (ModerContentEntry *)malloc (sizeof (ModerContentEntry));
  if (newentry == NULL)
    {
      OutOfMemory ();
      return (0);
    } /* if */
  newentry -> kind = kind;
  newentry -> name = name;
  newentry -> namelen = namelen;
  newentry -> location = NULL;
  newentry -> loclen = 0;
  newentry -> entry = entry;
  newentry -> entrylen = 0;
  newentry -> next = NULL;
  if (*last != NULL)
    (*last) -> next = newentry;
   else
    *list = newentry;
  *last = newentry;
  return (1);
} /* AddEntry */

/*
 * Extract a domain name from a site name.
 */
char	*ExtractDomain (site)
char	*site;
{
  int len;

  /* Skip to the start of the site name */
  while (*site != '\0' && isspace (*site))
    ++site;

  /* Find the end of the site name */
  len = 0;
  while (site[len] != '\0' && !isspace (site[len]))
    ++len;

  /* Scan back to strip off the last domain component */
  site[len] = '\0';
  while (len > 0 && site[len - 1] != '.')
    {
      --len;
      if (isupper (site[len]))
        site[len] = tolower (site[len]);
    } /* while */
  return (site + len);
} /* ExtractDomain */

/*
 * Determine if a domain name already occurs in a list
 * of content entries.
 */
int	DomainInList (list, domain)
ModerContentEntry *list;
char	*domain;
{
  int len = strlen (domain);
  while (list != NULL)
    {
      if (len == list -> namelen &&
      	  !STRNCMP ((char PTR *)domain, list -> name, len))
	return (1);
      list = list -> next;
    } /* while */
  return (0);
} /* DomainInList */

/*
 * Add a domain name as an entry to a content list.
 */
void	AddDomainEntry (list, last, domain)
ModerContentEntry **list, **last;
char	*domain;
{
  int len = strlen (domain);
  char *memdomain;
  if ((memdomain = (char *)malloc (len + 1)) == NULL)
    {
      OutOfMemory ();
      return;
    } /* if */
  strcpy (memdomain, domain);
  if (!AddEntry (list, last, MODER_DOMAIN, (char PTR *)memdomain, len,
  		 moderlst))
    free (memdomain);	/* Could not create entry: free up used space */
   else
    {
      /* Another entry has been added: determine the location */
      (*last) -> entrylen = moderlen;
      if ((memdomain = FindDomain (domain)) != NULL)
        {
	  (*last) -> location = (char PTR *)memdomain;
	  (*last) -> loclen = strlen (memdomain);
	} /* if */
    } /* else */
} /* AddDomainEntry */

/*
 * Extract a list of contents from the moder.lst file.
 */
ModerContentEntry *ExtractContents (moderlst, moderlen, kind)
char PTR *moderlst;
long	  moderlen;
int	  kind;
{
  ModerContentEntry *list, *last;
  int len, entrykind, fillin;
  char PTR *prevline;
  char PTR *thisline;
  char PTR *savelst;
  char *domain;
  long lastlen, prevlen, thislen, savelen;
  outofmem = 0;
  list = NULL;
  last = NULL;
  buffer2[0] = '\0';
  prevline = moderlst;
  thisline = moderlst;
  lastlen = moderlen;
  prevlen = moderlen;
  thislen = moderlen;
  fillin = 0;
  savelst = moderlst;
  savelen = moderlen;
  if (kind == MODER_MAIN)
    {
      moderdate[0] = '\0';
      moderversion[0] = '\0';
    } /* if */
  while (GetModerLine (&moderlst, &moderlen))
    {
      switch (kind)
        {
	  case MODER_MAIN:
	  	/* Parse the whole list into individual sections */
		if (!strncmp (buffer, "moder.lst ", 10) && last == NULL)
		  strcpy (moderdate, buffer + 10);
		 else if (!strncmp (buffer, "in garbo.", 9) && last == NULL)
		  {
		    /* Extract version number: ...:/pc/pd2/moder??.zip */
		    len = strlen (buffer) - 4;
		    if (strcmp (buffer + len, ".zip") != 0)
		      break;	/* This shouldn't happen!! */
		    buffer[len] = '\0';
		    while (len > 0 && isdigit (buffer[len - 1]))
		      --len;
		    strcpy (moderversion, buffer + len);
		  } /* then */
		 else if (!strncmp (buffer, "====", 4))
		  {
		    /* We've found the start of the introduction */
		    AddEntry (&list, &last, MODER_TEXTUAL,
		    	      (char PTR *)"Introduction", 12, moderlst);
		    lastlen = moderlen;
		  } /* then */
      		 else if (!strncmp (buffer, "----", 4))
      		  {
		    /* We've found the start of a new section */
		    if (last != NULL && last -> entrylen == 0)
		      last -> entrylen = lastlen - prevlen;
		    if (!strncmp (buffer2, "Main contents", 13))
		      break;		/* Skip the contents section */
		    if (!strncmp (buffer2, "Fill-in ", 8))
		      fillin = 1;
		     else
		      fillin = 0;
		    len = strlen (buffer2);
		    while (len > 0 && (isspace (buffer2[len - 1]) ||
		    		       buffer2[len - 1] == ':'))
		      --len;
		    if (!strncmp (buffer2, "Mirrors ", 8))
		      entrykind = MODER_MIRRORS;
		     else
		      entrykind = MODER_TEXTUAL;
		    AddEntry (&list, &last, entrykind, prevline, len,
		    		moderlst);
		    lastlen = moderlen;
		  } /* then */
		 else if (!strncmp (buffer, "Site:", 5) && last != NULL &&
		 	  !fillin)
		  last -> kind = MODER_SITELIST;
		break;
	  case MODER_SITELIST:
	  	/* Parse a list of sites into individual sites */
		if (!strncmp (buffer, "Site: ", 6))
		  {
		    /* We've found the start of a site entry */
		    len = 6;
		    while (buffer[len] != '\0' && isspace (buffer[len]))
		      ++len;
		    AddEntry (&list, &last, MODER_SITE, thisline + len,
		    	      strlen (buffer + len), thisline);
		    lastlen = thislen;
		  } /* then */
		 else if (buffer[0] == '\0')
		  {
		    /* We've found the end of a site entry */
		    if (last != NULL && last -> entrylen == 0)
		      last -> entrylen = lastlen - thislen;
		  } /* then */
		 else if (!strncmp (buffer, "Location:", 9) &&
		 	  last != NULL && last -> entrylen == 0)
		  {
		    /* We've found the location of the site */
		    len = 9;
		    while (buffer[len] != '\0' && isspace (buffer[len]))
		      ++len;
		    last -> location = thisline + len;
		    last -> loclen = strlen (buffer + len);
		  } /* then */
		break;
	  case MODER_MIRRORS:
	  	/* Parse a list of mirrored sites */
		if (!strncmp (buffer, "  ", 2) && buffer2[0] != ' ')
		  {
		    AddEntry (&list, &last, MODER_MIRRORLINKS, prevline,
		    	strlen (buffer2), thisline);
		    lastlen = thislen;
		  } /* then */
		 else if (buffer[0] == '\0' && last != NULL &&
		 	  last -> entrylen == 0)
		  last -> entrylen = lastlen - thislen;
		break;
          case MODER_MIRRORLINKS:
	  	/* Parse a list of site links */
		if (last != NULL)
		  last -> entrylen = lastlen - thislen;
		AddEntry (&list, &last, MODER_SITELINK, thisline + 2,
			strlen (buffer) - 2, thisline);
		lastlen = thislen;
		break;
	  case MODER_SITELINK:
	  	/* Find the data for a particular site */
		if (!strncmp (buffer, "Site: ", 6))
		  {
		    /* Check to see if this is the site we are looking for */
		    len = 6;
		    while (buffer[len] != '\0' && isspace (buffer[len]))
		      ++len;
		    if (STRNCMP (linktofind -> name, buffer + len,
		    		 linktofind -> namelen) != 0 ||
			(buffer[len + linktofind -> namelen] != '\0' &&
			 buffer[len + linktofind -> namelen] != ' '))
		      break;	/* Not the right entry */
		    AddEntry (&list, &last, MODER_SITE, thisline + len,
		    	      strlen (buffer + len), thisline);
		    lastlen = thislen;
		  } /* then */
		 else if (buffer[0] == '\0')
		  {
		    /* We've found the end of a site entry */
		    if (last != NULL && last -> entrylen == 0)
		      last -> entrylen = lastlen - thislen;
		  } /* then */
		break;
	  case MODER_DOMAINLIST:
	  	/* Parse the whole of moder.lst to find the domains */
		if (!strncmp (buffer, "Site: ", 6))
		  {
		    domain = ExtractDomain (buffer + 6);
		    if (!DomainInList (list, domain))
		      AddDomainEntry (&list, &last, domain);
		  } /* if */
		break;
	  case MODER_DOMAIN:
	  	/* Extract the sites in a particular domain */
		if (!strncmp (buffer, "Site: ", 6))
		  {
		    /* Check to see if this is the domain we are looking for */
		    len = 6;
		    while (buffer[len] != '\0' && isspace (buffer[len]))
		      ++len;
		    strcpy (buffer2, buffer + len);
		    domain = ExtractDomain (buffer2);
		    if (STRNCMP (linktofind -> name, domain,
		    		 linktofind -> namelen) != 0 ||
			strlen (domain) != linktofind -> namelen)
		      break;	/* Not int the right domain */
		    AddEntry (&list, &last, MODER_SITE, thisline + len,
		    	      strlen (buffer + len), thisline);
		    lastlen = thislen;
		  } /* then */
		 else if (buffer[0] == '\0')
		  {
		    /* We've found the end of a site entry */
		    if (last != NULL && last -> entrylen == 0)
		      last -> entrylen = lastlen - thislen;
		  } /* then */
		 else if (!strncmp (buffer, "Location:", 9) &&
		 	  last != NULL && last -> entrylen == 0)
		  {
		    /* We've found the location of the site */
		    len = 9;
		    while (buffer[len] != '\0' && isspace (buffer[len]))
		      ++len;
		    last -> location = thisline + len;
		    last -> loclen = strlen (buffer + len);
		  } /* then */
	  	break;
	} /* switch */
      strcpy (buffer2, buffer);
      prevline = thisline;
      thisline = moderlst;
      prevlen = thislen;
      thislen = moderlen;
    } /* while */
  if (last != NULL && last -> entrylen == 0)
    last -> entrylen = lastlen - moderlen;
  if (kind == MODER_MAIN)
    {
      /* Add dummy entries to the main contents for all sites */
      AddEntry (&list, &last, MODER_SITELIST, (char PTR *)"All sites", 9,
      		savelst);
      if (last != NULL && last -> entrylen == 0)
        last -> entrylen = savelen;
      AddEntry (&list, &last, MODER_DOMAINLIST,
      		(char PTR *)"All sites by top-level domain", 29, savelst);
      if (last != NULL && last -> entrylen == 0)
        last -> entrylen = savelen;
    } /* if */
  return (list);
} /* ExtractContents */

/*
 * Extract the information for a site given a link
 * or a domain record.
 */
ModerContentEntry *ExtractSite (link, kind)
ModerContentEntry *link;
int	kind;
{
  linktofind = link;
  return (ExtractContents (moderlst, moderlen, kind));
} /* ExtractSite */

/*
 * Destroy a contents list: it is no longer required.
 */
void	DestroyContents (list)
ModerContentEntry *list;
{
  ModerContentEntry *temp;
  while (list != NULL)
    {
      if (list -> kind == MODER_DOMAIN)
        free ((char *)(list -> name));
      temp = list -> next;
      free (list);
      list = temp;
    } /* while */
} /* DestroyContents */

/*
 * Compare two entries to determine which order they
 * should appear in a sorted list.
 */
int	CompareEntries (x, y)
ModerContentEntry *x, *y;
{
  int cmp;
  if (x -> namelen == y -> namelen)
    return (STRNCMP (x -> name, y -> name, x -> namelen));
   else if (x -> namelen > y -> namelen)
    {
      if ((cmp = STRNCMP (x -> name, y -> name, y -> namelen)) != 0)
        return (cmp);
       else
        return (1);
    } /* then */
   else
    {
      if ((cmp = STRNCMP (x -> name, y -> name, x -> namelen)) != 0)
        return (cmp);
       else
        return (-1);
    } /* else */
} /* CompareEntries */

/*
 * Sort a contents list into alphabetical order by name.
 * This uses a Bubble Sort (ack!), but since the lists
 * to be sorted are quite small, it really doesn't matter.
 */
ModerContentEntry *SortContents (list)
ModerContentEntry *list;
{
  int swapped = 1;
  ModerContentEntry *newlist, *last, *next;
  if (list == NULL || list -> next == NULL)
    return (list);
  while (swapped)
    {
      swapped = 0;
      newlist = list;
      last = NULL;
      while (list -> next != NULL)
        {
	  next = list -> next;
	  if (CompareEntries (list, list -> next) > 0)
	    {
	      /* Swap this entry with the next one */
	      swapped = 1;
	      if (last != NULL)
		last -> next = next;
	       else
	        newlist = next;
	      list -> next = next -> next;
	      last = next;
	    } /* then */
	   else
	    {
	      /* Entries are in order: advance to the next one */
	      if (last != NULL)
	        last -> next = list;
	      last = list;
	      list = next;
	    } /* else */
	} /* while */
      last -> next = list;
      list = newlist;
    } /* while */
  return (list);
} /* SortContents */

/*
 * Determine the length of a list of moder.lst entries.
 */
int	ModerListLength (display, list)
DisplayList *display;
ModerContentEntry *list;
{
  int length = 0;
  while (list != NULL)
    {
      ++length;
      if (list -> namelen > display -> extra)
        display -> extra = list -> namelen;
      list = list -> next;
    } /* while */
  return (length);
} /* ModerListLength */

/*
 * Find a particular item in a contents list.
 */
ModerContentEntry *FindModerItem (list, item)
ModerContentEntry *list;
int		item;
{
  while (item-- > 0)
    list = list -> next;
  return (list);
} /* FindModerItem */

/*
 * Draw an item from a list.
 */
void	DrawModerItem (display, param, item)
DisplayList *display;
void	*param;
int	item;
{
  ModerContentEntry *list = FindModerItem ((ModerContentEntry *)param, item);
  DrawChars ((DrawPtr)(list -> name), list -> namelen);
  if ((list -> kind == MODER_SITE || list -> kind == MODER_DOMAIN)
  	&& list -> location != NULL)
    {
      int len = list -> namelen - 2;
      while (len++ < display -> extra)
        DrawChars ((DrawPtr)" ", 1);
      DrawChars ((DrawPtr)(list -> location), list -> loclen);
    } /* if */
} /* DrawModerItem */

/*
 * Get the number of lines in a contents entry.
 */
int	ModerNumLines (entry)
ModerContentEntry *entry;
{
  int lines = 0;
  char PTR *ptr = entry -> entry;
  long len = entry -> entrylen;
  while (GetModerLine (&ptr, &len))
    ++lines;
  return (lines);
} /* ModerNumLines */

/*
 * Draw a line from a contents entry.
 */
void	DrawModerLine (display, param, item)
DisplayList *display;
void	*param;
int	item;
{
  char PTR *ptr = ((ModerContentEntry *)param) -> entry;
  long len = ((ModerContentEntry *)param) -> entrylen;
  buffer[0] = '\0';
  while (item-- >= 0 && GetModerLine (&ptr, &len))
    ;	/* Keep looping until we have the line we want */
  DrawChars ((DrawPtr)buffer, strlen (buffer));
} /* DrawModerLine */

/*
 * Show textual or site information.
 */
void	ShowTextual (entry, title, titlelen)
ModerContentEntry *entry;
DrawPtr	title;
int	titlelen;
{
  DisplayList display;
  InitList (&display, entry, DrawModerLine, ModerNumLines (entry),
  	    title, titlelen, 1);
  if (display.num != 0)
    SelectDisplay (&display);
} /* ShowTextual */

/*
 * Handle selections for the main contents list.
 */
void	ShowMainContents (list, title, titlelen)
ModerContentEntry *list;
DrawPtr	title;
int	titlelen;
{
  extern void DisplayHelp ();
  DisplayList display;
  ModerContentEntry *entry, *newlist;
  int item;
  if (list == NULL)
    return;
  if (list -> kind == MODER_SITE && list -> next == NULL)
    {
      ShowTextual (list, title, titlelen);
      return;
    } /* if */
  display.extra = 0;
  InitList (&display, list, DrawModerItem, ModerListLength (&display, list),
  		title, titlelen, 0);
  while ((item = SelectDisplay (&display)) != -1)
    {
      if (item == -2)
	{
	  DisplayHelp ();
	  continue;
	} /* if */
      entry = FindModerItem (list, item);
      switch (entry -> kind)
        {
	  case MODER_TEXTUAL: case MODER_SITE:
	  	ShowTextual (entry, entry -> name, entry -> namelen);
	  	break;
	  case MODER_SITELIST: case MODER_MIRRORS: case MODER_MIRRORLINKS:
	  case MODER_DOMAINLIST:
	  	newlist = ExtractContents (entry -> entry, entry -> entrylen,
					   entry -> kind);
		newlist = SortContents (newlist);
	  	ShowMainContents (newlist, entry -> name, entry -> namelen);
	  	DestroyContents (newlist);
	  	break;
	  case MODER_SITELINK:
	  	newlist = ExtractSite (entry, MODER_SITELINK);
		if (newlist == NULL)
		  {
		    ReportError ("Site information not present for ",
		    		 entry -> name, entry -> namelen);
		    break;
		  } /* if */
	  	ShowMainContents (newlist, entry -> name, entry -> namelen);
	  	DestroyContents (newlist);
	  	break;
	  case MODER_DOMAIN:
	  	newlist = SortContents (ExtractSite (entry, MODER_DOMAIN));
		if (entry -> location != NULL)
		  ShowMainContents (newlist, entry -> location,
		  			entry -> loclen);
		 else
		  ShowMainContents (newlist, entry -> name, entry -> namelen);
	  	DestroyContents (newlist);
	  	break;
	} /* switch */
    } /* while */
} /* ShowMainContents */
