--------------------------------------------------------------------------
--
--  Copyright (C) 1993, Peter J. Ashenden
--  Mail:       Dept. Computer Science
--              University of Adelaide, SA 5005, Australia
--  e-mail:     petera@cs.adelaide.edu.au
--
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program; if not, write to the Free Software
--  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--
--------------------------------------------------------------------------
--
--  $RCSfile: dlx_instr.vhdl,v $  $Revision: 2.1 $  $Date: 1993/10/31 22:34:41 $
--
--------------------------------------------------------------------------
--
--  Package specification for DLX instructions
--
--  8/94: Modified by Scott Thibault to compiler under the GM VHDL compiler.
--
--------------------------------------------------------------------------

library types;
use std.textio.line,
    types.dlx_types.all;

package dlx_instr is
  
-- A dlx instruction is 32 bits wide.  There are three instruction formats:
--
-- I-type:
--  0         5 6      10 11     15 16                            31
-- +----------------------------------------------------------------+
-- |  opcode   |   rs1   |   rd    |           immed16              |
-- +----------------------------------------------------------------+
--
-- R-type:
--  0         5 6      10 11     15 16     20 21     25 26        31
-- +----------------------------------------------------------------+
-- |  opcode   |   rs1   |   rs2   |   rd    |         |   func     |
-- +----------------------------------------------------------------+
--
-- J-type:
--  0         5 6                                                 31
-- +----------------------------------------------------------------+
-- |  opcode   |                     immed26                        |
-- +----------------------------------------------------------------+
--

  subtype dlx_opcode is bit_vector(0 to 5);
  subtype dlx_sp_func is bit_vector(0 to 5);
  subtype dlx_fp_func is bit_vector(0 to 4);
  subtype dlx_reg_addr is bit_vector(0 to 4);
  subtype dlx_immed16 is bit_vector(0 to 15);
  subtype dlx_immed26 is bit_vector(0 to 25);
  

  constant op_special   : dlx_opcode := B"000000";
  constant op_fparith   : dlx_opcode := B"000001";
  constant op_j         : dlx_opcode := B"000010";
  constant op_jal       : dlx_opcode := B"000011";
  constant op_beqz      : dlx_opcode := B"000100";
  constant op_bnez      : dlx_opcode := B"000101";
  constant op_bfpt      : dlx_opcode := B"000110";
  constant op_bfpf      : dlx_opcode := B"000111";
  constant op_addi      : dlx_opcode := B"001000";
  constant op_addui     : dlx_opcode := B"001001";
  constant op_subi      : dlx_opcode := B"001010";
  constant op_subui     : dlx_opcode := B"001011";
  constant op_andi      : dlx_opcode := B"001100";
  constant op_ori       : dlx_opcode := B"001101";
  constant op_xori      : dlx_opcode := B"001110";
  constant op_lhi       : dlx_opcode := B"001111";

  constant op_rfe       : dlx_opcode := B"010000";
  constant op_trap      : dlx_opcode := B"010001";
  constant op_jr        : dlx_opcode := B"010010";
  constant op_jalr      : dlx_opcode := B"010011";
  constant op_slli      : dlx_opcode := B"010100";
  constant op_undef_15  : dlx_opcode := B"010101";
  constant op_srli      : dlx_opcode := B"010110";
  constant op_srai      : dlx_opcode := B"010111";
  constant op_seqi      : dlx_opcode := B"011000";
  constant op_snei      : dlx_opcode := B"011001";
  constant op_slti      : dlx_opcode := B"011010";
  constant op_sgti      : dlx_opcode := B"011011";
  constant op_slei      : dlx_opcode := B"011100";
  constant op_sgei      : dlx_opcode := B"011101";
  constant op_undef_1E  : dlx_opcode := B"011110";
  constant op_undef_1F  : dlx_opcode := B"011111";

  constant op_lb        : dlx_opcode := B"100000";
  constant op_lh        : dlx_opcode := B"100001";
  constant op_undef_22  : dlx_opcode := B"100010";
  constant op_lw        : dlx_opcode := B"100011";
  constant op_lbu       : dlx_opcode := B"100100";
  constant op_lhu       : dlx_opcode := B"100101";
  constant op_lf        : dlx_opcode := B"100110";
  constant op_ld        : dlx_opcode := B"100111";
  constant op_sb        : dlx_opcode := B"101000";
  constant op_sh        : dlx_opcode := B"101001";
  constant op_undef_2A  : dlx_opcode := B"101010";
  constant op_sw        : dlx_opcode := B"101011";
  constant op_undef_2C  : dlx_opcode := B"101100";
  constant op_undef_2D  : dlx_opcode := B"101101";
  constant op_sf        : dlx_opcode := B"101110";
  constant op_sd        : dlx_opcode := B"101111";

  constant op_sequi     : dlx_opcode := B"110000";
  constant op_sneui     : dlx_opcode := B"110001";
  constant op_sltui     : dlx_opcode := B"110010";
  constant op_sgtui     : dlx_opcode := B"110011";
  constant op_sleui     : dlx_opcode := B"110100";
  constant op_sgeui     : dlx_opcode := B"110101";
  constant op_undef_36  : dlx_opcode := B"110110";
  constant op_undef_37  : dlx_opcode := B"110111";
  constant op_undef_38  : dlx_opcode := B"111000";
  constant op_undef_39  : dlx_opcode := B"111001";
  constant op_undef_3A  : dlx_opcode := B"111010";
  constant op_undef_3B  : dlx_opcode := B"111011";
  constant op_undef_3C  : dlx_opcode := B"111100";
  constant op_undef_3D  : dlx_opcode := B"111101";
  constant op_undef_3E  : dlx_opcode := B"111110";
  constant op_undef_3F  : dlx_opcode := B"111111";

  constant sp_func_nop          : dlx_sp_func := B"000000";
  constant sp_func_undef_01     : dlx_sp_func := B"000001";
  constant sp_func_undef_02     : dlx_sp_func := B"000010";
  constant sp_func_undef_03     : dlx_sp_func := B"000011";
  constant sp_func_sll          : dlx_sp_func := B"000100";
  constant sp_func_undef_05     : dlx_sp_func := B"000101";
  constant sp_func_srl          : dlx_sp_func := B"000110";
  constant sp_func_sra          : dlx_sp_func := B"000111";
  constant sp_func_undef_08     : dlx_sp_func := B"001000";
  constant sp_func_undef_09     : dlx_sp_func := B"001001";
  constant sp_func_undef_0A     : dlx_sp_func := B"001010";
  constant sp_func_undef_0B     : dlx_sp_func := B"001011";
  constant sp_func_undef_0C     : dlx_sp_func := B"001100";
  constant sp_func_undef_0D     : dlx_sp_func := B"001101";
  constant sp_func_undef_0E     : dlx_sp_func := B"001110";
  constant sp_func_undef_0F     : dlx_sp_func := B"001111";

  constant sp_func_sequ         : dlx_sp_func := B"010000";
  constant sp_func_sneu         : dlx_sp_func := B"010001";
  constant sp_func_sltu         : dlx_sp_func := B"010010";
  constant sp_func_sgtu         : dlx_sp_func := B"010011";
  constant sp_func_sleu         : dlx_sp_func := B"010100";
  constant sp_func_sgeu         : dlx_sp_func := B"010101";
  constant sp_func_undef_16     : dlx_sp_func := B"010110";
  constant sp_func_undef_17     : dlx_sp_func := B"010111";
  constant sp_func_undef_18     : dlx_sp_func := B"011000";
  constant sp_func_undef_19     : dlx_sp_func := B"011001";
  constant sp_func_undef_1A     : dlx_sp_func := B"011010";
  constant sp_func_undef_1B     : dlx_sp_func := B"011011";
  constant sp_func_undef_1C     : dlx_sp_func := B"011100";
  constant sp_func_undef_1D     : dlx_sp_func := B"011101";
  constant sp_func_undef_1E     : dlx_sp_func := B"011110";
  constant sp_func_undef_1F     : dlx_sp_func := B"011111";

  constant sp_func_add          : dlx_sp_func := B"100000";
  constant sp_func_addu         : dlx_sp_func := B"100001";
  constant sp_func_sub          : dlx_sp_func := B"100010";
  constant sp_func_subu         : dlx_sp_func := B"100011";
  constant sp_func_and          : dlx_sp_func := B"100100";
  constant sp_func_or           : dlx_sp_func := B"100101";
  constant sp_func_xor          : dlx_sp_func := B"100110";
  constant sp_func_undef_27     : dlx_sp_func := B"100111";
  constant sp_func_seq          : dlx_sp_func := B"101000";
  constant sp_func_sne          : dlx_sp_func := B"101001";
  constant sp_func_slt          : dlx_sp_func := B"101010";
  constant sp_func_sgt          : dlx_sp_func := B"101011";
  constant sp_func_sle          : dlx_sp_func := B"101100";
  constant sp_func_sge          : dlx_sp_func := B"101101";
  constant sp_func_undef_2E     : dlx_sp_func := B"101110";
  constant sp_func_undef_2F     : dlx_sp_func := B"101111";
  
  constant sp_func_movi2s       : dlx_sp_func := B"110000";
  constant sp_func_movs2i       : dlx_sp_func := B"110001";
  constant sp_func_movf         : dlx_sp_func := B"110010";
  constant sp_func_movd         : dlx_sp_func := B"110011";
  constant sp_func_movfp2i      : dlx_sp_func := B"110100";
  constant sp_func_movi2fp      : dlx_sp_func := B"110101";
  constant sp_func_undef_36     : dlx_sp_func := B"110110";
  constant sp_func_undef_37     : dlx_sp_func := B"110111";
  constant sp_func_undef_38     : dlx_sp_func := B"111000";
  constant sp_func_undef_39     : dlx_sp_func := B"111001";
  constant sp_func_undef_3A     : dlx_sp_func := B"111010";
  constant sp_func_undef_3B     : dlx_sp_func := B"111011";
  constant sp_func_undef_3C     : dlx_sp_func := B"111100";
  constant sp_func_undef_3D     : dlx_sp_func := B"111101";
  constant sp_func_undef_3E     : dlx_sp_func := B"111110";
  constant sp_func_undef_3F     : dlx_sp_func := B"111111";
  
  constant fp_func_addf         : dlx_fp_func := B"00000";
  constant fp_func_subf         : dlx_fp_func := B"00001";
  constant fp_func_multf        : dlx_fp_func := B"00010";
  constant fp_func_divf         : dlx_fp_func := B"00011";
  constant fp_func_addd         : dlx_fp_func := B"00100";
  constant fp_func_subd         : dlx_fp_func := B"00101";
  constant fp_func_multd        : dlx_fp_func := B"00110";
  constant fp_func_divd         : dlx_fp_func := B"00111";
  constant fp_func_cvtf2d       : dlx_fp_func := B"01000";
  constant fp_func_cvtf2i       : dlx_fp_func := B"01001";
  constant fp_func_cvtd2f       : dlx_fp_func := B"01010";
  constant fp_func_cvtd2i       : dlx_fp_func := B"01011";
  constant fp_func_cvti2f       : dlx_fp_func := B"01100";
  constant fp_func_cvti2d       : dlx_fp_func := B"01101";
  constant fp_func_mult         : dlx_fp_func := B"01110";
  constant fp_func_div          : dlx_fp_func := B"01111";
  
  constant fp_func_eqf          : dlx_fp_func := B"10000";
  constant fp_func_nef          : dlx_fp_func := B"10001";
  constant fp_func_ltf          : dlx_fp_func := B"10010";
  constant fp_func_gtf          : dlx_fp_func := B"10011";
  constant fp_func_lef          : dlx_fp_func := B"10100";
  constant fp_func_gef          : dlx_fp_func := B"10101";
  constant fp_func_multu        : dlx_fp_func := B"10110";
  constant fp_func_divu         : dlx_fp_func := B"10111";
  constant fp_func_eqd          : dlx_fp_func := B"11000";
  constant fp_func_ned          : dlx_fp_func := B"11001";
  constant fp_func_ltd          : dlx_fp_func := B"11010";
  constant fp_func_gtd          : dlx_fp_func := B"11011";
  constant fp_func_led          : dlx_fp_func := B"11100";
  constant fp_func_ged          : dlx_fp_func := B"11101";
  constant fp_func_undef_1E     : dlx_fp_func := B"11110";
  constant fp_func_undef_1F     : dlx_fp_func := B"11111";

  subtype dlx_opcode_num is natural range 0 to 63;
  subtype dlx_sp_func_num is natural range 0 to 63;
  subtype dlx_fp_func_num is natural range 0 to 31;
      
  subtype instr_name is string(1 to 8);
  type opcode_name_array is array (dlx_opcode_num) of instr_name;
  type sp_func_name_array is array (dlx_sp_func_num) of instr_name;
  type fp_func_name_array is array (dlx_fp_func_num) of instr_name;
  
  constant opcode_names : opcode_name_array;
  constant sp_func_names : sp_func_name_array;
  constant fp_func_names : fp_func_name_array;

  type immed_size is (immed_size_16, immed_size_26);
  
  subtype reg_index is natural range 0 to 31;
  
  constant link_reg : reg_index := 31;

  procedure write_instr (L : inout line; instr : in dlx_word);

end dlx_instr;


library arith1;
use std.textio.all,
    arith1.bv_arithmetic.bv_to_natural,
    arith1.bv_arithmetic.bv_to_integer;
    

package body dlx_instr is

  constant opcode_names : opcode_name_array :=
    ( "SPECIAL ",   "FPARITH ",   "J       ",   "JAL     ",
      "BEQZ    ",   "BNEZ    ",   "BFPT    ",   "BFPF    ",
      "ADDI    ",   "ADDUI   ",   "SUBI    ",   "SUBUI   ",
      "ANDI    ",   "ORI     ",   "XORI    ",   "LHI     ",
      "RFE     ",   "TRAP    ",   "JR      ",   "JALR    ",
      "SLLI    ",   "UNDEF_15",   "SRLI    ",   "SRAI    ",
      "SEQI    ",   "SNEI    ",   "SLTI    ",   "SGTI    ",
      "SLEI    ",   "SGEI    ",   "UNDEF_1E",   "UNDEF_1F",
      "LB      ",   "LH      ",   "UNDEF_22",   "LW      ",
      "LBU     ",   "LHU     ",   "LF      ",   "LD      ",
      "SB      ",   "SH      ",   "UNDEF_2A",   "SW      ",
      "UNDEF_2C",   "UNDEF_2D",   "SF      ",   "SD      ",
      "SEQUI   ",   "SNEUI   ",   "SLTUI   ",   "SGTUI   ",
      "SLEUI   ",   "SGEUI   ",   "UNDEF_36",   "UNDEF_37",
      "UNDEF_38",   "UNDEF_39",   "UNDEF_3A",   "UNDEF_3B",
      "UNDEF_3C",   "UNDEF_3D",   "UNDEF_3E",   "UNDEF_3F" );
      
  constant sp_func_names : sp_func_name_array :=
    ( "NOP     ",   "UNDEF_01",   "UNDEF_02",   "UNDEF_03",
      "SLL     ",   "UNDEF_05",   "SRL     ",   "SRA     ",
      "UNDEF_08",   "UNDEF_09",   "UNDEF_0A",   "UNDEF_0B",
      "UNDEF_0C",   "UNDEF_0D",   "UNDEF_0E",   "UNDEF_0F",
      "SEQU    ",   "SNEU    ",   "SLTU    ",   "SGTU    ",
      "SLEU    ",   "SGEU    ",   "UNDEF_16",   "UNDEF_17",
      "UNDEF_18",   "UNDEF_19",   "UNDEF_1A",   "UNDEF_1B",
      "UNDEF_1C",   "UNDEF_1D",   "UNDEF_1E",   "UNDEF_1F",
      "ADD     ",   "ADDU    ",   "SUB     ",   "SUBU    ",
      "AND     ",   "OR      ",   "XOR     ",   "UNDEF_27",
      "SEQ     ",   "SNE     ",   "SLT     ",   "SGT     ",
      "SLE     ",   "SGE     ",   "UNDEF_2E",   "UNDEF_2F",
      "MOVI2S  ",   "MOVS2I  ",   "MOVF    ",   "MOVD    ",
      "MOVFP2I ",   "MOVI2FP ",   "UNDEF_36",   "UNDEF_37",
      "UNDEF_38",   "UNDEF_39",   "UNDEF_3A",   "UNDEF_3B",
      "UNDEF_3C",   "UNDEF_3D",   "UNDEF_3E",   "UNDEF_3F" );

  constant fp_func_names : fp_func_name_array :=
    ( "ADDF    ",   "SUBF    ",   "MULTF   ",   "DIVF    ",
      "ADDD    ",   "SUBD    ",   "MULTD   ",   "DIVD    ",
      "CVTF2D  ",   "CVTF2I  ",   "CVTD2F  ",   "CVTD2I  ",
      "CVTI2F  ",   "CVTI2D  ",   "MULT    ",   "DIV     ",
      "EQF     ",   "NEF     ",   "LTF     ",   "GTF     ",
      "LEF     ",   "GEF     ",   "MULTU   ",   "DIVU    ",
      "EQD     ",   "NED     ",   "LTD     ",   "GTD     ",
      "LED     ",   "GED     ",   "UNDEF_1E",   "UNDEF_1F" );


  procedure write_reg (L : inout line; reg : reg_index) is
    
  begin
    write(L, 'R');
    write(L, reg);
  end write_reg;


  procedure write_freg (L : inout line; freg : reg_index) is
    
  begin
    write(L, 'F');
    write(L, freg);
  end write_freg;


  procedure write_special_reg (L : inout line; reg : reg_index) is
    
  begin
    case reg is
      when 0 =>
        write(L, string'("IAR"));
      WHEN 1 =>
        write(L, string'("FSR"));
      when others =>
        write(L, string'("SR"));
        write(L, reg);
    end case;
  end write_special_reg;


  procedure write_instr (L : inout line; instr : in dlx_word) is

    alias instr_opcode : dlx_opcode is instr(0 to 5);
    alias instr_sp_func : dlx_sp_func is instr(26 to 31);
    alias instr_fp_func : dlx_fp_func is instr(27 to 31);
    alias instr_rs1 : dlx_reg_addr is instr(6 to 10);
    alias instr_rs2 : dlx_reg_addr is instr(11 to 15);
    alias instr_Itype_rd : dlx_reg_addr is instr(11 to 15);
    alias instr_Rtype_rd : dlx_reg_addr is instr(16 to 20);
    alias instr_immed16 : dlx_immed16 is instr(16 to 31);
    alias instr_immed26 : dlx_immed26 is instr(6 to 31);

    variable instr_opcode_num : dlx_opcode_num;
    variable instr_sp_func_num : dlx_sp_func_num;
    variable instr_fp_func_num : dlx_fp_func_num;
    variable rs1 : reg_index;
    variable rs2 : reg_index;
    variable Itype_rd : reg_index;
    variable Rtype_rd : reg_index;

  begin
    instr_opcode_num := bv_to_natural(instr_opcode);
    instr_sp_func_num := bv_to_natural(instr_sp_func);
    instr_fp_func_num := bv_to_natural(instr_fp_func);
    rs1 := bv_to_natural(instr_rs1);
    rs2 := bv_to_natural(instr_rs2);
    Itype_rd := bv_to_natural(instr_Itype_rd);
    Rtype_rd := bv_to_natural(instr_Rtype_rd);
    --
    if (instr_opcode /= op_special) and (instr_opcode /= op_fparith) then
      write(L, opcode_names(instr_opcode_num));
      write(L, ' ');
    end if;
    case instr_opcode is
      when op_special =>
        write(L, sp_func_names(instr_sp_func_num));
        write(L, ' ');
        case instr_sp_func is
          when sp_func_nop =>
            null;
          when sp_func_sll | sp_func_srl | sp_func_sra |
              sp_func_sequ | sp_func_sneu | sp_func_sltu |
              sp_func_sgtu | sp_func_sleu | sp_func_sgeu |
              sp_func_add | sp_func_addu | sp_func_sub | sp_func_subu |
              sp_func_and | sp_func_or | sp_func_xor |
              sp_func_seq | sp_func_sne | sp_func_slt |
              sp_func_sgt | sp_func_sle | sp_func_sge =>
            write_reg(L, Rtype_rd); write(L, string'(", "));
            write_reg(L, rs1); write(L, string'(", "));
            write_reg(L, rs2);
          when sp_func_movi2s =>
            write_special_reg(L, Rtype_rd); write(L, string'(", "));
            write_reg(L, rs1);
          when sp_func_movs2i =>
            write_reg(L, Rtype_rd); write(L, string'(", "));
            write_special_reg(L, rs1);
          when sp_func_movf | sp_func_movd =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1);
          when sp_func_movfp2i =>
            write_reg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1);
          when sp_func_movi2fp =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_reg(L, rs1);
          when others =>
            null;
        end case;
      when op_fparith =>
        write(L, fp_func_names(instr_fp_func_num));
        write(L, ' ');
        case instr_fp_func is
          when fp_func_addf | fp_func_subf | fp_func_multf | fp_func_divf |
              fp_func_addd | fp_func_subd | fp_func_multd | fp_func_divd |
              fp_func_mult | fp_func_div | fp_func_multu | fp_func_divu =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1); write(L, string'(", "));
            write_freg(L, rs2);
          when fp_func_cvtf2d | fp_func_cvtd2f =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1);
          when fp_func_cvtf2i | fp_func_cvtd2i =>
            write_reg(L, Rtype_rd); write(L, string'(", "));
            write_freg(L, rs1);
          when fp_func_cvti2f | fp_func_cvti2d =>
            write_freg(L, Rtype_rd); write(L, string'(", "));
            write_reg(L, rs1);
          when fp_func_eqf | fp_func_nef | fp_func_ltf |
              fp_func_gtf | fp_func_lef | fp_func_gef |
              fp_func_eqd | fp_func_ned | fp_func_ltd |
              fp_func_gtd | fp_func_led | fp_func_ged =>
            write_freg(L, rs1); write(L, string'(", "));
            write_freg(L, rs2);
          when others =>
            null;
        end case;
      when op_j  | op_jal =>
        write(L, bv_to_integer(instr_immed26));
      when op_beqz | op_bnez =>
        write_reg(L, rs1); write(L, string'(", "));
        write(L, bv_to_integer(instr_immed16));
      when op_bfpt | op_bfpf =>
        write(L, bv_to_integer(instr_immed16));
      when op_slli | op_srli | op_srai =>
        write_reg(L, Itype_rd); write(L, string'(", "));
        write_reg(L, rs1); write(L, string'(", "));
        write(L, bv_to_natural(instr_immed16(11 to 15)));
      when op_addi | op_subi |
          op_seqi | op_snei | op_slti | op_sgti | op_slei | op_sgei =>
        write_reg(L, Itype_rd); write(L, string'(", "));
        write_reg(L, rs1); write(L, string'(", "));
        write(L, bv_to_integer(instr_immed16));
      when op_addui | op_subui | op_andi | op_ori | op_xori |
          op_sequi | op_sneui | op_sltui | op_sgtui | op_sleui | op_sgeui =>
        write_reg(L, Itype_rd); write(L, string'(", "));
        write_reg(L, rs1); write(L, string'(", "));
        write(L, bv_to_natural(instr_immed16));
      when op_lhi =>
        write_reg(L, Itype_rd); write(L, string'(", "));
        write(L, bv_to_natural(instr_immed16));
      when op_rfe =>
        null;
      when op_trap =>
        write(L, bv_to_natural(instr_immed26));
      when op_jr | op_jalr =>
        write_reg(L, rs1);
      when op_lb | op_lh | op_lw |
          op_lbu | op_lhu | op_lf | op_ld =>
        write_reg(L, Itype_rd); write(L, string'(", "));
        write(L, bv_to_integer(instr_immed16)); write(L, '(');
        write_reg(L, rs1); write(L, ')');
      when op_sb | op_sh | op_sw | op_sf | op_sd =>
        write(L, bv_to_integer(instr_immed16));
        write(L, '('); write_reg(L, rs1); write(L, string'("), "));
        write_reg(L, Itype_rd);
      when others =>
        null;
    end case;
  end write_instr;


end dlx_instr;
