#include <stdio.h>
#include <memory.h>
#include <string.h>
#include <ctype.h>
#include "cmdline.h"
#include "umem.h"
#include "module.h"
#include "data.h"
#include "section.h"
#include "public.h"
#include "extern.h"
#include "errors.h"
#include "lines.h"
#include "types.h"

extern SECTION **sectiontable;
extern BOOL prm_relocatable;
extern PUBLIC **publictable;
extern BOOL bigend;
extern int maus;
extern unsigned long lconfig;
extern char modname[];

SECTION *SectionPtr;
static long base_address;
static BYTE *memptr;
static int count;
static char *fixname;

/* type support for 8051 bits is hacked */

static void isloaded(SECTION *p,BOOL flag)
{
	if ((flag & !p) || (p && !p->parent->loaded))
		Error("Access to unloaded region %s in module %s",p->parent->name,modname);
}
static void outbyte(char ch)
{
	*memptr++ = ch;
	if (!--count) {
		memptr = PtrToEMSMem(SectionPtr->parent->buffer, base_address);
		base_address += MAX_EMS_READWRITE;
		count = MAX_EMS_READWRITE;
	}
}
EXPRESSION *readnumber(BYTE *bp)
{
	EXPRESSION *p;
	long val = 0;
	while (isxdigit(*bp)) {
		BYTE temp = *bp++ - '0';
		if (temp > 9)
			temp -= 7;
		val *= 16;
		val += temp;
	}
	if (*bp)
		return 0;
	p = AllocateMemory(sizeof(EXPRESSION));
	p->left = p->right = 0;
	p->value = val;
	return(p);
}
static EXPRESSION *onExpression(BYTE **ptr)
{
	BYTE buffer[120], *bp= buffer+19;
	EXPRESSION *p=0;
	*bp-- = 0;
  while(isalnum(**ptr) || **ptr == '+' || **ptr == '-' || **ptr == '/' || **ptr == '*')
		*bp-- = *(*ptr)--;
	if (*(++bp))
		if (*bp == '+' || *bp == '-' || *bp == '*' || *bp == '/') {
			if (**ptr != ',')
				return(0);
			(*ptr)--;
			p = AllocateMemory(sizeof(EXPRESSION));
			switch(*bp) {
				case '+':
					p->type = EXP_PLUS;
					break;
				case '-':
					p->type = EXP_MINUS;
					break;
				case '*':
					p->type = EXP_TIMES;
					break;
				case '/':
					p->type = EXP_DIVIDE;
					break;
			}
			p->left = p->right = 0;
			p->right = onExpression(ptr);
			if (**ptr != ',') {
				DeallocateMemory(p);
				return(0);
			}
			(*ptr)--;
			p->left = onExpression(ptr);
		}
		else
			if (isxdigit(*bp)) {
				p = readnumber(bp);
				if (p)
					p->type = EXP_NUMBER;
			}
			else
				if (*bp == 'P') {
					if (!*++bp ) {
						p = AllocateMemory(sizeof(EXPRESSION));
						p->left = p->right = 0;
						p->type = EXP_PC;
					}
				}
				else {
					BYTE type;
					p = 0;
					switch (*bp) {
						case 'X':
							type = EXP_EXTERN;
							break;
						case 'R':
							type = EXP_REL;
							break;
						case 'L':
							type = EXP_LOWLIMIT;
							break;
						case 'Y':
							type = EXP_EXTERNSEG;
							break;
						default:
							BadObjectFile();
					}
					p = readnumber(++bp);
					if (p)
						p->type = type;
				}
	return(p);
}
static BYTE *ReadExpression(EXPRESSION **q, BYTE *ptr)
{
	*q = onExpression(&ptr);
	if (!*q)
		BadObjectFile();
	return(ptr);
}
void DeleteExpression(EXPRESSION *exp)
{
	if (exp->left)
		DeleteExpression(exp->left);
	if (exp->right)
		DeleteExpression(exp->right);
	DeallocateMemory(exp);
}
void SectionStart(int mode)
{
	int id;
	FixLines();
	id = ReadNumber(0);
	CheckForPeriod();
	SectionPtr = sectiontable[id];
	base_address = SectionPtr->base;
	memptr = PtrToEMSMem(SectionPtr->parent->buffer,base_address);
	count = MAX_EMS_READWRITE;
}
void EnumeratedData(int mode)
{
	BYTE ch = ReadChar();
	while (ch != '.' && ch > 31) {
		int data;
		putbackch(ch);
		data = ReadNumber(2);
		if (mode == RESOLVE) {
			outbyte(data);
		}
		ch = ReadChar();
		base_address++;
	}
	if (ch < 31)
		BadObjectFile();
}
unsigned long doswap(long spec, long old)
{
  unsigned long val = 0;
	int i;
	for (i=0; i< 4; i++) {
		int grab = spec & 0x0f;
		spec = spec >> 4;
		val = val >> 8;
		switch (grab) {
			case 0:
				val |= (old & 0xff) << 24;
				break;
			case 1:
				val |= (old & 0xff00) <<16;
				break;
			case 2:
				val |= (old & 0xff0000L) << 8;
				break;
			case 3:
				val |= (old & 0xff000000L);
				break;
		}
	}
	return(val);
}
static int outfixupdata(long value, long len, long address, BOOL actual)
{
	if (len > 0xffff) {
		value = doswap(len & 0xffff,value);
		len = len >> 16;
	}
	if (len == 0x1100) {
		len = 1;
		if (address & 1) {
			Error("Fixup must be even in module %s section %s",SectionPtr->modname, SectionPtr->name);
			outbyte(0);
			return 1;
		}
	}
	if (len == 0x1202) {
		len = 1;
		if (((address+1)>> 8) != (value >> 8)) 
			goto rerr1;
	}
	if (len == 1) {
		if (value & 0xffffff00L) {
rerr1:
		  Error("Fixup to %s out of range in module %s section %s",fixname,SectionPtr->modname, SectionPtr->name);
		  outbyte(0);
		}
		else
			outbyte(value);
		return 1;
	}
	if (len == 2 || len == 0x1000 || len == 0x1001 || len == 0x1200 || len == 0x1201) {
		if (actual && (len == 0x1000 || len == 0x1001)) {
			if (((address + 2) & 0xf800) != (value & 0xf800))
				goto rerr;
			value = ((value & 0x700) << 5) + (value & 0xff) + 0x100 + ((len & 1) << 12);
out2:
			outbyte((value >> 8) & 0xff);
			outbyte((value) & 0xff);
		}		
		else if (actual && (len == 0x1200 || len == 0x1201)) {
			if (((address +2) & 0xf000) != (value & 0xf000))
				goto rerr;
			value = (value & 0xfff) + 0x2000 + ((len & 1) << 12);
			goto out2;
		}
		else if (bigend)
			if (((lconfig & CF_SIGNEDADDR) && ((value & 0x7fff) == value ) 
						|| ((value & 0xffff8000L) == 0xffff8000L))
					|| (!(lconfig & CF_SIGNEDADDR) && !(value &0xffff0000L))){
				outbyte((value >> 8) & 0xff);
				outbyte((value) & 0xff);
			}
			else
				goto rerr;
		else {
			if (!(value &0xffff0000L)) {
				outbyte((value) & 0xff);
				outbyte((value >> 8) & 0xff);
			}
			else {
rerr:
		  	Error("Fixup to %s out of range in module %s section %s",fixname,SectionPtr->modname, SectionPtr->name);
				outbyte(0);
				outbyte(0);
			}
		}
		return(2);
	}
	else
		if (len == 4) {
			if (bigend) {
				outbyte((value >> 24) & 0xff);
				outbyte((value >> 16) & 0xff);
				outbyte((value >> 8) & 0xff);
				outbyte((value) & 0xff);
			}
			else {
				outbyte((value) & 0xff);
				outbyte((value >> 8) & 0xff);
				outbyte((value >> 16) & 0xff);
				outbyte((value >> 24) & 0xff);
			}
			return(4);
		}
		else
			BadObjectFile();
	return(0);
}
static void math(EXPRESSION *e, BOOL rv)
{
	if (rv) {
		e->type = EXP_NUMBER;
		DeallocateMemory(e->right);
		DeallocateMemory(e->left);
		e->left = e->right = 0;
	}
}
static BOOL evalextern(EXPRESSION *e, BOOL mustabs, BOOL rv)
{
	EXTERN *x = GetExtern(e->value);
	if (x->pub) {
		PUBLIC *p = x->pub;
		isloaded(p->sect,FALSE);
		fixname = p->name;
		p->referenced = TRUE;
		if (!p->sect) {
			e->type = EXP_NUMBER;
			e->value = p->offset;
		}
		else if (mustabs) {
			if (lconfig & CF_8051 && p->datatype == TY_BIT) {
				e->value = p->offset;
				if (p->sect)
					e->value +=  (p->sect->base + p->sect->parent->absbase - p->sect->parent->virtual - 32) * 8;
				if (e->value >= 0x80)
				  Error("8051 bit fixup to %s out of range in module %s section %s",fixname,SectionPtr->modname, SectionPtr->name);
			}
			else {
				e->value = p->offset;
				if (p->sect)
					e->value +=  p->sect->base + p->sect->parent->absbase - p->sect->parent->virtual;
			}
		}
		else {
			e->type = EXP_PLUS;
			e->left = AllocateMemory(sizeof(EXPRESSION));
			e->left->left = e->left->right = 0;
			e->right = AllocateMemory(sizeof(EXPRESSION));
			e->right->left = e->right->right = 0;
			e->left->type = EXP_REL;
			e->left->value = p->sect->parent->section;
			e->right->type = EXP_NUMBER;
			e->right->value = p->offset +p->sect->base;
			rv = FALSE;
		}
	}
	else {
		if (!prm_relocatable)
			Error("Undefined external %s in module %s",x->name,x->modname);
		e->value = x->id;
		rv = FALSE;
	}
	return rv;
}
static int evalexternseg(EXPRESSION *e, BOOL mustabs,BOOL rv)
{
 	EXTERN *x = GetExtern(e->value);
 	if (x->pub) {
 		PUBLIC *p = x->pub;
		isloaded(p->sect,FALSE);
 		fixname = p->name;
 		p->referenced = TRUE;
 		if (!p->sect) {
 			e->type = EXP_NUMBER;
 			e->value = 0;
 		}
 		else if (mustabs) {
 			e->type = EXP_NUMBER;
 			e->value = p->sect->parent->absbase/16;
 		} else {
 			e->type = EXP_DIVIDE;
 			e->left = AllocateMemory(sizeof(EXPRESSION));
 			e->left->left = e->left->right = 0;
 			e->right = AllocateMemory(sizeof(EXPRESSION));
 			e->right->left = e->right->right = 0;
 			e->left->type = EXP_LOWLIMIT;
 			e->left->value = p->sect->parent->section;
 			e->right->type = EXP_NUMBER;
 			e->right->value = 16;
 		}
 	}
 	else {
 		if (!prm_relocatable)
 			Error("Undefined external %s in module %s",x->name,x->modname);
 		e->value = x->id;
 		rv = FALSE;
 	}
	return rv;
}
static BOOL evalExpr(EXPRESSION *e, long rel, BOOL mustabs)
{
	BOOL rv = TRUE;
	if (e->left)
  		rv = rv && evalExpr(e->left, rel, mustabs);
	if (e->right)
		rv = rv && evalExpr(e->right, rel, mustabs);
	switch(e->type) {
		case EXP_REL:
			rv = mustabs;
			if (mustabs) {
				SECTION *p = sectiontable[e->value];
				isloaded(p,TRUE);
				e->type = EXP_NUMBER;
				e->value = p->base;
				e->value +=p->parent->absbase - p->parent->virtual;
			}
			else {
				SECTION *s = sectiontable[e->value];
				isloaded(s,TRUE);
				e->left = AllocateMemory(sizeof(EXPRESSION));
				e->left->left = e->left->right = 0;
				e->right = AllocateMemory(sizeof(EXPRESSION));
				e->right->left = e->right->right = 0;
				e->left->value = s->parent->section;
				e->left->type = EXP_REL;
				e->right->value = s->base;
				e->right->value +=s->parent->absbase - s->parent->virtual;
				e->right->type = EXP_NUMBER;
				e->type = EXP_PLUS;
			}
			break;
		case EXP_PC:
			if (mustabs) {
				e->type = EXP_NUMBER;
				e->value = rel;
			}
			rv = mustabs;
			break;
		case EXP_EXTERN: 
			rv = evalextern(e,mustabs,rv);
			break;
		case EXP_EXTERNSEG: 
			rv = evalexternseg(e,mustabs,rv);
			break;
		case EXP_LOWLIMIT: 
			rv = mustabs;
			if (mustabs) {
				SECTION *s = sectiontable[e->value];
				isloaded(s,TRUE);
				e->type = EXP_NUMBER;
				e->value = s->parent->absbase;
			}
			break;
		case EXP_NUMBER:
			rv = TRUE;
			break;
		case EXP_PLUS:
			e->value = e->left->value + e->right->value;
			math(e,rv);
			break;
			
		case EXP_MINUS:
			e->value = e->left->value - e->right->value;
			math(e,rv);
			break;
		case EXP_TIMES:
			e->value = e->left->value * e->right->value;
			math(e,rv);
			break;
		case EXP_DIVIDE:
			e->value = e->left->value / e->right->value;
			math(e,rv);
			break;
	}
	return(rv);
}
static void ApplyFixup(FIXUPS *fixdata, long addend)
{
	if (!evalExpr(fixdata->size,0,TRUE))
		fatal("Internal error 2");
	if ((fixdata->size->value <100 || !prm_relocatable) && evalExpr(fixdata->exp,fixdata->address+addend,!prm_relocatable)) 
		fixdata->size->value = outfixupdata(fixdata->exp->value,fixdata->size->value,fixdata->address,TRUE);
	else {
		FIXUPS *fd;
		LIST *l;
		fixdata->size->value = outfixupdata(0L,fixdata->size->value,fixdata->address,FALSE);
		fd = AllocateMemory(sizeof(FIXUPS));
		l = AllocateMemory(sizeof(LIST));
		l->data = fd;
		l->link = 0;
		memcpy(fd,fixdata,sizeof(FIXUPS));
		*(SectionPtr->parent->fixupptr) = l;
		SectionPtr->parent->fixupptr =& l->link;
	}
	base_address += fixdata->size->value;
}
void ReadFixup(int mode)
{
	BYTE ch = ReadChar();
	BYTE buffer[100], *ptr = buffer;
	if (ch != '(')
		BadObjectFile();
	*ptr++=ch;
	while((ch = ReadChar()) != ')')
		*ptr++=ch;
	*ptr++ = 0;
	CheckForPeriod();
	if (mode == RESOLVE) {
		EXPRESSION *size, *data;
		FIXUPS fixdata;
			ptr -=2;
			ptr = ReadExpression(&size,ptr);
			if (*ptr == ',')
				ptr--;
			else
				BadObjectFile();
			ptr = ReadExpression(&data,ptr);
			fixdata.size = size;
			fixdata.exp = data;
			fixdata.address = base_address + SectionPtr->parent->absbase;
			fixdata.section = SectionPtr->parent->section;
			if (*ptr != '(')
				BadObjectFile();
			ApplyFixup(&fixdata,-SectionPtr->parent->virtual);
	}
}