WITH Ada.Text_IO;
WITH Ada.Integer_Text_IO;
WITH Ada.Float_Text_IO;
WITH Sort_Generic;
PROCEDURE Test_Sort_Generic IS
------------------------------------------------------------------
--| Demonstrates Sort_Generic using two unrelated kinds of lists;
--| this is not a realistic application, but rather just shows that
--| many instances of a generic can occur within one client program.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: September 1995                                     
------------------------------------------------------------------

  SUBTYPE Index IS Integer RANGE 1..10;
  TYPE    FloatVector IS ARRAY(Index RANGE <>) OF Float;

  V1 : FloatVector(1..10);

  SUBTYPE DayRange IS Natural  RANGE 0..6;
  SUBTYPE Weekdays IS DayRange RANGE 0..4;
  SUBTYPE Weekend  IS DayRange RANGE 5..6;

  TYPE Days IS (Mon, Tue, Wed, Thu, Fri, Sat, Sun);
  TYPE CallRecord IS RECORD
    DayOfWeek    : Days;
    NumberOfCalls: Natural;
  END RECORD;
    
  TYPE Callers IS ARRAY(DayRange RANGE <>) of CallRecord;

  PACKAGE Days_IO IS NEW Ada.Text_IO.Enumeration_IO(Enum => Days);

  ThisWeek:       Callers(DayRange);

  -- if we are going to sort CallRecords, 
  -- we need to know how to compare them

  FUNCTION "<" (L, R: CallRecord) RETURN Boolean IS
  BEGIN
    RETURN L.NumberOfCalls < R.NumberOfCalls;
  END "<";

  FUNCTION ">" (L, R: CallRecord) RETURN Boolean IS
  BEGIN
    RETURN L.NumberOfCalls > R.NumberOfCalls;
  END ">";

  -- local procedures to display the contents of two kinds of lists

  PROCEDURE DisplayCallers (List: Callers) IS
  BEGIN -- DisplayCallers
    FOR Count IN List'Range LOOP
      Days_IO.Put  (Item=>List(Count).DayOfWeek, Width=>3);
      Ada.Integer_Text_IO.Put(Item=>List(Count).NumberOfCalls, Width=>4);
      Ada.Text_IO.New_Line;
    END LOOP;
    Ada.Text_IO.New_Line;
  END DisplayCallers;

  PROCEDURE DisplayFloatVector (V: FloatVector) IS
  BEGIN
    FOR Count IN V'First..V'Last LOOP
      Ada.Float_Text_IO.Put(Item=>V(Count), Fore=>4, Aft=>2, Exp=>0);
    END LOOP;
    Ada.Text_IO.New_Line;
  END DisplayFloatVector;

  -- two instances of Sort_Generic for Float vectors;
  -- the first sorts in increasing order, the second in decreasing order

  PROCEDURE SortUpFloat IS NEW Sort_Generic
    (ElementType => Float,
     IndexType   => Index,
     ListType    => FloatVector,
     Compare     => "<");

  PROCEDURE SortDownFloat IS NEW Sort_Generic
    (ElementType => Float,
     IndexType   => Index,
     ListType    => FloatVector,
     Compare     => ">");

  -- two instances of Sort_Generic for Callers;
  -- the first sorts in increasing order, the second in decreasing order

  PROCEDURE SortUpCallers IS NEW Sort_Generic
    (ElementType => CallRecord,
     IndexType   => DayRange,
     ListType    => Callers,
     Compare     => "<");

  PROCEDURE SortDownCallers IS NEW Sort_Generic
    (ElementType => CallRecord,
     IndexType   => DayRange,
     ListType    => Callers,
     Compare     => ">");

BEGIN -- Test_Sort_Generic

  V1 := (0.7, 1.5, 6.9, -3.2, 0.0, 5.1, 2.0, 7.3, 2.2, -5.9);
  Ada.Text_IO.New_Line;
  Ada.Text_IO.Put(Item=> "Testing Sort_Generic for float vectors");
  Ada.Text_IO.New_Line;
  Ada.Text_IO.Put(Item=> "Here is the vector before sorting.");
  Ada.Text_IO.New_Line;
  DisplayFloatVector(V => V1);
  Ada.Text_IO.New_Line;
  
  SortUpFloat(List => V1);
  Ada.Text_IO.Put(Item=> "Here is the vector after upward sorting.");
  Ada.Text_IO.New_Line;
  DisplayFloatVector(V => V1);
  Ada.Text_IO.New_Line;

  SortDownFloat(List => V1);
  Ada.Text_IO.Put(Item=> "Here is the vector after downward sorting.");
  Ada.Text_IO.New_Line;
  DisplayFloatVector(V => V1);
  Ada.Text_IO.New_Line;

  ThisWeek := ((Mon, 12), (Tue, 23), (Wed, 100), (Thu, 40), 
               (Fri, 52), (Sat, 17), (Sun,   2));

  Ada.Text_IO.Put(Item=> "Testing Sort_Generic for telephone callers");
  Ada.Text_IO.New_Line;
  Ada.Text_IO.Put(Item=> "Here is ThisWeek before sorting.");
  Ada.Text_IO.New_Line;
  DisplayCallers(List => ThisWeek);
  Ada.Text_IO.New_Line;
  
  SortUpCallers(List => ThisWeek);
  Ada.Text_IO.Put(Item=> "Here is ThisWeek after upward sorting.");
  Ada.Text_IO.New_Line;
  DisplayCallers(List => ThisWeek);
  Ada.Text_IO.New_Line;

  SortDownCallers(List => ThisWeek);
  Ada.Text_IO.Put(Item=> "Here is ThisWeek after downward sorting.");
  Ada.Text_IO.New_Line;
  DisplayCallers(List => ThisWeek);
  Ada.Text_IO.New_Line;


END Test_Sort_Generic;
