WITH Ada.Text_IO;
WITH Ada.Integer_Text_IO;
PROCEDURE Test_Blob_Size IS
------------------------------------------------------------------
--| Illustrates the recursive function BlobSize, which computes
--| size of a "blob" or group of filled cells on a grid.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: July 1995                                     
------------------------------------------------------------------
  
  TYPE Fill IS (Empty, Filled);
  TYPE BlobArray IS ARRAY (Integer RANGE <>,Integer RANGE <>) OF Fill;

  Test: BlobArray(1..5,1..5);

  PROCEDURE DisplayGrid(Grid: BlobArray) IS
  -- Pre:  Grid is defined
  -- Post: displays Grid on the screen

  BEGIN -- DisplayGrid

    FOR Column IN Grid'Range(2) LOOP -- top border
      Ada.Text_IO.Put(Item => "----");
    END LOOP;
    Ada.Text_IO.Put(Item => '-');
    Ada.Text_IO.New_Line;

    FOR Row IN Grid'Range(1) LOOP
      FOR Column IN Grid'Range(2) LOOP
        IF Grid(Row, Column) = Filled THEN
          Ada.Text_IO.Put(Item => "| X ");
        ELSE
          Ada.Text_IO.Put(Item => "|   ");
        END IF;
      END LOOP;
      Ada.Text_IO.Put(Item => '|');
      Ada.Text_IO.New_Line;

      FOR Column IN Grid'Range(2) LOOP -- after each row
        Ada.Text_IO.Put(Item => "----");
      END LOOP;
      Ada.Text_IO.Put(Item => '-');
      Ada.Text_IO.New_Line;
    END LOOP;

  END DisplayGrid;

  FUNCTION BlobSize(Grid : BlobArray; X, Y: Integer) RETURN Natural IS
  -- Counts the number of filled cells in the blob containing
  -- point (X, Y).
  -- Pre : Blob array Grid and point (X,Y) are defined.
  -- Post: Returns the size of the blob containing point (X, Y).
  --       Resets the status of each cell in the blob to Empty.

    CopyOfGrid : BlobArray(Grid'Range(1),Grid'Range(2)); 
                             -- because functions can't modify
                             -- their parameters, in Ada

    FUNCTION Blob (X, Y : Integer) RETURN Natural IS
      
    -- Inner function that performs the counting operation for BlobSize
    -- Pre : Global array CopyOfGrid and point (X,Y) are defined.
    -- Post: Returns the size of the blob containing point (X, Y).
    --       Resets the status of each cell in the blob to Empty.
      
    Result: Natural;

    BEGIN -- Blob   

      IF (X NOT IN CopyOfGrid'Range(1)) OR
         (Y NOT IN CopyOfGrid'Range(2)) THEN
        Result := 0;                        -- cell not in grid   
      ELSIF CopyOfGrid(X, Y) = Empty THEN
        Result := 0;                        -- cell is empty   
      ELSE -- cell is filled   
        -- recursive step   
        CopyOfGrid(X, Y) := Empty;
        Result := 1 + Blob(X-1, Y+1) + Blob(X, Y+1) +
                      Blob(X+1, Y+1) + Blob(X+1, Y) +
                      Blob(X+1, Y-1) + Blob(X, Y-1) +
                      Blob(X-1, Y-1) + Blob(X-1, Y);
      END IF;

      RETURN Result;

    END Blob;

  BEGIN

    CopyOfGrid := Grid;
    RETURN Blob(X,Y);

  END BlobSize;

BEGIN -- Test_Blob_Size

  Test := ((Empty,  Filled, Empty,  Empty,  Filled),
           (Filled, Empty,  Empty,  Filled, Filled),
           (Empty,  Empty,  Filled, Filled, Empty ),
           (Filled, Empty,  Empty,  Empty,  Empty ),
           (Filled, Filled, Filled, Empty,  Empty ));

  DisplayGrid (Grid => Test);

  Ada.Text_IO.Put(Item => "BlobSize(3,4) is ");
  Ada.Integer_Text_IO.Put(Item => BlobSize(Test,3,4), Width => 1);
  Ada.Text_IO.New_Line;

  Ada.Text_IO.Put(Item => "BlobSize(1,2) is ");
  Ada.Integer_Text_IO.Put(Item => BlobSize(Test,1,2), Width => 1);
  Ada.Text_IO.New_Line;

  Ada.Text_IO.Put(Item => "BlobSize(5,5) is ");
  Ada.Integer_Text_IO.Put(Item => BlobSize(Test,5,5), Width => 1);
  Ada.Text_IO.New_Line;

  Ada.Text_IO.Put(Item => "BlobSize(5,1) is ");
  Ada.Integer_Text_IO.Put(Item => BlobSize(Test,5,1), Width => 1);
  Ada.Text_IO.New_Line;

END Test_Blob_Size;
