WITH Ada.Calendar;
WITH Ada.Text_IO;
WITH Ada.Integer_Text_IO;
PACKAGE BODY Dates.IO IS
------------------------------------------------------------------
--| body for child package to read and display calendar dates
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: July 1995                                     
------------------------------------------------------------------

  PACKAGE Month_IO IS 
    NEW Ada.Text_IO.Enumeration_IO(Enum => Months);

  PROCEDURE Get(File: IN Ada.Text_IO.File_Type; Item: OUT Date) IS

    M:    Months;
    D:    Day_Number;
    Y:    Year_Number;

  BEGIN -- Get

    Month_IO.Get (File => File, Item => M);
    Ada.Integer_Text_IO.Get(File => File, Item => D);
    Ada.Integer_Text_IO.Get(File => File, Item => Y);
    -- assert: M, D, and Y are well-formed and in range
    --         otherwise one of the Get's would raise an exception

    Item := Date_Of (Month => M, Year => Y, Day => D);
    -- assert: Item is a valid date if Date_Error not raised

  EXCEPTION

    WHEN Ada.Text_IO.Data_Error =>
      RAISE Date_Error;
    WHEN Constraint_Error =>
      RAISE Date_Error;
    WHEN Date_Error =>
      RAISE Date_Error;

  END Get;

  PROCEDURE WriteShort(File: IN Ada.Text_IO.File_Type; Item: IN Date) IS
  -- Pre: Item is assigned a value
  -- Post: Writes a date in dd MMM yy form

    Last2Digits : Natural;

  BEGIN -- WriteShort

    Last2Digits := Item.Year MOD 100;

    IF Item.Day < 10 THEN
      Ada.Text_IO.Put(File => File, Item => '0');
    END IF;
    Ada.Integer_Text_IO.Put(File => File, Item => Item.Day, Width => 1);
    Ada.Text_IO.Put(File => File, Item => ' '); 
    Month_IO.Put (File => File, Item => Item.Month, Width => 1);
    Ada.Text_IO.Put(File => File, Item => ' ');
    IF Last2Digits < 10 THEN
      Ada.Text_IO.Put(File => File, Item => '0');
    END IF;
    Ada.Integer_Text_IO.Put(File => File, Item => Last2Digits, Width => 1);

  END WriteShort;

  PROCEDURE WriteFull(File: IN Ada.Text_IO.File_Type; Item: IN Date) IS
  -- Pre: Item is assigned a value
  -- Post: Writes a date in Monthname dd, yyyy form

  BEGIN

    CASE Item.Month IS
      WHEN Jan =>
        Ada.Text_IO.Put(File => File, Item => "January");
      WHEN Feb =>
        Ada.Text_IO.Put(File => File, Item => "February");
      WHEN Mar =>
        Ada.Text_IO.Put(File => File, Item => "March");
      WHEN Apr =>
        Ada.Text_IO.Put(File => File, Item => "April");
      WHEN May =>
        Ada.Text_IO.Put(File => File, Item => "May");
      WHEN Jun =>
        Ada.Text_IO.Put(File => File, Item => "June");
      WHEN Jul =>
        Ada.Text_IO.Put(File => File, Item => "July");
      WHEN Aug =>
        Ada.Text_IO.Put(File => File, Item => "August");
      WHEN Sep =>
        Ada.Text_IO.Put(File => File, Item => "September");
      WHEN Oct =>
        Ada.Text_IO.Put(File => File, Item => "October");
      WHEN Nov =>
        Ada.Text_IO.Put(File => File, Item => "November");
      WHEN Dec =>
        Ada.Text_IO.Put(File => File, Item => "December");
    END CASE;

    Ada.Text_IO.Put(File => File, Item => ' ');
    Ada.Integer_Text_IO.Put(File => File, Item => Item.Day, Width => 1);
    Ada.Text_IO.Put(File => File, Item => ", "); 
    Ada.Integer_Text_IO.Put(File => File, Item => Item.Year, Width => 1);

  END WriteFull;

  PROCEDURE WriteNumeric(File: IN Ada.Text_IO.File_Type; Item: IN Date) IS
  -- Pre: Item is assigned a value
  -- Post: Writes a date in mm/dd/yy form

    Last2Digits : Natural;

  BEGIN

    Last2Digits := Item.Year MOD 100;

    Ada.Integer_Text_IO.Put 
      (File => File, Item => Months'Pos(Item.Month)+1, Width => 1);
    Ada.Text_IO.Put(File => File, Item => '/');
    Ada.Integer_Text_IO.Put(File => File, Item => Item.Day, Width => 1);
    Ada.Text_IO.Put(File => File, Item => '/'); 
    IF Last2Digits < 10 THEN
      Ada.Text_IO.Put(File => File, Item => '0');
    END IF;
    Ada.Integer_Text_IO.Put(File => File, Item => Last2Digits, Width => 1);

  END WriteNumeric;

  PROCEDURE Put(File: IN Ada.Text_IO.File_Type; 
                Item: IN Date; Format: IN Formats) IS
  BEGIN -- Put
    CASE Format IS
      WHEN Short =>
        WriteShort(File => File, Item => Item);
      WHEN Full =>
        WriteFull(File => File, Item => Item);
      WHEN Numeric =>
        WriteNumeric(File => File, Item => Item);
    END CASE;
  END Put;

  PROCEDURE Get(Item: OUT Date) IS
  BEGIN -- Get
    Get(File => Ada.Text_IO.Standard_Input, Item => Item);
  END Get;

  PROCEDURE Put(Item: IN Date; Format: IN Formats) IS
  BEGIN -- Put
    Put(File => Ada.Text_IO.Standard_Output, Item => Item, Format => Format);
  END Put;

END Dates.IO;
