WITH Ada.Text_IO;
WITH Screen;
PROCEDURE Task_Array IS
------------------------------------------------------------------
--| Shows tasks writing into their respective columns on the
--| screen. This time we use a protected type, whose procedure
--| can be executed by only one task at a time.
--| The task objects are stored in an array, and receive their
--| configuration parameters through "start buttons" rather than
--| discriminants.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: December 1995                                     
------------------------------------------------------------------

  PROTECTED TYPE ScreenManagerType IS

  -- If multiple calls of Write are made simultaneously, each is 
  -- executed in its entirety before the next is begun.
  -- The Ada standard does not specify an order of execution.

    PROCEDURE Write (Item:   IN String; 
                     Row:    IN Screen.Depth; 
                     Column: IN Screen.Width);

  END ScreenManagerType;

  PROTECTED BODY ScreenManagerType IS

    PROCEDURE Write (Item:   IN String; 
                     Row:    IN Screen.Depth; 
                     Column: IN Screen.Width) IS
    BEGIN -- Write

      Screen.MoveCursor(Row => Row, Column => Column);
      Ada.Text_IO.Put(Item => Item);

    END Write;

  END ScreenManagerType;

  Manager: ScreenManagerType;

  TASK TYPE SimpleTask IS
                        
    -- Task receives its parameters through a start-button entry 
    -- instead of discriminants This is more flexible..
    ENTRY StartRunning (Message: Character; 
                        HowMany: Screen.Depth;
                        Column:  Screen.Width);

  END SimpleTask;

  TASK BODY SimpleTask IS

    MyMessage: Character;
    MyCount  : Screen.Depth;
    MyColumn : Screen.Width;

  BEGIN -- SimpleTask
    
    -- Each task will write its message in its own column
    -- Now the task locks the screen before moving the cursor,
    -- unlocking it when writing is completed.

    ACCEPT StartRunning (Message: Character; 
                        HowMany: Screen.Depth;
                        Column:  Screen.Width) DO
      MyMessage := Message;
      MyCount   := HowMany;
      MyColumn  := Column;

    END StartRunning;

    FOR Count IN 1..MyCount LOOP

      -- No need to lock the screen explicitly; just call the
      -- protected procedure.
      Manager.Write(Row => Count, Column => MyColumn,
                    Item => "Hello from Task " & MyMessage);

      DELAY 0.5;            -- lets another task have the CPU
    END LOOP;

  END SimpleTask;

  Family: ARRAY (1..3) OF SimpleTask;
  Char  : CONSTANT Character := 'A';

BEGIN -- Task_Array;
  
  Screen.ClearScreen;
  FOR Which IN Family'Range LOOP
    Family(Which).StartRunning 
      (Message => Character'Val(Character'Pos(Char) + Which),
       HowMany => 3 * Which,
       Column  => 3 + (24 * (Which - 1)));
  END LOOP;

END Task_Array;
