WITH Society;
WITH Room;
WITH Ada.Numerics.Discrete_Random;
PACKAGE BODY Phil IS
------------------------------------------------------------------------
--| Dining Philosophers - Ada 95 edition
--| Philosopher is an Ada 95 task type with discriminant.
--| Chopsticks are assigned by a higher authority, which
--|   can vary the assignments to show different algorithms.
--| Philosopher always grabs First_Grab, then Second_Grab.
--| Philosopher is oblivious to outside world, but needs to
--|   communicate its life-cycle events to the Maitre_D.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: January 1996                                     
------------------------------------------------------------------------

  SUBTYPE Think_Times IS Positive RANGE 1..8;
  PACKAGE Think_Length IS 
    new Ada.Numerics.Discrete_Random (Result_Subtype => Think_Times);

  SUBTYPE Meal_Times IS Positive RANGE 1..10;
  PACKAGE Meal_Length IS
    new Ada.Numerics.Discrete_Random (Result_Subtype => Meal_Times);

  TASK BODY Philosopher IS  -- My_ID is discriminant
 
    SUBTYPE Life_Time IS Positive RANGE 1..5;
 
    Who_Am_I    : Society.Unique_DNA_Codes := My_ID; -- discrim
    First_Grab  : Positive;
    Second_Grab : Positive;
    Meal_Time   : Meal_Times; 
    Think_Time  : Think_Times;
    T           : Think_Length.Generator;
    M           : Meal_Length.Generator;

  BEGIN

    Think_Length.Reset(T);
    Meal_Length.Reset(M);

     -- get assigned the first and second chopsticks here

    ACCEPT Start_Eating (Chopstick1 : IN Positive;
                         Chopstick2 : IN Positive) do
      First_Grab  := Chopstick1;
      Second_Grab := Chopstick2;
    END Start_Eating;
 
    Room.Maitre_D.Report_State (Who_Am_I, Breathing);
 
    FOR Meal IN Life_Time LOOP
 
      Room.Sticks (First_Grab).Pick_Up;
      Room.Maitre_D.Report_State (Who_Am_I, Got_One_Stick, First_Grab);
 
      Room.Sticks (Second_Grab).Pick_Up;
      Room.Maitre_D.Report_State (Who_Am_I, Got_Other_Stick, Second_Grab);
 
      Meal_Time := Meal_Length.Random(M);
      Room.Maitre_D.Report_State (Who_Am_I, Eating, Meal_Time, Meal);
 
      DELAY Duration (Meal_Time);
 
      Room.Maitre_D.Report_State (Who_Am_I, Done_Eating);

      Room.Sticks (First_Grab).Put_Down;
      Room.Sticks (Second_Grab).Put_Down;
 
      Think_Time := Think_Length.Random(T); 
      Room.Maitre_D.Report_State (Who_Am_I, Thinking, Think_Time);
      DELAY Duration (Think_Time);
 
    END LOOP;
 
    Room.Maitre_D.Report_State (Who_Am_I, Dying);

  END Philosopher;
 
END Phil;
