
------------------------------------------------------------
--        Name: Alden Dima
--      E-mail: dimaaa@seas.gwu.edu
--      School: The George Washington University
--              School of Engineering and Applied Science
--              Washington, D.C.
--       Class: CSci 298 - Independent Study
--     Project: Ada Curses Binding and Textual User Interface
--        File: tuidemo.adb 
--        Date: 12/24/95 
-- Description: Demo program for AdaTUI. It is an Ada 95
--              version of P.J. Kunst's demo program for TUI
--              version 1.02. Demonstrates AdaTUI's user
--              interface and implements a simple file browser. 
--   Revisions: 5/31/96 - AAD - Rewrote Ada-Curses binding to
--              enhance portability and maintainability.  Made
--              necessary changes in AdaTUI/TUIDemo to
--              accomodate new binding.  Made minor changes
--              to correct a problem using access types with
--              unconstrained arrays discovered by GNAT 3.03.
--              Eliminated several unused variables.
------------------------------------------------------------

with ada.characters.handling;
with adatui;
with bstrings;
with c;
with pdcurses;
with interfaces.c;
with text_io;

use type bstrings.bounded_string;
use interfaces.c;

procedure tuidemo is

   FIELD_WIDTH : constant positive := 50;
   FILE_NAME   : constant string := "tuidemo.adb";

   package tui is new adatui;

--
-- strings entry box
--
   procedure Address is
      name   : bstrings.bounded_string := bstrings.to_bounded_string("Name");
      street : bstrings.bounded_string := bstrings.to_bounded_string("Street");
      city   : bstrings.bounded_string := bstrings.to_bounded_string("City");
      state  : bstrings.bounded_string := bstrings.to_bounded_string("State");
      country: bstrings.bounded_string := bstrings.to_bounded_string("Country");

      name_buffer   : bstrings.bounded_string := bstrings.to_bounded_string("");
      street_buffer : bstrings.bounded_string := bstrings.to_bounded_string("");
      city_buffer   : bstrings.bounded_string := bstrings.to_bounded_string("");
      state_buffer  : bstrings.bounded_string := bstrings.to_bounded_string("");
      country_buffer: bstrings.bounded_string := bstrings.to_bounded_string("");
      print_buffer  : bstrings.bounded_string := bstrings.to_bounded_string("");
      blank_buffer  : bstrings.bounded_string := bstrings.to_bounded_string("");

      items : tui.input_items := (
	 0 => ( description => name,
	        buffer      => name_buffer ),

	 1 => ( description => street,
	        buffer      => street_buffer ),

	 2 => ( description => city,
	        buffer      => city_buffer ),

	 3 => ( description => state,
	        buffer      => state_buffer ),

	 4 => ( description => country,
	        buffer      => country_buffer ) );

	 wbody : pdcurses.A_WINDOW_T;
	 key   : integer;

	 temp_charv : aliased c.charv(1..80);
	 temp_charp : c.charp;
	 void       : c.signed_int;

   begin
      wbody := tui.body_win;

      tui.get_strings (
	 items => items, 
	 field => FIELD_WIDTH,
	 key   => key );

      if key /= tui.KEY_ESC then
	 for index in items'range loop
	    print_buffer := items(index).description;

	    bstrings.head (
	       source => print_buffer,
	       count  => 10 );

	    bstrings.append (
	       source   => print_buffer,
	       new_item => " : " );

	    bstrings.append (
	       source   => print_buffer,
	       new_item => items(index).buffer );

	    bstrings.append (
	       source   => print_buffer,
	       new_item => character'val(tui.NEWLINE) );
	 --
	 -- Using procedure form of interfaces.c.to_c to avoid raising
	 -- Constraint_Error due to mismatch in lengths of item and target.
	 -- 
	    declare
	       count : interfaces.c.size_t;
	    begin
	       interfaces.c.to_c (
		  item   => bstrings.to_string(print_buffer),
		  target => temp_charv,
		  count  => count );
	    end;

  	    temp_charp := temp_charv'unchecked_access;
  	    void := pdcurses.wprintw ( wbody, temp_charp );
	 end loop;
	 void := pdcurses.wrefresh ( wbody );
      end if;

      for index in items'range loop
	 items(index).buffer := blank_buffer;
      end loop;
   end Address;

--
-- string entry box
--
   procedure get_file_name (
      description : in     bstrings.bounded_string;
      file_name   :    out bstrings.bounded_string;
      field       : in     integer;
      key         :    out integer )
   is
      file_data    : tui.input_items(0..0);
      temp_buffer  : bstrings.bounded_string;
      temp_char    : character;

      function to_upper ( item : character) return character
	 renames ada.characters.handling.to_upper;

   begin
      file_data(0).description := description;
      file_data(0).buffer := bstrings.to_bounded_string("");
      tui.get_strings (
	 items => file_data, 
	 field => FIELD_WIDTH,
	 key   => key );

      if key = tui.KEY_ESC then
	 file_name := bstrings.to_bounded_string("");

      else
	 temp_buffer := file_data(0).buffer;
	 for buffer_index in 1..bstrings.length(temp_buffer) loop
	    temp_char := bstrings.element (
	       source => temp_buffer,
	       index  => buffer_index );

	    temp_char := to_upper ( temp_char );
	    bstrings.Replace_Element (
	       source => temp_buffer,
	       index  => buffer_index,
	       by     => temp_char );
	 end loop;

	 file_name := temp_buffer;
      end if;
   exception
      when others =>
	 tui.error_message ( msg => "Error getting file name" );
   end get_file_name;

--
-- a very simple text file browser
--
   procedure show_file ( file_name : in bstrings.bounded_string ) is
      input_file : text_io.file_type;
      input_data : character;
      temp_str   : bstrings.bounded_string;
      BODY_LEN   : constant integer := tui.body_length;
      row        : integer := 0;
      key        : integer;

   begin
      tui.clear_body_win;

      text_io.open (
	 file => input_file,
	 mode => text_io.in_file,
	 name => bstrings.to_string (file_name) );

      while not text_io.end_of_file ( file => input_file ) loop
	 text_io.get ( 
	    file => input_file,
	    item => input_data );

	 if ada.characters.handling.is_graphic ( item => input_data ) then 
	    tui.body_window_addch ( item => input_data );
	 elsif input_data = ASCII.HT then
	    tui.body_window_addch ( item => ASCII.HT );
	 else
	    tui.body_window_addch ( item => '@' );
	 end if;

	 if text_io.end_of_line ( file => input_file ) then
	    tui.body_window_addch ( item => character'val(tui.NEWLINE) );
	    if row = BODY_LEN - 1 then
	       key := tui.wait_for_key;
	       if key = tui.KEY_ESC then
		  exit;
	       end if;
	    end if;
	    row := ( row + 1 ) rem BODY_LEN;
	 end if;
      end loop;
      text_io.close ( file => input_file );
      tui.refresh_body_win;

   exception
      when text_io.name_error =>
	 temp_str := bstrings.to_bounded_string("Can't open file: ");
	 temp_str := temp_str & file_name;
	 tui.error_message ( msg => bstrings.to_string(temp_str) );

      when others =>
	 temp_str := bstrings.to_bounded_string("File error : ");
	 temp_str := temp_str & file_name;
	 tui.error_message ( msg => bstrings.to_string(temp_str) );

   end show_file;

--
-- Forward Declarations
--
   procedure sub0;
   procedure sub1;
   procedure sub2;
   procedure sub3;
   procedure func1;
   procedure func2;
   procedure subfunc1;
   procedure subfunc2;
   procedure subsub;

--
-- Menu Initialization
--
-- Modified menus to use constrained array type to accomodate 
-- increased pointer restrictions in GNAT 3.03 - AAD 5/31/96
--
   MainMenu : aliased tui.menu :=             -- tuidemo.c:136
   ( 0 => ( item_name => new string'("Asub"),
	    item_func => sub0'Access, 
	    item_desc => new string'("Go inside first submenu")  ),

     1 => ( item_name => new string'("Bsub"),
	    item_func => sub1'Access, 
	    item_desc => new string'("Go inside second submenu") ),

     2 => ( item_name => new string'("Csub"),
	    item_func => sub2'Access, 
	    item_desc => new string'("Go inside third submenu")  ),

     3 => ( item_name => new string'("Dsub"),
	    item_func => sub3'Access, 
            item_desc => new string'("Go inside fourth submenu") ),

     others => ( item_name => NULL, item_func => NULL, item_desc => NULL ));


   SubMenu0 : aliased tui.menu :=             -- tuidemo.c:145
   ( 0 => ( item_name => new string'("Exit"),
	    item_func => tui.do_exit'Access, 
            item_desc => new string'("Terminate program") ),

     others => ( item_name => NULL, item_func => NULL, item_desc => NULL ));

   SubMenu1 : aliased tui.menu :=             -- tuidemo.c:151
   ( 0 => ( item_name => new string'("OneBeep"),
	    item_func => func1'Access, 
	    item_desc => new string'("Sound one beep") ),   

     1 => ( item_name => new string'("TwoBeeps"),
	    item_func => func2'Access, 
            item_desc => new string'("Sound two beeps") ),

     others => ( item_name => NULL, item_func => NULL, item_desc => NULL ));

   SubMenu2 : aliased tui.menu :=             -- tuidemo.c:158
   ( 0 => ( item_name => new string'("Browse"),
	    item_func => subfunc1'Access, 
	    item_desc => new string'("Source file lister") ),

     1 => ( item_name => new string'("Input"),
	    item_func => subfunc2'Access, 
	    item_desc => new string'("Interactive file lister")  ),

     2 => ( item_name => new string'("Address"),
	    item_func => Address'Access, 
            item_desc => new string'("Get address data") ),

     others => ( item_name => NULL, item_func => NULL, item_desc => NULL ));

   SubMenu3 : aliased tui.menu :=             -- tuidemo.c:166
   ( 0 => ( item_name => new string'("SubSub"),
	    item_func => subsub'Access, 
            item_desc => new string'("Go inside sub-submenu") ),

     others => ( item_name => NULL, item_func => NULL, item_desc => NULL ));

   ptrMainMenu : tui.a_menu_t := MainMenu'Access;
   ptrSubMenu0 : tui.a_menu_t := SubMenu0'Access;
   ptrSubMenu1 : tui.a_menu_t := SubMenu1'Access;
   ptrSubMenu2 : tui.a_menu_t := SubMenu2'Access;
   ptrSubMenu3 : tui.a_menu_t := SubMenu3'Access;

--
-- Main Menu Functions
--
   procedure sub0 is
   begin
      tui.do_menu(mp => ptrSubMenu0);
   end sub0;

   procedure sub1 is
   begin
      tui.do_menu(mp => ptrSubMenu1);
   end sub1;

   procedure sub2 is
   begin
      tui.do_menu(mp => ptrSubMenu2);
   end sub2;

   procedure sub3 is
   begin
      tui.do_menu(mp => ptrSubMenu3);
   end sub3;

--
-- SubMenu1 Functions
--
   procedure func1 is
      msg  : aliased string := "One beep!";
      temp : c.signed_int;
   begin
      temp := pdcurses.beep;
      tui.body_message(msg => msg);
   end func1;

   procedure func2 is
      msg  : aliased string := "Two beeps!";
      temp : c.signed_int;
   begin
      temp := pdcurses.beep;
      tui.body_message(msg => msg); 
      temp := pdcurses.beep;
   end func2;

--
-- SubMenu2 Functions
--
   procedure subfunc1 is
   begin
      show_file ( file_name => bstrings.to_bounded_string ( FILE_NAME ) );
   end subfunc1;

   procedure subfunc2 is
      description : bstrings.bounded_string;
      file_name   : bstrings.bounded_string;
      key : integer;
   begin
      description := bstrings.to_bounded_string("File to browse:");
      get_file_name (
	 description => description,
	 file_name   => file_name,
	 field       => FIELD_WIDTH,
	 key         => key );

      if key /= tui.KEY_ESC then
	 show_file ( file_name => file_name );
      else
	 tui.error_message ( msg => "Could not get file name" );
      end if;

   exception
      when others =>
	 tui.error_message ( msg => "Error: tuidemo.subfunc2" );
   end subfunc2;

--
-- SubMenu3 Functions
--
   procedure subsub is
   begin
      tui.do_menu(mp => ptrSubMenu2);
   end subsub;

  title : aliased string := "AdaTUI Demonstration Program";

begin -- tuidemo
   tui.start_menu (
      mp    => ptrMainMenu,
      title => title ); 
end tuidemo;
