/* ppm2bbc - PPM to BBC

	written by SeDi (vmgsedi@msn.com)

  Type "ppm2bbc" on the command line without parameters to get instructions on using the program.

*/

#include <stdio.h>
#include <stdlib.h>

int ReadPPM(char *);
void WritePPM(char *,int);

int *red,*green,*blue,width,height;
int curmode;
int aspect,htone,stagger;

struct RGB {
	int r,g,b;
	};

struct RGB modeclrs[6][8]={
	{{0,0,0},{1,1,1},NULL,NULL,NULL,NULL,NULL,NULL},
	{{0,0,0},{1,0,0},{1,1,0},{1,1,1},NULL,NULL,NULL,NULL},
	{{0,0,0},{1,0,0},{0,1,0},{1,1,0},{0,0,1},{1,0,1},{0,1,1},{1,1,1}},
	{{0,0,0},NULL,NULL,NULL,NULL,NULL,NULL,NULL},
	{{0,0,0},{1,1,1},NULL,NULL,NULL,NULL,NULL,NULL},
	{{0,0,0},{1,0,0},{1,1,0},{1,1,1},NULL,NULL,NULL,NULL}
	};

/* now comes our halftoning matrices */
int mat[16][16],
sub1[4][4]={				//fine halftoning
	 8, 7,11, 4,
	 2,13, 1,14,
	10, 5, 9, 6,
	 0,15, 3,12},
sub2[4][4]={				//coarse halftoning
	 9,13, 0, 2,
	11,15, 6, 4,
	 1, 3, 8,12,
	 7, 5,10,14},
sub3[4][4]={				//spiral halftoning
	 3, 4, 5, 6,
	 2,13,14, 7,
	 1,12,15, 8,
	 0,11,10, 9},
sub4[4][4]={				//diagonal halftoning
	 0, 4, 8,12,
	13, 1, 5, 9,
	10,14, 2, 6,
	 7,11,15, 3};

void initmat() {
	int t,r;
	if(htone==0) {
		for(t=0; t<16; t++)
			for(r=0; r<16; r++)
				mat[t][r]=0;
		return;
		}
	for(t=0; t<4; t++)
		for(r=0; r<4; r++) {
			switch(htone) {
				case 1:	//fine
					mat[t][r]=sub1[t][r]*16;
					mat[t+4][r+4]=sub1[t][r]*16+4;
					mat[t+4][r]=sub1[t][r]*16+8;
					mat[t][r+4]=sub1[t][r]*16+12;
					break;
				case 2:	//coarse
					mat[t][r]=sub2[t][r]*16;
					mat[t+4][r+4]=sub2[t][r]*16+4;
					mat[t+4][r]=sub2[t][r]*16+8;
					mat[t][r+4]=sub2[t][r]*16+12;
					break;
				case 3:	//spiral
					mat[t][r]=sub3[t][r]*16;
					mat[t+4][r+4]=sub3[t][r]*16+4;
					mat[t+4][r]=sub3[t][r]*16+8;
					mat[t][r+4]=sub3[t][r]*16+12;
					break;
				case 4:	//diagonal
					mat[t][r]=sub4[t][r]*16;
					mat[t+4][r+4]=sub4[t][r]*16+4;
					mat[t+4][r]=sub4[t][r]*16+8;
					mat[t][r+4]=sub4[t][r]*16+12;
					break;
				}
			}
	for(t=0; t<8; t++)
		for(r=0; r<8; r++) {
			mat[t+8][r+8]=mat[t][r]+1;
			mat[t+8][r]=mat[t][r]+2;
			mat[t][r+8]=mat[t][r]+3;
			}
	for(t=0; t<16; t++)
		for(r=0; r<16; r++)
			mat[t][r]=((mat[t][r]-128)*9)/10;
	}

int main(int argc,char **argv) {
	int t;
	char *fname;
	if(argc<2) {
		printf("%s%s%s%s%s%s%s%s%s%s",
			"Usage:-\n  ppm2bbc [<opts>]<ppm-file> [ [<opts>]<ppm-file>... ]\n",
			"<opts> are -0,-1,-2,-4,-5 to set BBC MODE. -2 is default.\n",
			"-a2,-a0 for aspect-correction, -hX1,-hsXa2 for halftoning:-\n",
			"\tX=0 for no halftoning,\n\tX=1 for fine halftoning (default),\n",
			"\tX=2 for coarse halftoning,\n\tX=3 for spiral halftoning,\n",
			"\tX=4 for diagonal halftoning.\n",
			"\t\'s\' between the -h and the X value staggers the RGB values.\n",
			"\tStaggering often works well for MODE 2, but not for 2-colour modes.\n",
			"\nLeave no spaces between the options and the filename.\neg. ppm2bbc -a2house.ppm -hs4a0picture.ppm\n",
			"\nTip: Use GNU bash if you want to use wildcards with this program...\n");
		exit(0);
		}

	initmat();

	for(t=1; t<argc; t++) {
		fname=argv[t];
		curmode=2;
		aspect=0;
		htone=1;
		stagger=0;
		if((*fname)=='-') {
			if(*(fname+1)=='h') {	//note: -h switch specifies NO halftoning
				if(*(fname+2)=='s') {
					fname++;
					stagger=1;
					}
				htone=(*(fname+2))-'0';
				fname+=2;
				if((htone<0)||(htone>4)) {
					printf("Halftoning switch -h%c out of range. Using -h1 as default.\n",(char)(htone+'0'));
					htone=1;
					}
				}
			if(*(fname+1)=='a') {
				aspect=1;
				fname++;
				}
			curmode=(*(fname+1))-'0';
			switch(curmode) {
				case 0:	case 1:	case 2:	case 4:	case 5:
					/* OK! */
					break;
				default:
					printf("Invalid MODE %c for %s. Using MODE 2 as default.\n",*(fname+1),fname+2);
					curmode=2;
					break;
				}
			fname+=2;
			}
		if(ReadPPM(fname)) {
			initmat();
			WritePPM(fname,255);
			free(red); free(green); free(blue);
			}
		printf("%d%% completed.\n",(t*100)/(argc-1));
		}
	}

int ReadPPM(char *fileName) {
	int x,y;
	FILE *f;
	int temp,t2;
	printf("Attempting to open %s\n",fileName);
	if(f=fopen(fileName,"rb")) {
		temp=1;
		if(fgetc(f)!='P') temp=0;
		if(fgetc(f)!='6') temp=0;
		if(fgetc(f)!='\n') temp=0;
		if(temp==0) {
			printf("Not a PPM6 (binary-encoded) file!\n");
			fclose(f);
			return(0==1);
			}
		if((t2=fgetc(f))!='#') {
			printf("This PPM file has no comment.\n");
			ungetc(t2,f);
			}
		else {
			printf("File comment:-\n");
			while(((temp=fgetc(f))!='\n')&&(!feof(f))) {
				if((temp>=' ')&&(temp<='z'))
					fputc(temp,stdout);
				}
			}
		fputc('\n',stdout);
		fscanf(f,"%d %d\n",&width,&height);
		printf("File is %d x %d\n",width,height);
		red=malloc(width*height*sizeof(int));
		green=malloc(width*height*sizeof(int));
		blue=malloc(width*height*sizeof(int));
		if((!red)||(!green)||(!blue)) {
			if(red) free(red);
			if(green) free(green);
			if(blue) free(blue);
			fclose(f);
			printf("Out of memory (%d bytes required).\n",width*height*sizeof(int));
			return(0==1);
			}
		while((fgetc(f)!='\n')&&(!feof(f)));
		printf("Now analysing file...\n");
		for(y=0; y<height; y++)
			for(x=0; x<width; x++) {
				red[x*height+y]=fgetc(f);
				green[x*height+y]=fgetc(f);
				blue[x*height+y]=fgetc(f);
				}
		fclose(f);
		}
	else {
		printf("Unable to open this file.\n");
		return(0==1);
		}
	return(1==1);
	}

int matchcol(int r,int g,int b,int clrs) {	//which colour in the current MODE matches best?
	int t,minerr,mint,err;
	struct RGB *rgb=modeclrs[curmode];
	r=r>127;
	g=g>127;
	b=b>127;
	minerr=10000;
	mint=-1;
	for(t=0; t<clrs; t++) {
		err=0;
		if(!(rgb[t].r && r)) err++;
		if(!(rgb[t].g && g)) err++;
		if(!(rgb[t].b && b)) err++;
		if(err<minerr) {
			minerr=err;
			mint=t;
			}
		}
	return mint;
	}

 void	WritePPM(char *fileName, int maxVal)
    {
	register int x, y;
	int r, g, b;
	int wbytes,clrs,pixpb;
	FILE *f;
	int row,byte,boff;
	char *x1,*x2;
	char byteclrs[8];
	unsigned char finalbyte;

	x1=malloc(strlen(fileName)+5);
	x2=malloc(strlen(fileName)+10);
	strcpy(x1,fileName);
	strcat(x1,".bbc");
	strcpy(x2,x1);
	strcat(x2,".inf");

	printf("Opening output file \"%s\"...\n",x1);
    if(f=fopen(x1,"wb")) {

	wbytes=((curmode==2)||(curmode==0)||(curmode==1))?80:
		40;
	pixpb=	((curmode==0)||(curmode==4))?8:
		((curmode==1)||(curmode==5))?4:
		2;
	clrs=	((curmode==0)||(curmode==4))?2:
		((curmode==1)||(curmode==5))?4:
		8;
	printf("MODE %ld has %ld bytes per row, %ld steady colours\nand %ld pixels per byte.\n",curmode,wbytes,clrs,pixpb);

	for(row=0; row<32; row++) {
		for(byte=0; byte<(wbytes*8); byte++) {
			x=(byte/8)*pixpb;		//byte's left edge in pixels
			y=(row*8)+(byte%8);		//byte's Y coordinate
			if(y>=height)
				finalbyte=0;	//blank byte
			else {
				for(boff=0; boff<pixpb; boff++) {
					int xp=boff+x;
					if(aspect) {
						if((curmode==2)||(curmode==5))	xp*=2;
						else if(curmode==0)	xp/=2;
						}
					if(xp>=width)
						byteclrs[boff]=0;
					else {
						r=red[xp*height+y];
						g=green[xp*height+y];
						b=blue[xp*height+y];
						if(aspect)
							if((curmode==2)||(curmode==5)) {
								r=(r+red[(xp+1)*height+y])/2;
								g=(g+green[(xp+1)*height+y])/2;
								b=(b+blue[(xp+1)*height+y])/2;
								}
						if(stagger) {
							r+=mat[(boff+x)%16][y%16];
							g+=mat[(boff+x+1)%16][(y+2)%16];
							b+=mat[(boff+x+2)%16][(y+1)%16];
							}
						else {
							r+=mat[(boff+x)%16][y%16];
							g+=mat[(boff+x)%16][y%16];
							b+=mat[(boff+x)%16][y%16];
							}
						r=(r<0)?0:(r>255)?255:r;
						g=(g<0)?0:(g>255)?255:g;
						b=(b<0)?0:(b>255)?255:b;
						byteclrs[boff]=matchcol(r,g,b,clrs);
						}
					}
				finalbyte=0;
				switch(curmode) {
					case 0:	case 4:	//the easy one...
						for(boff=0; boff<8; boff++)
							finalbyte=(finalbyte<<1)|(byteclrs[boff]?1:0);
						break;
					case 1:	case 5:	//a bit harder...
						for(boff=0; boff<4; boff++)
							finalbyte=(finalbyte<<1)|((byteclrs[boff]&2)?1:0);
						for(boff=0; boff<4; boff++)
							finalbyte=(finalbyte<<1)|((byteclrs[boff]&1)?1:0);
						break;
					case 2:			//now we need to take a different tack...
						for(boff=4; boff>=1; boff/=2) {
							finalbyte=(finalbyte<<1)|((byteclrs[0]&boff)?1:0);
							finalbyte=(finalbyte<<1)|((byteclrs[1]&boff)?1:0);
							}
						break;
					}		//phew!
				}
			fwrite(&finalbyte,1,1,f);
			}
		}

	fclose(f);
        }

	printf("Writing BBCIm INF file...\n");
	f=fopen(x2,"wb");
	fprintf(f,"$.SCREEN %06lx %06lx\n",
			((curmode==0)||(curmode==1)||(curmode==2))?0x3000:0x5800,
			0);
	fclose(f);

	free(x1);
	free(x2);

	printf("Done!!!\n");

    }
