#include <math.h>
#include "player.hpp"
#include "map.hpp"
#include "keyboard.hpp"
#include "mouse.hpp"
#include "zox3d.hpp"
#include "vga.hpp"			// for screen dimensions
#include "joystick.hpp"

/*
 * TEST_KEY performs the test to see if a certain key has been pressed.
 * If POLL_KEYS is defined the best method is used, in which several
 * keys can be detected as pressed during one frame.  If not, the
 * value in the key variable is used, and only one key can be detected
 * each frame.  (see below for how to use it.)
 *
 * POLL_KEYS must be defined in KEYBOARD.HPP to work properly.
 */

#ifdef POLL_KEYS
# define TEST_KEY(s, c) (keyboard[s].tacCount() || keyboard[s].isPressed())
#else
# define TEST_KEY(s, c) (key == c)
#endif

// quit is set to 1 when player wants to leave.
int playerQuit;

Fixed	virtualGuyX, virtualGuyY,	// player position - can be manipulated
									// during tracing
		guyX, guyY,					// player position - constant during
									// tracing
		moveX, moveY,				// player movement
		guyElev, prevElev,			// player elevation
		guyElevScreen,
		cosGuy, sinGuy;				// cos/sin of player face angle
FixAngle guyAngle, prevAngle;		// player orientation

// wallMargin is the closest distance the player is allowed to approach a
// wall before a collision is detected.
const Fixed wallMargin = BLOCKSIZE_WORLD >> 4;

// walkStep is the distance the player moves when walking by using the
// keyboard.  The mouse and joystick are analogue, and thus let the
// player choose the speed he likes.
const Fixed walkStep = BLOCKSIZE_WORLD >> 2;

// maxWalkStep reflects the limits inherent in the wall collision detection
// routines, which only checks walls actually touching the square occupied
// by the player.  Thus the player is allowed to move at most a little less
// than a full square for one frame.
const Fixed maxWalkStep = BLOCKSIZE_WORLD - (BLOCKSIZE_WORLD >> 8);

// rotateStep dictates the increment of the player's facing angle when
// turning using the keypad arrows.
const FixAngle rotateStep = DEG45/4;

// features dictates which special features are enabled.
int features = DRAW_FLOOR | DRAW_SKY; // | SHOW_HELP;


// initPlayer() positions the player on the map.  Note that this is specific
// to the map defined in MAP.CPP.

void initPlayer()
	{
	guyElev = BLOCKSIZE_WORLD >> 1;			prevElev = guyElev+1;
	guyX = 11 * BLOCKSIZE_WORLD >> 2;
	guyY = 15 * BLOCKSIZE_WORLD >> 1;
	guyAngle = DEG180;						prevAngle = guyAngle+1;
	}

/*
 * To fully understand the following, have a look at this figure:
 *
 * This is the map square at (x,y):
 *
 * vertical wall -------+
 *                      |  
 *                      |                 (x+1,y+1)
 * horiz. margin ---->--|--                  |
 *                    : v :                  v
 *                    : :Ŀ
 *                    :  :                  
 *                    :  :                                   N
 *                    :  :                  
 *                    :  :    Square                       W + E
 * vert. margins ---->:  :     (x,y)        
 *               -------->:                                   S
 *                    :  :                  
 *                    :  :                  
 *                    :  :                  
 *                    +---+---------------------
 *                    :  :      ^            : 
 *                    : :|;<----------------+
 *                    : ^ :      |             :               |
 * horiz. margin ---->----+------|--------------               |
 *                    ^ |        |^            ^               |
 *                    |(x,y)     ||            |               |
 *                    |          ||            |               |
 *                    |         horiz. margins |               |
 *                 vert. margin                |          horizontal wall
 *                                         vert. margin
 *
 *
 * The square is associated with one horizontal and one vertical wall.
 * Both walls are surrounded by invisible margins, whose thickness is
 * given by wallMargin, the idea being to prevent the player from getting
 * "too" close to a wall.
 *
 * The following functions make use of the terms in this figure.
 */


// withinWall() returns 1 if a is within the margins if the wall running
// from min to min+len.  (A simple bounds check.)

inline int withinWall(Fixed a, int min, int len)
	{
	return a > INT2FIX(min)-wallMargin
		&& a < INT2FIX(min+len)+wallMargin;
	}

// clipHorzWall() returns 1 if the horizontal wall in the map square given
// by (x,y) is solid, and if the inters parameter is within the wall's
// vertical margins.

int clipHorzWall(int mapX, int mapY, Fixed inters)
	{
	return map[mapY][mapX].horz.type & SOLID && withinWall(inters, mapX, 1);
	}

// clipVertWall() returns 1 if the vertical wall in the map square given
// by (x,y) is solid, and if the inters parameter is within the wall's
// horizontal margins.

int clipVertWall(int mapX, int mapY, Fixed inters)
	{
	return map[mapY][mapX].vert.type & SOLID && withinWall(inters, mapY, 1);
	}

// clipHorzEdge() returns 1 if the vertical wall in the map square given
// by (x,y) is solid, and if the inters parameter is within the wall's
// vertical margins.

int clipHorzEdge(int mapX, int mapY, Fixed inters)
	{
	return map[mapY][mapX].horz.type & SOLID && withinWall(inters, mapY, 0);
	}

// clipVertEdge() returns 1 if the horizontal wall in the map square given
// by (x,y) is solid, and if the inters parameter is within the wall's
// horizontal margins.

int clipVertEdge(int mapX, int mapY, Fixed inters)
	{
	return map[mapY][mapX].vert.type & SOLID && withinWall(inters, mapX, 0);
	}


/*
 * processPlayer() should be called once each frame to update the player's
 * position in accordance to the user input.
 *
 * processPlayer() also interprets commands like quit, toggle help,
 * toggle floor/ceiling texture, and so on.
 */


void processPlayer()
	{
	moveX = 0;
	moveY = 0;

	/*
	 * Process user commands from keyboard.
	 */

#ifndef POLL_KEYS
	KeyCode key = keyboard.flush();
#endif

//			case 'd':	// column debug on
//				rowDebug = 1;
//				break;

	Fixed walkForward = 0, walkRight = 0;

	if (TEST_KEY(0x47, 0x4700))	// HOME: slow turn left
		guyAngle += 0x0040;

	if (TEST_KEY(0x48, 0x4800))	// UP: move forward
		walkForward += walkStep;

	if (TEST_KEY(0x49, 0x4900))	// PGUP: slow turn right
		guyAngle -= 0x0040;

	if (TEST_KEY(0x4B, 0x4B00))	// LEFT: turn left
		guyAngle += rotateStep;

	if (TEST_KEY(0x4c, 0x4c00))	// keypad 5: U-turn
		guyAngle += DEG180;

	if (TEST_KEY(0x4d, 0x4D00))	// RIGHT: turn right
		guyAngle -= rotateStep;

	if (TEST_KEY(0x4f, 0x4f00))	// END: move left
		walkRight -= walkStep;

	if (TEST_KEY(0x50, 0x5000))	// DOWN: move backwards
		walkForward -= walkStep;

	if (TEST_KEY(0x51, 0x5100))	// PGDN: move right
		walkRight += walkStep;

	if (TEST_KEY(0x4a, '-'))	// grey minus: hover up
		guyElev -= (BLOCKSIZE_WORLD >> 3);

	if (TEST_KEY(0x4e, '+'))	// grey plus: hover down
		guyElev += (BLOCKSIZE_WORLD >> 3);

#ifndef POLL_KEYS
	if (key == 'b')				// breakpoint, only if not polling keys
		asm nop;
#endif

	if (TEST_KEY(0x01, 27))		// Escape: quit demo
		playerQuit = 1;

	if (TEST_KEY(0x21, 'f'))	// toggle floor
		{
		features ^= DRAW_FLOOR;
		prevElev = guyElev-1;
		}

	if (TEST_KEY(0x1f, 's'))	// toggle sky
		{
		features ^= DRAW_SKY;
		prevElev = guyElev-1;
		}

	if (TEST_KEY(0x23, 'h'))	// toggle help
		features ^= SHOW_HELP;

	if (TEST_KEY(0x33, ','))	// shrink window dimensions
		{
		windowWidth -= screenWidth >> 4;
		windowHeight -= screenHeight >> 4;
		windowChanged = 1;
		}

	if (TEST_KEY(0x34, '.'))	// increase window dimension
		{
		windowWidth += screenWidth >> 4;
		windowHeight += screenHeight >> 4;
		windowChanged = 1;
		}

	if (TEST_KEY(0x24, 'j'))	// recalibrate joystick
		{
		Gamecard::reset();
		if (Gamecard::isPresent())
			joystickA.calibrate();
		}

	/*
	 * Process mouse movement.
	 *
	 * No buttons held:
	 * - Up/down moves player forward/backwards.
	 * - Left/right turns player correspondingly.
	 *
	 * Right button held:
	 * - Up/down elevates player up/down.
	 * - Left/right moves player left/right
	 */

	int mouseX, mouseY;
	mouse.getMotion(mouseX, mouseY);
	if (mouse.getButtons() & Mouse::RIGHT)
		{
		walkRight += INT2FIX(mouseX) >> 7;
		guyElev += INT2FIX(mouseY)>>8;
		}
	else
		{
		guyAngle -= mouseX << 6;
		if (mouseY)
			walkForward -= INT2FIX(mouseY) >> 7;
		}

	/*
	 * Process joystick input.  First poll the gamecard to update the
	 * joystick state variables, then read selected joystick.
	 */

	int joyX, joyY;
	Gamecard::poll();
	joystickA.getPosition(joyX, joyY);
	if (abs(joyX) > (joystickA.getResolution()>>4))
		guyAngle -= joyX>>3;
	if (abs(joyY) > (joystickA.getResolution()>>4))
		walkForward -= joyY;


	/*
	 * Prohibit movement across more than one map square at a time
	 */

	bound(walkForward, -maxWalkStep, maxWalkStep);
	bound(walkRight, -maxWalkStep, maxWalkStep);
	bound(guyElev, 0L, BLOCKSIZE_WORLD-1);


	if (walkForward)
		{
		FixMulAdd(cosGuy, walkForward, moveX);
		FixMulAdd(sinGuy, walkForward, moveY);
		}
	if (walkRight)
		{
		FixMulAdd(sinGuy, walkRight, moveX);
		FixMulAdd(-cosGuy, walkRight, moveY);
		}


	/*
	 * Restrict (clip) movement to surrounding walls.  A maximum of 8
	 * neighbouring walls are checked, in the quadrant of the player's
	 * general direction:
	 *
	 *       |     |                                  |
	 *       A     C          ----- horizontal wall   | vertical wall
	 *       |     |                                  |
	 *  --A-- --B-- --C--
	 *             |          * player
	 *          /  B          / direction of movement
	 *         *   |
	 *              --A--
	 *             |          ABC refers to order of checking (see below)
	 *             A
	 *             |
	 *
	 * The player can have a minimum distance from any wall equal to
	 * wallMargin.  (Imagine each wall surrounded by a 'force field'.)
	 * This is to avoid dealing with special cases like if the player
	 * stand exactly *on* a wall, and it looks pretty natural.
	 *
	 * First, we check wether the movement will cross any of the
	 * horizontal or vertical margins.  If not, we skip intersection
	 * tests for the corresponding walls.
	 *
	 * Second, the collision tests are performed for the walls indicated
	 * in the check variable.  Positive tests might turn off tests to
	 * be performed later on.
	 *
	 * Third, the movement vector itself is clipped according to the
	 * tests.
	 *
	 * Fourth, the players angle is adjusted to 'home in' on the new
	 * direction if a wall was hit.  (Like in Ultima Underworld.)
	 */

	// Map square occupied by our hero.
	int mapX = FIX2INT(guyX), mapY = FIX2INT(guyY);

	// The general direction of his movement.
	int dirX = SIGN(moveX), dirY = SIGN(moveY);

	// Intersection coordinates with closest vertical/horizontal
	// grid lines.

	Fixed vix, viy, hix, hiy;

	// Logic mask for which neighbouring walls remain to check for collision.
	// BACK -> A walls
	// SIDE -> B walls  in the sketch above
	// FRONT -> C walls

	enum { NONE=0,
		XBACK=1, XSIDE=2, XFRONT=4,
		YFRONT=8, YSIDE=16, YBACK=32,
		XCLIP = 64, YCLIP=128,
		XCHECK = XBACK | XSIDE | XFRONT,
		YCHECK = YBACK | YSIDE | YFRONT };
	int check = NONE;

	// Check for intersection with vertical and horizontal margins
	// respectively.  Activate wall checking with 'check' variable
	// if neccessary.

	if (dirX)
		{
		vix = INT2FIX(mapX + (dirX==1)) - wallMargin * dirX;
		if (SIGN(guyX+moveX-vix) == dirX)
			{
			FixMul(vix-guyX, moveY, viy);
			FixDiv(viy, moveX, viy);
			viy += guyY;
			check |= XCHECK | XCLIP;
			}
		}

	if (dirY)
		{
		hiy = INT2FIX(mapY + (dirY==1)) - wallMargin * dirY;
		if (SIGN(guyY+moveY-hiy) == dirY)
			{
			FixMul(hiy-guyY, moveX, hix);
			FixDiv(hix, moveY, hix);
			hix += guyX;
			check |= YCHECK | YCLIP;
			}
		}

	// If further collision checking is needed, start with the 2 blocks
	// most 'behind' the player (A blocks above), then the 2
	// immediately neighbouring blocks (B), then the one block in
	// 'front' of the player (C).

	// Note that when a collision with a certain wall has been detected,
	// it usually implies that some other collisions are impossible,
	// so we turn off impossible tests in such cases.  This is the beauty
	// of using bit patterns.

	if (check)
		{
		int hit = 0;

		// Step from front to back (-dir, 0, +dir) if applicable,
		// otherwise just use (-1, 0, +1).

		int stepX = dirX?dirX:1, stepY = dirY?dirY:1;

		// (mapVX,mapY) and (mapX,mapHY) are the coordinates of the facing
		// vertical and horizontal walls respectively.

		int mapVX = mapX + (stepX>0), mapHY = mapY + (stepY>0);

		// Check A walls

		if (check & XBACK
			&& (clipVertWall(mapVX, mapY-stepY, viy)
				|| clipHorzEdge(mapX+stepX, mapY+(stepY<0), viy)))
			check &= ~(XCHECK | YBACK | YFRONT);

		if (check & YBACK
			&& (clipHorzWall(mapX-stepX, mapHY, hix)
				|| clipVertEdge(mapX+(stepX<0), mapY+stepY, hix)))
			check &= ~(YCHECK | XBACK | XFRONT);

		// Check B walls - the two closest neighbouring walls to the
		// player's square.

	checkB:

		if (check & XSIDE && clipVertWall(mapVX, mapY, viy))
			{
			check &= ~XCHECK;
			hix = vix;
			}

		if (check & YSIDE && clipHorzWall(mapX, mapHY, hix))
			{
			check &= ~YCHECK;
			viy = hiy;
			goto checkB;
			}

		// Check C walls

		if (check & XFRONT && clipVertWall(mapVX, mapY+stepY, viy))
			check &= ~(XCHECK | YFRONT);

		if (check & YFRONT && clipHorzWall(mapX+stepX, mapHY, hix))
			check &= ~(YCHECK | XFRONT);

		if (check & XFRONT && clipHorzEdge(mapX+stepX, mapHY, viy))
			check &= ~XCHECK;

		if (check & YFRONT && clipVertEdge(mapVX, mapY+stepY, hix))
			check &= ~YCHECK;

		// A move coordinate is clipped if the corresponding XCLIP or
		// YCLIP bit is the only bit left in the pattern.

		if ((check & (XCHECK | XCLIP)) == XCLIP)
			{
			moveX = vix-guyX;
			hit = 1;
			}
		if ((check & (YCHECK | YCLIP)) == YCLIP)
			{
			moveY = hiy-guyY;
			hit = 1;
			}

		// If a wall was hit, try to turn the player towards the new
		// movement direction.  This eases navigation around corners etc.

		if (hit && (moveX || moveY))
			{
			FixAngle hitAngle = atan2(-moveY, moveX)*DEG180/M_PI;
			if (abs(int(hitAngle-guyAngle)) < DEG90)
				guyAngle += int(hitAngle - guyAngle) / 4;
			}
		}
	// Do the actual movement.

	guyX += moveX;
	guyY += moveY;
	}


