#include <mem.h>
#include "map.hpp"

/*
 * MAP_SIZE defines the length of an edge of the map.  Local to this module.
 */

#define MAP_SIZE 10

/*
 * mapWidth and mapHeight are the dimensions used in other modules .
 */

int mapWidth=MAP_SIZE, mapHeight=MAP_SIZE;

/*
 * This array (below) defines the layout of the map, and is used to build the
 * actual map array below.  I.e. you would want to put this array out
 * in a file, instead of hardcoding it into the program.
 *
 * Each number defines the walls associated with one map sqare.  Each
 * square may contain two walls: one on the southern edge, and one on the
 * western edge.  Note that the actual map uses a right-handed coordinate
 * system:
 *
 *      North          y
 *        ^            |
 * West <-+-> East     |
 *        v            |
 *      South          +----x
 *                  (0,0)
 *
 * ...while this array is listed in a left-handed fashion:
 *
 *                  (0,0)
 *      South          +----x
 *        ^            |
 * West <-+-> East     |
 *        v            |
 *      North          y
 *
 * One byte is used for each wall.  Line 1 defines the southern walls
 * of the squares in the top line of the map.  Line 2 defines the
 * western walls to the same squares.  Line 3 and 4 is for the next line
 * of the map.  See PLAYER.CPP for a closeup of a map square
 *
 * The byte is essentially a bitmap number + 1.  A value of 0 means
 * no bitmap, i.e. no wall.  If the highest bit (bit 7) is set, this
 * wall's bitmap is drawn in the reverse direction.  Use this to avoid
 * mirroring of bitmaps with text etc.
 *
 * The south walls of the easternmost squares is outside the map, and should
 * be set to all 0s.  The same goes for the last line, which defines the
 * west walls of the northernmost squares.
 *
 * I repeat, this array is used only for initializing the REAL map
 * structure.  mapData should not be used outside MAP.CPP.
 */

typedef unsigned char MapCell;	// Local typedef only

MapCell mapData[MAP_SIZE][2][MAP_SIZE] =
	{ // south is down, west is right ---->
	   0x04, 0x07, 0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x00,

	0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01,

	   0x00, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,

	0x02, 0x00, 0x00, 0x08, 0x08, 0x00, 0x00, 0x04, 0x01, 0x01,

	   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,

	0x01, 0x00, 0x00, 0x08, 0x08, 0x00, 0x00, 0x00, 0x01, 0x01,

	   0x00, 0x83, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00,

	0x02, 0x03, 0x03, 0x00, 0x00, 0x00, 0x00, 0x8A, 0x01, 0x01,

	   0x00, 0x83, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8A, 0x00, 0x00,

	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,

	   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x00, 0x00,

	0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x01,

	   0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00,

	0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x86, 0x01,

	   0x00, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x00,

	0x07, 0x07, 0x0B, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x01,

	   0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x01,

	   0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x00,

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	};

/*
 * The map[][] array is the actual map being used in the program, being
 * a 2D table of Squares (defined in MAP.HPP).
 */

Square map[MAX_MAP][MAX_MAP];


/*
 * Wall::init() initializes a Wall structure given a value v from the
 * mapData array above.  If v = 0, no bitmap is used, otherwise bitmap
 * number (v-1) is used.  Bit 7 is stored as part of the bitmap number.
 * It is the responsibility of the wall drawing routines in WSLICE.CPP
 * to handle wall bitmap directions.
 *
 * The switch statement selects a wall type appropriate for the given
 * bitmap.  This is SPECIFIC for the bitmaps listed in RESOURCE.DAT!
 */

void Wall::init(int v)
	{
	bmap = v ? v-1 : NO_BITMAP;

	switch (v & 0x7f)
		{
		case 3:
			type = GLASS;
			break;
		case 5:
		case 6:
			type = GLASS | SOLID;
			break;
		case 1:
		case 2:
		case 4:
		case 8:
		case 10:
		case 11:
			type = OPAQUE | SOLID;
			break;
		case 7:
			type = MIRROR | SOLID;
			break;
		default:
			type = NOWALL;
			break;
		}
	}


/*
 * Square::init() initializes a Square, given values for the horizontal and
 * vertical walls from the mapData array above.
 *
 * Currently only one floor bitmap and one ceiling bitmap is supported,
 * so we merely set those bitmap numbers to zero.
 *
 * Currently objects are not implemented, so we set the object pointer to 0.
 */

void Square::init(int h, int v)
	{
	horz.init(h);
	vert.init(v);
	floor.bmap = 0;
	ceil.bmap = 0;
	}


/*
 * initMap() is called by the main program to set up the real map array
 * used by other modules.
 */

void initMap()
	{
	for (int x=0; x<mapWidth; x++)
		for (int y=0; y<mapHeight; y++)
			map[y][x].init(mapData[y][0][x], mapData[y][1][x]);
	}

