/* -------------------------------------------------------------------------- */
/*                                                                            */
/* (C) Copyright D.C.Devenport 1997. All right reserved.                      */
/*                                                                            */
/* THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY      */
/* KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE        */
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR      */
/* PURPOSE.                                                                   */
/*                                                                            */
/* This code, and no part of this code, may not be used in any                */
/* commercial or for-profit venture without the express written               */
/* permission of D.C.Devenport. (DDevenp666@aol.com)                          */
/*                                                                            */
/* Credit must be given within any program that uses any of this code         */
/* OR in the accompanying documentation. (And mail me a copy :) )             */
/*                                                                            */
/*----------------------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include "bbc.h"
#include "sysvia.h"
#include "main.h"

extern unsigned long TotalClockCycles;
extern int    CyclesToGo,OriginalCyclesToGo;

// 0xffff * 2 = 0x1fffe +2 = 0x20000
#define TIMERWRAPVALUE 0x20000

// only 16 regs, but 4 are different on reads
static
BYTE ORB,IRB,
     ORA,IRA,
     DDRB, // 0 bit means input, a 1 output
     DDRA, // 0 bit means input, a 1 output
     T1CL, // read = t1 low order counter
     T1CH,
     T1LL,
     T1LH,
     T2CL, // read = t2 low order counter
     T2CH,
     T2LL,
     SR,
     ACR,
     PCR,
     IFR,
     IER=0,

     PortA,PortB; // these variables contain the actual values on the lines...

extern unsigned long TotalClockCycles;

extern BYTE InterruptPendingFlag;

// these values are multiplied by 2, so they can be decremented at 2MHz
static int Timer1,Timer2;
static BYTE SetInterruptOnTimer2;


extern unsigned long TotalClockCycles;


void ResetUserVIA()
{
  Timer1=TIMERWRAPVALUE;
  Timer2=TIMERWRAPVALUE;
  T1LL=0xff;
  T1LH=0xff;
}



BYTE ReadUserVIA(WORD Address)
{
  BYTE Result;

// ASSUMPTION - all cycles upto the actual read have been done
  SyncClocks();  // its a slow device so sync with it

    switch (Address & 0xf) // now the correct register
    {
      case 0  : PortB=(BYTE) TotalClockCycles;
                Result=ORB & DDRB;
                  if ((ACR & 0x02)==0) // latching disabled
                    Result|=((DDRB ^ 255) & PortB);
                  else // latching enabled
                    Result|=((DDRB ^ 255) & IRB);

                   if ((PCR & 0xe0)==0x20 || (PCR & 0xe0)==0x60) // independant mode
                     IFR&=0xef; // clear bit 4
                   else
                     IFR&=0xe7; // clear bits 3&4
                UpdateClocksAndInterrupts();
                return (Result);
                break;
      case 1  : // IRQ handshaking
                   if ((PCR & 0x0e)==2 || (PCR & 0x0e)==6) // independant mode
                     IFR&=0xfd; // clear bit 1
                   else           
                     IFR&=0xfc; // clear bits 0 & 1
                UpdateClocksAndInterrupts();
                // fall thru
      case 15 :

                  if ((ACR & 0x01)==0) // latching disabled
                    Result=PortA;
                  else // latching enabled
                    Result=IRA;

                return (Result);
                break;
      case 2  : return (DDRB);
                break;
      case 3  : return (DDRA);
                break;
      case 4  : UpdateClocksAndInterrupts();
                IFR&=0xbf; // Bit 6 cleared in IFR
                T1CL=(BYTE) (Timer1>>1);
                return (T1CL);
                break;
      case 5  : UpdateClocksAndInterrupts();
                T1CH=(BYTE) (Timer1>>9);
                return (T1CH);
                break;
      case 6  : return (T1LL);
                break;
      case 7  : return (T1LH);
                break;
      case 8  : UpdateClocksAndInterrupts();
                IFR&=0xdf; // bit 5 cleared in IFR
                T2CL=(BYTE) (Timer2>>1);
                return (T2CL);
                break;
      case 9  : UpdateClocksAndInterrupts();
                T2CH=(BYTE) (Timer2>>9);
                return (T2CH);
                break;
      case 10 : IFR&=0xfb; // clear bit 2
                UpdateClocksAndInterrupts();
                return (SR); // SR does nothing
                break;
      case 11 : return (ACR);
                break;
      case 12 : return (PCR);
                break;
      case 13 :  // unset/set bit 7 before read
                  if ((IFR & IER & 0x7f)==0)
                    IFR&=0x7f;
                  else
                    IFR|=0x80;
                return (IFR);
                break;

      case 14 : return (128 | IER); // bit 7 always set on read
                break;
    }
  return 0;
}


void WriteUserVIA(WORD Address,BYTE Value)
{
// ASSUMPTION - all cycles upto the actual write have been done
  SyncClocks();  // its a slow device so sync with it

    switch (Address & 0xf)
    {
      case 0  : ORB=Value;
      NewPortB :

                PortB=(PortB & (DDRB ^ 255)) + (ORB & DDRB);

                  if ((PCR & 0xe0)==0x20 || (PCR & 0xe0)==0x60) // independant mode
                    IFR&=0xef; // clear bit 4
                  else
                    IFR&=0xe7; // clear bits 3 & 4
                UpdateClocksAndInterrupts();
                break;

      case 1  : // IRQ handshaking
                   if ((PCR & 0x0e)==2 || (PCR & 0x0e)==6) // independant mode
                     IFR&=0xfd; // clear bit 1
                   else
                     IFR&=0xfc; // clear bits 0 & 1
                UpdateClocksAndInterrupts();
                // fall thru
      case 15 : // same as case 1 - no handshaking interrupt
                ORA=Value;
      NewPortA :
                PortA=(PortA & (DDRA ^ 255)) + (ORA & DDRA);
                break;
      case 2  : DDRB=Value;
                goto NewPortB; // change line status for port
      case 3  : DDRA=Value; 
                goto NewPortA; // change line status for port
      case 4  : T1LL=Value;
                break;
      case 5  :   if ((ACR & 0xc0)==0x80) // one shot mode
                  {
                    ORB|=128; // PB7 goes lo (presumably logic 0)
                    PortB=(PortB & (DDRB ^ 255)) + (ORB & DDRB);
                  }
                  else 
                    T1LH=Value;
                UpdateClocksAndInterrupts();
                T1CL=T1LL;
                T1CH=T1LH;
                IFR&=0xbf; // Bit 6 cleared in IFR

                Timer1=(T1CH*256 + T1CL)*2+2; // was +2

                break;
      case 6  : T1LL=Value;
                break;
      case 7  :   if ((ACR & 0xc0)==0x80) // one shot mode
                    return; // no effect in one shot mode
                T1LH=Value;
                break;
      case 8  : T2LL=Value;
                break;
      case 9  : UpdateClocksAndInterrupts();
                T2CH=Value;
                T2CL=T2LL;
                IFR&=0xdf; // clear bit 5

                Timer2=(T2CH*256 + T2CL)*2+2; // was +2

                SetInterruptOnTimer2=TRUE;
                break;
                
      case 10 : IFR&=0xfb; // clear bit 2
                UpdateClocksAndInterrupts();
                SR=Value;
                break;
      case 11 : ACR=Value;
                break;
      case 12 ://   if ( ((PCR & 0xe0)==0xc0) && ((Value & 0xe0)==0xe0) )
               //   { // CB2 pulsed high - so fake an interrupt
               //     SETUserCB2(); // a fix?
               //   }
                PCR=Value;
                break;
      case 13 : Value&=0x7f; // bit 7 cannot be just 'cleared'
                IFR&=(Value ^ 0x7f) ;  // clear 'set' bits
                  if (IFR)
                    IFR|=128;
                UpdateClocksAndInterrupts();
                break;
      case 14 :   if (Value>=128) // setting bits
                    IER|=(Value & 0x7f);
                  else // clearing bits
                    IER&=(Value ^ 0x7f);
                UpdateClocksAndInterrupts();
                break;
    }
}


int UserViaMinClock()
{
#ifdef DEV
  printf("USER T1=%d, T2=%d\n",Timer1,Timer2);
#endif

    if (Timer1<Timer2)
      return (Timer1);
    else
      return (Timer2);
}


void CalcUserVia(int Cycles)
{
  // GENERATE Interrupts if necessary

// PB7 goes hi on T1 time out (single shot mode)
// IFR bit set

 //decrement Timer1
  Timer1-=Cycles;
    if (Timer1<=0)
    {  
        switch (ACR & 0xc0)
        {
           case 0x80 :  // one shot mode
                        ORB&=0x7f; // PB7 goes hi (presumably logic 1)
                        PortB=(PortB & (DDRB ^ 255)) + (ORB & DDRB);
                        // fall thru
           case 0x00 :  Timer1+=TIMERWRAPVALUE;
                        break;

           case 0xc0 :  // continuous mode
                        ORB^=128; // PB7 goes toggles (square wave)
                        PortB=(PortB & (DDRB ^ 255)) + (ORB & DDRB);
                        // fall thru
           case 0x40 :  Timer1+=(((T1LH<<8) + T1LL)<<1);              
                          if (TotalClockCycles % 2==1)
                            Timer1++;
        }
      IFR|=64; // t1 interrupt flag set
    }


  // test timer2 action
    if ((ACR & 0x20)==0) // one shot mode
    {
     // T2 time out (1 shot mode) disables its interupt
     //decrement Timer2
      Timer2-=Cycles;
        if (Timer2<=0)
        {
          Timer2+=TIMERWRAPVALUE;
            if (SetInterruptOnTimer2)
            {
              IFR|=32; // t2 interrupt flag set
              SetInterruptOnTimer2=FALSE;
            }
        }
    }
    else // pulse counting mode
    {
      // do nothing unless PB6 is pulsed
      // then decrement T2
/*
        if (Timer2<0 && SetInterruptOnTimer2)
        {
          Timer2+=TIMERWRAPVALUE;
          IFR|=32; // t2 interrupt flag set
          SetInterruptOnTimer2=FALSE;
        }
*/
    }

  // generate interrupts if occurred & enabled

// IFR bit 7 cleared if IFR & 0x7f ==0 || IER==0
    if ((IFR & IER & 0x7f)>0)  // set interrupt flag in IFR
    {
      IFR|=128;
      InterruptPendingFlag=TRUE;
    }
    else // clear interrupt flag in IFR
      IFR&=0x7f;
}


void SETUserCB1()
{
  IFR|=0x10; // set CB1 interrupt
  IRB=(PortB & (DDRB ^ 255)) + (ORB & DDRB);
  UpdateClocksAndInterrupts();
}


void SETUserCB2()
{
  IFR|=0x8; // set CB2 interrupt
  UpdateClocksAndInterrupts();
}

// 
void SETUserCA1()
{
  IFR|=0x2; // set CA1 interrupt
  UpdateClocksAndInterrupts();
}

// Printer strobe
void SETUserCA2()
{
  IFR|=0x1; // set CA2 interrupt
  UpdateClocksAndInterrupts();
}


void DumpUserVIA(FILE * FileHandle)
{
  fprintf(FileHandle,"ORB = %.2x\n",ORB);
  fprintf(FileHandle,"IRB = %.2x\n",IRB);
  fprintf(FileHandle,"ORA = %.2x\n",ORA);
  fprintf(FileHandle,"IRA = %.2x\n",IRA);
  fprintf(FileHandle,"DDRB = %.2x\n",DDRB);
  fprintf(FileHandle,"DDRA = %.2x\n",DDRA);
  fprintf(FileHandle,"T1CL = %.2x\n",T1CL);
  fprintf(FileHandle,"T1CH = %.2x\n",T1CH);
  fprintf(FileHandle,"T1LL = %.2x\n",T1LL);
  fprintf(FileHandle,"T1LH = %.2x\n",T1LH);
  fprintf(FileHandle,"T2CL = %.2x\n",T2CL);
  fprintf(FileHandle,"T2CH = %.2x\n",T2CH);
  fprintf(FileHandle,"T2LL = %.2x\n",T2LL);
  fprintf(FileHandle,"SR  = %.2x\n",SR);
  fprintf(FileHandle,"ACR = %.2x\n",ACR);
  fprintf(FileHandle,"PCR = %.2x\n",PCR);
  fprintf(FileHandle,"IFR = %.2x\n",IFR);
  fprintf(FileHandle,"IER = %.2x\n",IER);
  fprintf(FileHandle,"Port A = %.2x  - Port B = %.2x\n",PortA,PortB);
  fprintf(FileHandle,"Timer1 = %d  - Timer2 = %d\n",Timer1,Timer2);
}
