/* -------------------------------------------------------------------------- */
/*                                                                            */
/* (C) Copyright D.C.Devenport 1997. All right reserved.                      */
/*                                                                            */
/* THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY      */
/* KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE        */
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR      */
/* PURPOSE.                                                                   */
/*                                                                            */
/* This code, and no part of this code, may not be used in any                */
/* commercial or for-profit venture without the express written               */
/* permission of D.C.Devenport. (DDevenp666@aol.com)                          */
/*                                                                            */
/* Credit must be given within any program that uses any of this code         */
/* OR in the accompanying documentation. (And mail me a copy :) )             */
/*                                                                            */
/*----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include "6502.h"
#include "bbc.h"
#include "main.h"
#include "fs-file.h"
#include "fs-fsc.h"
#include "fs-rom.h"
#include "fs.h"

#include "uservia.h"
#include "sysvia.h"

#define __INCLUDEUNDOCUMENTED__ 1
// #define __TESTFLAGS__
// #define __LOGBCD__ 1
// #define __TRAPADDRESS__ 0x7413

BYTE   AddressSpace[0x10000];
struct ProcessorType Processor;


void Reset6502()
{
// All registers except the Program Counter remain unmodified after -RESET.
// (This is why you must preset D and I flags in the RESET handler.)

  Processor.PC.ByteValue.Lo=AddressSpace[0xfffc]; // reset vectors
  Processor.PC.ByteValue.Hi=AddressSpace[0xfffd];

// PC always points to the next byte to be executed

  SETIFLAG; // bodge
}


static void InterruptOccurs(bool NMIFlag)
{
  WAIT_CYCLES(7);

  PUSH(Processor.PC.ByteValue.Hi);
  PUSH(Processor.PC.ByteValue.Lo);
  PUSH(MAKEIRQSTATUSREG);
  SETIFLAG;

    if (NMIFlag)
    {
      Processor.PC.ByteValue.Lo=AddressSpace[0xfffa]; // PC = NMI vector contents
      Processor.PC.ByteValue.Hi=AddressSpace[0xfffb];
    }
    else
    {
      Processor.PC.ByteValue.Lo=AddressSpace[0xfffe]; // PC = IRQ vector contents
      Processor.PC.ByteValue.Hi=AddressSpace[0xffff];
    }
}


void Write(register WORD Address,register BYTE Value)
{
    if (Address < 0x8000)          
      AddressSpace[Address]=Value; 
    else if (Address<0xc000)
    {
        if (!CurrentROMReadOnly)
        {
          AddressSpace[Address]=Value;    
          CurrentSidewaysRAMAltered=TRUE;
        }
    }                                 
    else if ((Address & 0xff00)==0xfe00) 
      WriteShiela(Address,Value); 
}


BYTE Read(register WORD Address)
{
    if (Address<0xfe00 || Address>0xfeff)
      return(AddressSpace[Address]);

  return (ReadShiela(Address));
}


//////////////
// Below refer to processor thru a pointer
//////////////

#define GETBYTE (AddressSpace[P6502->PC.WordValue++])

///////
// All AddressOf_ leave the result in variable Address
// can destroy variable Byte
//////

#define AddressOf_ZEROPAGE() \
 Address.WordValue=(WORD) (GETBYTE);

// page 67 of AMCT explains below
#define AddressOf_ZEROPAGEX() \
  AddressOf_ZEROPAGE(); \
  Address.ByteValue.Lo+=P6502->XReg;


#define AddressOf_ZEROPAGEY() \
  AddressOf_ZEROPAGE(); \
  Address.ByteValue.Lo+=P6502->YReg;

#define AddressOf_ABSOLUTE() \
  Address.ByteValue.Lo=AddressSpace[P6502->PC.WordValue++]; \
  Address.ByteValue.Hi=AddressSpace[P6502->PC.WordValue++];

#define AddressOf_ABSOLUTEX() \
  AddressOf_ABSOLUTE(); \
  Address.WordValue+= (WORD) P6502->XReg;

#define AddressOf_ABSOLUTEXPLUS() \
  AddressOf_ABSOLUTE(); \
  Byte=Address.ByteValue.Hi; \
  Address.WordValue+=(WORD) P6502->XReg; \
  WAIT_CYCLES(Address.ByteValue.Hi-Byte); 

#define AddressOf_ABSOLUTEY() \
  AddressOf_ABSOLUTE(); \
  Address.WordValue+= (WORD) P6502->YReg;

// page37 of AUG explains below
#define AddressOf_INDIRECT() \
  AddressOf_ABSOLUTE(); \
  SpareAddress=Address; \
  Address.ByteValue.Lo=AddressSpace[SpareAddress.WordValue]; \
  SpareAddress.ByteValue.Lo++; \
  Address.ByteValue.Hi=AddressSpace[SpareAddress.WordValue];

#define AddressOf_PRE_INDEXED() \
  AddressOf_ZEROPAGEX(); \
  SpareAddress=Address; \
  Address.ByteValue.Lo=AddressSpace[SpareAddress.WordValue]; \
  SpareAddress.ByteValue.Lo++; \
  Address.ByteValue.Hi=AddressSpace[SpareAddress.WordValue];

#define AddressOf_POST_INDEXED() \
  AddressOf_ZEROPAGE(); \
  SpareAddress=Address; \
  Address.ByteValue.Lo=AddressSpace[SpareAddress.WordValue]; \
  SpareAddress.ByteValue.Lo++; \
  Address.ByteValue.Hi=AddressSpace[SpareAddress.WordValue];\
  Address.WordValue+=P6502->YReg;



///////
// All ValueOf_ leave the result in variable Byte and Address read from in
// Address
//////

// assume immediate   *never* reads from FRED/JIM/SHIELA
// zeropage\zeropagex *never* reads from FRED/JIM/SHIELA
#define ValueOf_IMMEDIATE() \
  Byte=GETBYTE;

#define ValueOf_ZEROPAGE() \
  AddressOf_ZEROPAGE(); \
  Byte=AddressSpace[Address.WordValue];

#define ValueOf_ZEROPAGEX() \
  AddressOf_ZEROPAGEX(); \
  Byte=AddressSpace[Address.WordValue];

#define ValueOf_ZEROPAGEY() \
  AddressOf_ZEROPAGEY(); \
  Byte=AddressSpace[Address.WordValue];

#define ValueOf_ABSOLUTE() \
  AddressOf_ABSOLUTE(); \
  Byte=Read(Address.WordValue);

// if crossed boundary=1 extra cycle, no branch needed :)
#define ValueOf_ABSOLUTEXPLUS() \
  AddressOf_ABSOLUTEXPLUS(); \
  Byte=Read(Address.WordValue);

#define ValueOf_ABSOLUTEYPLUS() \
  AddressOf_ABSOLUTE(); \
  Byte=Address.ByteValue.Hi; \
  Address.WordValue+= (WORD) P6502->YReg; \
  WAIT_CYCLES(Address.ByteValue.Hi-Byte); \
  Byte=Read(Address.WordValue);

#define ValueOf_PRE_INDEXED() \
  AddressOf_PRE_INDEXED(); \
  Byte=Read(Address.WordValue); 


#define ValueOf_POST_INDEXEDPLUS() \
  AddressOf_ZEROPAGE(); \
  SpareAddress=Address; \
  Address.ByteValue.Lo=AddressSpace[SpareAddress.WordValue]; \
  SpareAddress.ByteValue.Lo++; \
  Address.ByteValue.Hi=AddressSpace[SpareAddress.WordValue];\
  Byte=Address.ByteValue.Hi; \
  Address.WordValue+=P6502->YReg; \
  WAIT_CYCLES(Address.ByteValue.Hi-Byte); \
  Byte=Read(Address.WordValue);


//////////////////////////////////////////////////////////////////
// Multiple commands
//////////////////////////////////////////////////////////////////

#define ANDCommand(B) \
  Byte=B;\
  P6502->AReg&=Byte; \
  P6502->NFlag=P6502->AReg & 128; \
  SETZ(P6502->AReg);

  // bits 6 and 7 from Byte to V and N
#define BITCommand(B) \
  Byte=B;\
  Result=(P6502->AReg & Byte); \
  SETZ(Result); \
  P6502->NFlag=(Byte & 128); \
  P6502->VFlag=(Byte & 64); 


#define CMPCommand(B) \
{  \
  Byte=B;\
  Result=P6502->AReg-Byte; \
  SETFLAG((P6502->AReg>=Byte),P6502->CFlag); \
  SETZ(Result); \
  P6502->NFlag=(Result & 128); \
}


#define CPXCommand(B) \
{ \
  Byte=B;\
  Result=P6502->XReg-Byte; \
  SETFLAG((P6502->XReg>=Byte),P6502->CFlag); \
  SETZ(Result); \
  P6502->NFlag=(Result & 128); \
}


#define CPYCommand(B) \
{ \
  Byte=B;\
  Result=P6502->YReg-Byte; \
  SETFLAG((P6502->YReg>=Byte),P6502->CFlag); \
  SETZ(Result); \
  P6502->NFlag=(Result & 128); \
}


#define EORCommand(B) \
  Byte=B;\
  P6502->AReg^=Byte; \
  P6502->NFlag=P6502->AReg & 128; \
  SETZ(P6502->AReg); 


#define LDACommand(B) \
  Byte=B;\
  P6502->AReg=Byte; \
  P6502->NFlag=P6502->AReg & 128; \
  SETZ(P6502->AReg);


#define LDXCommand(B) \
  Byte=B;\
  P6502->XReg=Byte; \
  P6502->NFlag=P6502->XReg & 128; \
  SETZ(P6502->XReg); 


#define LDYCommand(B) \
  Byte=B;\
  P6502->YReg=Byte; \
  P6502->NFlag=P6502->YReg & 128; \
  SETZ(P6502->YReg);


#define ORACommand(B) \
  Byte=B;\
  P6502->AReg|=Byte; \
  P6502->NFlag=P6502->AReg & 128; \
  SETZ(P6502->AReg);


///////////////////////////////////////////////////////////////////////
// BELOW have action on memory/registers - Result in variable Result //
///////////////////////////////////////////////////////////////////////

//C = old bit 7
#define ASLCommand(B) \
  Byte=B; \
  SETFLAG(((Byte & 128)==128),P6502->CFlag);\ 
  Result=Byte<<1; \
  P6502->NFlag=Result & 128; \
  SETZ(Result);

// c = old bit 0
#define LSRCommand(B) \
  Byte=B; \
  SETFLAG(((Byte & 1)==1),P6502->CFlag); \
  Result=Byte>>1; \
  UNSET(P6502->NFlag); \
  SETZ(Result);

 // c = old bit 7
#define ROLCommand(B) \
  Byte=B; \
  OldCFlag=P6502->CFlag; \
  SETFLAG(((Byte & 128)==128),P6502->CFlag); \
  Result=(Byte<<1) | OldCFlag; \
  P6502->NFlag=Result & 128; \
  SETZ(Result); 

 // c = old bit 7
#define RORCommand(B) \
  Byte=B; \
  OldCFlag=P6502->CFlag; \
  SETFLAG(((Byte & 1)==1),P6502->CFlag); \
  Result=(Byte>>1) | (OldCFlag<<7); \
  P6502->NFlag=Result & 128; \
  SETZ(Result); 

#define PullStatusByte() \
  POP(Byte); \
  P6502->NFlag=(Byte & SIGNFLAG); \
  P6502->VFlag=(Byte & OVERFLOWFLAG); \
  P6502->DFlag=(Byte & DECIMALFLAG); \
  P6502->IFlag=(Byte & INTERRUPTFLAG); \
  SETFLAG(((Byte & 2)==2) ,P6502->ZFlag); \
  SETFLAG(((Byte & 1)==1) ,P6502->CFlag);


////////////////////////////////////////////
// 'undocumented' (I found a document :) )
// alter memory
////////////////////////////////////////////

//C = old bit 7
#define ASOCommand(B) \
  Byte=B; \
  SETFLAG(((Byte & 128)>0),P6502->CFlag);\ 
  Result=Byte<<1; \
  Write(Address.WordValue,Result); \
  P6502->AReg|=Result; \
  P6502->NFlag=P6502->AReg & 128; \
  SETZ(P6502->AReg);

#define RLACommand(B) \
  Byte=B; \
  OldCFlag=P6502->CFlag; \
  SETFLAG(((Byte & 128)>0),P6502->CFlag); \
  Result=(Byte<<1) + (OldCFlag ? 1 : 0); \
  Write(Address.WordValue,Result); \
  P6502->AReg&=Result; \
  P6502->NFlag=P6502->AReg & 128; \
  SETZ(P6502->AReg);

#define LSECommand(B) \
  Byte=B; \
  SETFLAG((Byte & 1),P6502->CFlag); \
  Result=Byte>>1; \
  Write(Address.WordValue,Result); \
  P6502->AReg^=Result; \
  P6502->NFlag=P6502->AReg & 128; \
  SETZ(P6502->AReg); 

// flags set before ADCCommand, because D mode doesn't set them!
#define RRACommand(B) \
  Byte=B; \
  OldCFlag=P6502->CFlag; \
  SETFLAG((Byte & 1),P6502->CFlag); \
  Result=((Byte>>1) + (OldCFlag ? 128 : 0)); \
  Write(Address.WordValue,Result); \
  P6502->NFlag=Result & 128; \
  SETZ(Result); \
  ADCCommand(Result);

// basically same as AND#imm, but C set to NFlag value
#define ANCCommand(B) \
  Byte=B; \
  P6502->AReg&=Byte; \
  P6502->NFlag=P6502->AReg & 128; \
  P6502->CFlag=(P6502->AReg>>7); \
  SETZ(P6502->AReg);



//////////////////////////////////////////////////////////////////
// BRANCH needs cycles adding
//////////////////////////////////////////////////////////////////
// normal 2 cycles
// +2 if branch succeeds and page crossed
// +1 if branch succeeds otherwise
        // extra wait if page boundary crossed
        // little speed up doesn't work here
        // Byte-P6502->PC.ByteValue.Hi could be
        // -1,0 or 1

#define BranchCommand(BranchFlag) \
if (BranchFlag) \
{ \
  Address.WordValue=(WORD) GETBYTE; \
  Byte=P6502->PC.ByteValue.Hi; \
    if (Address.ByteValue.Lo>127) \
      Address.ByteValue.Hi=0xff; \
  P6502->PC.WordValue+=Address.WordValue; \
    if (Byte!=P6502->PC.ByteValue.Hi) \
    { \
      WAIT_CYCLES(4); \
    } \
    else \
    { \
      WAIT_CYCLES(3); \
    } \
} \
else \
{ \
  WAIT_CYCLES(2); \
  P6502->PC.WordValue++; \
} 


void Do6502Instructions()
{
  static   union  WordField SpareAddress;
  static   BYTE   OldCFlag;
  register union  WordField Address;
  register BYTE   Byte,Result;
  register struct ProcessorType * P6502=&Processor;
  

    if (InterruptPendingFlag && (P6502->IFlag==0))
      InterruptOccurs(FALSE); // generate IRQ

    do
    {
        switch (GETBYTE)
        {
          // ADC - only affects A + flags
          case 0x69 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      ADCCommand(Byte);
                      break;
          case 0x65 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      ADCCommand(Byte);
                      break;
          case 0x75 : WAIT_CYCLES(4);
                      ValueOf_ZEROPAGEX();
                      ADCCommand(Byte);
                      break;
          case 0x6d : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      ADCCommand(Byte);
                      break;     
          case 0x7d : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEXPLUS();
                      ADCCommand(Byte);
                      break;
          case 0x79 : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEYPLUS();
                      ADCCommand(Byte);
                      break;
          case 0x61 : WAIT_CYCLES(6);
                      ValueOf_PRE_INDEXED();
                      ADCCommand(Byte);
                      break;
          case 0x71 : WAIT_CYCLES(5);
                      ValueOf_POST_INDEXEDPLUS();
                      ADCCommand(Byte);
                      break;
    
          // AND - only affects A + flags
          case 0x29 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      ANDCommand(Byte);
                      break;
          case 0x25 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      ANDCommand(Byte);
                      break;
          case 0x35 : WAIT_CYCLES(4);
                      ValueOf_ZEROPAGEX();
                      ANDCommand(Byte);
                      break;
          case 0x2d : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      ANDCommand(Byte);
                      break;
          case 0x3d : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEXPLUS();
                      ANDCommand(Byte);
                      break;
          case 0x39 : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEYPLUS();
                      ANDCommand(Byte);
                      break;
          case 0x21 : WAIT_CYCLES(6);
                      ValueOf_PRE_INDEXED();
                      ANDCommand(Byte);
                      break;
          case 0x31 : WAIT_CYCLES(5);
                      ValueOf_POST_INDEXEDPLUS();
                      ANDCommand(Byte);
                      break;
    
          // ASL - affects A/memory + flags
          case 0x0a : WAIT_CYCLES(2);
                      ASLCommand(P6502->AReg);
                      P6502->AReg=Result;
                      break;
          case 0x06 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      ASLCommand(AddressSpace[Address.WordValue]);
                      AddressSpace[Address.WordValue]=Result;
                      break;
          case 0x16 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      ASLCommand(AddressSpace[Address.WordValue]);
                      AddressSpace[Address.WordValue]=Result;
                      break;
          case 0x0e : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();
                      Byte=Read(Address.WordValue);
                      ASLCommand(Byte);
                      Write(Address.WordValue,Result);
                      break;
          case 0x1e : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEX();
                      Byte=Read(Address.WordValue);
                      ASLCommand(Byte);
                      Write(Address.WordValue,Result);
                      break;
    
          // BCC - affects PC
          case 0x90 : BranchCommand(P6502->CFlag==0);
                      break;
    
          // BCS - affects PC
          case 0xb0 : BranchCommand(P6502->CFlag);
                      break;
    
          // BEQ - affects PC
          case 0xf0 : BranchCommand(P6502->ZFlag);
                      break;
    
          // BIT - affects flags
          case 0x24 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      BITCommand(Byte);
                      break;
          case 0x2c : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      BITCommand(Byte);
                      break;
    
          // BMI - affects PC
          case 0x30 : BranchCommand(P6502->NFlag);
                      break;
    
          // BNE - affects PC
          case 0xd0 : BranchCommand(P6502->ZFlag==0);
                      break;
    
          // BPL - affects PC
          case 0x10 : BranchCommand(P6502->NFlag==0);
                      break;
    
          // BRK - affects PC,S,I flag
          case 0x00 : WAIT_CYCLES(7);
                      P6502->PC.WordValue++; // skip error number
                      PUSH(P6502->PC.ByteValue.Hi);
                      PUSH(P6502->PC.ByteValue.Lo);
                      PUSH(MAKESTATUSREG);
                      SETIFLAG;
                      P6502->PC.ByteValue.Hi=AddressSpace[0xffff];
                      P6502->PC.ByteValue.Lo=AddressSpace[0xfffe];
                      break;
    
          // BVC - affects PC
          case 0x50 : BranchCommand(P6502->VFlag==0);
                      break;
    
          // BVS - affects PC
          case 0x70 : BranchCommand(P6502->VFlag);
                      break;
    
          // CLC - affects C flag
          case 0x18 : WAIT_CYCLES(2);
                      UNSET(P6502->CFlag);
                      break;
    
          // CLD - affects D flag
          case 0xd8 : WAIT_CYCLES(2);
                      UNSET(P6502->DFlag);
                      break;
    
          // CLI - affects I flag
          case 0x58 : WAIT_CYCLES(2);
                        if (P6502->IFlag!=0)
                        {
                          UNSET(P6502->IFlag);
                            if (InterruptPendingFlag) // cause an interrupt
                              UpdateClocksAndInterrupts();
                        }
                      break;
    
          // CLV - affects V flag
          case 0xb8 : WAIT_CYCLES(2);
                      UNSET(P6502->VFlag);
                      break;
    
          // CMP - affects flags
          case 0xc9 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      CMPCommand(Byte);
                      break;
          case 0xc5 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      CMPCommand(Byte);
                      break;
          case 0xd5 : WAIT_CYCLES(4);
                      ValueOf_ZEROPAGEX();
                      CMPCommand(Byte);
                      break;
          case 0xcd : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      CMPCommand(Byte);
                      break;
          case 0xdd : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEXPLUS();
                      CMPCommand(Byte);
                      break;
          case 0xd9 : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEYPLUS();
                      CMPCommand(Byte);
                      break;
          case 0xc1 : WAIT_CYCLES(6);
                      ValueOf_PRE_INDEXED();
                      CMPCommand(Byte);
                      break;
          case 0xd1 : WAIT_CYCLES(5);
                      ValueOf_POST_INDEXEDPLUS();
                      CMPCommand(Byte);
                      break;
    
          // CPX - affects flags
          case 0xe0 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      CPXCommand(Byte);
                      break;
          case 0xe4 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      CPXCommand(Byte);
                      break;
          case 0xec : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      CPXCommand(Byte);
                      break;
    
          // CPY - affects flags
          case 0xc0 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      CPYCommand(Byte);
                      break;
          case 0xc4 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      CPYCommand(Byte);
                      break;
          case 0xcc : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      CPYCommand(Byte);
                      break;
    
          // DEC - affects memory + flags
          case 0xc6 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      AddressSpace[Address.WordValue]--;
                      Result=AddressSpace[Address.WordValue];
                      SETZ(Result);
                      P6502->NFlag=Result & 128;
                      break;
          case 0xd6 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      AddressSpace[Address.WordValue]--;
                      Result=AddressSpace[Address.WordValue];
                      SETZ(Result);
                      P6502->NFlag=Result & 128;
                      break;
          case 0xce : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();
                      Result=Read(Address.WordValue)-1;
                      Write(Address.WordValue,Result);
                      SETZ(Result);
                      P6502->NFlag=Result & 128;                      
                      break;
          case 0xde : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEX();
                      Result=Read(Address.WordValue)-1;
                      Write(Address.WordValue,Result);
                      SETZ(Result);
                      P6502->NFlag=Result & 128;
                      break;
    
          // DEX - affects X + flags
          case 0xca : WAIT_CYCLES(2);
                      P6502->XReg--;
                      SETZ(P6502->XReg);
                      P6502->NFlag=P6502->XReg & 128;
                      break;
    
          // DEY - affects Y + flags
          case 0x88 : WAIT_CYCLES(2);
                      P6502->YReg--;
                      SETZ(P6502->YReg);
                      P6502->NFlag=P6502->YReg & 128;
                      break;
    
          // EOR - affects A + flags
          case 0x49 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      EORCommand(Byte);
                      break;
          case 0x45 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      EORCommand(Byte);
                      break;
          case 0x55 : WAIT_CYCLES(4);
                      ValueOf_ZEROPAGEX();
                      EORCommand(Byte);
                      break;
          case 0x4d : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      EORCommand(Byte);
                      break;
          case 0x5d : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEXPLUS();
                      EORCommand(Byte);
                      break;
          case 0x59 : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEYPLUS();
                      EORCommand(Byte);
                      break;
          case 0x41 : WAIT_CYCLES(6);
                      ValueOf_PRE_INDEXED();
                      EORCommand(Byte);
                      break;
          case 0x51 : WAIT_CYCLES(7);
                      ValueOf_POST_INDEXEDPLUS();
                      EORCommand(Byte);
                      break;
    
          // INC - affects memory + flags
          case 0xe6 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      AddressSpace[Address.WordValue]++;
                      Result=AddressSpace[Address.WordValue];
                      SETZ(Result);
                      P6502->NFlag=Result & 128;
                      break;
          case 0xf6 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      AddressSpace[Address.WordValue]++;
                      Result=AddressSpace[Address.WordValue];
                      SETZ(Result);
                      P6502->NFlag=Result & 128;
                      break;
          case 0xee : WAIT_CYCLES(6); 
                      AddressOf_ABSOLUTE();
                      Result=Read(Address.WordValue)+1;
                      Write(Address.WordValue,Result);
                      SETZ(Result);
                      P6502->NFlag=Result & 128;
                      break;
          case 0xfe : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEX();
                      Result=Read(Address.WordValue)+1;
                      Write(Address.WordValue,Result);
                      SETZ(Result);
                      P6502->NFlag=Result & 128;
                      break;
    
          // INX - affects X + flags
          case 0xe8 : WAIT_CYCLES(2);
                      P6502->XReg++;
                      SETZ(P6502->XReg);
                      P6502->NFlag=P6502->XReg & 128;
                      break;
    
          // INY - affects X + flags
          case 0xc8 : WAIT_CYCLES(2);
                      P6502->YReg++;
                      SETZ(P6502->YReg);
                      P6502->NFlag=P6502->YReg & 128;
                      break;
    
          // JMP - affects PC
          case 0x4c : WAIT_CYCLES(3);
                      AddressOf_ABSOLUTE();
                      P6502->PC.WordValue=Address.WordValue;
                      break;
          case 0x6c : WAIT_CYCLES(5);
                      AddressOf_INDIRECT();
                      P6502->PC.WordValue=Address.WordValue;
                      break;
    
          // JSR - affects PC,stack
          case 0x20 : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();  // increment PC
                      P6502->PC.WordValue--; // PC -1 is pushed on stack
                                             // it took me ages to sus that
                      PUSH(P6502->PC.ByteValue.Hi);
                      PUSH(P6502->PC.ByteValue.Lo);
                      P6502->PC.WordValue=Address.WordValue;
                      break;
    
          // LDA - affects A + flags
          case 0xa9 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      LDACommand(Byte);
                      break;
          case 0xa5 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      LDACommand(Byte);
                      break;
          case 0xb5 : WAIT_CYCLES(4);
                      ValueOf_ZEROPAGEX();
                      LDACommand(Byte);
                      break;
          case 0xad : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      LDACommand(Byte);
                      break;
          case 0xbd : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEXPLUS();
                      LDACommand(Byte);
                      break;
          case 0xb9 : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEYPLUS();
                      LDACommand(Byte);
                      break;
          case 0xa1 : WAIT_CYCLES(6);
                      ValueOf_PRE_INDEXED();
                      LDACommand(Byte);
                      break;
          case 0xb1 : WAIT_CYCLES(5);
                      ValueOf_POST_INDEXEDPLUS();
                      LDACommand(Byte);
                      break;
    
          // LDX - affects X + flags
          case 0xa2 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      LDXCommand(Byte);
                      break;
          case 0xa6 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      LDXCommand(Byte);
                      break;
          case 0xb6 : WAIT_CYCLES(4);
                      ValueOf_ZEROPAGEY();
                      LDXCommand(Byte);
                      break;
          case 0xae : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      LDXCommand(Byte);
                      break;
          case 0xbe : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEYPLUS();
                      LDXCommand(Byte);
                      break;
    
          // LDY - affects Y + flags
          case 0xa0 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      LDYCommand(Byte);
                      break;
          case 0xa4 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      LDYCommand(Byte);
                      break;
          case 0xb4 : WAIT_CYCLES(4);
                      ValueOf_ZEROPAGEX();
                      LDYCommand(Byte);
                      break;
          case 0xac : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      LDYCommand(Byte);
                      break;
          case 0xbc : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEXPLUS();
                      LDYCommand(Byte);
                      break;
    
          // LSR - affects A/memory + flags
          case 0x4a : WAIT_CYCLES(2);
                      LSRCommand(P6502->AReg);
                      P6502->AReg=Result;
                      break;
          case 0x46 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      LSRCommand(AddressSpace[Address.WordValue]);
                      AddressSpace[Address.WordValue]=Result;
                      break;
          case 0x56 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      LSRCommand(AddressSpace[Address.WordValue]);
                      AddressSpace[Address.WordValue]=Result;
                      break;
          case 0x4e : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();
                      Byte=Read(Address.WordValue);
                      LSRCommand(Byte);
                      Write(Address.WordValue,Result);
                      break;
          case 0x5e : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEXPLUS();
                      Byte=Read(Address.WordValue);
                      LSRCommand(Byte);
                      Write(Address.WordValue,Result);
                      break;
    
#ifdef __INCLUDEUNDOCUMENTED__
          //NOP - affects nothing (0xea is official others 'undocumented')
          case 0x1a :
          case 0x3a :
          case 0x5a :
          case 0x7a :
          case 0xda :
          case 0xfa :
#endif    // NOP
          case 0xea : WAIT_CYCLES(2);
                      break;
    
          // ORA - affects A + flags
          case 0x09 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      ORACommand(Byte);
                      break;
          case 0x05 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      ORACommand(Byte);
                      break;
          case 0x15 : WAIT_CYCLES(4);
                      ValueOf_ZEROPAGEX();
                      ORACommand(Byte);
                      break;
          case 0x0d : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      ORACommand(Byte);
                      break;
          case 0x1d : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEXPLUS();
                      ORACommand(Byte);
                      break;
          case 0x19 : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEYPLUS();
                      ORACommand(Byte);
                      break;
          case 0x01 : WAIT_CYCLES(6);
                      ValueOf_PRE_INDEXED();
                      ORACommand(Byte);
                      break;
          case 0x11 : WAIT_CYCLES(5);
                      ValueOf_POST_INDEXEDPLUS();
                      ORACommand(Byte);
                      break;
    
          // PHA - affects stack
          case 0x48 : WAIT_CYCLES(3);
                      PUSH(P6502->AReg);
                      break;
    
          // PHP - affects stack
          case 0x08 : WAIT_CYCLES(3);
                      PUSH(MAKESTATUSREG);
                      break;
    
          // PLA - affects A, stack + flags
          case 0x68 : WAIT_CYCLES(4);
                      POP(P6502->AReg);
                      SETZ(P6502->AReg);
                      P6502->NFlag=P6502->AReg & 128;
                      break;
    
          // PLP - affects stack + flags
          case 0x28 : WAIT_CYCLES(4);
                      PullStatusByte();
                         // cause an interrupt
                        if (InterruptPendingFlag && (P6502->IFlag==0))
                          UpdateClocksAndInterrupts();
                      break;
      
          // ROL - affects A/memory + flags
          case 0x2a : WAIT_CYCLES(2);
                      ROLCommand(P6502->AReg);
                      P6502->AReg=Result;
                      break;
          case 0x26 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      ROLCommand(AddressSpace[Address.WordValue]);
                      AddressSpace[Address.WordValue]=Result;
                      break;
          case 0x36 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      ROLCommand(AddressSpace[Address.WordValue]);
                      AddressSpace[Address.WordValue]=Result;
                      break;
          case 0x2e : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();
                      Byte=Read(Address.WordValue);
                      ROLCommand(Byte);
                      Write(Address.WordValue,Result);
                      break;
          case 0x3e : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEX();
                      Byte=Read(Address.WordValue);
                      ROLCommand(Byte);
                      Write(Address.WordValue,Result);
                      break;
    
          // ROR - affects A/memory + flags
          case 0x6a : WAIT_CYCLES(2);
                      RORCommand(P6502->AReg);
                      P6502->AReg=Result;
                      break;
          case 0x66 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      RORCommand(AddressSpace[Address.WordValue])
                      AddressSpace[Address.WordValue]=Result;
                      break;
          case 0x76 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      RORCommand(AddressSpace[Address.WordValue]);
                      AddressSpace[Address.WordValue]=Result;
                      break;
          case 0x6e : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();
                      Byte=Read(Address.WordValue);
                      RORCommand(Byte);
                      Write(Address.WordValue,Result);
                      break;
          case 0x7e : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEX();
                      Byte=Read(Address.WordValue);
                      RORCommand(Byte);
                      Write(Address.WordValue,Result);
                      break;
    
          // RTI
          case 0x40 : WAIT_CYCLES(6);
                      PullStatusByte();
                      POP(P6502->PC.ByteValue.Lo);
                      POP(P6502->PC.ByteValue.Hi);
                        // cause an interrupt
                        if (InterruptPendingFlag && (P6502->IFlag==0) )
                          UpdateClocksAndInterrupts();
                      break;
    
          // RTS
          case 0x60 : WAIT_CYCLES(6);
                      POP(P6502->PC.ByteValue.Lo);
                      POP(P6502->PC.ByteValue.Hi);
                      P6502->PC.WordValue++; // see JSR
                      break;
    
          // SBC
#ifdef __INCLUDEUNDOCUMENTED__
          case 0xeb :
#endif
          case 0xe9 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      SBCCommand(Byte);
                      break;
          case 0xe5 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      SBCCommand(Byte);
                      break;
          case 0xf5 : WAIT_CYCLES(4);
                      ValueOf_ZEROPAGEX();
                      SBCCommand(Byte);
                      break;
          case 0xed : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      SBCCommand(Byte);
                      break;
          case 0xfd : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEXPLUS();
                      SBCCommand(Byte);
                      break;
          case 0xf9 : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEYPLUS();
                      SBCCommand(Byte);
                      break;
          case 0xe1 : WAIT_CYCLES(6);
                      ValueOf_PRE_INDEXED();
                      SBCCommand(Byte);
                      break;
          case 0xf1 : WAIT_CYCLES(5);
                      ValueOf_POST_INDEXEDPLUS();
                      SBCCommand(Byte);
                      break;
    
          // SEC - affects C flag
          case 0x38 : WAIT_CYCLES(2);
                      SET(P6502->CFlag);
                      break;
    
          // SED - affects D flag
          case 0xf8 : WAIT_CYCLES(2);
                      SETDFLAG;
                      break;
    
          // SEI - affects I flag
          case 0x78 : WAIT_CYCLES(2);
                      SETIFLAG;
                      break;
    
          // STA - affects memory
          case 0x85 : WAIT_CYCLES(3);
                      AddressOf_ZEROPAGE();
                      AddressSpace[Address.WordValue]=P6502->AReg;
                      break;
          case 0x95 : WAIT_CYCLES(4);
                      AddressOf_ZEROPAGEX();
                      AddressSpace[Address.WordValue]=P6502->AReg;
                      break;
          case 0x8d : WAIT_CYCLES(4);
                      AddressOf_ABSOLUTE();
                      Write(Address.WordValue,P6502->AReg);
                      break;
          case 0x9d : WAIT_CYCLES(5);
                      AddressOf_ABSOLUTEX();
                      Write(Address.WordValue,P6502->AReg);
                      break;
          case 0x99 : WAIT_CYCLES(5);
                      AddressOf_ABSOLUTEY();
                      Write(Address.WordValue,P6502->AReg);
                      break;
          case 0x81 : WAIT_CYCLES(6);
                      AddressOf_PRE_INDEXED();
                      Write(Address.WordValue,P6502->AReg);
                      break;
          case 0x91 : WAIT_CYCLES(6);
                      AddressOf_POST_INDEXED();
                      Write(Address.WordValue,P6502->AReg);
                      break;
    
          // STX - affects memory
          case 0x86 : WAIT_CYCLES(3);
                      AddressOf_ZEROPAGE();
                      AddressSpace[Address.WordValue]=P6502->XReg;
                      break;
          case 0x96 : WAIT_CYCLES(4);
                      AddressOf_ZEROPAGEY();
                      AddressSpace[Address.WordValue]=P6502->XReg;
                      break;
          case 0x8e : WAIT_CYCLES(4);
                      AddressOf_ABSOLUTE();
                      Write(Address.WordValue,P6502->XReg);
                      break;
    
          // STY - affects memory
          case 0x84 : WAIT_CYCLES(3);
                      AddressOf_ZEROPAGE();
                      AddressSpace[Address.WordValue]=P6502->YReg;
                      break;
          case 0x94 : WAIT_CYCLES(4);
                      AddressOf_ZEROPAGEX();
                      AddressSpace[Address.WordValue]=P6502->YReg;
                      break;
          case 0x8c : WAIT_CYCLES(4);
                      AddressOf_ABSOLUTE();
                      Write(Address.WordValue,P6502->YReg);
                      break;
    
          // TAX - affects X + flags
          case 0xaa : WAIT_CYCLES(2);
                      P6502->XReg=P6502->AReg;
                      SETZ(P6502->XReg);
                      P6502->NFlag=P6502->XReg & 128;
                      break;
    
          // TAY - affects Y + flags
          case 0xa8 : WAIT_CYCLES(2);
                      P6502->YReg=P6502->AReg;
                      SETZ(P6502->YReg);
                      P6502->NFlag=P6502->YReg & 128;
                      break;
    
          // TSX - affects X + flags
          case 0xba : WAIT_CYCLES(2);
                      P6502->XReg=P6502->StackPointer;
                      SETZ(P6502->XReg);
                      P6502->NFlag=P6502->XReg & 128;
                      break;
    
          // TXA - affects A + flags
          case 0x8a : WAIT_CYCLES(2);
                      P6502->AReg=P6502->XReg;
                      SETZ(P6502->AReg);
                      P6502->NFlag=P6502->AReg & 128;
                      break;
    
          // TXS - affects S - NO FLAGS!
          case 0x9a : WAIT_CYCLES(2);
                      P6502->StackPointer=P6502->XReg;
                      break;
    
          // TYA - affects A + flags
          case 0x98 : WAIT_CYCLES(2);
                      P6502->AReg=P6502->YReg;
                      SETZ(P6502->AReg);
                      P6502->NFlag=P6502->AReg & 128;
                      break;


/////////////////////////////////
// 'Undocumented' instructions //
/////////////////////////////////


#ifdef __INCLUDEUNDOCUMENTED__
          // ASO (SLO) - same as ASL addr followed by ORA addr
          // affects A/memory + flags
          case 0x07 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      ASOCommand(AddressSpace[Address.WordValue]);
                      break;
          case 0x17 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      ASOCommand(AddressSpace[Address.WordValue]);
                      break;
          case 0x0f : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();
                      ASOCommand(Read(Address.WordValue));
                      break;
          case 0x1f : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEX();
                      ASOCommand(Read(Address.WordValue));
                      break;
          case 0x1b : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEY();
                      ASOCommand(Read(Address.WordValue));
                      break;
          case 0x03 : WAIT_CYCLES(8);
                      AddressOf_PRE_INDEXED();
                      ASOCommand(Read(Address.WordValue));
                      break;
          case 0x13 : WAIT_CYCLES(8);
                      AddressOf_POST_INDEXED();
                      ASOCommand(Read(Address.WordValue));
                      break;

          // RLA - same as ROL addr followed by AND addr
          // affects A/memory + flags
          case 0x27 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      RLACommand(AddressSpace[Address.WordValue]);
                      break;
          case 0x37 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      RLACommand(AddressSpace[Address.WordValue]);
                      break;
          case 0x2f : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();
                      RLACommand(Read(Address.WordValue));
                      break;
          case 0x3f : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEX();
                      RLACommand(Read(Address.WordValue));
                      break;
          case 0x3b : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEY();
                      RLACommand(Read(Address.WordValue));
                      break;
          case 0x23 : WAIT_CYCLES(8);
                      AddressOf_PRE_INDEXED();
                      RLACommand(Read(Address.WordValue));
                      break;
          case 0x33 : WAIT_CYCLES(8);
                      AddressOf_POST_INDEXED();
                      RLACommand(Read(Address.WordValue));
                      break;

          // LSE (SRE) - same as LSR addr followed by EOR addr
          // affects A/memory + flags
          case 0x47 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      LSECommand(AddressSpace[Address.WordValue]);
                      break;
          case 0x57 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      LSECommand(AddressSpace[Address.WordValue]);
                      break;
          case 0x4f : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();
                      LSECommand(Read(Address.WordValue));
                      break;
          case 0x5f : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEX();
                      LSECommand(Read(Address.WordValue));
                      break;
          case 0x5b : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEY();
                      LSECommand(Read(Address.WordValue));
                      break;
          case 0x43 : WAIT_CYCLES(8);
                      AddressOf_PRE_INDEXED();
                      LSECommand(Read(Address.WordValue));
                      break;
          case 0x53 : WAIT_CYCLES(8);
                      AddressOf_POST_INDEXED();
                      LSECommand(Read(Address.WordValue));
                      break;

          // RRA - same as ROR addr followed by ADC addr
          // affects A/memory + flags
          case 0x67 : WAIT_CYCLES(5);
                      AddressOf_ZEROPAGE();
                      RRACommand(AddressSpace[Address.WordValue]);
                      break;
          case 0x77 : WAIT_CYCLES(6);
                      AddressOf_ZEROPAGEX();
                      RRACommand(AddressSpace[Address.WordValue]);
                      break;
          case 0x6f : WAIT_CYCLES(6);
                      AddressOf_ABSOLUTE();
                      RRACommand(Read(Address.WordValue));
                      break;
          case 0x7f : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEX();
                      RRACommand(Read(Address.WordValue));
                      break;
          case 0x7b : WAIT_CYCLES(7);
                      AddressOf_ABSOLUTEY();
                      RRACommand(Read(Address.WordValue));
                      break;
          case 0x63 : WAIT_CYCLES(8);
                      AddressOf_PRE_INDEXED();
                      RRACommand(Read(Address.WordValue));
                      break;
          case 0x73 : WAIT_CYCLES(8);
                      AddressOf_POST_INDEXED();
                      RRACommand(Read(Address.WordValue));
                      break;

          // SAX stores (A AND X) at address
          case 0x87 : WAIT_CYCLES(3);
                      AddressOf_ZEROPAGE();
                      AddressSpace[Address.WordValue]=( P6502->AReg & P6502->XReg );
                      break;
          case 0x97 : WAIT_CYCLES(4);
                      AddressOf_ZEROPAGEY();
                      AddressSpace[Address.WordValue]=( P6502->AReg & P6502->XReg );
                      break;
          case 0x8f : WAIT_CYCLES(4);
                      AddressOf_ABSOLUTE();
                      Write(Address.WordValue,( P6502->AReg & P6502->XReg ) );
                      break;
          case 0x83 : WAIT_CYCLES(6);
                      AddressOf_PRE_INDEXED();
                      Write(Address.WordValue,( P6502->AReg & P6502->XReg ) );
                      break;

          // LAX (loads A and X with contents of address)
          case 0xa7 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      LDACommand(Byte);
                      P6502->XReg=P6502->AReg;
                      break;
          case 0xb7 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGEY();
                      LDACommand(Byte);
                      P6502->XReg=P6502->AReg;
                      break;
          case 0xaf : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTE();
                      LDACommand(Byte);
                      P6502->XReg=P6502->AReg;
                      break;
          case 0xbf : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEYPLUS();
                      LDACommand(Byte);
                      P6502->XReg=P6502->AReg;
                      break;
          case 0xa3 : WAIT_CYCLES(6);
                      ValueOf_PRE_INDEXED();
                      LDACommand(Byte);
                      P6502->XReg=P6502->AReg;
                      break;
          case 0xb3 : WAIT_CYCLES(5);
                      ValueOf_POST_INDEXEDPLUS();
                      LDACommand(Byte);
                      P6502->XReg=P6502->AReg;
                      break;



          // SKB (skip next BYTE)
          case 0x80 :
          case 0x82 :
          case 0xc2 :
          case 0xe2 : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      break;
          case 0x04 : // SKB zp
          case 0x44 :
          case 0x64 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGE();
                      break;
          case 0x14 : // SKB zp,X
          case 0x34 :
          case 0x54 :
          case 0x74 :
          case 0xd4 :
          case 0xf4 : WAIT_CYCLES(3);
                      ValueOf_ZEROPAGEX();
                      break;
                

          // SKW (skip next word)
          case 0x0c : WAIT_CYCLES(4); // *special case*
                      ValueOf_ABSOLUTE();
                      break;
          case 0x1c :
          case 0x3c :
          case 0x5c :
          case 0x7c :
          case 0xdc :
          case 0xfc : WAIT_CYCLES(4);
                      ValueOf_ABSOLUTEXPLUS();
                      break;





          case 0x4b : WAIT_CYCLES(2); // ALR (ASR) imm
                      ValueOf_IMMEDIATE();
                      P6502->AReg&=Byte;
                      LSRCommand(P6502->AReg);
                      P6502->AReg=Result;
                      break;

          case 0x6b : WAIT_CYCLES(2); // ARR imm (could be wrong)
                      ValueOf_IMMEDIATE();
                      P6502->AReg&=Byte;
                      RORCommand(P6502->AReg);
                      P6502->AReg=Result;
                      break;

          // ANC imm (both of em!)
          case 0x0b :
          case 0x2b : WAIT_CYCLES(2);
                      ValueOf_IMMEDIATE();
                      ANDCommand(Byte);
                      break;


#endif
          case HLT_ROM :  OSROM();
                          break;

          case HLT_FILE : OSFILE();
                          break;

          case HLT_FSC :  OSFSC();
                          break;

          case HLT_TEXT : Processor.AReg=DoAFileTextChar();
                          break;

          default :
                    P6502->PC.WordValue--; // point to instruction
                    FatalError("Bad Opcode %.2x",
                               AddressSpace[P6502->PC.WordValue]);
        }

#ifdef __TESTFLAGS__
        if ( ((P6502->NFlag!=0) && (P6502->NFlag!=SIGNFLAG)) ||
             ((P6502->VFlag!=0) && (P6502->VFlag!=OVERFLOWFLAG)) ||
             ((P6502->DFlag!=0) && (P6502->DFlag!=DECIMALFLAG)) ||
             ((P6502->IFlag!=0) && (P6502->IFlag!=INTERRUPTFLAG)) ||
             ((P6502->ZFlag!=0) && (P6502->ZFlag!=1)) ||
             ((P6502->CFlag!=0) && (P6502->CFlag!=1)) )
        {
          Dump6502(stdout);
          FatalError("Bad flag!");
        }
#endif

#ifdef __TRAPADDRESS__
        if (P6502->PC.WordValue==__TRAPADDRESS__ )
        {
          FILE * FileHandle=fopen("cracked","wb");
          fwrite(&AddressSpace[0x3000],0x5000,1,FileHandle);
          fclose(FileHandle);
          exit(-1);
        }
#endif
    } while (CyclesToGo>0);
}


//////////////
// Below refer to processor without the pointer
//////////////


static void ADCCommand(BYTE Byte)
{
  BYTE Org=Processor.AReg;
  BYTE OrgCFlag=Processor.CFlag;

    if (Processor.DFlag==0)
    {  // normal
      union WordField Result;

      Result.WordValue=Processor.AReg+Byte+Processor.CFlag;

      Processor.AReg=Result.ByteValue.Lo;
      SETFLAG((Result.ByteValue.Hi>0),Processor.CFlag);
      Processor.NFlag=Processor.AReg & 128;
      SETZ(Processor.AReg);

      Processor.VFlag=0;
        if ( (((Org ^ Byte) & 128)==0) &&
             (((Org ^ Processor.AReg) & 128)==128) ) // maybe set V
        {
            if (Processor.NFlag==128) //negative result, set V
              SETVFLAG;
            else // positive result
            {
                if (OrgCFlag) // C was 1, so set V if Result>0
                {
                    if (Processor.ZFlag==0)
                      SETVFLAG;
    
                }
                else // C was 0, so set V if Result>=0
                  SETVFLAG;
            }
        }
    }
    else
    {
      BYTE AL, // low nybble of accumulator
           AH=0; // high nybble of accumulator

      union WordField Result;
      Result.WordValue=Processor.AReg+Byte+Processor.CFlag;
      SETZ(Result.ByteValue.Lo); // bizarre I know, but correct!

      AL = (Processor.AReg & 0xf) + (Byte & 0xf) +
           (Processor.CFlag);  // Calculate the lower nybble.

        if (AL>9)// BCD fixup for lower nybble.
        {
          AL-=10;
          AL&=0xf;
          AH=1;
        }

         // Calculate the upper nybble.
      AH += ((Processor.AReg >> 4) + (Byte >> 4));

      // before fix up of top nibble, do N+V
      Processor.AReg=(AH<<4) | AL;
      Processor.NFlag=Processor.AReg & 128;

      Processor.VFlag=0;
        if ( (((Org ^ Byte) & 128)==0) &&
             (((Org ^ Processor.AReg) & 128)==128) ) // maybe set V
        {
            if (Processor.NFlag==128) //negative result, set V
              SETVFLAG;
            else // positive result
            {
                if (OrgCFlag) // C was 1, so set V if Result>0
                {
                    if (Processor.ZFlag==0)
                      SETVFLAG;
    
                }
                else // C was 0, so set V if Result>=0
                  SETVFLAG;
            }
        }

      UNSET(Processor.CFlag);
        if (AH>9)
        {
          SET(Processor.CFlag);
          AH-=10;
          AH&=0xf;
        }

      Processor.AReg=(AH<<4) | AL;


      #ifdef __LOGBCD__
      {
/*
        FILE * FH=fopen("BCD.LOG","at+");
        fprintf(FH,"\nADC #&%.2x ",Byte);
        Dump6502(FH);
        fprintf(FH,"Result   ");
        Dump6502(FH);
        fclose(FH);
*/
        FILE * FH=fopen("RESULT.LOG","ab+");
        fputc(Processor.AReg,FH);
        fclose(FH);
        FH=fopen("FLAGS.LOG","ab+");
        fputc(MAKESTATUSREG,FH);
        fclose(FH);
      }
      #endif
    }
}


static void SBCCommand(BYTE Byte)
{
  BYTE Org=Processor.AReg;
  BYTE OrgCFlag=Processor.CFlag;

    if (Processor.DFlag==0)
    { // normal
      Processor.AReg=Processor.AReg-Byte-(1-Processor.CFlag);
      Processor.NFlag=Processor.AReg & 128;
      SETZ(Processor.AReg);

        if (OrgCFlag)
          SETFLAG((Byte<=Org),Processor.CFlag);
        else
          SETFLAG((Byte<Org),Processor.CFlag);

      Processor.VFlag=0;

        if ( (((Org ^ Byte) & 128)==128) &&
             (((Byte ^ Processor.AReg) & 128)==0) ) // maybe set V
        {
            if (Processor.NFlag==128) //negative result, set V
              SETVFLAG;
            else // positive result
            {
                if (OrgCFlag) // C was 1, so set V if Result>0
                {
                    if (Processor.ZFlag==0)
                      SETVFLAG;
    
                }
                else // C was 0, so set V if Result>=0
                  SETVFLAG;
            }
        }
    }
    else
    { 
      BYTE  AL, // low nybble of accumulator
            AH; // high nybble of accumulator
      BYTE  Borrow=0,Result;
      Result=Processor.AReg-Byte-(1-Processor.CFlag);
      SETZ(Result); // bizarre I know, but correct!


      AL = (Processor.AReg & 0xf) - (Byte & 0xf) - (1 - Processor.CFlag);
           // Calculate the lower nybble.

        if ((AL & 16)==16)
        {
          AL-=6;// BCD fixup for lower nybble.
          AL&=0xf;
          Borrow=1; // 16?
        }

      // Calculate the upper nybble.
      AH = (Processor.AReg>>4) - (Byte>>4) - Borrow;

      // do N+V before fixing top nibble
      Processor.AReg = (AH<<4) | AL;
      Processor.NFlag=Processor.AReg & 128;
      Processor.VFlag=0;

        if ( (((Org ^ Byte) & 128)==128) &&
             (((Byte ^ (AH<<4)) & 128)==0) ) // maybe set V
        {
            if (Processor.NFlag==128) //negative result, set V
              SETVFLAG;
            else // positive result
            {
                if (OrgCFlag) // C was 1, so set V if Result>0
                {
                    if (Processor.ZFlag==0)
                      SETVFLAG;
    
                }
                else // C was 0, so set V if Result>=0
                  SETVFLAG;
            }
        }

      SET(Processor.CFlag);

        if ((AH & 16)==16)
        {
          UNSET(Processor.CFlag);
          AH-=6;
        }

      Processor.AReg = ((AH<<4) | AL);

      #ifdef __LOGBCD__
      {
/*
        FILE * FH=fopen("BCD.LOG","at+");
        fprintf(FH,"\nSBC #&%.2x ",Byte);
        Dump6502(FH);
        fprintf(FH,"Result   ");
        Dump6502(FH);
        fclose(FH);
*/
        FILE * FH=fopen("RESULT.LOG","ab+");
        fputc(Processor.AReg,FH);
        fclose(FH);
        FH=fopen("FLAGS.LOG","ab+");
        fputc(MAKESTATUSREG,FH);
        fclose(FH);
      }
      #endif
    }
}


void Dump6502(FILE * FileHandle)
{
  fprintf(FileHandle,"PC=%4x A=%2x X=%2x Y=%2x S=%2x Flags:%c%c-B%c%c%c%c\n",
                     Processor.PC.WordValue,
                     Processor.AReg,
                     Processor.XReg,
                     Processor.YReg,
                     Processor.StackPointer,
                     Processor.NFlag ? 'N' : ' ',
                     Processor.VFlag ? 'V' : ' ',
                     Processor.DFlag ? 'D' : ' ',
                     Processor.IFlag ? 'I' : ' ',
                     Processor.ZFlag ? 'Z' : ' ',
                     Processor.CFlag ? 'C' : ' ');
/*
  fprintf(FileHandle,"N=%.2x  V=%.2x  D=%.2x  I=%.2x  Z=%.2x  C=%.2x  \n",
                     Processor.NFlag,
                     Processor.VFlag,
                     Processor.DFlag,
                     Processor.IFlag,
                     Processor.ZFlag,
                     Processor.CFlag);
*/                      
}
