/*****************************************************************************
 *****************************************************************************
 Copyright (c) 2001 - 2002, VIA Technologies, Inc.

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation 
    and/or other materials provided with the distribution.

 3. Neither the name of VIA Technologies, Inc. nor the names of its contributors 
    may be used to endorse or promote products derived from this software 
    without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
 ****************************************************************************/

#ifndef RHINE_H
#define RHINE_H

#ifdef MODULE
#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif /* MODVERSIONS */
#include <linux/module.h>
#endif /* MODULE */

#include <linux/types.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/version.h>
#include <linux/string.h>
#include <linux/wait.h>
#include <asm/io.h>

#include <linux/if.h>
#include <linux/config.h>
#include <asm/uaccess.h>
#include <linux/proc_fs.h>

#include "kcompat.h"
#include "rhine_cfg.h"

#ifdef CONFIG_PROC_FS
#include "rhine_proc.h"
#endif


#ifdef RHINE_DEBUG
#define ASSERT(x) { \
	if (!(x)) { \
		printk(KERN_ERR "assertion %s failed: file %s line %d\n", #x,\
		__FUNCTION__, __LINE__);\
		*(int*) 0=0;\
	}\
}
#else
#define ASSERT(x)	
#endif

typedef enum __rhine_msg_level {
	MSG_LEVEL_ERR=0,            //Errors that will cause abnormal operation.
	MSG_LEVEL_NOTICE=1,			//Some errors need users to be notified.
	MSG_LEVEL_INFO=2,           //Normal message.
	MSG_LEVEL_VERBOSE=3,        //Will report all trival errors.
	MSG_LEVEL_DEBUG=4           //Only for debug purpose.
} RHINE_MSG_LEVEL, *PRHINE_MSG_LEVEL;

#define RHINE_PRT(l, p, args...) {if (l<=msglevel) printk( p ,##args);}

#ifdef VMNS
#include "rhine_vmns.h"
#endif
#define W_MAX_TIMEOUT		0x3fff
//
// Registers in the MAC  
//
#define MAC_REG_PAR         0x00        // physical address
#define MAC_REG_RCR         0x06        // 
#define MAC_REG_TCR         0x07        // 
#define MAC_REG_CR0         0x08        // 
#define MAC_REG_CR1         0x09        //
#define MAC_REG_TQWK        0x0A
#define MAC_REG_ISR         0x0C        //
#define MAC_REG_IMR         0x0E        // 
#define MAC_REG_MAR         0x10        //
#define MAC_REG_MCAM		0x10        //
#define MAC_REG_VCAM		0x16		//
#define MAC_REG_CUR_RD_ADDR 0x18        // 
#define MAC_REG_CUR_TD_ADDR 0x1C        // 
#define MAC_REG_RX_DMA_PTR  0x60        // 
#define MAC_REG_MPHY        0x6C        // 
#define MAC_REG_MIISR       0x6D        // 
#define MAC_REG_BCR0        0x6E        // 
#define MAC_REG_BCR1        0x6F        // 
#define MAC_REG_MIICR       0x70        // 
#define MAC_REG_MIIAD       0x71        // 
#define MAC_REG_MIIDATA     0x72        // 
#define MAC_REG_EECSR       0x74        // 
#define MAC_REG_TEST        0x75        // 
#define MAC_REG_CFGA        0x78        // 
#define MAC_REG_CFGB        0x79        // 
#define MAC_REG_CFGC        0x7A        // 
#define MAC_REG_CFGD        0x7B        // 
#define MAC_REG_CNTR_MPA    0x7C        // 
#define MAC_REG_CNTR_CRC    0x7E        // 

// Registers for VT6102
#define MAC_REG_GFTEST      0x54    //
#define MAC_REG_RFTCMD      0x55    //
#define MAC_REG_TFTCMD      0x56    //
#define MAC_REG_GFSTATUS    0x57    //
#define MAC_REG_BNRY        0x58    // 
#define MAC_REG_CURR        0x5A    //
#define MAC_REG_FIFO_DATA   0x5C    //
#define MAC_REG_CUR_RXDMA   0x60    //
#define MAC_REG_CUR_TXDMA   0x64    //

#define MAC_REG_MISC_CR0    0x80    //
#define MAC_REG_MISC_CR1    0x81    //
#define MAC_REG_PMCPORT     0x82    //
#define MAC_REG_STICKHW     0x83    //
#define MAC_REG_MISR        0x84    //
#define MAC_REG_MIMR        0x86    //
#define MAC_REG_CAMMSK      0x88    //
#define MAC_REG_BPMA        0x8C    //
#define MAC_REG_BPMD        0x8F    //
#define MAC_REG_BPCMD       0x90    //
#define MAC_REG_BPIN_DATA   0x91    //
#define MAC_REG_CAMC        0x92    //
#define MAC_REG_CAMADD   	0x93    //
#define MAC_REG_EECHKSUM    0x93    //
#define MAC_REG_SUS_MII_AD  0x94    //
#define MAC_REG_MIBCR       0x94
#define MAC_REG_MIBDATA		0x96	//
#define MAC_REG_SUS_PHY_ID  0x96    //
#define MAC_REG_MIBPORT     0x96    //
#define MAC_REG_MIBDAT      0x97    //
#define MAC_REG_PAUSE_TIMER 0x98    //
#define MAC_REG_FLOWCR0		0x98
#define MAC_REG_SOFT_TIMER0 0x9C    //
#define MAC_REG_SOFT_TIMER1 0x9E    //
#define MAC_REG_WOLCR_SET   0xA0    //
#define MAC_REG_PWCFG_SET   0xA1    //
#define MAC_REG_TSTREG_SET  0xA2    //
#define MAC_REG_WOLCG_SET   0xA3    //
#define MAC_REG_WOLCR_CLR   0xA4    //
#define MAC_REG_PWCFG_CLR   0xA5    //
#define MAC_REG_TSTREG_CLR  0xA6    //
#define MAC_REG_WOLCG_CLR   0xA7    //
#define MAC_REG_PWRCSR_SET  0xA8    //
#define MAC_REG_PWRCSR_CLR  0xAC    //
#define MAC_REG_PATRN_CRC0  0xB0    //
#define MAC_REG_PATRN_CRC1  0xB4    //
#define MAC_REG_PATRN_CRC2  0xB8    //
#define MAC_REG_PATRN_CRC3  0xBC    //
#define MAC_REG_BYTEMSK0_0  0xC0    //
#define MAC_REG_BYTEMSK0_1  0xC4    //
#define MAC_REG_BYTEMSK0_2  0xC8    //
#define MAC_REG_BYTEMSK0_3  0xCC    //
#define MAC_REG_BYTEMSK1_0  0xD0    //
#define MAC_REG_BYTEMSK1_1  0xD4    //
#define MAC_REG_BYTEMSK1_2  0xD8    //
#define MAC_REG_BYTEMSK1_3  0xDC    //
#define MAC_REG_BYTEMSK2_0  0xE0    //
#define MAC_REG_BYTEMSK2_1  0xE4    //
#define MAC_REG_BYTEMSK2_2  0xE8    //
#define MAC_REG_BYTEMSK2_3  0xEC    //
#define MAC_REG_BYTEMSK3_0  0xF0    //
#define MAC_REG_BYTEMSK3_1  0xF4    //
#define MAC_REG_BYTEMSK3_2  0xF8    //
#define MAC_REG_BYTEMSK3_3  0xFC    //

#define BYTE_REG_BITS_ON(x,p)	do { writeb(readb((p))|(x),(p));} while (0)

#define WORD_REG_BITS_ON(x,p)	do { writew(readw((p))|(x),(p));} while (0)
#define DWORD_REG_BITS_ON(x,p)	do { writel(readl((p))|(x),(p));} while (0)

#define BYTE_REG_BITS_IS_ON(x,p) (readb((p)) & (x))
#define WORD_REG_BITS_IS_ON(x,p) (readw((p)) & (x))
#define DWORD_REG_BITS_IS_ON(x,p) (readl((p)) & (x))

#define BYTE_REG_BITS_OFF(x,p)	do { writeb(readb((p)) & (~(x)),(p));} while (0)
#define WORD_REG_BITS_OFF(x,p)	do { writew(readw((p)) & (~(x)),(p));} while (0)
#define DWORD_REG_BITS_OFF(x,p)	do { writel(readl((p)) & (~(x)),(p));} while (0)

#define BYTE_REG_BITS_SET(x,m,p)	do { writeb( (readb((p)) & (~(m))) |(x),(p));} while (0)
#define WORD_REG_BITS_SET(x,m,p)	do { writew( (readw((p)) & (~(m))) |(x),(p));} while (0)
#define DWORD_REG_BITS_SET(x,m,p)	do { writel( (readl((p)) & (~(m)))|(x),(p));}  while (0)


//
// Bits in the RCR register
//
#define RCR_RRFT2           0x80        //
#define RCR_RRFT1           0x40        //
#define RCR_RRFT0           0x20        //
#define RCR_PROM            0x10        //
#define RCR_AB              0x08        //
#define RCR_AM              0x04        //
#define RCR_AR              0x02        //
#define RCR_SEP             0x01        //

//
// Bits in the TCR register
//
#define TCR_RTSF            0x80        //
#define TCR_RTFT1           0x40        //
#define TCR_RTFT0           0x20        //
#define TCR_RTGOPT			0x10		//
#define TCR_OFSET           0x08        //
#define TCR_LB1             0x04        // loopback[1]
#define TCR_LB0             0x02        // loopback[0]
#define	TCR_PQEN			0x01

//
// Bits in the CR0 register
//
#define CR0_RDMD            0x40        // rx descriptor polling demand
#define CR0_TDMD            0x20        // tx descriptor polling demand
#define CR0_TXON            0x10        // 
#define CR0_RXON            0x08        //
#define CR0_STOP            0x04        // stop MAC, default = 1
#define CR0_STRT            0x02        // start MAC
#define CR0_INIT            0x01        // start init process

#define CR0_SFRST           0x8000      // software reset
#define CR0_RDMD1           0x4000      //
#define CR0_TDMD1           0x2000      //
#define CR0_KEYPAG          0x1000      //
#define CR0_DPOLL           0x0800      // disable rx/tx auto polling
#define CR0_FDX             0x0400      // full duplex mode
#define CR0_ETEN            0x0200      // early tx mode
#define CR0_EREN            0x0100      // early rx mode
//
// Bits in the CR1 register
//
#define CR1_SFRST           0x80        // software reset
#define CR1_RDMD1           0x40        //
#define CR1_TDMD1           0x20        //
#define CR1_REAUTO			0x10		// for VT6105
#define CR1_KEYPAG          0x10        //
#define CR1_DPOLL           0x08        // disable rx/tx auto polling
#define CR1_FDX             0x04        // full duplex mode
#define CR1_DISAU			0x02		// for VT6105
#define CR1_ETEN            0x02        // early tx mode
#define CR1_EREN            0x01        // early rx mode

//
// Bits in the IMR register
//
#define IMR_KEYM            0x8000      //
#define IMR_SRCM            0x4000      //
#define IMR_ABTM            0x2000      //
#define IMR_NORBFM          0x1000      //
#define IMR_PKTRAM          0x0800      //
#define IMR_OVFM            0x0400      //
#define IMR_ETM             0x0200      //
#define IMR_ERM             0x0100      //
#define IMR_CNTM            0x0080      //
#define IMR_BEM             0x0040      //
#define IMR_RUM             0x0020      //
#define IMR_TUM             0x0010      //
#define IMR_TXEM            0x0008      //
#define IMR_RXEM            0x0004      //
#define IMR_PTXM            0x0002      //
#define IMR_PRXM            0x0001      //

#define IMR_TDWBRAI			0x00080000UL //
//
// Bits in the ISR (MISR) register
//
#define ISR_GENI            0x00008000UL    //  for 6102
#define ISR_SRCI            0x00004000UL    //
#define ISR_ABTI            0x00002000UL    //
#define ISR_NORBF           0x00001000UL    //
#define ISR_PKTRA           0x00000800UL    //
#define ISR_OVFI            0x00000400UL    //
#define ISR_UDFI            0x00000200UL    //  for 6102
#define ISR_ERI             0x00000100UL    //
#define ISR_CNT             0x00000080UL    //
#define ISR_BE              0x00000040UL    //
#define ISR_RU              0x00000020UL    //
#define ISR_TU              0x00000010UL    //
#define ISR_TXE             0x00000008UL    //
#define ISR_RXE             0x00000004UL    //
#define ISR_PTX             0x00000002UL    //
#define ISR_PRX             0x00000001UL    //
// Bits in MISR
#define ISR_PWEINT          0x00800000UL    // power event report in test mode
#define ISR_UDPINT_CLR      0x00400000UL    // userdefined, host driven interrupt.clear
#define ISR_UDPINT_SET      0x00200000UL    // userdefined, host driven interrupt.Set
#define ISR_SSRCI           0x00100000UL    // suspend well mii polling status change interrupt
#define ISR_TDWBRAI         0x00080000UL    // TD WB queue race
#define ISR_PHYINT          0x00040000UL    // PHY state change interrupt, active by
					    // PHYINTEN (misc.cr[9]) in normal mode
#define ISR_TM1_INT         0x00020000UL    //
#define ISR_TM0_INT         0x00010000UL    //


//
// Bits in the MIISR register
//
#define MIISR_N_FDX			0x04
#define MIISR_LNKFL         0x02        //
#define MIISR_SPEED         0x01        //

//
// Bits in the MIICR register
//
#define MIICR_MAUTO         0x80        //
#define MIICR_RCMD          0x40        //
#define MIICR_WCMD          0x20        //
#define MIICR_MDPM          0x10        //
#define MIICR_MOUT          0x08        //
#define MIICR_MDO           0x04        //
#define MIICR_MDI           0x02        //
#define MIICR_MDC           0x01        //

//
// Bits in the MIIAD register
//
#define MIIAD_MIDLE         0x80        //
#define MIIAD_MSRCEN        0x40        //
#define MIIAD_MDONE         0x20        //
//
// Bits in the MIBCR register
//
#define MIBCR_MIBEN			0x10
#define MIBCR_MIBHALF		0x20
#define MIBCR_MIBINC		0x40
#define MIBCR_MIBRTN		0x80
//
// Bits in the EECSR register
//
#define EECSR_EEPR          0x80        // eeprom programed status, 73h means programed
#define EECSR_EMBP          0x40        // eeprom embeded programming
#define EECSR_AUTOLD        0x20        // eeprom content reload
#define EECSR_DPM           0x10        // eeprom direct programming
#define EECSR_CS            0x08        // eeprom CS pin
#define EECSR_SK            0x04        // eeprom SK pin
#define EECSR_DI            0x02        // eeprom DI pin
#define EECSR_DO            0x01        // eeprom DO pin

//
// Bits in the BCR0 register
//
#define BCR0_BOOT_MASK		((BYTE) 0xC0)
#define BCR0_BOOT_INT19		((BYTE) 0x00)
#define BCR0_BOOT_INT18     ((BYTE) 0x40)
#define BCR0_BOOT_LOCAL		((BYTE) 0x80)
#define BCR0_BOOT_BEV		((BYTE) 0xC0)

#define BCR0_MED2           0x80        //
#define BCR0_LED100M        0x40        //
#define BCR0_CRFT2          0x20        //
#define BCR0_CRFT1          0x10        //
#define BCR0_CRFT0          0x08        //
#define BCR0_DMAL2          0x04        //
#define BCR0_DMAL1          0x02        //
#define BCR0_DMAL0          0x01        //

//

// Bits in the BCR1 register
//
#define BCR1_MED1           0x80        // for VT6102
#define BCR1_MED0           0x40        // for VT6102
#define BCR1_VIDFR			0x80		// for VT6105
#define BCR1_TXQNOBK		0x40		// for VT6105
#define BCR1_CTSF           0x20        //
#define BCR1_CTFT1          0x10        //
#define BCR1_CTFT0          0x08        //
#define BCR1_POT2           0x04        //
#define BCR1_POT1           0x02        //
#define BCR1_POT0           0x01        //

//
// Bits in the CFGA register
//
#define CFGA_EELOAD         0x80        // enable eeprom embeded and direct programming
#define CFGA_LED0S0         0x01        //
//
// Bits in the CFGB register
//
#define CFGB_QPKTDIS        0x80        //
#define CFGB_MRLDIS         0x20        //

//
// Bits in the CFGC register
//
#define CFGC_BOOT_RPL		((BYTE) 0x80) //Boot method selection for VT3106
#define CFGC_MEDEN          0x80        //
#define CFGC_BROPT          0x40        //
#define CFGC_DLYEN          0x20        //
#define CFGC_DTSEL          0x10        //
#define CFGC_BTSEL          0x08        //
#define CFGC_BPS2           0x04        // bootrom select[2]
#define CFGC_BPS1           0x02        // bootrom select[1]
#define CFGC_BPS0           0x01        // bootrom select[0]

typedef enum {
	RHINE_VLAN_ID_CAM,
	RHINE_MULTICAST_CAM
} RHINE_CAM_TYPE, *PRHINE_CAM_TYPE;
//
// Bits in the CAMCR register
//
#define CAMC_CAMRD          0x08    
#define CAMC_CAMWR          0x04    
#define CAMC_VCAMSL         0x02    
#define CAMC_CAMEN          0x01    

//
// Bits in the CFGD register
//
#define CFGD_GPIOEN         0x80        //
#define CFGD_DIAG           0x40        //
#define CFGD_MAGIC          0x10        //
#define CFGD_CRADOM         0x08        //
#define CFGD_CAP            0x04        //
#define CFGD_MBA            0x02        //
#define CFGD_BAKOPT         0x01        //

// for VT6102 

// Bits in STICKHW
#define STICKHW_LEGWOLEN        0x0080  // status for software reference
#define STICKHW_LEGACY_WOLSR    0x0008
#define STICKHW_LEGACY_WOLEN    0x0004
#define STICKHW_DS1_SHADOW      0x0002  // R/W by software/cfg cycle
#define STICKHW_DS0_SHADOW      0x0001  // suspend well DS write port

// Bits in MISC.CR0
#define MISC_CR0_TM0US          0x20
#define MISC_CR0_FDXTFEN        0x10    // Full-duplex flow control TX enable
#define MISC_CR0_FDXRFEN        0x08    // Full-duplex flow control RX enable
#define MISC_CR0_HDXFEN         0x04    // Half-duplex flow control enable
#define MISC_CR0_TIMER0_SUSPEND 0x02
#define MISC_CR0_TIMER0_EN      0x01

// Bits in MISC.CR1
#define MISC_CR1_FORSRST        0x40
#define MISC_CR1_VAUXJMP        0x20
#define MISC_CR1_PHYINT         0x02
#define MISC_CR1_TIMER1_EN      0x01

// Bits in BPCMD
#define BPCMD_BPDNE             0x80
#define BPCMD_EBPWR             0x02
#define BPCMD_EBPRD             0x01

// Bits in MISR
#define MISR_PWEINT             0x80    // power event report in test mode
#define MISR_UDPINT_CLR         0x40    // userdefined, host driven interrupt.clear
#define MISR_UDPINT_SET         0x20    // userdefined, host driven interrupt.Set
#define MISR_SSRCI              0x10    // suspend well mii polling status change interrupt
#define MISR_TDWBRAI            0x08    // TD WB queue race
#define MISR_PHYINT             0x04    // PHY state change interrupt, active by
					// PHYINTEN (misc.cr[9]) in normal mode
#define MISR_TM1_INT            0x02
#define MISR_TM0_INT            0x01

// Bits in WOLCR
#define WOLCR_LNKOFF_EN         0x80    // link off detected enable
#define WOLCR_LNKON_EN          0x40    // link on detected enable
#define WOLCR_MAGPKT_EN         0x20    // magic packet filter enable
#define WOLCR_UNICAST_EN        0x10    // unicast filter enable
#define WOLCR_MSWOLEN3          0x08    // enable pattern match filtering
#define WOLCR_MSWOLEN2          0x04
#define WOLCR_MSWOLEN1          0x02
#define WOLCR_MSWOLEN0          0x01


// Bits in PWCFG
#define PWCFG_SMIITIME          0x80    // internal MII I/F timing
#define PWCFG_PCISTICK          0x40    // PCI sticky R/W enable
#define PWCFG_WOLTYPE           0x20    // pulse(1) or button (0)
#define PWCFG_LEGCY_WOL                 0x10    
#define PWCFG_PMCSR_PME_SR      0x08  
#define PWCFG_PMCSR_PME_EN      0x04    // control by PCISTICK
#define PWCFG_LEGACY_WOLSR      0x02    // Legacy WOL_SR shadow
#define PWCFG_LEGACY_WOLEN      0x01    // Legacy WOL_EN shadow

// Bits in TestReg
#define TSTREG_SGENWATCH        0x08  
#define TSTREG_SMCSNOOP         0x04
#define TSTREG_SMACTEST         0x02
#define TSTREG_SNORMAL          0x01

// Bits in WOLCFG
#define WOLCFG_PME_OVR          0x80    // for legacy use, force PMEEN always
#define WOLCFG_SFDX             0x40    // full duplex while in WOL mode
#define WOLCFG_SAM              0x20    // accept multicast case reset, default=0
#define WOLCFG_SAB              0x10    // accept broadcast case reset, default=0
#define WOLCFG_SMIIACC          0x08    // ??
#define WOLCFG_SMIIOPT          0x04    // MIIOPT to extend clock in suspendwell
#define WOLCFG_SSRCEN           0x02    // suspend well mii status change enable
#define WOLCFG_PHYINTEN         0x01    // 0:PHYINT trigger enable, 1:use internal MII
					// to report status change

// Ethernet address filter type
#define PKT_TYPE_NONE               0x0000  // turn off receiver
#define PKT_TYPE_DIRECTED           0x0001  // obselete, directed address is always accepted
#define PKT_TYPE_MULTICAST          0x0002
#define PKT_TYPE_ALL_MULTICAST      0x0004
#define PKT_TYPE_BROADCAST          0x0008
#define PKT_TYPE_PROMISCUOUS        0x0020
#define PKT_TYPE_LONG               0x2000  // NOTE.... the definition of LONG is >2048 bytes in our chip
#define PKT_TYPE_RUNT               0x4000
#define PKT_TYPE_ERROR              0x8000  // accept error packets, e.g. CRC error


// Loopback mode
#define MAC_LB_NONE         0x00        //
#define MAC_LB_INTERNAL     0x01        //
#define MAC_LB_PHY          0x02        // MII or Internal-10BaseT loopback


// Forced media type
#define FORCED_MEDIA_NONE       0x00    //
#define FORCED_MEDIA_AUTO       0x01    //
#define FORCED_MEDIA_100M_HALF  0x02    // hub card
#define FORCED_MEDIA_100M_FULL  0x03    // fiber channel
#define FORCED_MEDIA_10M_HALF   0x04    //
#define FORCED_MEDIA_10M_FULL   0x05    //

//
// Registers in the MII (offset unit is WORD)
//
#define MII_REG_BMCR        0x00        // physical address
#define MII_REG_BMSR        0x01        // 
#define MII_REG_PHYID1      0x02        // OUI
#define MII_REG_PHYID2      0x03        // OUI + Module ID + REV ID
#define MII_REG_ANAR        0x04        // 
#define MII_REG_ANLPAR      0x05        //
#define MII_REG_MODCFG		0x10
// NS, MYSON only
#define MII_REG_PCR         0x17        // 
// ESI only
#define MII_REG_PCSR        0x17        // 

//
// Bits in the BMCR register
//
#define BMCR_RESET          0x8000      // 
#define BMCR_LBK            0x4000      // 
#define BMCR_SPEED          0x2000      // 
#define BMCR_AUTO           0x1000      // 
#define BMCR_PD             0x0800      // 
#define BMCR_ISO            0x0400      // 
#define BMCR_REAUTO         0x0200      // 
#define BMCR_FDX            0x0100      // 

//
// Bits in the BMSR register
//
#define BMSR_AUTOCM         0x0020      // 
#define BMSR_LNK            0x0004      // 

//
// Bits in the ANAR register
//
#define ANAR_ASMDIR         0x0800      // Asymmetric PAUSE support
#define ANAR_PAUSE          0x0400      // Symmetric PAUSE Support
#define ANAR_T4             0x0200      // 
#define ANAR_TXFD           0x0100      // 
#define ANAR_TX             0x0080      // 
#define ANAR_10FD           0x0040      // 
#define ANAR_10             0x0020      // 

//
// Bits in the ANLPAR register
//
#define ANLPAR_ASMDIR       0x0800      // Asymmetric PAUSE support
#define ANLPAR_PAUSE        0x0400      // Symmetric PAUSE Support
#define ANLPAR_T4           0x0200      // 
#define ANLPAR_TXFD         0x0100      // 
#define ANLPAR_TX           0x0080      // 
#define ANLPAR_10FD         0x0040      // 
#define ANLPAR_10           0x0020      // 

//
// Bits in the RSR0 register
//
#define RSR0_BUFF           0x80        //
#define RSR0_FRAG			0x40
#define RSR0_SERR           0x40        //
#define RSR0_RUNT           0x20        //
#define RSR0_LONG           0x10        //
#define RSR0_FOV            0x08        //
#define RSR0_FAE            0x04        //
#define RSR0_CRC            0x02        //
#define RSR0_RERR           0x01        //

//
// Bits in the RSR1 register
//
#define RSR1_RXOK           0x80        // rx OK
#define RSR1_VIDHIT         0x40        // VID Hit
#define RSR1_MAR            0x20        // MAC accept multicast address packet
#define RSR1_BAR            0x10        // MAC accept broadcast address packet
#define RSR1_PHY            0x08        // MAC accept physical address packet
#define RSR1_CHN            0x04        // chain buffer, always = 1
#define RSR1_STP            0x02        // start of packet
#define RSR1_EDP            0x01        // end of packet

#define PQSTS_RXLERR		0x80
#define PQSTS_SNPTAG		0x40
#define PQSTS_IPOK          0x20        //IP Checkusm validatiaon ok
#define PQSTS_TUOK          0x10        //TCP/UDP Checkusm validatiaon ok
#define PQSTS_IPKT          0x08        //Received an IP packet
#define PQSTS_TCPKT         0x04        //Received a TCP packet
#define PQSTS_UDPKT         0x02        //Received a UDP packet
#define PQSTS_TAG           0x01        //Received a tagged packet

//
// Bits in the TSR0 register
//
#define TSR0_CDH            0x80        // AQE test fail (CD heartbeat)
#define TSR0_COLS           0x10        // experience collision in this transmit event
#define TSR0_NCR3           0x08        // collision retry counter[3]
#define TSR0_NCR2           0x04        // collision retry counter[2]
#define TSR0_NCR1           0x02        // collision retry counter[1]
#define TSR0_NCR0           0x01        // collision retry counter[0]

//
// Bits in the TSR1 register
//
#define TSR1_TERR           0x80        //
#define TSR1_JAB            0x40        // jabber condition occured
#define TSR1_SERR           0x20        //
#define TSR1_TBUFF          0x10        //
#define TSR1_UDF            0x08        //
#define TSR1_CRS            0x04        //
#define TSR1_OWC            0x02        // late collision
#define TSR1_ABT            0x01        //

//
// Bits in the TCR register
//
#define TCR_IC              0x80        // assert interrupt immediately 
                                        // while descriptor has been send complete
#define TCR_EDP             0x40        // end of packet
#define TCR_STP             0x20        // start of packet
#define TCR_TCPCK           0x10        // request TCP checksum calculation.
#define TCR_UDPCK           0x08        // request UDP checksum calculation.
#define TCR_IPCK            0x04        // request TCP checksum calculation.
#define TCR_TAG				0x02		// Do insert tag
#define TCR_CRC             0x01        // disable CRC generation

//
// Bits in the FlowCR1 register
//
#define FLOWCR1_XHITH1		0x80
#define FLOWCR1_XHITH0		0x40
#define FLOWCR1_XLTH1		0x20
#define FLOWCR1_XLTH0		0x10
#define FLOWCR1_XONEN		0x08
#define FLOWCR1_FDXTFCEN	0x04
#define FLOWCR1_FDXRFCEN	0x02
#define FLOWCR1_HDXFCEN		0x01

//
// revision id
//
#define REV_ID_VT86C100A_E  0x04
#define REV_ID_VT6102_A     0x40
#define REV_ID_VT6102_C     0x42
#define REV_ID_VT6105_A0	0x80
#define REV_ID_VT6105_B0	0x83
#define REV_ID_VT6105M_A0	0x90
#define REV_ID_VT6105M_B1	0x94

typedef struct __chip_info_tbl{
	CHIP_TYPE	chip_id;
	char*		name;	
	int			io_size;
	int			nTxQueue;
	U32			flags;
} CHIP_INFO, *PCHIP_INFO;

typedef struct __rdesc0 {
	BYTE	byRSR0;
	BYTE	byRSR1;
	U16		f15Length:15;
	U16		f1Owner:1;	
} __attribute__ ((__packed__))
RDESC0, *PRDESC0;

typedef struct __rdesc1 {
	volatile	U16		f15BufLen:15;
	volatile	U16		f1Reserved:1;
	volatile	U8		byPQSTS;
	volatile	U8		byIPKT;	
} __attribute__ ((__packed__))
RDESC1, *PRDESC1;

typedef struct {
	struct sk_buff*		skb;
	dma_addr_t			skb_dma;
	dma_addr_t			curr_desc;
} RHINE_RD_INFO,	*PRHINE_RD_INFO;

static inline PRHINE_RD_INFO alloc_rd_info(void) {
	PRHINE_RD_INFO	ptr;
	if ((ptr=kmalloc(sizeof(RHINE_RD_INFO),GFP_ATOMIC))==NULL)
		return NULL;
	else {
		memset(ptr,0,sizeof(RHINE_RD_INFO));
		return ptr;
	}
}

typedef enum {
	OWNED_BY_HOST=0,
	OWNED_BY_NIC=1
} RHINE_OWNER_TYPE, *PRHINE_OWNER_TYPE;

typedef struct __rx_desc {
	volatile RDESC0				rdesc0;
	volatile RDESC1				rdesc1;
	volatile U32				buff_addr;
	volatile U32				next_desc;
	struct __rx_desc			*next;		//4 bytes
	volatile PRHINE_RD_INFO		pInfo;		//4 bytes
	volatile U32				Reserved[2];//8 bytes
} __attribute__ ((__packed__))
RX_DESC, *PRX_DESC;

typedef struct __tx_desc0 {
	volatile	BYTE	byTSR0;
	volatile	BYTE	byTSR1;
	volatile	U16		f12VID:12;
	volatile	U16		f3Priority:3;
	volatile	U16		f1Owner:1;
} __attribute__ ((__packed__))
TDESC0, *PTDESC0;

typedef struct __tx_desc1 {
	volatile	U16			f15BufLen:15;
	volatile	U16			f1Chain:1;
	volatile	BYTE		byTCR;
	volatile	BYTE		byReserved;
} __attribute ((__packed__))
TDESC1, *PTDESC1;

typedef struct {
	struct sk_buff*		skb;
	PU8					buf;
	dma_addr_t			skb_dma;
	dma_addr_t			buf_dma;
	dma_addr_t			curr_desc;
} RHINE_TD_INFO,	*PRHINE_TD_INFO;

static inline PRHINE_TD_INFO alloc_td_info(void) {
	PRHINE_TD_INFO	ptr;
	if ((ptr=kmalloc(sizeof(RHINE_TD_INFO),GFP_ATOMIC))==NULL)
		return NULL;
	else {
		memset(ptr,0,sizeof(RHINE_TD_INFO));
		return ptr;
	}
}

typedef struct _tx_desc {
	volatile TDESC0				tdesc0;
	volatile TDESC1				tdesc1;
	volatile U32				buff_addr;
	volatile U32				next_desc;
	struct _tx_desc* 			next;		//4 bytes
	PRHINE_TD_INFO				pInfo;		//4 bytes
	volatile U32				Reserved[2];//8 bytes
} __attribute__ ((__packed__)) 
TX_DESC, *PTX_DESC;

typedef enum _speed_opt {
	SPD_DPX_AUTO=0,
	SPD_DPX_100_HALF=1,
	SPD_DPX_100_FULL=2,
	SPD_DPX_10_HALF=3,
	SPD_DPX_10_FULL=4
} SPD_DPX_OPT, *PSPD_DPX_OPT;

//flags for options
#define		RHINE_FLAGS_TAGGING			0x00000001UL
#define		RHINE_FLAGS_TX_CSUM			0x00000002UL
#define		RHINE_FLAGS_RX_CSUM			0x00000004UL
#define		RHINE_FLAGS_IP_ALIGN		0x00000008UL
#define		RHINE_FLAGS_VAL_PKT_LEN		0x00000010UL

//flags for driver status
#define		RHINE_FLAGS_OPENED			0x00010000UL
#define		RHINE_FLAGS_VMNS_CONNECTED	0x00020000UL
#define		RHINE_FLAGS_VMNS_COMMITTED	0x00040000UL

//flags for capbilities
#define		RHINE_FLAGS_TX_ALIGN		0x01000000UL
#define		RHINE_FLAGS_HAVE_CAM		0x02000000UL
#define 	RHINE_FLAGS_FLOW_CTRL		0x04000000UL

//flags for MII status
#define 	RHINE_LINK_FAIL				0x00000001UL
#define		RHINE_SPEED_10				0x00000002UL
#define		RHINE_SPEED_100				0x00000004UL
#define		RHINE_SPEED_1000			0x00000008UL
#define		RHINE_DUPLEX_FULL			0x00000010UL
#define		RHINE_AUTONEG_ENABLE		0x00000020UL
#define		RHINE_FORCED_BY_EEPROM		0x00000040UL


typedef struct __rhine_opt {
	int			nRxDescs;		//Number of RX descriptors
	int			nTxDescs;		//Number of TX descriptors
	SPD_DPX_OPT	spd_dpx;		//Media link mode
	int			vid;            //vlan id
	int			tx_thresh;      //Tx Fifo threshold
	int			rx_thresh;		//Rx fifo threshold
	int			DMA_length;     //DMA length
	int			flow_cntl;
	U32			flags;
} OPTIONS, *POPTIONS;

typedef struct __mac_regs {
    volatile U8    abyPAR[6];                  // 0x00
    volatile U8    byRCR;
    volatile U8    byTCR;

    volatile U8    byCR0;                      // 0x08
    volatile U8    byCR1;
    volatile U8    byTXQWAK;
    volatile U8    wReserve0B;

    volatile U8    byISR0;                     // 0x0C
    volatile U8    byISR1;
    volatile U8    byIMR0;
    volatile U8    byIMR1;

    volatile U8    abyMAR[8];                  // 0x10

    volatile U32   dwCurrRxDescAddr;           // 0x18
    volatile U32   adwCurrTxDescAddr[8];

    volatile U32   dwCurrentRDSE0;             // 0x3C
    volatile U32   dwCurrentRDSE1;
    volatile U32   dwCurrentRDSE2;
    volatile U32   dwCurrentRDSE3;
    volatile U32   dwCurrentTDSE0;             // 0x4C
    volatile U32   dwCurrentTDSE1;
    volatile U8    byGFTEST;
    volatile U8    byRFTCMD;
    volatile U8    byTFTCMD;
    volatile U8    byCFSTATUS;					// 0x57
    volatile U16   wBNRY;						// 0x58
    volatile U16   wCURR;						// 0x5a
    volatile U32   dwFIFODataPort;				// 0x5c
    volatile U32   dwCurrentTDSE2;				// 0x60
    volatile U32   dwCurrentTDSE3;				// 0x64
	volatile U32   dwReserverd_68;

    volatile U8    byMIICFG;					// 0x6C
    volatile U8    byMIISR;
    volatile U8    byBCR0;
    volatile U8    byBCR1;
    volatile U8    byMIICR;
    volatile U8    byMIIAD;
    volatile U16   wMIIDATA;

    volatile U8    byEECSR;						// 0x74
    volatile U8    byTEST;
    volatile U8    byDEBUG0;
    volatile U8    byDEBUG1;

    volatile U8    byCFGA;                     // 0x78
    volatile U8    byCFGB;
    volatile U8    byCFGC;
    volatile U8    byCFGD;

    // tally counter will be reset when read each time
    volatile U16   wReserved_7C;               // 0x7C
    volatile U16   wReserved_7E;

    // for VT6102
    volatile U8    byMISCCR0;                  // 0x80
    volatile U8    byMISCCR1;
    volatile U8    byPMCPORT;
    volatile U8    bySTICKHW;
    volatile U8    byMISR;
    volatile U8    byReserved_85;
    volatile U8    byMIMR;
    volatile U8    byReserved_87;
    volatile U32   dwCAMMASK;                  //
    volatile U16   wBPMA;
    volatile U8    byRamBist;
    volatile U8    byBPMD;
    volatile U8    byBPCMD;
    volatile U8    byBPINDATA;
    volatile U8    byCAMCR;
    volatile U8    byCAMADD;
    volatile U8    byMIBCR;
    volatile U8    byPHY_ANR;
    volatile U8    byMIBTestPort;
    volatile U8    byMIBContent;
    volatile U8    byFlowCR0;
    volatile U8    byFlowCR1;
    volatile U16   wTxPasueTimer;
    volatile U16   wSOFTTIMER[2];

    volatile U8    byWOLCRSet;                 // 0xA0
    volatile U8    byPWCFGSet;
    volatile U8    byReserved_A2;
    volatile U8    byWOLCGSet;
    volatile U8    byWOLCRClr;
    volatile U8    byPWCFGClr;
    volatile U8    byReserved_A6;
    volatile U8    byWOLCGClr;
    volatile U8    byPWRCSRSet;
    volatile U8    byReserved_A9[3];
    volatile U8    byPWRCSRClr;
    volatile U8    wReserved_AD[3];
    volatile U32   dwPatternCRC[4];
    volatile U32   adwByteMask[4][4];          // 0xC0
} 
__attribute__ ((__packed__))
MAC_REGS,	*PMAC_REGS;


typedef struct __rhine_info {
	struct __rhine_info*		next;
	struct __rhine_info*		prev;

	struct pci_dev*				pcid;
	struct net_device*			dev;
	struct net_device_stats		stats;
	
	dma_addr_t					pool_dma;
	dma_addr_t					rd_pool_dma;
	dma_addr_t					td_pool_dma[TX_QUEUE_NO];	

	dma_addr_t					tx_bufs_dma;
	PU8							tx_bufs;
			
	CHIP_TYPE					chip_id;
	
	PMAC_REGS					pMacRegs;
	U32							memaddr;
	U32							ioaddr;
	U32							io_size;
	
	U8							byRevId;
	U16							SubSystemID;
	U16							SubVendorID;

#define	AVAIL_TD(p,q)	((p)->sOpts.nTxDescs-((p)->iTDUsed[(q)]))

	int							nTxQueues;
	volatile int				iTDUsed[TX_QUEUE_NO];
	volatile PTX_DESC			apCurrTD[TX_QUEUE_NO];
	volatile PTX_DESC			apTailTD[TX_QUEUE_NO];
	PTX_DESC					apTDRings[TX_QUEUE_NO];

	PRX_DESC					pCurrRD;
	PRX_DESC					aRDRing;

	OPTIONS						sOpts;
		
	U32							IntMask;	

	U32							flags; 
	
	int							rx_buf_sz;
	U32							mii_status;
	int							multicast_limit;
	spinlock_t					lock;
	
#ifdef CONFIG_PROC_FS
	struct proc_dir_entry*		pProcDir;
#endif
	
#ifdef	VMNS
	PVMNS_DRV_PRIVATE			vmns_priv;		
#endif
	
} RHINE_INFO, *PRHINE_INFO;


static inline void rhine_disable_int(PRHINE_INFO pInfo) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	writew(0, &pMacRegs->byIMR0);
	if (pInfo->byRevId>REV_ID_VT6102_A)
		writeb(0,	&pMacRegs->byMIMR);
}

static inline void rhine_enable_int(PRHINE_INFO pInfo) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	writew(pInfo->IntMask, &pMacRegs->byIMR0);

	if (pInfo->byRevId>REV_ID_VT6102_A)
		writeb(pInfo->IntMask>>16, &pMacRegs->byMIMR);
}

static inline void enable_mmio(PRHINE_INFO pInfo) {
    int n;
    if (pInfo->chip_id == VT86C100A) {
		n = inb(pInfo->ioaddr + MAC_REG_CFGA) | 0x20;
		outb(n, pInfo->ioaddr + MAC_REG_CFGA);
	} else {
		n = inb(pInfo->ioaddr + MAC_REG_CFGD) | CFGD_GPIOEN;
		outb(n, pInfo->ioaddr + MAC_REG_CFGD);
    }                                                        
}

static inline void	reload_eeprom(PRHINE_INFO pInfo) {
	int i;
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	
	writeb(readb(&pMacRegs->byEECSR)|EECSR_AUTOLD, &pMacRegs->byEECSR);
	
	    /* Typically 2 cycles to reload. */
    for (i = 0; i < 150; i++)
    	if (!(readb(&pMacRegs->byEECSR) & EECSR_AUTOLD))
		    break;
	
	enable_mmio(pInfo);
}




static inline void enable_flow_control(PRHINE_INFO pInfo) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;

	/* Set {XHITH1, XHITH0, XLTH1, XLTH0} in FlowCR1 to {1, 0, 1, 1} 
		depend on RD=64, and Turn on XNOEN in FlowCR1*/		
	BYTE_REG_BITS_SET(
		(FLOWCR1_XONEN|FLOWCR1_XHITH1|FLOWCR1_XLTH1|FLOWCR1_XLTH0), 
		0xFF,&pMacRegs->byFlowCR1);
		
    /* Set TxPauseTimer to 0xFFFF */    
	writew(0xFFFF, &pMacRegs->wTxPasueTimer);
	
    /* Initialize RBRDU to Rx buffer count.*/	
	writeb(pInfo->sOpts.nRxDescs,&pMacRegs->byFlowCR0);
}
                 
 #define MII_REG_BITS_ON(x,i,p) do {\
	U16	w;\
	rhine_mii_read((p),(i),&(w));\
	(w)|=(x);\
	rhine_mii_write((p),(i),(w));\
} while (0)

#define MII_REG_BITS_OFF(x,i,p) do {\
	U16	w;\
	rhine_mii_read((p),(i),&(w));\
	(w)&=(~(x));\
	rhine_mii_write((p),(i),(w));\
} while (0)

#define MII_REG_BITS_IS_ON(x,i,p) ({\
	U16	w;\
	rhine_mii_read((p),(i),&(w));\
	((BOOL) ((w) & (x)));})

static inline void rhine_set_duplex(PRHINE_INFO pInfo, BOOL bFlag) {
	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
	
	if (bFlag) 
		BYTE_REG_BITS_ON(CR1_FDX, &pMacRegs->byCR1);
	else 
		BYTE_REG_BITS_OFF(CR1_FDX, &pMacRegs->byCR1);	
			
	if (pInfo->byRevId>=REV_ID_VT6102_A)
		writeb(WOLCFG_SFDX, &pMacRegs->byWOLCGSet);
	else	
		writeb(WOLCFG_SFDX, &pMacRegs->byWOLCGClr);
		
}

static inline void
rhine_set_tx_thresh(PRHINE_INFO pInfo, int tx_thresh) {
	PMAC_REGS pMacRegs=pInfo->pMacRegs;
	BYTE_REG_BITS_SET(tx_thresh <<3, 
		(BCR1_CTSF|BCR1_CTFT1|BCR1_CTFT0), &pMacRegs->byBCR1);
			
	BYTE_REG_BITS_SET(tx_thresh <<5, 
		(TCR_RTSF|TCR_RTFT1|TCR_RTFT0), &pMacRegs->byTCR);
}

static inline void
rhine_set_rx_thresh(PRHINE_INFO pInfo, int rx_thresh) {
	PMAC_REGS pMacRegs=pInfo->pMacRegs;
	BYTE_REG_BITS_SET(rx_thresh <<3, 
		(BCR0_CRFT2|BCR0_CRFT1|BCR0_CRFT0), &pMacRegs->byBCR0);
		
	BYTE_REG_BITS_SET(rx_thresh <<5, 
		(RCR_RRFT2|RCR_RRFT1|RCR_RRFT0), &pMacRegs->byRCR);
}

static inline void
rhine_set_DMA_length(PRHINE_INFO pInfo, int DMA_length) {
	PMAC_REGS pMacRegs=pInfo->pMacRegs;
	BYTE_REG_BITS_SET(DMA_length,
		(BCR0_DMAL2|BCR0_DMAL1|BCR0_DMAL0),&pMacRegs->byBCR0);
}

static inline void
rhine_ClearISR(PMAC_REGS	pMacRegs) {
	writew(0xFFFF,&pMacRegs->byISR0);
	writeb(0xDF,&pMacRegs->byMISR);	
}

#define WAIT_MAC_TX_OFF(pMacRegs)	do {udelay(5);} while (BYTE_REG_BITS_IS_ON(CR0_TXON,&(pMacRegs)->byCR0))

inline static U32 rhine_ReadISR(PMAC_REGS pMacRegs, U8 byRevId) {
	U32	status=0;
	status=readw(&pMacRegs->byISR0);
	if (byRevId>REV_ID_VT6102_A) 
		status|=(readb(&pMacRegs->byMISR) <<16);
	return status;
}

inline static void rhine_WriteISR(int status,PMAC_REGS pMacRegs, U8 byRevId)	{
	writew((status & 0xFFFF),&pMacRegs->byISR0);
	if (byRevId>REV_ID_VT6102_A) 
		writeb((status>>16) & 0xFF,&pMacRegs->byMISR);
}

U32 rhine_check_media_mode(PRHINE_INFO pInfo);
void rhine_get_cam_mask(PRHINE_INFO pInfo, PU32 pMask, RHINE_CAM_TYPE cam_type);
void rhine_set_cam_mask(PRHINE_INFO pInfo, U32 Mask, RHINE_CAM_TYPE cam_type);
void rhine_set_cam(PRHINE_INFO pInfo, int idx, PU8 addr, RHINE_CAM_TYPE cam_type);
void rhine_get_cam(PRHINE_INFO pInfo, int idx, PU8 addr, RHINE_CAM_TYPE cam_type);
#endif
