#include "cmaudio.h"


#define SIS7018_PCI_REVISION_ID        0x08
#define SIS7018_PCI_LEGACY_IO          0x44

#define SIS7018_AUDIO_MPUR0            0x20
#define SIS7018_AUDIO_MPUR1            0x21
#define SIS7018_AUDIO_MPUR2            0x22
#define SIS7018_AUDIO_MPUR3            0x23

#define SIS7018_AUDIO_ACWR             0x40
#define SIS7018_AUDIO_ACRD             0x44
#define SIS7018_AUDIO_SCTRL            0x48

#define SIS7018_AUDIO_AIN              0x98
#define SIS7018_AUDIO_GC               0xa0
#define SIS7018_AUDIO_AINTEN           0xa4
#define SIS7018_AUDIO_WAVEVOL          0xa8

#define SIS7018_AUDIO_MISCINT          0xb0
#define SIS7018_AUDIO_START_B          0xb4
#define SIS7018_AUDIO_STOP_B           0xb8

#define SIS7018_AUDIO_CSO_ALPHA_FMS    0xe0
#define SIS7018_AUDIO_LBA              0xe4
#define SIS7018_AUDIO_ESO_DELTA        0xe8
#define SIS7018_AUDIO_ATTRIBUTE_B      0xec
#define SIS7018_AUDIO_GVSEL_PAN_VOL    0xf0


void RequestRegion_SiS_7018(struct cmedia_card *card)
{
    request_region(card->iobase, 4096, card_names[card->driver_data]);
    request_region(card->opbase, 256, card_names[card->driver_data]);
}

void ReleaseRegion_SiS_7018(struct cmedia_card *card)
{
    release_region(card->iobase, 4096);
    release_region(card->opbase, 256);
}

int Initial_SiS_7018(struct cmedia_card *card)
{
    u16 eid;
    u32 dwValue, dwTemp;
    
    // 1.Set PCI_CFG+44h
    pci_read_config_dword(card->pci_dev, SIS7018_PCI_LEGACY_IO, &dwValue);
    dwValue = (dwValue&0xff00ff00)|0x000800aa;
    pci_write_config_dword(card->pci_dev, SIS7018_PCI_LEGACY_IO, dwValue);

    // 2. Set AudioBase+A0h
    dwTemp = inl(card->opbase+SIS7018_AUDIO_GC);
    dwTemp = (dwTemp&0xffff00ff)|0x0100;
    outl(dwTemp, card->opbase+SIS7018_AUDIO_GC);
        
    // 3. Disable INT which were used by 32 channels
    outl(0, card->opbase+SIS7018_AUDIO_AINTEN);

    // 4. Clear INT 0f 32 channels
    outl(0xffffffff, card->opbase+SIS7018_AUDIO_AIN);

    // 5. Connect MPUR0 to MIDI-OUT FIFO
    outb(0x10, card->opbase+SIS7018_AUDIO_MPUR2);

    // 6. Codec Cold Reset
    dwTemp = inl(card->opbase+SIS7018_AUDIO_SCTRL);
    outl(dwTemp|0x00000002, card->opbase+SIS7018_AUDIO_SCTRL);
    udelay(10);
    outl(dwTemp&0xfffffffd, card->opbase+SIS7018_AUDIO_SCTRL);
    udelay(10);

    /* Don't attempt to get eid until powerup is complete */
    eid = card->ReadRegisterWord(card, AC97_EXTENDED_ID);
		
    if(eid==0xFFFFFF)
    {
        printk(KERN_WARNING "CMedia Audio: no codec attached ?\n");
        return 0;
    }
		
    card->ac97_features = eid;
				
    /* Now check the codec for useful features to make up for
       the dumbness of the 810 hardware engine */

    if(eid&0x0001)
    {
        printk(KERN_WARNING "CMedia Audio: Codec refused to allow VRA, using 48Khz only.\n");
        card->ac97_features&=~1;
    }

    card->ac97_vender_id1 = card->ReadRegisterWord(card, AC97REG_VENDOR_ID1);   		
    printk(KERN_INFO "CMedia Audio: AC97REG_VENDOR_ID1=%x\n", card->ac97_vender_id1);
    card->ac97_vender_id2 = card->ReadRegisterWord(card, AC97REG_VENDOR_ID2);   		
    printk(KERN_INFO "CMedia Audio: AC97REG_VENDOR_ID2=%x\n", card->ac97_vender_id2);

    if (card->ac97_vender_id1 != 0x434d)
    {
        printk(KERN_ERR "CMedia Audio: CMedia codec not found.\n");
        return 0;
    }

    return 1;
}

u16 ReadRegisterWord_SiS_7018(struct cmedia_card *card, u8 reg)
{
    unsigned short wTemp, ReadIndex, CodecIOAddr;
    int Count;

    CodecIOAddr = card->opbase;
    CodecIOAddr += SIS7018_AUDIO_ACRD;

    ReadIndex = 0x8000|reg;
    
    // Wait for CodecIOAddr[bit15] until it's 0 (Ready to read).
    wTemp = inw(CodecIOAddr);
    Count=0;
    while (wTemp&0x8000 && Count < 100) { wTemp = inw(CodecIOAddr); Count++; udelay(10); }

    // Output the WORD "Index" to CodecIOAddr
    outw(ReadIndex, CodecIOAddr);

    // Wait for the bit in Audiobase+44h[bit15] until it's 1. (returned data is available)
    wTemp = inw(CodecIOAddr);
    Count=0;
    while (wTemp&0x8000 && Count < 100) { wTemp = inw(CodecIOAddr); Count++; udelay(10); }
    
    // input a WORD from AudioBase+46h, and return the WORD as result.
    return inw(CodecIOAddr+0x02);
}

void WriteRegisterWord_SiS_7018(struct cmedia_card *card, u8 reg, u16 data)
{
    unsigned short CodecIOAddr;
    unsigned short wTemp;
    int Count;
    u8 RevisionID=0;
    unsigned short WriteIndex=0;

    // Read PCI Configuration Byte - Read Revision ID
    pci_read_config_byte(card->pci_dev, SIS7018_PCI_REVISION_ID, &RevisionID);

    WriteIndex = 0x8100|reg;
    if (RevisionID <= 1) WriteIndex = 0x8000|reg;

    // Wait for [bit15] until it's 0 (Ready to read).
    CodecIOAddr = card->opbase+SIS7018_AUDIO_ACWR;
    wTemp = inw(CodecIOAddr);
    Count=0;
    while (wTemp&0x8000 && Count < 100) { wTemp = inw(CodecIOAddr); Count++; udelay(10); }

    // Output the BYTE in "Index" bit0-7 to CodecIOAddr
    wTemp = (wTemp&0xff00)|(WriteIndex&0x00ff);
    outw(wTemp, CodecIOAddr);

    // Output the WORD "Data" to CodecIOAddr+0x02
    outw(data, CodecIOAddr+0x02);
    
    // Output the BYTE in "Index" bit8-15 to CodecIOAddr
    wTemp = (wTemp&0x00ff)|(WriteIndex&0xff00);
    outw(wTemp, CodecIOAddr);
}

void StartADC_SiS_7018(struct cmedia_card *card)
{
    unsigned long flags, dwTemp;

    if (!(card->dma_state & ADC_RUNNING)) 
    {
        spin_lock_irqsave(&card->lock, flags);
        card->dma_state |= ADC_RUNNING;
        dwTemp = inl(card->opbase+SIS7018_AUDIO_START_B);
        dwTemp |= 0x00000008;
        outl(dwTemp, card->opbase+SIS7018_AUDIO_START_B);
        spin_unlock_irqrestore(&card->lock, flags);
    }
}

/* Stop recording (lock held) */
void StopADC_SiS_7018(struct cmedia_card *card)
{
    unsigned long dwTemp = 0x00000008;
    card->dma_state &= ~ADC_RUNNING;
    outl(dwTemp, card->opbase+SIS7018_AUDIO_STOP_B);
    memset(card->pcmin_rawbuf, 0, PCMIN_DMABUF_SIZE);
}

void StartDAC_SiS_7018(struct cmedia_card *card)
{
    u16 val;
    unsigned long flags, dwTemp;

    if (!(card->dma_state & DAC_RUNNING)) 
    {
        spin_lock_irqsave(&card->lock, flags);
        card->dma_state |= DAC_RUNNING;
        dwTemp = inl(card->opbase+SIS7018_AUDIO_START_B);
        dwTemp |= 0x00000001;
        if (output_channels>=4) dwTemp |= 0x00000002;
        if (output_channels>=6) dwTemp |= 0x00000004;
        outl(dwTemp, card->opbase+SIS7018_AUDIO_START_B);
        spin_unlock_irqrestore(&card->lock, flags);
    }

    if (card->ac97_vender_id2==0x4961 && card->bMonitorSPDIF)
    {
        val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
        card->WriteRegisterWord(card , AC97REG_SPDIF_FUNCTION, val&(~0x08));
    }
}

/* stop playback (lock held) */
static inline void __StopDAC_SiS_7018(struct cmedia_card *card)
{
    u16 val;
    unsigned long dwTemp = 0x00000001; // 2 channel
    if (output_channels>=4) dwTemp |= 0x00000002;
    if (output_channels>=6) dwTemp |= 0x00000004;
    card->dma_state &= ~DAC_RUNNING;
    outl(dwTemp, card->opbase+SIS7018_AUDIO_STOP_B);
    memset(card->pcmout_rawbuf, 0, PCMOUT_DMABUF_SIZE);
    memset(card->dmaout_rawbuf, 0, PCMOUT_DMABUF_SIZE);

    if (card->ac97_vender_id2==0x4961 && card->bMonitorSPDIF)
    {
        val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
        card->WriteRegisterWord(card , AC97REG_SPDIF_FUNCTION, val|0x08);
    }
}

/* update buffer manangement pointers */
void UpdatePtr_SiS_7018(struct cmedia_card *card)
{
    int port; 
    unsigned long dwTemp;
    if(!card->dma_state) return;

    port = card->opbase;
    if (card->dma_state & ADC_RUNNING) 
    {
        dwTemp = inl(port+SIS7018_AUDIO_GC);
        dwTemp = (dwTemp&0xffffffe0)|35;      // Recording
        outl(dwTemp, port+SIS7018_AUDIO_GC);

        card->pcmin_hwptr = inl(port+SIS7018_AUDIO_CSO_ALPHA_FMS);
        card->pcmin_hwptr = (card->pcmin_hwptr&0xffff0000)>>14;
        card->pcmin_hwptr -= card->pcmin_hwptr%4;
    }

    if (card->dma_state & DAC_RUNNING) 
    {
        dwTemp = inl(port+SIS7018_AUDIO_GC);
        dwTemp = (dwTemp&0xffffffe0)|32;      // Front Left/Right
        outl(dwTemp, port+SIS7018_AUDIO_GC);

        card->pcmout_hwptr = inl(port+SIS7018_AUDIO_CSO_ALPHA_FMS);
        card->pcmout_hwptr = (card->pcmout_hwptr&0xffff0000)>>14;
        card->pcmout_hwptr -= card->pcmout_hwptr%4;

        card->pcmout_hwptr = card->pcmout_hwptr*3;
    }
}

int DrainDAC_SiS_7018(struct cmedia_card *card)
{
    __StopDAC_SiS_7018(card);

    return 0;
}

int ProgramDMABuffer_SiS_7018(struct cmedia_card *card)
{
    int port,i; 
    unsigned long flags=0, dwTemp=0, ESO=0, DELTA=0;

    if(card->dma_state) return -1;

    port = card->opbase;
    spin_lock_irqsave(&card->lock, flags);
    card->pcmout_count = card->pcmin_count = 0;
    card->pcmout_swptr = card->pcmout_hwptr = card->pcmin_swptr = card->pcmin_hwptr = 0;

    memset(card->pcmout_rawbuf, 0, PCMOUT_DMABUF_SIZE);
    memset(card->dmaout_rawbuf, 0, PCMOUT_DMABUF_SIZE);
    memset(card->pcmin_rawbuf, 0, PCMIN_DMABUF_SIZE);

    // --------------------------------------------------------
    // Playback
    for(i=0; i<(output_channels/2); i++)
    {
        dwTemp = inl(port+SIS7018_AUDIO_GC);
        dwTemp = (dwTemp&0xffffffe0)|(32+i);      // Front Left/Right
        dwTemp |= 0x00010000;
        outl(dwTemp, port+SIS7018_AUDIO_GC);

        // Set LBA attribute (E4h) of channel to point the address of played buffer
        dwTemp = inl(port+SIS7018_AUDIO_LBA);
        dwTemp = (dwTemp&0x80000000)|(virt_to_bus(card->dmaout_rawbuf+(PCMOUT_DMABUF_SIZE*i/3)));
        outl(dwTemp, port+SIS7018_AUDIO_LBA);

        // Set DELTA attribute (E8h[bit0-15]) of channel tp ((Sample Rate<<12)/48000)
        // Set ESO attribute (E8h[bit16-31]) of channel to the length of played buffer in samples
        ESO = (PCMOUT_DMABUF_SIZE>>2)/3-1;
        DELTA = 4096;
        dwTemp = (ESO<<16)|DELTA;
        outl(dwTemp, port+SIS7018_AUDIO_ESO_DELTA);

        // Set Current Sample Offset
        outl(0, port+SIS7018_AUDIO_CSO_ALPHA_FMS);
    
        // Set CTRL attribute (F0h[bit13-15]) of channel by the format of wave (stereo/mono and 8/16 bit)
        // Set the channel into loop mode (f0h[bit12]=1)
        // Set PAN (F0h[bit16-23]) and VOL (F0h[bit24-30]) if necessary
        dwTemp = 0x8000f000; // loop mode, WAVEVOL, 16 bit, stereo, signed
        outl(dwTemp, port+SIS7018_AUDIO_GVSEL_PAN_VOL);
    
        // Set Playback channel attribute
        if (i==0) outl(0x48000000, port+SIS7018_AUDIO_ATTRIBUTE_B);
        else if (i==1) outl(0x58000000, port+SIS7018_AUDIO_ATTRIBUTE_B);
        else if (i==2) outl(0x54000000, port+SIS7018_AUDIO_ATTRIBUTE_B);
    }
    
    // Disable all interrupt of audio controller
    dwTemp = inl(port+SIS7018_AUDIO_MISCINT);
    dwTemp &= 0xff7bffff;
    outl(dwTemp, port+SIS7018_AUDIO_MISCINT);

    // Disable MIDLP_IE, ENPLP_IE
    dwTemp = inl(port+SIS7018_AUDIO_GC);
    dwTemp &= 0xffffcfff; dwTemp |= 0x00010000;
    outl(dwTemp, port+SIS7018_AUDIO_GC);

    // Set Global Volume
    outl(0, port+SIS7018_AUDIO_WAVEVOL);

    // Enable PCM OUT L/R
    dwTemp = inl(port+SIS7018_AUDIO_SCTRL);
    dwTemp &= ~0x000f0000;
    if (output_channels>=2) dwTemp |= 0x00010000;
    if (output_channels>=4) dwTemp |= 0x00020000;
    if (output_channels>=6) dwTemp |= 0x000c0000;
    outl(dwTemp, port+SIS7018_AUDIO_SCTRL);

    card->pcmout_count = 0;

    // --------------------------------------------------------
    // Recording
    dwTemp = inl(port+SIS7018_AUDIO_GC);
    dwTemp = (dwTemp&0xffffffe0)|35;
    outl(dwTemp, port+SIS7018_AUDIO_GC);

    // Set LBA attribute (E4h) of channel to point the address of played buffer
    dwTemp = inl(port+SIS7018_AUDIO_LBA);
    dwTemp = (dwTemp&0x80000000)|(virt_to_bus(card->pcmin_rawbuf));
    outl(dwTemp, port+SIS7018_AUDIO_LBA);

    // Set DELTA attribute (E8h[bit0-15]) of channel tp ((Sample Rate<<12)/48000)
    // Set ESO attribute (E8h[bit16-31]) of channel to the length of played buffer in samples
    ESO = (PCMIN_DMABUF_SIZE>>2)-1;
    DELTA = 4096;
    dwTemp = (ESO<<16)|DELTA;
    outl(dwTemp, port+SIS7018_AUDIO_ESO_DELTA);

    // Set Current Sample Offset
    outl(0, port+SIS7018_AUDIO_CSO_ALPHA_FMS);
    
    // Set CTRL attribute (F0h[bit13-15]) of channel by the format of wave (stereo/mono and 8/16 bit)
    // Set the channel into loop mode (f0h[bit12]=1)
    // Set PAN (F0h[bit16-23]) and VOL (F0h[bit24-30]) if necessary
    dwTemp = 0x8000f000; // loop mode, WAVEVOL, 16 bit, stereo, signed
    outl(dwTemp, port+SIS7018_AUDIO_GVSEL_PAN_VOL);
    
    // Set Recording channel attribute
    outl(0x8a000000, port+SIS7018_AUDIO_ATTRIBUTE_B);
    
    card->pcmin_count = 0;
    spin_unlock_irqrestore(&card->lock, flags);

    return 0;
}

/*
int PrintCount=0;
unsigned long OutputPosition=0;
unsigned long OutPos=0;
short OutValue[48] = { 
    0x0000, 0x10b5, 0x2120, 0x30fb, 0x4000, 0x4deb, 0x5a82, 0x658c, 0x6ed9, 0x7641, 0x7ba3, 0x7ee7,
    0x7fff, 0x7ee7, 0x7ba3, 0x7641, 0x6ed9, 0x658c, 0x5a82, 0x4deb, 0x4000, 0x30fb, 0x2120, 0x10b5,
    0x0000, 0xef4b, 0xdee0, 0xcf05, 0xc000, 0xb215, 0xa57e, 0x9a74, 0x9127, 0x89bf, 0x845d, 0x8119,
    0x8000, 0x8119, 0x845d, 0x89bf, 0x9127, 0x9a74, 0xa57e, 0xb215, 0xc000, 0xcf05, 0xdee0, 0xef4b };
*/

void ConvertPCMOUT_SiS_7018(struct cmedia_card *card)
{
    short *ptrSource, *ptrTarget;
    unsigned long i, len1, len2;
    len1=len2=0;

    if (card->pcmout_swptr < card->pcmout_hwptr)
    {
        len1 = ((card->pcmout_hwptr-card->pcmout_swptr)>>1)/6;
    }
    else
    {
        len1 = ((PCMOUT_DMABUF_SIZE-card->pcmout_swptr)>>1)/6;
        len2 = ((card->pcmout_hwptr)>>1)/6;
    }

    if (output_channels>=2) // Front Left/Right
    {
        ptrSource=(short *)(card->pcmout_rawbuf+card->pcmout_swptr);
        ptrTarget=(short *)(card->dmaout_rawbuf+card->pcmout_swptr/3);
        if (card->XearMode)
        {
            for(i=0; i<len1; i++)
            {
                ptrSource++;ptrSource++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++;
            }
        }
        else
        {
            for(i=0; i<len1; i++)
            {
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++;ptrSource++;ptrSource++;
            }
        }

        ptrSource=(short *)(card->pcmout_rawbuf);
        ptrTarget=(short *)(card->dmaout_rawbuf);
        if (card->XearMode)
        {
            for(i=0; i<len2; i++)
            {
                ptrSource++;ptrSource++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++;
            }
        }
        else
        {
            for(i=0; i<len2; i++)
            {
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++;ptrSource++;ptrSource++;
            }
        }
    }

    if (output_channels>=4) // Rear Left/Right
    {
        ptrSource=(short *)(card->pcmout_rawbuf+card->pcmout_swptr);
        ptrTarget=(short *)(card->dmaout_rawbuf+card->pcmout_swptr/3+(PCMOUT_DMABUF_SIZE*1/3));
        if (card->XearMode)
        {
            for(i=0; i<len1; i++)
            {
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++; ptrSource++;ptrSource++;
            }
        }
        else
        {
            for(i=0; i<len1; i++)
            {
                ptrSource++;ptrSource++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++;
            }
        }

        ptrSource=(short *)(card->pcmout_rawbuf);
        ptrTarget=(short *)(card->dmaout_rawbuf+(PCMOUT_DMABUF_SIZE*1/3));
        if (card->XearMode)
        {
            for(i=0; i<len2; i++)
            {
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++; ptrSource++;ptrSource++;
            }
        }
        else
        {
            for(i=0; i<len2; i++)
            {
                ptrSource++;ptrSource++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++;
            }
        }
    }

    if (output_channels>=6) // Center/Bass
    {
        ptrSource=(short *)(card->pcmout_rawbuf+card->pcmout_swptr);
        ptrTarget=(short *)(card->dmaout_rawbuf+card->pcmout_swptr/3+(PCMOUT_DMABUF_SIZE*2/3));
        for(i=0; i<len1; i++)
        {
            ptrSource++;ptrSource++;ptrSource++;ptrSource++;
            *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
            *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
        }

        ptrSource=(short *)(card->pcmout_rawbuf);
        ptrTarget=(short *)(card->dmaout_rawbuf+(PCMOUT_DMABUF_SIZE*2/3));
        for(i=0; i<len2; i++)
        {
            ptrSource++;ptrSource++;ptrSource++;ptrSource++;
            *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
            *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
        }
    }
}
