#include "cmaudio.h"

#define DRIVER_VERSION "0.36"

u16 output_channels;

char * chip_names[] = 
{
    "CMI9738",
    "CMI9739"
};

char * card_names[] = 
{
    "Intel ICH4",
    "VIA VT8233",
    "SiS 7012",
    "SiS 7018",
    "ALI 5451",
    "nVIDIA nForce",
    "Intel ICH2",
    "VIA VT82686",
    "Intel ICH5",
    "AMD 8111"
};

static struct pci_device_id cmedia_pci_tbl [] __initdata = {
    {PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH4, PCI_ANY_ID, PCI_ANY_ID, 0, 0, INTELICH4},
    {PCI_VENDOR_ID_VIA, PCI_DEVICE_ID_VIA_8233_5,   PCI_ANY_ID, PCI_ANY_ID, 0, 0, VIA8233},
    {PCI_VENDOR_ID_SIS, PCI_DEVICE_ID_SIS_7012,     PCI_ANY_ID, PCI_ANY_ID, 0, 0, SIS7012},
    {PCI_VENDOR_ID_SIS, PCI_DEVICE_ID_SIS_7018,     PCI_ANY_ID, PCI_ANY_ID, 0, 0, SIS7018},
    {PCI_VENDOR_ID_ALI, PCI_DEVICE_ID_ALI_5451,     PCI_ANY_ID, PCI_ANY_ID, 0, 0, ALI5451},
    {PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_NFORCE_0,     PCI_ANY_ID, PCI_ANY_ID, 0, 0, NVIDIA_NFORCE},
    {PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_NVIDIA_NFORCE_1,     PCI_ANY_ID, PCI_ANY_ID, 0, 0, NVIDIA_NFORCE},
    {PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH2, PCI_ANY_ID, PCI_ANY_ID, 0, 0, INTELICH2},
    {PCI_VENDOR_ID_VIA, PCI_DEVICE_ID_VIA_82686,   PCI_ANY_ID, PCI_ANY_ID, 0, 0, VIA82686},
    {PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH5, PCI_ANY_ID, PCI_ANY_ID, 0, 0, INTELICH5},
    {PCI_VENDOR_ID_AMD, PCI_DEVICE_ID_AMD_8111, PCI_ANY_ID, PCI_ANY_ID, 0, 0, AMD8111},
    {0,}
};

MODULE_DEVICE_TABLE (pci, cmedia_pci_tbl);

static const unsigned int MulRatio[21][3] = 
{
    {17, 17, 17}, {5, 6, 9}, {4, 5, 8}, {3, 6, 7}, {3, 4, 7}, {2, 17, 17}, {2, 5, 6}, {2, 4, 5}, 
    {2, 3, 6}, {2, 3, 4}, {1, 17, 17}, {1, 5, 6}, {1, 4, 5}, {1, 3, 6}, {1, 3, 4}, {1, 2, 17},
    {1, 2, 5}, {1, 2, 4}, {1, 2, 3}, {1, 2, 3}, {0, 17, 17}
};

int PrintCount=0;
unsigned long OutputPosition=0;
unsigned long OutPos=0;
short OutValue[48] = { 
    0x0000, 0x10b5, 0x2120, 0x30fb, 0x4000, 0x4deb, 0x5a82, 0x658c, 0x6ed9, 0x7641, 0x7ba3, 0x7ee7,
    0x7fff, 0x7ee7, 0x7ba3, 0x7641, 0x6ed9, 0x658c, 0x5a82, 0x4deb, 0x4000, 0x30fb, 0x2120, 0x10b5,
    0x0000, 0xef4b, 0xdee0, 0xcf05, 0xc000, 0xb215, 0xa57e, 0x9a74, 0x9127, 0x89bf, 0x845d, 0x8119,
    0x8000, 0x8119, 0x845d, 0x89bf, 0x9127, 0x9a74, 0xa57e, 0xb215, 0xc000, 0xcf05, 0xdee0, 0xef4b };

static struct cmedia_card *devs = NULL;
struct timer_list cmedia_timer;

/*---------------------------------------------------------------------------------------*/
static inline long GetNewSampleLen(INDEX doppler_rate, INDEX index, long OldLen)
{
    long dwTemp = OldLen;
    if (OldLen <= 0) return 0;
    return ((dwTemp<<16)-index.s.fract)/doppler_rate.dw;
}

//int pcount=0;

short *RecordingSamplingRateConvert
(
    short   *pSource,        // Source Data starting pointer
    short   *MaxSource,      // Source Data maximun pointer
    short   *pTarget,        // Target Data starting pointer
    long     MaxCount,       // Target buffer samples count
    INDEX    doppler_rate,   // Sampling rate ratio
    INDEX   *index,          // Current sampling rate ratio
    unsigned nChannels,      // Channels count
    long     *Count          // Process sample count
)
{
    long NORMAL_DOPPLER;
    unsigned i, j;
    int fract;
    short *p1, *p2, *Source, *Target;

    Source=pSource; Target=pTarget;
    NORMAL_DOPPLER=0x10000;
    *Count=0;

    if (doppler_rate.dw==NORMAL_DOPPLER) // 48000, just copy to target
    {
        i = (MaxSource-Source)/2/nChannels;
        if (i > MaxCount) i = MaxCount;
        memcpy(Target, Source, i*2*nChannels);
        *Count = i;
        return (Source+i*nChannels);
    }
    else
    {
        p1 = Source + nChannels*index->s.whole;
        p2 = p1+nChannels;

        while((p2+nChannels) < MaxSource && *Count < MaxCount) // process the sample rate convert
        {
            fract = index->s.fract;
            for (j=0; j<nChannels; j++)
            {
                *Target = (short)(((*p1) * (NORMAL_DOPPLER-fract) + (*p2) * fract) >> 16);
                Target++; p1++; p2++;
            }

            (*Count)++;
            index->dw += doppler_rate.dw;
            p1 = Source + nChannels*index->s.whole;
            p2 = p1+nChannels;
         }
    }

    index->s.whole = 0; // reset to the first sample
    return p1; // return the last source buffer position
}

/*---------------------------------------------------------------------------------------*/
static inline void ProcessData(struct cmedia_channel *channel, int rec, void *TargetBuf, long count)
{
    struct cmedia_card *card = devs;
    short *ptrSource, *ptrTarget;
    long ProcessCount, cnt;
    long value;

    if (count <= 0) return;

    if (!rec)
    {
        if (channel->out_count < count) count = channel->out_count;
        count = count>>1;

        ptrSource = (short *)(channel->outbuf+channel->out_swptr);
        ptrTarget = (short *)TargetBuf;
        while (count)
        {
            cnt=0;
            ProcessCount = channel->out_total_bytes-channel->out_swptr;
            ProcessCount >>= 1;
            while(count && ProcessCount)
            {
                if (card->IsAC3 == 1) 
                {
                    if (channel->IsAC3==0) value = *ptrTarget;
                    else value = *ptrSource;
                }
                else
                {
                    value = *ptrSource; value += *ptrTarget;
                }

                if (value > 32767) *ptrTarget = 32767;
                else if (value < -32768) *ptrTarget = -32768;
                else *ptrTarget = value;
                ptrSource++; ProcessCount--;
                ptrTarget++;cnt++;
                count--; 
            }

            channel->out_swptr = (channel->out_swptr+(cnt<<1)) % channel->out_total_bytes;
            ptrSource = (short *)(channel->outbuf+channel->out_swptr);
        }
    }
    else
    {
        if ((channel->in_total_bytes-channel->in_count) < count) 
            count = channel->in_total_bytes-channel->in_count;
        count = count>>1;

        ptrSource = (short *)TargetBuf;
        ptrTarget = (short *)(channel->inbuf+channel->in_hwptr);
        
        while (count)
        {
            cnt=0;
            ProcessCount = channel->in_total_bytes-channel->in_hwptr;
            ProcessCount >>= 1;
            while(count && ProcessCount)
            {
                *ptrTarget = *ptrSource;
                ptrSource++; ptrTarget++;cnt++;
                count--; ProcessCount--;
            }

            channel->in_hwptr = (channel->in_hwptr+(cnt<<1)) % channel->in_total_bytes;
            ptrTarget = (short *)(channel->inbuf+channel->in_hwptr);
        }
    }
}

static void cmedia_timeout(unsigned long ptr)
{
    struct cmedia_card *card = devs;
    struct cmedia_channel *channel;
    int i, left=0, right=0, left1=0, right1=0;
    int l1, l2, l3, r1, r2, r3, l4, l5, l6, r4, r5, r6;
    unsigned long flags;
    short *ptrTarget, *ptrSource, *MaxPos;
    long InProcessCount1=0, InProcessCount2=0, OutProcessCount1=0, OutProcessCount2=0;
    long value=0;

    l1=r1=l4=r4=0;
    l2=l3=r2=r3=l5=l6=r5=r6=17;

    if (!card->dma_state) goto RESET_TIMER;

    //printk("cmedia_timeout\n");

    spin_lock_irqsave(&card->lock, flags);
    card->UpdatePtr(card);

    if (card->dma_state & ADC_RUNNING) 
    {
        if (card->pcmin_swptr < card->pcmin_hwptr)
        {
            InProcessCount1 = card->pcmin_hwptr-card->pcmin_swptr;
            InProcessCount2=0;
        }
        else
        {
            InProcessCount1 = PCMIN_DMABUF_SIZE-card->pcmin_swptr;
            InProcessCount2 = card->pcmin_hwptr;
        }

        if (card->ac97_vender_id2 == 0x4961 && card->ReadRegisterWord(card, AC97_RECORD_SELECT)==0x0505 &&
            (card->dma_state & DAC_RUNNING) && card->IsAC3 == 0)
        {
            ptrTarget = (short *)(card->pcmin_rawbuf+card->pcmin_swptr);
            MaxPos=(short *)(card->pcmout_rawbuf+PCMOUT_DMABUF_SIZE);
            ptrSource = (short *)(card->pcmout_rawbuf+card->pcmout_swptr);
            for(i=0; i<(InProcessCount1>>1); i++)
            {
                value = *ptrSource; value += *ptrTarget;
                if (value > 32767) *ptrTarget = 32767;
                else if (value < -32768) *ptrTarget = -32768;
                else *ptrTarget = value;

                ptrTarget++; ptrSource++;
                if ((i%2)==1) {ptrSource++;ptrSource++;ptrSource++;ptrSource++;}
                if (ptrSource>=MaxPos) ptrSource=card->pcmout_rawbuf;
            }

            ptrTarget = (short *)(card->pcmin_rawbuf);
            for(i=0; i<(InProcessCount2>>1); i++)
            {
                value = *ptrSource; value += *ptrTarget;
                if (value > 32767) *ptrTarget = 32767;
                else if (value < -32768) *ptrTarget = -32768;
                else *ptrTarget = value;

                ptrTarget++; ptrSource++;
                if ((i%2)==1) {ptrSource++;ptrSource++;ptrSource++;ptrSource++;}
                if (ptrSource>=MaxPos) ptrSource=card->pcmout_rawbuf;
            }
        }
    }

    if (card->dma_state & DAC_RUNNING) 
    {
        if (card->pcmout_swptr < card->pcmout_hwptr)
        {
            memset((card->pcmout_rawbuf+card->pcmout_swptr), 0, (card->pcmout_hwptr-card->pcmout_swptr));
            OutProcessCount1 = card->pcmout_hwptr-card->pcmout_swptr; OutProcessCount2=0;
        }
        else
        {
            memset((card->pcmout_rawbuf+card->pcmout_swptr), 0, (PCMOUT_DMABUF_SIZE-card->pcmout_swptr));
            OutProcessCount1 = PCMOUT_DMABUF_SIZE-card->pcmout_swptr;
            memset((card->pcmout_rawbuf), 0, card->pcmout_hwptr);
            OutProcessCount2 = card->pcmout_hwptr;
        }
    }

    for(i=0; i<NR_VR_CH; i++)
    {
        channel = card->channels[i];
        if (channel && (channel->enable & ADC_RUNNING))
        {
            ptrTarget = (void *)(card->pcmin_rawbuf+card->pcmin_swptr);
            ProcessData(channel, 1, ptrTarget, InProcessCount1);

            if (InProcessCount2)
            {
                ptrTarget = card->pcmin_rawbuf;
                ProcessData(channel, 1, ptrTarget, InProcessCount2);
            }

            if ((channel->in_count+InProcessCount1+InProcessCount2) < channel->in_total_bytes) 
                channel->in_count += (InProcessCount1+InProcessCount2);
            else 
                channel->in_count = channel->in_total_bytes;
        }

        if (channel && (channel->enable & DAC_RUNNING))
        {
            ptrTarget = (void *)(card->pcmout_rawbuf+card->pcmout_swptr);
            ProcessData(channel, 0, ptrTarget, OutProcessCount1);

            if (OutProcessCount2)
            {
                ptrTarget = card->pcmout_rawbuf;
                ProcessData(channel, 0, ptrTarget, OutProcessCount2);
            }

            if (channel->out_count > (OutProcessCount1+OutProcessCount2)) 
                channel->out_count -= (OutProcessCount1+OutProcessCount2);
            else 
                channel->out_count = 0;

            //printk("i=%d Timeout %ld\n", i, channel->out_count);
        }

    }

    if (card->ac97_vender_id2==0x4941) 
    {
        left1 = card->mixer_state[SOUND_MIXER_PCM]&0x00ff;
        right1 =(card->mixer_state[SOUND_MIXER_PCM]&0xff00)>>8;
        left = left1/5; right = right1/5;
        if (left==0 && left1!=0) left=1;
        if (right==0 && right1!=0) right=1;
        l1 = MulRatio[left][0]; l2 = MulRatio[left][1]; l3 = MulRatio[left][2];
        r1 = MulRatio[right][0]; r2 = MulRatio[right][1]; r3 = MulRatio[right][2];

        l4 = MulRatio[20][0]; l5 = MulRatio[20][1]; l6 = MulRatio[20][2];
        r4 = MulRatio[20][0]; r5 = MulRatio[20][1]; r6 = MulRatio[20][2];
    }
    else if (card->ac97_vender_id2==0x4961 && (card->IsAC3==0))
    {
        left1 = (card->mixer_state[SOUND_MIXER_VOLUME]&0x00ff);
        right1 =((card->mixer_state[SOUND_MIXER_VOLUME]&0xff00)>>8);
        left = left1/5; right = right1/5;
        if (left==0 && left1!=0) left=1;
        if (right==0 && right1!=0) right=1;
        l1 = MulRatio[left][0]; l2 = MulRatio[left][1]; l3 = MulRatio[left][2];
        r1 = MulRatio[right][0]; r2 = MulRatio[right][1]; r3 = MulRatio[right][2];

        left1 = (card->mixer_state[SOUND_MIXER_PCM]&0x00ff);
        right1 =((card->mixer_state[SOUND_MIXER_PCM]&0xff00)>>8);
        left = left1/5; right = right1/5;
        if (left==0 && left1!=0) left=1;
        if (right==0 && right1!=0) right=1;
        l4 = MulRatio[left][0];  l5 = MulRatio[left][1];  l6 = MulRatio[left][2];
        r4 = MulRatio[right][0]; r5 = MulRatio[right][1]; r6 = MulRatio[right][2];
    }

    if ((card->ac97_vender_id1==0x434d) && (card->dma_state&DAC_RUNNING) && (card->IsAC3==0)) 
    {
        ptrTarget = (void *)(card->pcmout_rawbuf+card->pcmout_swptr);
        i=OutProcessCount1>>2;

        for(; i>0; i--)
        {
            *ptrTarget=(*ptrTarget>>l1)+(*ptrTarget>>l2)+(*ptrTarget>>l3);
            *ptrTarget=(*ptrTarget>>l4)+(*ptrTarget>>l5)+(*ptrTarget>>l6);
            ptrTarget++;

            *ptrTarget=(*ptrTarget>>r1)+(*ptrTarget>>r2)+(*ptrTarget>>r3);
            *ptrTarget=(*ptrTarget>>r4)+(*ptrTarget>>r5)+(*ptrTarget>>r6);
            ptrTarget++;
        }

        ptrTarget = (void *)(card->pcmout_rawbuf);
        i=OutProcessCount2>>2;
        for(; i>0; i--)
        {
            *ptrTarget=(*ptrTarget>>l1)+(*ptrTarget>>l2)+(*ptrTarget>>l3);
            *ptrTarget=(*ptrTarget>>l4)+(*ptrTarget>>l5)+(*ptrTarget>>l6);
            ptrTarget++;

            *ptrTarget=(*ptrTarget>>r1)+(*ptrTarget>>r2)+(*ptrTarget>>r3);
            *ptrTarget=(*ptrTarget>>r4)+(*ptrTarget>>r5)+(*ptrTarget>>r6);
            ptrTarget++;
        }
    }

    card->ConvertPCMOUT(card);

    if (card->dma_state & DAC_RUNNING) card->pcmout_swptr = card->pcmout_hwptr;
    if (card->dma_state & ADC_RUNNING) card->pcmin_swptr = card->pcmin_hwptr;
    spin_unlock_irqrestore(&card->lock, flags);

    for(i=0; i<NR_VR_CH; i++)
    {
        channel = card->channels[i];
        if (channel && (channel->enable & ADC_RUNNING)) wake_up(&channel->in_wait);
        if (channel && (channel->enable & DAC_RUNNING)) wake_up(&channel->out_wait);
    }

RESET_TIMER:
    if (devs)
    {
        cmedia_timer.expires = jiffies + PROCESS_TIMER;
        add_timer(&cmedia_timer);
    }
}

/*---------------------------------------------------------------------------------------*/
/* allocate DMA buffer, playback and recording buffer should be allocated seperately */
static int __init alloc_dmabuf(struct cmedia_card *card)
{
    void *rawbuf= NULL;
    int order;
    struct page *page, *pend;

    //--------------------------------------------------------	
    /* alloc enough to satisfy the PCMOUT_DMABUF_SIZE*/
    for (order = DMABUF_DEFAULTORDER; order >= DMABUF_MINORDER; order--) 
    {
        if ( (PAGE_SIZE<<order) > PCMOUT_DMABUF_SIZE ) continue;
        order++;
        if ((rawbuf = pci_alloc_consistent(card->pci_dev,
                                           PAGE_SIZE << order,
                                           &card->pcmout_dma_handle))) break;
    }

    if (!rawbuf) return -ENOMEM;

    //printk("CMedia Audio: allocated pcmout_rawbuf %ld (order = %d) bytes at %p\n", 
    //       PAGE_SIZE << order, order, rawbuf);

    card->pcmout_rawbuf = rawbuf;
    card->pcmout_buforder = order;

    /* now mark the pages as reserved; otherwise remap_page_range doesn't do what we want */
    pend = virt_to_page(rawbuf + (PAGE_SIZE << order) - 1);
    for (page = virt_to_page(rawbuf); page <= pend; page++) mem_map_reserve(page);

    //--------------------------------------------------------	
    /* alloc enough to satisfy the PCMOUT_DMABUF_SIZE*/
    rawbuf= NULL;
    rawbuf = pci_alloc_consistent(card->pci_dev,
                                           PAGE_SIZE << order,
                                           &card->dmaout_dma_handle);

    if (!rawbuf) return -ENOMEM;

    card->dmaout_rawbuf = rawbuf;

    /* now mark the pages as reserved; otherwise remap_page_range doesn't do what we want */
    pend = virt_to_page(rawbuf + (PAGE_SIZE << order) - 1);
    for (page = virt_to_page(rawbuf); page <= pend; page++) mem_map_reserve(page);

    //--------------------------------------------------------	
    /* alloc enough to satisfy the PCMIN_DMABUF_SIZE*/
    rawbuf= NULL;
    for (order = DMABUF_DEFAULTORDER; order >= DMABUF_MINORDER; order--) 
    {
        if ( (PAGE_SIZE<<order) > PCMIN_DMABUF_SIZE ) continue;
        order++;
        if ((rawbuf = pci_alloc_consistent(card->pci_dev,
                                           PAGE_SIZE << order,
                                           &card->pcmin_dma_handle))) break;
    }

    if (!rawbuf) return -ENOMEM;

    //printk("CMedia Audio: allocated pcmin_rawbuf %ld (order = %d) bytes at %p\n", 
    //       PAGE_SIZE << order, order, rawbuf);

    card->pcmin_rawbuf = rawbuf;
    card->pcmin_buforder = order;
	
    /* now mark the pages as reserved; otherwise remap_page_range doesn't do what we want */
    pend = virt_to_page(rawbuf + (PAGE_SIZE << order) - 1);
    for (page = virt_to_page(rawbuf); page <= pend; page++) mem_map_reserve(page);

    return 0;
}

/* free DMA buffer */
static void __exit dealloc_dmabuf(struct cmedia_card *card)
{
    struct page *page, *pend;

    if (card->pcmout_rawbuf) 
    {
        /* undo marking the pages as reserved */
        pend = virt_to_page(card->pcmout_rawbuf + (PAGE_SIZE << card->pcmout_buforder) - 1);
        for (page = virt_to_page(card->pcmout_rawbuf); page <= pend; page++)	mem_map_unreserve(page);
        pci_free_consistent(card->pci_dev, PAGE_SIZE << card->pcmout_buforder, card->pcmout_rawbuf, 
            card->pcmout_dma_handle);
    }

    card->pcmout_rawbuf = NULL;

    if (card->dmaout_rawbuf) 
    {
        /* undo marking the pages as reserved */
        pend = virt_to_page(card->dmaout_rawbuf + (PAGE_SIZE << card->pcmout_buforder) - 1);
        for (page = virt_to_page(card->dmaout_rawbuf); page <= pend; page++)	mem_map_unreserve(page);
        pci_free_consistent(card->pci_dev, PAGE_SIZE << card->pcmout_buforder, card->dmaout_rawbuf, 
            card->dmaout_dma_handle);
    }

    card->dmaout_rawbuf = NULL;

    if (card->pcmin_rawbuf) 
    {
        /* undo marking the pages as reserved */
        pend = virt_to_page(card->pcmin_rawbuf + (PAGE_SIZE << card->pcmin_buforder) - 1);
        for (page = virt_to_page(card->pcmin_rawbuf); page <= pend; page++)	mem_map_unreserve(page);
        pci_free_consistent(card->pci_dev, PAGE_SIZE << card->pcmin_buforder, card->pcmin_rawbuf, 
            card->pcmin_dma_handle);
    }

    card->pcmin_rawbuf = NULL;
}

/*---------------------------------------------------------------------------------------*/
static loff_t cmedia_llseek(struct file *file, loff_t offset, int origin)
{
	return -ESPIPE;
}

/* in this loop, dmabuf.count signifies the amount of data that is waiting to be copied to
   the user's buffer.  it is filled by the dma machine and drained by this loop. */
static ssize_t cmedia_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
    struct cmedia_channel *channel = (struct cmedia_channel *)file->private_data;
    struct cmedia_card *card = channel->card;
    ssize_t ret;
    unsigned long flags;
    long swptr;
    long cnt, new_cnt, left;
    int opened, samplesize;
    void *NewP;

    //printk("CMedia Audio: cmedia_read called, count = %d\n", count);
    samplesize=channel->PhysicalChannels<<1;

    if (ppos != &file->f_pos) return -ESPIPE;
    if(!(card->dma_state & ADC_RUNNING)) card->StartADC(card);
    if (!access_ok(VERIFY_WRITE, buffer, count)) return -EFAULT;
    spin_lock_irqsave(&card->lock, flags);
    if (channel->opened)
    {
        channel->enable |= ADC_RUNNING;
        channel->trigger |= PCM_ENABLE_INPUT;
    }
    spin_unlock_irqrestore(&card->lock, flags);
    ret = left = 0;

    while (count > 0) 
    {
        spin_lock_irqsave(&card->lock, flags);
        cnt = channel->in_count;
        swptr = channel->in_swptr;
        if(cnt > (channel->in_total_bytes - swptr)) cnt = channel->in_total_bytes - swptr;
        opened = channel->opened;
        spin_unlock_irqrestore(&card->lock, flags);
        if (!opened) return -EFAULT;

        if (cnt <= 0) 
        {
            unsigned long tmo;
            if(!(card->dma_state & ADC_RUNNING)) 
            {
                  channel->trigger |= PCM_ENABLE_INPUT;
                  card->StartADC(card);
            }

            if (file->f_flags & O_NONBLOCK) 
            {
                if (!ret) ret = -EAGAIN;
                return ret;
            }

            tmo = PROCESS_TIMER*10;
            interruptible_sleep_on_timeout(&channel->in_wait, tmo);

            if (signal_pending(current)) 
            {
                ret = ret ? ret : -ERESTARTSYS;
                return ret;
            }

            continue;
        }

        new_cnt = cnt;

        if (channel->rate != 48000) 
        {
            // Copy to the temp buffer until temp buffer is full
            if (cnt > channel->pcmin_tempbuf1_count) cnt = new_cnt = channel->pcmin_tempbuf1_count;

            memcpy(channel->pcmin_tempbuf1+TEMPBUF_SIZE-channel->pcmin_tempbuf1_count, 
                   channel->inbuf + swptr, cnt);
            channel->pcmin_tempbuf1_count -= cnt;

            if (channel->pcmin_tempbuf1_count <= 0)
            {
                NewP = RecordingSamplingRateConvert(
                           channel->pcmin_tempbuf1,
                           channel->pcmin_tempbuf1+TEMPBUF_SIZE,
                           channel->pcmin_tempbuf,
                           count/samplesize,
                           channel->pcmin_doppler_rate,
                           &(channel->pcmin_index), 
                           channel->PhysicalChannels,
                           &cnt);
                cnt *= samplesize;

                memcpy(channel->pcmin_tempbuf1, NewP, channel->pcmin_tempbuf1+TEMPBUF_SIZE-NewP);
                channel->pcmin_tempbuf1_count = NewP-channel->pcmin_tempbuf1;

                if (copy_to_user(buffer, channel->pcmin_tempbuf, cnt)) 
                {
                    if (!ret) ret = -EFAULT;
                    return ret;
                }
            }
            else
            {
                cnt = 0;
            }

            swptr = (swptr + new_cnt) % channel->in_total_bytes;
        }
        else
        {
            if (cnt > count) cnt = new_cnt = count;

            if (copy_to_user(buffer, channel->inbuf + swptr, cnt)) 
            {
                if (!ret) ret = -EFAULT;
                return ret;
            }

            swptr = (swptr + new_cnt) % channel->in_total_bytes;
        }

        spin_lock_irqsave(&card->lock, flags);
        channel->in_swptr = swptr;
        channel->in_count -= new_cnt;
        spin_unlock_irqrestore(&card->lock, flags);

        count -= cnt;
        buffer += cnt;
        ret += cnt;
    }

    //printk("ret = %d channel->in_count=%d\n", ret, channel->in_count);

    return ret;
}

/* in this loop, dmabuf.count signifies the amount of data that is waiting to be dma to
   the soundcard.  it is drained by the dma machine and filled by this loop. */
static ssize_t cmedia_write(struct file *file, const char *buffer, size_t count, loff_t *ppos)
{
    struct cmedia_channel *channel = (struct cmedia_channel *)file->private_data;
    struct cmedia_card *card = channel->card;
    ssize_t ret;
    unsigned long flags, i;
    long hwptr=0;
    long cnt, new_cnt, left;
    int opened, samplesize;
    short *ptrSource, *ptrTarget;
    void *NewP;
    u16 wTemp;
    u8 Reg;

    //printk("cmedia_write , count = %d card->dma_state=%x %d\n", count, card->dma_state, channel->rate);

    if (ppos != &file->f_pos) return -ESPIPE;

    if (!(card->dma_state & DAC_RUNNING)) card->StartDAC(card);
    if (!access_ok(VERIFY_READ, buffer, count)) return -EFAULT;
    spin_lock_irqsave(&card->lock, flags);
    if (channel->opened)
    {
        if (!(channel->enable&DAC_RUNNING))
        {
            if (card->FourChannelCount<0) card->FourChannelCount=0;
            if (card->SixChannelCount<0) card->SixChannelCount=0;

            //printk("cmedia_write , %d %d %d\n", channel->PhysicalChannels, card->FourChannelCount,
            //       card->SixChannelCount);

            if (channel->IsAC3 == 1)
            {
                card->IsAC3++;
                if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4961) 
                {
                    wTemp = card->ReadRegisterWord(card, AC97REG_RESERVED2);
                    wTemp |= 0x0002;
                    card->WriteRegisterWord(card, AC97REG_RESERVED2, wTemp);

                    wTemp = card->ReadRegisterWord(card, AC97REG_PCMOUT_VOLUME);
                    wTemp |= 0x8000;
                    card->WriteRegisterWord(card, AC97REG_PCMOUT_VOLUME, wTemp);
                }
            }

            if (channel->PhysicalChannels == 4) 
            {
                card->FourChannelCount++;
                if (card->FourChannelCount==1 && card->SixChannelCount==0 && AUTO_PHONEJACK_SHARE)
                {
                    card->LineInAsRear = 1;
                    Reg=AC97REG_MULTI_CHANNEL_CTRL;
                    if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4941) 
                        Reg=AC97REG_CM9738_EXT1;
                    wTemp = card->ReadRegisterWord(card, Reg);
                    wTemp |= 0x0400;
                    card->WriteRegisterWord(card, Reg, wTemp);
                }
            }
            else if (channel->PhysicalChannels == 6) 
            {
                card->SixChannelCount++;
                if (card->FourChannelCount==0 && card->SixChannelCount==1 && AUTO_PHONEJACK_SHARE)
                {
                    card->LineInAsRear = 1;
                    Reg=AC97REG_MULTI_CHANNEL_CTRL;
                    if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4941) 
                        Reg=AC97REG_CM9738_EXT1;
                    wTemp = card->ReadRegisterWord(card, Reg);
                    wTemp |= 0x0400;
                    card->WriteRegisterWord(card, Reg, wTemp);
                }

                if (card->SixChannelCount==1 && AUTO_PHONEJACK_SHARE)
                {
                    card->MicAsCenterBass = 1;

                    wTemp = card->ReadRegisterWord(card, 0x20);
                    wTemp |= 0x0100;
                    card->WriteRegisterWord(card, 0x20, wTemp);

                    wTemp = card->ReadRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL);
                    wTemp &= (~0x3000);
                    wTemp |= 0x1000;
                    card->WriteRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL, wTemp);
                }
            }

            channel->enable |= DAC_RUNNING;
            channel->trigger |= PCM_ENABLE_OUTPUT;
        }
    }
    spin_unlock_irqrestore(&card->lock, flags);
    ret = 0;
    samplesize=channel->PhysicalChannels<<1;

    while (count > 0) 
    {
        spin_lock_irqsave(&card->lock, flags);
        cnt = channel->out_total_bytes - channel->out_count;
        hwptr = channel->out_hwptr;

        // this is to make the copy_to_user simpler below
        if(cnt > (channel->out_total_bytes - hwptr)) cnt = channel->out_total_bytes - hwptr;
        opened = channel->opened;
        spin_unlock_irqrestore(&card->lock, flags);
        if (!opened) return -EFAULT;

        if (cnt == 0) 
        {
            unsigned long tmo;
    
            if (file->f_flags & O_NONBLOCK) 
            {
                if (!ret) ret = -EAGAIN;
                return ret;
            }

            tmo = PROCESS_TIMER*10;
            interruptible_sleep_on_timeout(&channel->out_wait, tmo);

            if (signal_pending(current)) 
            {
                if (!ret) ret = -ERESTARTSYS;
                return ret;
            }
            continue;
        }

        cnt = cnt*channel->PhysicalChannels/6;
        new_cnt = cnt;

        if (channel->rate != 48000) 
        {
            left = TEMPBUF_SIZE-channel->pcmout_leftsize;
            if (left > count) cnt = count;
            else cnt = left;

            ret = copy_from_user(channel->pcmout_tempbuf+channel->pcmout_leftsize,buffer,cnt);
            if (ret) return -EFAULT;

            NewP = RecordingSamplingRateConvert(
                       channel->pcmout_tempbuf,
                       channel->pcmout_tempbuf+cnt+channel->pcmout_leftsize,
                       channel->outbuf+hwptr,
                       new_cnt/samplesize,
                       channel->pcmout_doppler_rate,
                       &(channel->pcmout_index), 
                       channel->PhysicalChannels,
                       &new_cnt);
            new_cnt *= samplesize;

            channel->pcmout_leftsize = channel->pcmout_tempbuf+cnt+channel->pcmout_leftsize-NewP;
            if (left>0) memcpy(channel->pcmout_tempbuf, NewP, channel->pcmout_leftsize);
            else channel->pcmout_leftsize = 0;
        }
        else
        {
            if (cnt > count) cnt = new_cnt = count;

            if (copy_from_user(channel->outbuf+hwptr,buffer,cnt))
            {
                if (!ret) ret = -EFAULT;
                return ret;
            }
        }

        // Process to the correct position
        {
            switch (channel->PhysicalChannels)
            {
            case 1:
                i = new_cnt/2;
                ptrSource = (unsigned short *)(channel->outbuf+hwptr+new_cnt-2);
                ptrTarget = (unsigned short *)(channel->outbuf+hwptr+new_cnt*6-2);

                for(;i>0;i--)
                {
                    *ptrTarget = 0;ptrTarget--; *ptrTarget = 0;ptrTarget--; 
                    *ptrTarget = 0;ptrTarget--; *ptrTarget = 0;ptrTarget--; 
                    *ptrTarget = *ptrSource; ptrTarget--;
                    *ptrTarget = *ptrSource; ptrTarget--; ptrSource--;
                }
                break;
            case 2:
                i = new_cnt/4;
                ptrSource = (unsigned short *)(channel->outbuf+hwptr+new_cnt-2);
                ptrTarget = (unsigned short *)(channel->outbuf+hwptr+new_cnt*3-2);
                for(;i>0;i--)
                {
                    *ptrTarget = 0;ptrTarget--; *ptrTarget = 0;ptrTarget--; 
                    *ptrTarget = 0;ptrTarget--; *ptrTarget = 0;ptrTarget--; 
                    *ptrTarget = *ptrSource; ptrTarget--; ptrSource--;
                    *ptrTarget = *ptrSource; ptrTarget--; ptrSource--;
                }
                break;
            case 4:
                i = new_cnt/8;
                ptrSource = (unsigned short *)(channel->outbuf+hwptr+new_cnt-2);
                ptrTarget = (unsigned short *)(channel->outbuf+hwptr+new_cnt*3/2-2);

                for(;i>0;i--)
                {
                    *ptrTarget = 0;ptrTarget--; *ptrTarget = 0;ptrTarget--; 
                    *ptrTarget = *ptrSource; ptrTarget--; ptrSource--;
                    *ptrTarget = *ptrSource; ptrTarget--; ptrSource--;
                    *ptrTarget = *ptrSource; ptrTarget--; ptrSource--;
                    *ptrTarget = *ptrSource; ptrTarget--; ptrSource--;
                }
                break;
            case 6:
                break;
            }
        }

        hwptr = (hwptr + new_cnt*6/channel->PhysicalChannels) % channel->out_total_bytes;

        spin_lock_irqsave(&card->lock, flags);
        channel->out_hwptr = hwptr;
        channel->out_count += new_cnt*6/channel->PhysicalChannels;
        spin_unlock_irqrestore(&card->lock, flags);

        count -= cnt;
        buffer += cnt;
        ret += cnt;
    }

    return ret;
}

static int cmedia_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
    struct cmedia_channel *channel = (struct cmedia_channel *)file->private_data;
    struct cmedia_card *card = channel->card;
    unsigned long flags;
    audio_buf_info abinfo;
    long swptr=0;
    long cnt;
    u16 wTemp;
    u8 Reg;
    //count_info cinfo;
    int val;
    //int ret;
    INDEX index;

    //printk("CMedia Audio: cmedia_ioctl, arg=0x%x, cmd=0x%x\n", arg ? *(int *)arg : 0, cmd);

    switch (cmd) 
    {
    case OSS_GETVERSION:
        //printk("OSS_GETVERSION\n");

        return put_user(SOUND_VERSION, (int *)arg);

    case SNDCTL_DSP_RESET:
        //printk("SNDCTL_DSP_RESET\n");

        return 0;

    case SNDCTL_DSP_SYNC:
        //printk("SNDCTL_DSP_SYNC\n");

        return -EFAULT;

    case SNDCTL_DSP_SPEED: /* set smaple rate */
        //printk("SNDCTL_DSP_SPEED\n");
        if (get_user(val, (int *)arg) || (val < 8000 || val > 48000)) return -EFAULT;
        channel->rate = val;
        channel->pcmout_doppler_rate.dw=val; channel->pcmout_doppler_rate.dw <<= 16; 
        channel->pcmout_doppler_rate.dw /= 48000;
        channel->pcmout_index.dw=0;
        channel->pcmin_doppler_rate.dw=48000; channel->pcmin_doppler_rate.dw <<= 16; 
        channel->pcmin_doppler_rate.dw /= val;
        channel->pcmin_index.dw=0;
        printk("SNDCTL_DSP_SPEED %d\n", val);
        
        return put_user(channel->rate, (int *)arg); 

    case SNDCTL_DSP_STEREO: /* set stereo or mono channel */
        //printk("SNDCTL_DSP_STEREO\n");
        if (get_user(val, (int *)arg)) return -EFAULT;
        spin_lock_irqsave(&card->lock, flags);
        if (val) { channel->fmt |= CMEDIA_FMT_STEREO; channel->PhysicalChannels=2; }
        else { channel->fmt &= ~CMEDIA_FMT_STEREO; channel->PhysicalChannels=1; }
        spin_unlock_irqrestore(&card->lock, flags);

        return 0;

    case SNDCTL_DSP_GETBLKSIZE:
        //printk("SNDCTL_DSP_GETBLKSIZE\n");

        return put_user(4096, (int *)arg);

    case SNDCTL_DSP_GETFMTS: /* Returns a mask of supported sample format*/
        //printk("SNDCTL_DSP_GETFMTS\n");

        return put_user(AFMT_S16_LE|AFMT_AC3, (int *)arg);

    case SNDCTL_DSP_SETFMT: /* Select sample format */
        if (get_user(val, (int *)arg)) return -EFAULT;
        //printk("SNDCTL_DSP_SETFMT %x  %x  %x\n", val, AFMT_S16_LE, AFMT_AC3);

        if (val == AFMT_AC3) 
        {
            spin_lock_irqsave(&card->lock, flags);
            val=48000;
            channel->rate = val;
            channel->pcmout_doppler_rate.dw=val; channel->pcmout_doppler_rate.dw <<= 16; 
            channel->pcmout_doppler_rate.dw /= 48000;
            channel->pcmout_index.dw=0;
            channel->pcmin_doppler_rate.dw=48000; channel->pcmin_doppler_rate.dw <<= 16; 
            channel->pcmin_doppler_rate.dw /= val;
            channel->pcmin_index.dw=0;
            channel->PhysicalChannels=2;
            channel->IsAC3=1;
            channel->fmt |= CMEDIA_FMT_STEREO;
            spin_unlock_irqrestore(&card->lock, flags);
            return put_user(AFMT_AC3, (int *)arg);
        }
        else 
        {
            spin_lock_irqsave(&card->lock, flags);
            channel->IsAC3=0;
            spin_unlock_irqrestore(&card->lock, flags);
            return put_user(AFMT_S16_LE, (int *)arg);
        }

    case SNDCTL_DSP_CHANNELS:
        val=channel->PhysicalChannels;
        if (get_user(val, (int *)arg)) return -EFAULT;
        if (!(val == 1 || val == 2 || val == 4 || val == 6)) return -EFAULT;
        if (val > output_channels) return -EFAULT;
        //printk("SNDCTL_DSP_CHANNELS %d\n", val);

        spin_lock_irqsave(&card->lock, flags);
        channel->PhysicalChannels=val;
        spin_unlock_irqrestore(&card->lock, flags);

        return put_user(val, (int *)arg);

    case SNDCTL_DSP_POST: /* the user has sent all data and is notifying us */
        //printk("SNDCTL_DSP_POST\n");

        return 0;

    case SNDCTL_DSP_SUBDIVIDE:
        //printk("SNDCTL_DSP_SUBDIVIDE\n");

        return -EFAULT;

    case SNDCTL_DSP_SETFRAGMENT:
        //printk("SNDCTL_DSP_SETFRAGMENT\n");

        return 0;

    case SNDCTL_DSP_GETOSPACE:
        //printk("SNDCTL_DSP_GETOSPACE\n");

        if (!(file->f_mode & FMODE_WRITE)) return -EINVAL;

        cnt=0;
        abinfo.fragsize = PAGE_SIZE;
        abinfo.fragstotal = PCMOUT_DMABUF_SIZE/PAGE_SIZE;
        abinfo.fragments=0;
        index.dw=0;

        while (cnt==0 || abinfo.fragments==0) 
        {
            spin_lock_irqsave(&card->lock, flags);
            cnt = channel->out_total_bytes - channel->out_count;
            spin_unlock_irqrestore(&card->lock, flags);

            cnt = cnt/12;
            spin_lock_irqsave(&card->lock, flags);
            cnt = GetNewSampleLen(channel->pcmin_doppler_rate, index, cnt);
            spin_unlock_irqrestore(&card->lock, flags);
            cnt = (cnt)*2*channel->PhysicalChannels;
            abinfo.bytes = cnt;
            abinfo.fragments = abinfo.bytes / PAGE_SIZE;
            //if (PrintCount < 100) {printk("cnt=%ld\n", cnt); PrintCount++;}

            if (cnt == 0 || abinfo.fragments == 0)
            {
                unsigned long tmo;
                tmo = PROCESS_TIMER*10;
                interruptible_sleep_on_timeout(&channel->out_wait, tmo);
            }

            if (signal_pending(current)) return -ERESTARTSYS;
        }

        abinfo.bytes = cnt;
        abinfo.fragments = abinfo.bytes / PAGE_SIZE;
        
        //printk("SNDCTL_DSP_GETOSPACE %d  %d\n", abinfo.bytes, abinfo.fragments);
        return copy_to_user((void *)arg, &abinfo, sizeof(abinfo)) ? -EFAULT : 0;

    case SNDCTL_DSP_GETOPTR:
        printk("SNDCTL_DSP_GETOPTR\n");
        return -EFAULT;

    case SNDCTL_DSP_GETISPACE:
        //printk("SNDCTL_DSP_GETISPACE\n");
        if (!(file->f_mode & FMODE_READ)) return -EINVAL;

        cnt=0;
        while (cnt==0) 
        {
            spin_lock_irqsave(&card->lock, flags);
            cnt = channel->in_count;
            swptr = channel->in_swptr;
            if(cnt > (channel->in_total_bytes - swptr)) cnt = channel->in_total_bytes - swptr;
            spin_unlock_irqrestore(&card->lock, flags);

            if (cnt == 0)
            {
                unsigned long tmo;
                tmo = PROCESS_TIMER*10;
                interruptible_sleep_on_timeout(&channel->in_wait, tmo);
            }

            if (signal_pending(current)) return -ERESTARTSYS;
        }

        abinfo.fragsize = PAGE_SIZE;
        abinfo.fragstotal = PCMIN_DMABUF_SIZE/PAGE_SIZE;
        abinfo.bytes = cnt;
        abinfo.fragments = abinfo.bytes / PAGE_SIZE;

        return copy_to_user((void *)arg, &abinfo, sizeof(abinfo)) ? -EFAULT : 0;

    case SNDCTL_DSP_GETIPTR:
        //printk("SNDCTL_DSP_GETIPTR\n");

        return -EFAULT;

    case SNDCTL_DSP_NONBLOCK:
        //printk("SNDCTL_DSP_NONBLOCK\n");

        return -EFAULT;

    case SNDCTL_DSP_GETCAPS:
        //printk("SNDCTL_DSP_GETCAPS\n");
        return put_user(DSP_CAP_DUPLEX|DSP_CAP_BATCH|DSP_CAP_MULTI,(int *)arg);

    case SNDCTL_DSP_GETTRIGGER:
        //printk("SNDCTL_DSP_GETTRIGGER\n");

        return -EFAULT;

    case SNDCTL_DSP_SETTRIGGER:
        //printk("SNDCTL_DSP_SETTRIGGER\n");

        return -EFAULT;

    case SNDCTL_DSP_SETDUPLEX:
        //printk("SNDCTL_DSP_SETDUPLEX\n");

        return 0;

    case SNDCTL_DSP_GETODELAY:
        //printk("SOUND_DSP_GETODELAY\n");
        return -EFAULT;

    case SOUND_PCM_READ_RATE:
        //printk("SOUND_PCM_READ_RATE\n");

        return put_user(48000, (int *)arg);

    case SOUND_PCM_READ_CHANNELS:
        //printk("SOUND_PCM_READ_CHANNELS\n");

        return put_user(2, (int *)arg);

    case SOUND_PCM_READ_BITS:
        //printk("SOUND_PCM_READ_BITS\n");

        return put_user(16, (int *)arg);

    case SNDCTL_DSP_MAPINBUF:
    case SNDCTL_DSP_MAPOUTBUF:
    case SNDCTL_DSP_SETSYNCRO:
    case SOUND_PCM_WRITE_FILTER:
    case SOUND_PCM_READ_FILTER:

        //printk("SNDCTL_* -EINVAL\n");

        return -EINVAL;

    //case SNDCTL_DSP_GETCHANNELMASK:
    //    printk("SNDCTL_DSP_GETCHANNELMASK\n");
    //    return put_user(DSP_BIND_FRONT|DSP_BIND_SURR|DSP_BIND_CENTER_LFE|DSP_BIND_SPDIF, (int *)arg);

    case SNDCTL_DSP_LINEINASREAR:
        if (get_user(val, (int *)arg)) return -EFAULT;
        if (output_channels < 4) return -EINVAL;
        if (FORCE_LINEINASREAR_MODE) return -EINVAL;

        if (val == 0 || val == 1)
        {
            card->LineInAsRear = val;
            Reg=AC97REG_MULTI_CHANNEL_CTRL;
            if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4941) 
                Reg=AC97REG_CM9738_EXT1;
            wTemp = card->ReadRegisterWord(card, Reg);
            if (val) wTemp |= 0x0400;
            else wTemp &= (~0x0400);
            card->WriteRegisterWord(card, Reg, wTemp);
        }
        else val = card->LineInAsRear;
        
        return put_user(val, (int *)arg);

    case SNDCTL_DSP_MICASCENTERBASS:
        if (get_user(val, (int *)arg)) return -EFAULT;
        if (output_channels < 6) return -EINVAL;
        if (FORCE_MICASBASS_MODE) return -EINVAL;

        if (val == 0 || val == 1)
        {
            card->MicAsCenterBass = val;

            wTemp = card->ReadRegisterWord(card, 0x20);
            if (val || FORCE_MICSELECT2_MODE) wTemp |= 0x0100;
            else wTemp &= ~0x0100;
            card->WriteRegisterWord(card, 0x20, wTemp);

            wTemp = card->ReadRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL);
            wTemp &= (~0x3000);
            if (val) wTemp |= 0x1000;
            else wTemp |= 0x2000;
            card->WriteRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL, wTemp);
        }
        else val = card->MicAsCenterBass;

        return put_user(val, (int *)arg);

    case SNDCTL_DSP_XEAR:
        if (get_user(val, (int *)arg)) return -EFAULT;
        if (output_channels < 4) return -EINVAL;
        if (FORCE_XEAR_MODE) return -EINVAL;

        if (val == 0 || val == 1)
        {
            if (val) card->XearMode=1;
            else card->XearMode=0;
        }
        else val = card->XearMode;

        return put_user(val, (int *)arg);

    case SNDCTL_DSP_MICSELECT2:
        if (get_user(val, (int *)arg)) return -EFAULT;
        if (FORCE_MICSELECT2_MODE) return -EINVAL;

        if (val == 0 || val == 1)
        {
            if ((output_channels >= 6) && card->MicAsCenterBass && (val==0)) val=1;
            card->MicSelect2 = val;
            wTemp = card->ReadRegisterWord(card, 0x20);
            wTemp &= (~0x0100);
            if (val) wTemp |= 0x0100;
            card->WriteRegisterWord(card, 0x20, wTemp);
        }
        else val = card->MicSelect2;

        return put_user(val, (int *)arg);

    case SNDCTL_DSP_MIX2SURROUND:
        printk("SNDCTL_DSP_MIX2SURROUND ");
        if (get_user(val, (int *)arg)) return -EFAULT;
        printk("arg = %x ", val);

        if (val == 0 || val == 1)
        {
            if (card->ac97_vender_id2 == 0x4941)
            {
                wTemp = card->ReadRegisterWord(card, AC97REG_CM9738_EXT1);
                if (val) card->WriteRegisterWord(card , AC97REG_CM9738_EXT1, wTemp|0x2000);
                else card->WriteRegisterWord(card , AC97REG_CM9738_EXT1, wTemp&(~0x2000));
                card->Mix2Surround=val;
            }
            else
            {
                wTemp = card->ReadRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL);
                printk("wTemp1=%x ", wTemp);
                if (val) card->WriteRegisterWord(card , AC97REG_MULTI_CHANNEL_CTRL, wTemp|0x0200);
                else card->WriteRegisterWord(card , AC97REG_MULTI_CHANNEL_CTRL, wTemp&(~0x0200));
                card->Mix2Surround=val;
                printk("wTemp2=%x\n", wTemp);
            }
        }
        else val = card->Mix2Surround;

        return put_user(val, (int *)arg);
    }

    return -EINVAL;
}

static int cmedia_open(struct inode *inode, struct file *file)
{
    int i = 0;
    struct cmedia_card *card = devs;
    struct cmedia_channel *channel = NULL;
    struct page *page, *pend;
    unsigned long flags;

    //printk("cmedia_open\n");

    /* find an avaiable virtual channel (instance of /dev/dsp) */
    if (card != NULL) 
    {
        for (i = 0; i < 50 && card->initializing; i++) 
        {
            current->state = TASK_UNINTERRUPTIBLE;
            schedule_timeout(HZ/20);
        }

        for (i = 0; i < NR_VR_CH && !card->initializing; i++) 
        {
            if (card->channels[i] == NULL) 
            {
                channel = (struct cmedia_channel *) kmalloc(sizeof(struct cmedia_channel), GFP_KERNEL);
                if (channel == NULL) return -ENOMEM;

                pend = virt_to_page(channel + sizeof(struct cmedia_channel) - 1);
                for (page = virt_to_page(channel); page <= pend; page++) mem_map_reserve(page);

                memset(channel, 0, sizeof(struct cmedia_channel));
                goto found_virt;
            }
        }
    }

    /* no more virtual channel avaiable */
    if (!channel) return -ENODEV;

found_virt:

    /* initialize the virtual channel */
    channel->virt = i;
    channel->PhysicalChannels=2;
    channel->card = card;
    init_waitqueue_head(&channel->in_wait);
    init_waitqueue_head(&channel->out_wait);
    init_MUTEX(&channel->open_sem);
    file->private_data = channel;
    channel->rate = 48000;
    channel->pcmout_doppler_rate.dw=65536;
    channel->pcmout_index.dw=0;
    channel->pcmout_leftsize=0;
    channel->pcmin_doppler_rate.dw=65536;
    channel->pcmin_index.dw=0;
    channel->fmt = CMEDIA_FMT_16BIT | CMEDIA_FMT_STEREO;
    channel->IsAC3 = 0;
    channel->open_mode |= file->f_mode & (FMODE_READ | FMODE_WRITE);

    /* allocate hardware channels */
    if ((channel->outbuf = (void *) kmalloc(PCMOUT_DMABUF_SIZE, GFP_KERNEL)) == NULL)
    {
        kfree (channel);
        return -ENOMEM;
    }

    pend = virt_to_page(channel->outbuf + PCMOUT_DMABUF_SIZE - 1);
    for (page = virt_to_page(channel->outbuf); page <= pend; page++) mem_map_reserve(page);

    channel->out_hwptr = channel->out_swptr = channel->out_count = 0;
    channel->out_total_bytes = PCMOUT_DMABUF_SIZE;

    if ((channel->inbuf = (void *) kmalloc(PCMIN_DMABUF_SIZE, GFP_KERNEL)) == NULL)
    {
        kfree (channel->outbuf);
        kfree (channel);
        return -ENOMEM;
    }

    pend = virt_to_page(channel->inbuf + PCMIN_DMABUF_SIZE - 1);
    for (page = virt_to_page(channel->inbuf); page <= pend; page++) mem_map_reserve(page);

    channel->in_hwptr = channel->in_swptr = channel->in_count = 0;
    channel->in_total_bytes = PCMIN_DMABUF_SIZE;

    // for sampling rate covert
    if ((channel->pcmout_tempbuf = (void *) kmalloc(TEMPBUF_SIZE, GFP_KERNEL)) == NULL)
    {
        kfree (channel->outbuf);
        kfree (channel->inbuf);
        kfree (channel);
        return -ENOMEM;
    }

    pend = virt_to_page(channel->pcmout_tempbuf + TEMPBUF_SIZE - 1);
    for (page = virt_to_page(channel->pcmout_tempbuf); page <= pend; page++) mem_map_reserve(page);

    if ((channel->pcmin_tempbuf = (void *) kmalloc(TEMPBUF_SIZE, GFP_KERNEL)) == NULL)
    {
        kfree (channel->outbuf);
        kfree (channel->inbuf);
        kfree (channel->pcmout_tempbuf);
        kfree (channel);
        return -ENOMEM;
    }

    pend = virt_to_page(channel->pcmin_tempbuf + TEMPBUF_SIZE - 1);
    for (page = virt_to_page(channel->pcmin_tempbuf); page <= pend; page++) mem_map_reserve(page);

    if ((channel->pcmin_tempbuf1 = (void *) kmalloc(TEMPBUF_SIZE, GFP_KERNEL)) == NULL)
    {
        kfree (channel->outbuf);
        kfree (channel->inbuf);
        kfree (channel->pcmout_tempbuf);
        kfree (channel->pcmin_tempbuf);
        kfree (channel);
        return -ENOMEM;
    }

    pend = virt_to_page(channel->pcmin_tempbuf1 + TEMPBUF_SIZE - 1);
    for (page = virt_to_page(channel->pcmin_tempbuf1); page <= pend; page++) mem_map_reserve(page);
    channel->pcmin_tempbuf1_count = TEMPBUF_SIZE;

    channel->opened=1;

    spin_lock_irqsave(&card->lock, flags);
    if (file->f_mode & FMODE_WRITE) { card->pcmout_count++; channel->trigger |= PCM_ENABLE_OUTPUT; }
    if (file->f_mode & FMODE_READ) { card->pcmin_count++; channel->trigger |= PCM_ENABLE_INPUT; }
    card->channels[i] = channel;
    spin_unlock_irqrestore(&card->lock, flags);
    return 0;
}

static int cmedia_release(struct inode *inode, struct file *file)
{
    struct cmedia_channel *channel = (struct cmedia_channel *)file->private_data;
    struct cmedia_card *card = channel->card;
    unsigned long flags;
    unsigned long tmo;
    int enable;
    u16 wTemp;
    u8 Reg;

    //printk("cmedia_release\n");
    if (channel->PhysicalChannels == 4) 
    {
        card->FourChannelCount--;
        if (card->FourChannelCount==0 && card->SixChannelCount==0 && AUTO_PHONEJACK_SHARE)
        {
            card->LineInAsRear = 0;
            Reg=AC97REG_MULTI_CHANNEL_CTRL;
            if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4941) 
                Reg=AC97REG_CM9738_EXT1;
            wTemp = card->ReadRegisterWord(card, Reg);
            wTemp &= (~0x0400);
            card->WriteRegisterWord(card, Reg, wTemp);
        }
    }
    else if (channel->PhysicalChannels == 6) 
    {
        card->SixChannelCount--;
        if (card->FourChannelCount==0 && card->SixChannelCount==0 && AUTO_PHONEJACK_SHARE)
        {
            card->LineInAsRear = 0;
            Reg=AC97REG_MULTI_CHANNEL_CTRL;
            if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4941) 
                Reg=AC97REG_CM9738_EXT1;
            wTemp = card->ReadRegisterWord(card, Reg);
            wTemp &= (~0x0400);
            card->WriteRegisterWord(card, Reg, wTemp);
        }

        if (card->SixChannelCount==0 && AUTO_PHONEJACK_SHARE)
        {
            card->MicAsCenterBass = 0;

            wTemp = card->ReadRegisterWord(card, 0x20);
            if (FORCE_MICSELECT2_MODE) wTemp |= 0x0100;
            else wTemp &= ~0x0100;
            card->WriteRegisterWord(card, 0x20, wTemp);

            wTemp = card->ReadRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL);
            wTemp &= (~0x3000);
            wTemp |= 0x2000;
            card->WriteRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL, wTemp);
        }
    }

    spin_lock_irqsave(&card->lock, flags);
    card->channels[channel->virt] = NULL;
    enable = channel->enable;
    channel->enable = channel->opened = 0;
    if (file->f_mode & FMODE_WRITE) card->pcmout_count--;
    if (file->f_mode & FMODE_READ) card->pcmin_count--;

    /* stop DMA state machine and free DMA buffers/channels */
    if((enable & ADC_RUNNING) && (card->pcmin_count == 0)) card->StopADC(card);
    if((enable & DAC_RUNNING) && (card->pcmout_count == 0)) card->DrainDAC(card);
    card->IsAC3 -= channel->IsAC3;
    if (card->IsAC3 <= 0)
    {
        if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4961) 
        {
            wTemp = card->ReadRegisterWord(card, AC97REG_RESERVED2);
            wTemp &= ~0x0002;
            card->WriteRegisterWord(card, AC97REG_RESERVED2, wTemp);

            if (card->mixer_state[SOUND_MIXER_PCM] != 0)
            {
                wTemp = card->ReadRegisterWord(card, AC97REG_PCMOUT_VOLUME);
                wTemp &= ~0x8000;
                card->WriteRegisterWord(card, AC97REG_PCMOUT_VOLUME, wTemp);
            }
        }
        card->IsAC3 = 0;
    }
    spin_unlock_irqrestore(&card->lock, flags);

    tmo = PROCESS_TIMER*10;
    if (file->f_mode & FMODE_WRITE) interruptible_sleep_on_timeout(&channel->out_wait, tmo);
    else interruptible_sleep_on_timeout(&channel->in_wait, tmo);

    kfree (channel->outbuf);
    kfree (channel->inbuf);
    kfree (channel->pcmout_tempbuf);
    kfree (channel->pcmin_tempbuf);
    kfree (channel->pcmin_tempbuf1);
    kfree(channel);

    return 0;
}

static /*const*/ struct file_operations cmedia_audio_fops = {
	owner:		THIS_MODULE,
	llseek:		cmedia_llseek,
	read:		cmedia_read,
	write:		cmedia_write,
	ioctl:		cmedia_ioctl,
	open:		cmedia_open,
	release:	cmedia_release,
};

/* OSS /dev/mixer file operation methods */

static int cmedia_open_mixdev(struct inode *inode, struct file *file)
{
    int i;
    int minor = MINOR(inode->i_rdev);
    struct cmedia_card *card = devs;

    if (card != NULL) 
    {
        for (i = 0; i < 50 && card->initializing; i++) 
        {
            current->state = TASK_UNINTERRUPTIBLE;
            schedule_timeout(HZ/20);
        }

        if (!card->initializing)
        {
            if (card->dev_mixer == minor) 
            {
                file->private_data = card;
                return 0;
            }
        }
    }

    return -ENODEV;
}

static int cmedia_ioctl_mixdev(struct inode *inode, struct file *file, unsigned int cmd,
				unsigned long arg)
{
    int i, val = 0;
    struct cmedia_card *card = devs;

    if (cmd == SOUND_MIXER_INFO) 
    {
        mixer_info info;
        if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4941)
        {
             strncpy(info.id, chip_names[0], sizeof(info.id));
             strncpy(info.name, chip_names[0], sizeof(info.name));
        }
        else if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4961)
        {
             strncpy(info.id, chip_names[1], sizeof(info.id));
             strncpy(info.name, chip_names[1], sizeof(info.name));
        }
        info.modify_counter = card->modify_counter;
        if (copy_to_user((void *)arg, &info, sizeof(info))) return -EFAULT;
        return 0;
    }

    if (cmd == SOUND_OLD_MIXER_INFO) 
    {
        _old_mixer_info info;
        if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4941)
        {
             strncpy(info.id, chip_names[0], sizeof(info.id));
             strncpy(info.name, chip_names[0], sizeof(info.name));
        }
        else if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4961)
        {
             strncpy(info.id, chip_names[1], sizeof(info.id));
             strncpy(info.name, chip_names[1], sizeof(info.name));
        }
        if (copy_to_user((void *)arg, &info, sizeof(info))) return -EFAULT;
        return 0;
    }

    if (_IOC_TYPE(cmd) != 'M' || _SIOC_SIZE(cmd) != sizeof(int)) return -EINVAL;

    if (cmd == OSS_GETVERSION) return put_user(SOUND_VERSION, (int *)arg);

    if (_SIOC_DIR(cmd) == _SIOC_READ) 
    {
        switch (_IOC_NR(cmd)) 
        {
        case SOUND_MIXER_RECSRC: /* give them the current record source */
            if (!card->RecMask_io) val = 0;
            else val = card->RecMask_io(card, 1, 0);
            break;

        case SOUND_MIXER_DEVMASK: /* give them the supported mixers */
            val = card->supported_mixers;
            break;

        case SOUND_MIXER_RECMASK: /* Arg contains a bit for each supported recording source */
            val = card->record_sources;
            break;

        case SOUND_MIXER_STEREODEVS: /* Mixer channels supporting stereo */
            val = card->stereo_mixers;
            break;

        case SOUND_MIXER_CAPS:
            val = SOUND_CAP_EXCL_INPUT;
            break;

        default: /* read a specific mixer */
            i = _IOC_NR(cmd);

            if (!card_supported_mixer(card, i)) return -EINVAL;

            /* do we ever want to touch the hardware? */
            /* val = card->ReadRegisterWord(card, i); */
            val = card->mixer_state[i];
            break;
        }

        return put_user(val, (int *)arg);
    }

    if (_SIOC_DIR(cmd) == (_SIOC_WRITE|_SIOC_READ)) 
    {
        card->modify_counter++;
        if (get_user(val, (int *)arg)) return -EFAULT;

        switch (_IOC_NR(cmd)) 
        {
        case SOUND_MIXER_RECSRC: /* Arg contains a bit for each recording source */
            if (!card->RecMask_io) return -EINVAL;
            if (!val) return 0;
            if (!(val &= card->record_sources)) return -EINVAL;

            card->RecMask_io(card, 0, val);

            return 0;

        default: /* write a specific mixer */
            i = _IOC_NR(cmd);

            if (!card_supported_mixer(card, i)) return -EINVAL;

            card->Set_Mixer(card, i, val);

            return 0;
        }
    }

    return -EINVAL;
}

static /*const*/ struct file_operations cmedia_mixer_fops = {
	owner:		THIS_MODULE,
	llseek:		cmedia_llseek,
	ioctl:		cmedia_ioctl_mixdev,
	open:		cmedia_open_mixdev,
};

/* CMedia Audio initialisation. */
static int __init cmedia_audio_init(struct cmedia_card *card)
{
    u16 val;
    u8 Reg;
    u16 wTemp;

    //printk(KERN_INFO "CMedia Audio, cmedia_audio_init()\n");

    if (card->Initial(card) < 1)
    {
        //printk(KERN_ERR "CMedia Audio: couldn't initial!\n");
        return 0;
    }

    if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4961)
    {
        card->InitialMixer=InitialMixerCMI9739;
        card->RecMask_io=RecMask_io_CMI9739;
        card->Set_Mixer=Set_Mixer_CMI9739;
        output_channels=6;
    }
    else if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4941)
    {
        card->InitialMixer=InitialMixerCMI9738;
        card->RecMask_io=RecMask_io_CMI9738;
        card->Set_Mixer=Set_Mixer_CMI9738;
        output_channels=4;
    }

    card->LineInAsRear=0;
    card->MicAsCenterBass=0;
    if (FORCE_XEAR_MODE) card->XearMode=1;
    else card->XearMode=0;
    card->FourChannelCount=0;
    card->SixChannelCount=0;

    if (FORCE_LINEINASREAR_MODE)
    {
        card->LineInAsRear=1;
        Reg=AC97REG_MULTI_CHANNEL_CTRL;
        if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4941) Reg=0x5a;
        wTemp = card->ReadRegisterWord(card, Reg);
        wTemp |= 0x0400;
        card->WriteRegisterWord(card, Reg, wTemp);
    }

    wTemp = card->ReadRegisterWord(card, 0x20);
    card->MicSelect2=0;
    wTemp &= ~0x0100;
    if (FORCE_MICSELECT2_MODE) { card->MicSelect2=1; wTemp |= 0x0100; }
    card->WriteRegisterWord(card, 0x20, wTemp);

    if (FORCE_MICASBASS_MODE && card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4961)
    {
        card->MicAsCenterBass=1;
        wTemp = card->ReadRegisterWord(card, 0x20);
        wTemp |= 0x0100;
        card->WriteRegisterWord(card, 0x20, wTemp);

        wTemp = card->ReadRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL);
        wTemp &= (~0x3000);
        wTemp |= 0x1000;
        card->WriteRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL, wTemp);
    }

    if (card->InitialMixer) card->InitialMixer(card);

    if ((card->dev_mixer = register_sound_mixer(&cmedia_mixer_fops, -1)) < 0) 
    {
        //printk(KERN_ERR "CMedia Audio: couldn't register mixer!\n");
        return 0;
    }

    // Set GPIO and PC BEEP
    if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4961)
    {
        val = card->ReadRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL);
        card->WriteRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL, val&(~0x8000));

        card->WriteRegisterWord(card, 0x70, 0x0100);
        card->WriteRegisterWord(card, 0x72, 0x0020);

        val = card->ReadRegisterWord(card, 0x70);
        printk(KERN_ERR "Set GPIO: 0x70 %x!\n", val);
        val = card->ReadRegisterWord(card, 0x72);
        printk(KERN_ERR "Set GPIO: 0x72 %x!\n", val);
    }

    return 1;
}

/* install the driver, we do not allocate hardware channel nor DMA buffer now, they are defered 
   until "ACCESS" time (in prog_dmabuf called by open/read/write/ioctl/mmap) */
   
static int __init cmedia_probe(struct pci_dev *pci_dev, const struct pci_device_id *pci_id)
{
    struct cmedia_card *card;
    struct page *page, *pend;

    if (pci_enable_device(pci_dev)) return -EIO;

    if (pci_set_dma_mask(pci_dev, CMEDIA_DMA_MASK)) 
    {
        //printk(KERN_ERR "CMedia Audio: architecture does not support 32bit PCI busmaster DMA\n");
        return -ENODEV;
    }

    if ((card = kmalloc(sizeof(struct cmedia_card), GFP_KERNEL)) == NULL) 
    {
        //printk(KERN_ERR "CMedia Audio: out of memory\n");
        return -ENOMEM;
    }
    memset(card, 0, sizeof(*card));

    pend = virt_to_page(card + sizeof(struct cmedia_card) - 1);
    for (page = virt_to_page(card); page <= pend; page++) mem_map_reserve(page);

    card->initializing = 1;
    card->iobase = pci_resource_start (pci_dev, 1);
    card->opbase = pci_resource_start (pci_dev, 0);
    card->pci_dev = pci_dev;
    card->pci_id = pci_id->device;
    card->driver_data = pci_id->driver_data;
    spin_lock_init(&card->lock);
    devs = card;

    pci_set_master(pci_dev);

    printk(KERN_INFO "CMedia Audio: %s found at IO 0x%04lx and 0x%04lx\n", card_names[pci_id->driver_data], card->iobase, card->opbase);

    if (alloc_dmabuf(card))
    {
        kfree(card);
        return -ENOMEM;
    }

    /* point to correct functions */
    switch(card->driver_data)
    {
    case INTELICH2:
    case INTELICH4:
    case INTELICH5:
    case SIS7012:
    case NVIDIA_NFORCE:
    case AMD8111:
        card->RequestRegion = RequestRegion_INTEL_ICHx;
        card->ReleaseRegion = ReleaseRegion_INTEL_ICHx;
        card->Initial = Initial_INTEL_ICHx;
        card->ReadRegisterWord = ReadRegisterWord_INTEL_ICHx;
        card->WriteRegisterWord = WriteRegisterWord_INTEL_ICHx;
        card->StartADC = StartADC_INTEL_ICHx;
        card->StopADC = StopADC_INTEL_ICHx;
        card->StartDAC = StartDAC_INTEL_ICHx;
        card->DrainDAC = DrainDAC_INTEL_ICHx;
        card->UpdatePtr = UpdatePtr_INTEL_ICHx;
        card->ProgramDMABuffer = ProgramDMABuffer_INTEL_ICHx;
        card->ConvertPCMOUT = ConvertPCMOUT_INTEL_ICHx;
        break;

    case ALI5451:
        card->RequestRegion = RequestRegion_ALI_5451;
        card->ReleaseRegion = ReleaseRegion_ALI_5451;
        card->Initial = Initial_ALI_5451;
        card->ReadRegisterWord = ReadRegisterWord_ALI_5451;
        card->WriteRegisterWord = WriteRegisterWord_ALI_5451;
        card->StartADC = StartADC_ALI_5451;
        card->StopADC = StopADC_ALI_5451;
        card->StartDAC = StartDAC_ALI_5451;
        card->DrainDAC = DrainDAC_ALI_5451;
        card->UpdatePtr = UpdatePtr_ALI_5451;
        card->ProgramDMABuffer = ProgramDMABuffer_ALI_5451;
        card->ConvertPCMOUT = ConvertPCMOUT_ALI_5451;
        break;

    case VIA8233:      
    case VIA82686:     
        card->RequestRegion = RequestRegion_VIA_82cxxx;
        card->ReleaseRegion = ReleaseRegion_VIA_82cxxx;
        card->Initial = Initial_VIA_82cxxx;
        card->ReadRegisterWord = ReadRegisterWord_VIA_82cxxx;
        card->WriteRegisterWord = WriteRegisterWord_VIA_82cxxx;
        card->StartADC = StartADC_VIA_82cxxx;
        card->StopADC = StopADC_VIA_82cxxx;
        card->StartDAC = StartDAC_VIA_82cxxx;
        card->DrainDAC = DrainDAC_VIA_82cxxx;
        card->UpdatePtr = UpdatePtr_VIA_82cxxx;
        card->ProgramDMABuffer = ProgramDMABuffer_VIA_82cxxx;
        card->ConvertPCMOUT = ConvertPCMOUT_VIA_82cxxx;
        break;

    case SIS7018:
        card->RequestRegion = RequestRegion_SiS_7018;
        card->ReleaseRegion = ReleaseRegion_SiS_7018;
        card->Initial = Initial_SiS_7018;
        card->ReadRegisterWord = ReadRegisterWord_SiS_7018;
        card->WriteRegisterWord = WriteRegisterWord_SiS_7018;
        card->StartADC = StartADC_SiS_7018;
        card->StopADC = StopADC_SiS_7018;
        card->StartDAC = StartDAC_SiS_7018;
        card->DrainDAC = DrainDAC_SiS_7018;
        card->UpdatePtr = UpdatePtr_SiS_7018;
        card->ProgramDMABuffer = ProgramDMABuffer_SiS_7018;
        card->ConvertPCMOUT = ConvertPCMOUT_SiS_7018;
        break;
    }
    
    /* claim our iospace and irq */
    card->RequestRegion(card);

    /* register /dev/dsp */
    if ((card->dev_audio = register_sound_dsp(&cmedia_audio_fops, -1)) < 0) 
    {
        //printk(KERN_ERR "CMedia Audio: couldn't register DSP device!\n");
        card->ReleaseRegion(card);
        //free_irq(card->irq, card);
        kfree(card);
        return -ENODEV;
    }

    /* initialize cmedia audio and register /dev/mixer */
    if (cmedia_audio_init(card) <= 0) 
    {
        unregister_sound_dsp(card->dev_audio);
        card->ReleaseRegion(card);
        //free_irq(card->irq, card);
        kfree(card);
        return -ENODEV;
    }

    pci_dev->driver_data = card;

    /* initail the DMA buffer */
    card->ProgramDMABuffer(card);

    card->initializing = 0;

    return 0;
}

static void __exit cmedia_remove(struct pci_dev *pci_dev)
{
    u16 val;
    unsigned long flags;
    struct cmedia_card *card = (struct cmedia_card *) pci_dev->driver_data;

    del_timer(&cmedia_timer);

    spin_lock_irqsave(&card->lock, flags);

    /* stop everything */
    if (card->dma_state & DAC_RUNNING) card->DrainDAC(card);
    if (card->dma_state & ADC_RUNNING) card->StopADC(card);

    devs = NULL;
    spin_unlock_irqrestore(&card->lock, flags);

    // Set GPIO and PC BEEP
    if (card->ac97_vender_id1 == 0x434d && card->ac97_vender_id2 == 0x4961)
    {
        val = card->ReadRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL);
        card->WriteRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL, val|0x8000);

        card->WriteRegisterWord(card, 0x70, 0x0100);
        card->WriteRegisterWord(card, 0x72, 0x0021);

        val = card->ReadRegisterWord(card, 0x70);
        printk(KERN_ERR "Set GPIO: 0x70 %x!\n", val);
        val = card->ReadRegisterWord(card, 0x72);
        printk(KERN_ERR "Set GPIO: 0x72 %x!\n", val);
    }

    card->ReleaseRegion(card);

    dealloc_dmabuf(card);

    /* unregister audio devices */
    unregister_sound_mixer(card->dev_mixer);

    unregister_sound_dsp(card->dev_audio);
    kfree(card);
}

MODULE_AUTHOR("Min-Chih Feng");
MODULE_DESCRIPTION("C-Media audio support");
MODULE_LICENSE("GPL");

#define CMEDIA_MODULE_NAME "cmedia_audio"

static struct pci_driver cmedia_pci_driver = {
    name:	CMEDIA_MODULE_NAME,
    id_table:	cmedia_pci_tbl,
    probe:	cmedia_probe,
    remove:	cmedia_remove,
};

static int __init cmedia_init_module (void)
{
    printk(KERN_INFO "\n\nCMedia Audio, version " DRIVER_VERSION ", " __TIME__ " " __DATE__ "\n");
    //printk(KERN_INFO "CMedia Audio, cmedia_init_module()\n");

    /* No PCI bus in this machine! */
    if (!pci_present()) return -ENODEV;

    if (!pci_register_driver(&cmedia_pci_driver)) 
    {
        pci_unregister_driver(&cmedia_pci_driver);
        return -ENODEV;
    }

    init_timer(&cmedia_timer);
    cmedia_timer.function = cmedia_timeout;
    cmedia_timer.data = (unsigned long)NULL;
    cmedia_timer.expires = jiffies + PROCESS_TIMER;
    add_timer(&cmedia_timer);

    return 0;
}

static void __exit cmedia_cleanup_module (void)
{
    pci_unregister_driver(&cmedia_pci_driver);

    printk(KERN_INFO "CMedia Audio exit!\n\n");
}

module_init(cmedia_init_module);
module_exit(cmedia_cleanup_module);
