#include "cmaudio.h"

/* interrupts for a dma engine */
#define DMA_INT_FIFO            (1<<4)  /* fifo under/over flow */
#define DMA_INT_COMPLETE        (1<<3)  /* buffer read/write complete and ioc set */
#define DMA_INT_LVI             (1<<2)  /* last valid done */
#define DMA_INT_CELV            (1<<1)  /* last valid is current */
#define DMA_INT_DCH             (1)     /* DMA Controller Halted (happens on LVI interrupts) */
#define DMA_INT_MASK (DMA_INT_FIFO|DMA_INT_COMPLETE|DMA_INT_LVI)

// Native audio bus master control registers (offsets)
#define PI_BDBAR         0x00 // PCM In Buffer Descriptor Base Address Register
#define PI_CIV           0x04 // PCM In Current Index Value
#define PI_LVI           0x05 // PCM In Last Valid Index
#define PI_SR            0x06 // PCM In Status Register
#define PI_PICB          0x08 // PCM In Position In Current Buffer
#define PI_PIV           0x0A // PCM In Prefetch Index Value
#define PI_CR            0x0B // PCM In Control Register
#define PO_BDBAR         0x10 // PCM Out Buffer Descriptor Base Address Register
#define PO_CIV           0x14 // PCM Out Current Index Value
#define PO_LVI           0x15 // PCM Out Last Valid Index
#define PO_SR            0x16 // PCM Out Status Register
#define PO_PICB          0x18 // PCM Out Position In Current Buffer
#define PO_PIV           0x1A // PCM Out Prefetch Index Value
#define PO_CR            0x1B // PCM Out Control Register
#define MC_BDBAR         0x20 // Mic In Buffer Descriptor Base Address Register
#define MC_CIV           0x24 // Mic In Current Index Value
#define MC_LVI           0x25 // Mic In Last Valid Index
#define MC_SR            0x26 // Mic In Status Register
#define MC_PICB          0x28 // Mic In Position In Current Buffer
#define MC_PIV           0x2A // Mic In Prefetch Index Value
#define MC_CR            0x2B // Mic In Control Register
#define GLOB_CNT         0x2C // Global Control
#define GLOB_STA         0x30 // Global Status
#define CAS              0x34 // Codec Access Semiphore

#define OFF_BDBAR        0x00 // Buffer Descriptor Base Address Register Offset
#define OFF_CIV          0x04 // Current Index Value Offset
#define OFF_LVI          0x05 // Last Valid Index Offset
#define OFF_SR           0x06 // Status Register Offset
#define OFF_PICB         0x08 // Position In Current Buffer Offset
#define OFF_PIV          0x0A // Prefetch Index Value Offset
#define OFF_CR           0x0B // Control Register Offset

#define INT_PO           0x00000040   // PCM Out Interrupt
#define INT_PI           0x00000020   // PCM In Interrupt

void RequestRegion_INTEL_ICHx(struct cmedia_card *card)
{
    request_region(card->iobase, 64, card_names[card->driver_data]);
    request_region(card->opbase, 256, card_names[card->driver_data]);
}

void ReleaseRegion_INTEL_ICHx(struct cmedia_card *card)
{
    release_region(card->iobase, 64);
    release_region(card->opbase, 256);
}

int Initial_INTEL_ICHx(struct cmedia_card *card)
{
    u16 eid;
    int i=0;
    u32 reg;

    reg = inl(card->iobase + GLOB_CNT);
	
    if((reg&2)==0)	/* Cold required */
        reg|=2;
    else
        reg|=4;	/* Warm */
		
    reg&=~8;	/* ACLink on */
    outl(reg , card->iobase + GLOB_CNT);
	
    while(i<10)
    {
        if((inl(card->iobase+GLOB_CNT)&4)==0) break;
        current->state = TASK_UNINTERRUPTIBLE;
        schedule_timeout(HZ/20);
        i++;
    }

    if(i==10)
    {
        printk(KERN_ERR "CMedia Audio: AC'97 reset failed.\n");
        return 0;
    }

    current->state = TASK_UNINTERRUPTIBLE;
    schedule_timeout(HZ/5);
		
    inw(card->opbase);

    /* power up everything, modify this when implementing power saving */
    card->WriteRegisterWord(card, AC97_POWER_CONTROL, 
        card->ReadRegisterWord(card, AC97_POWER_CONTROL) & ~0x7f00);

    /* wait for analog ready */
    for (i=10; i && ((card->ReadRegisterWord(card, AC97_POWER_CONTROL) & 0xf) != 0xf); i--)
    {
        current->state = TASK_UNINTERRUPTIBLE;
        schedule_timeout(HZ/20);
    }

    /* Don't attempt to get eid until powerup is complete */
    eid = card->ReadRegisterWord(card, AC97_EXTENDED_ID);
		
    if(eid==0xFFFFFF)
    {
        printk(KERN_WARNING "CMedia Audio: no codec attached ?\n");
        return 0;
    }
		
    card->ac97_features = eid;
				
    /* Now check the codec for useful features to make up for
       the dumbness of the 810 hardware engine */

    if(eid&0x0001)
    {
        printk(KERN_WARNING "CMedia Audio: Codec refused to allow VRA, using 48Khz only.\n");
        card->ac97_features&=~1;
    }

    card->ac97_vender_id1 = card->ReadRegisterWord(card, AC97REG_VENDOR_ID1);   		
    printk(KERN_INFO "CMedia Audio: AC97REG_VENDOR_ID1=%x\n", card->ac97_vender_id1);
    card->ac97_vender_id2 = card->ReadRegisterWord(card, AC97REG_VENDOR_ID2);   		
    printk(KERN_INFO "CMedia Audio: AC97REG_VENDOR_ID2=%x\n", card->ac97_vender_id2);

    if (card->ac97_vender_id1 != 0x434d)
    {
        printk(KERN_ERR "CMedia Audio: CMedia codec not found.\n");
        return 0;
    }

    return 1;
}

u16 ReadRegisterWord_INTEL_ICHx(struct cmedia_card *card, u8 reg)
{
    int count = 100;

    while(count-- && (inb(card->iobase + CAS) & 1)) udelay(1);
    return inw(card->opbase + (reg&0x7f));
}

void WriteRegisterWord_INTEL_ICHx(struct cmedia_card *card, u8 reg, u16 data)
{
    int count = 100;

    while(count-- && (inb(card->iobase + CAS) & 1)) udelay(1);
    outw(data, card->opbase + (reg&0x7f));
}

void StartADC_INTEL_ICHx(struct cmedia_card *card)
{
    //u16 val;
    unsigned long flags;

    if (!(card->dma_state & ADC_RUNNING)) 
    {
        spin_lock_irqsave(&card->lock, flags);
        card->dma_state |= ADC_RUNNING;
        outb(1, card->iobase + PI_CR);
        spin_unlock_irqrestore(&card->lock, flags);
    }
}

/* Stop recording (lock held) */
void StopADC_INTEL_ICHx(struct cmedia_card *card)
{
    card->dma_state &= ~ADC_RUNNING;
    outb(0, card->iobase + PI_CR);

    // wait for the card to acknowledge shutdown 
    while( inb(card->iobase + PI_CR) != 0 ) {};

    // now clear any latent interrupt bits (like the halt bit) 
    outb( inb(card->iobase + PI_SR), card->iobase + PI_SR );
    outl( inl(card->iobase + GLOB_STA) & INT_PI, card->iobase + GLOB_STA);

    outb(0, card->iobase+PI_CIV);
    outb(SG_LEN-1, card->iobase+PI_LVI);

    memset(card->pcmin_rawbuf, 0, PCMIN_DMABUF_SIZE);
}

void StartDAC_INTEL_ICHx(struct cmedia_card *card)
{
    u16 val;
    unsigned long flags;
    if (!(card->dma_state & DAC_RUNNING)) 
    {
        spin_lock_irqsave(&card->lock, flags);

        card->dma_state |= DAC_RUNNING;
        outb(inb(card->iobase + PO_CR)|(0x01), card->iobase + PO_CR);

        spin_unlock_irqrestore(&card->lock, flags);
    }

    if (card->ac97_vender_id2==0x4961 && card->bMonitorSPDIF)
    {
        val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
        card->WriteRegisterWord(card , AC97REG_SPDIF_FUNCTION, val&(~0x08));
    }
}

/* stop playback (lock held) */
static inline void __StopDAC_INTEL_ICHx(struct cmedia_card *card)
{
    u16 val;
    card->dma_state &= ~DAC_RUNNING;
    outb(inb(card->iobase + PO_CR)&(~0x01), card->iobase + PO_CR);

    // wait for the card to acknowledge shutdown
    while( inb(card->iobase + PO_CR) != 0 ) {};

    // now clear any latent interrupt bits (like the halt bit)
    outb( inb(card->iobase + PO_SR), card->iobase + PO_SR );
    outl( inl(card->iobase + GLOB_STA) & INT_PO, card->iobase + GLOB_STA);

    memset(card->pcmout_rawbuf, 0, PCMOUT_DMABUF_SIZE);
    memset(card->dmaout_rawbuf, 0, PCMOUT_DMABUF_SIZE);

    if (card->ac97_vender_id2==0x4961 && card->bMonitorSPDIF)
    {
        val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
        card->WriteRegisterWord(card , AC97REG_SPDIF_FUNCTION, val|0x08);
    }
}

/* update buffer manangement pointers */
void UpdatePtr_INTEL_ICHx(struct cmedia_card *card)
{
    int port; 
    unsigned long civ, BUFFER_SIZE;
    if(!card->dma_state) return;

    port = card->iobase;
    if (card->dma_state & ADC_RUNNING) 
    {
        do
        {
            civ = inb(port+PI_CIV);
            if (card->driver_data == SIS7012) 
            {
                card->pcmin_hwptr = inl(port+PI_CIV);
                card->pcmin_hwptr = (PCMIN_DMABUF_SIZE>>5)-(card->pcmin_hwptr>>16);
            }
            else 
            {
                card->pcmin_hwptr = inw(port+PI_PICB);
                card->pcmin_hwptr = (PCMIN_DMABUF_SIZE>>5)-(card->pcmin_hwptr<<1);
            }
            card->pcmin_hwptr -= card->pcmin_hwptr%4;
            card->pcmin_hwptr += civ*(PCMIN_DMABUF_SIZE>>5);
        } while (civ != inb(port+PI_CIV));

        if (civ == 0) civ = SG_LEN-1; else civ--;
        outb(civ,port+PI_LVI);
    }

    if (card->dma_state & DAC_RUNNING) 
    {
        BUFFER_SIZE=PCMOUT_DMABUF_SIZE*output_channels/6;

        do
        {
            civ = inb(port+PO_CIV);
            if (card->driver_data == SIS7012) 
            {
                card->pcmout_hwptr = inl(port+PO_CIV);
                card->pcmout_hwptr = (BUFFER_SIZE>>5)-(card->pcmout_hwptr>>16);
            }
            else
            { 
                card->pcmout_hwptr = inw(port+PO_PICB);
                card->pcmout_hwptr = (BUFFER_SIZE>>5)-(card->pcmout_hwptr<<1);
            }
            
            card->pcmout_hwptr += civ*(BUFFER_SIZE>>5);

            if (output_channels == 2)
            {
                card->pcmout_hwptr -= card->pcmout_hwptr%4;
                card->pcmout_hwptr = card->pcmout_hwptr*3;
            }
            else if (output_channels == 4) 
            { 
                card->pcmout_hwptr -= card->pcmout_hwptr%8;
                card->pcmout_hwptr = card->pcmout_hwptr*3/2;
            }
            else if (output_channels == 6) card->pcmout_hwptr -= card->pcmout_hwptr%12;

        } while (civ != inb(port+PO_CIV));

        if (civ == 0) civ = SG_LEN-1; else civ--;
        outb(civ,port+PO_LVI);
    }
}

int DrainDAC_INTEL_ICHx(struct cmedia_card *card)
{
    unsigned long flags;
    u32 reg;

    if (!(card->dma_state&DAC_RUNNING)) return 0;

    // Set PCM output channels (for nVidia FIFO not clean problem)
    reg = inl(card->iobase + GLOB_CNT);
    if (card->driver_data != SIS7012) outl(reg&(~0x00300000), card->iobase + GLOB_CNT);

    __StopDAC_INTEL_ICHx(card);

    // Set PCM output channels (for nVidia FIFO not clean problem)
    if (card->driver_data != SIS7012) outl(reg, card->iobase + GLOB_CNT);

    spin_lock_irqsave(&card->lock, flags);
    ProgramDMABuffer_INTEL_ICHx(card);
    spin_unlock_irqrestore(&card->lock, flags);

    return 0;
}

int ProgramDMABuffer_INTEL_ICHx(struct cmedia_card *card)
{
    struct sg_item *sg;
    unsigned long flags=0, reg, BUFFER_SIZE;
    int i;
    u32 busaddr, dwTemp;

    spin_lock_irqsave(&card->lock, flags);
    card->pcmout_count = card->pcmin_count = 0;
    card->pcmout_swptr = card->pcmout_hwptr = card->pcmin_swptr = card->pcmin_hwptr = 0;

    memset(card->pcmout_rawbuf, 0, PCMOUT_DMABUF_SIZE);
    memset(card->dmaout_rawbuf, 0, PCMOUT_DMABUF_SIZE);
    memset(card->pcmin_rawbuf, 0, PCMIN_DMABUF_SIZE);

    dwTemp = inl(card->iobase+GLOB_STA);
    if((dwTemp&0x00200000)==0 && output_channels == 6) output_channels = 4;
    if((dwTemp&0x00100000)==0 && output_channels == 4) output_channels = 2;
 
    sg=&card->pcmout_buffer_descriptor.sg[0];
    busaddr=virt_to_bus(card->dmaout_rawbuf);
    BUFFER_SIZE=PCMOUT_DMABUF_SIZE*output_channels/6;
    for(i=0; i<SG_LEN; i++) 
    {
        sg->busaddr=busaddr+(BUFFER_SIZE>>5)*i;
        if (card->driver_data == SIS7012) sg->control=BUFFER_SIZE>>5;
        else sg->control=BUFFER_SIZE>>6;
        sg->control|=CON_BUFPAD;
        sg++;
    }

    outb(2, card->iobase+PO_CR);   // reset DMA machine 
    outl(virt_to_bus(&card->pcmout_buffer_descriptor.sg[0]), card->iobase+PO_BDBAR);
    outb(0, card->iobase+PO_CIV);
    outb(SG_LEN-1, card->iobase+PO_LVI);
    card->pcmout_count = 0;

    // Set PCM output channels
    reg = inl(card->iobase + GLOB_CNT);
    if (card->driver_data == SIS7012) 
    {
        reg &= ~0x000000c0;
        if (output_channels == 6) reg |= 0x00000080;
        else if (output_channels == 4) reg |= 0x00000040;
    }
    else
    {
        reg &= ~0x00300000;
        if (output_channels == 6) reg |= 0x00200000;
        else if (output_channels == 4) reg |= 0x00100000;
    }
    outl(reg, card->iobase + GLOB_CNT);

    // -------------------------------------------------------------------------------------------------
    sg=&card->pcmin_buffer_descriptor.sg[0];
    busaddr=virt_to_bus(card->pcmin_rawbuf);
    for(i=0; i<SG_LEN; i++) 
    {
        sg->busaddr=busaddr+(PCMIN_DMABUF_SIZE>>5)*i;
        if (card->driver_data == SIS7012) sg->control=PCMIN_DMABUF_SIZE>>5;
        else sg->control=PCMIN_DMABUF_SIZE>>6;
        sg++;
    }

    outb(2, card->iobase+PI_CR);   // reset DMA machine 
    outl(virt_to_bus(&(card->pcmin_buffer_descriptor.sg[0])), card->iobase+PI_BDBAR);
    outb(0, card->iobase+PI_CIV);
    outb(SG_LEN-1, card->iobase+PI_LVI);
    card->pcmin_count = 0;

    spin_unlock_irqrestore(&card->lock, flags);

    return 0;
}


void ConvertPCMOUT_INTEL_ICHx(struct cmedia_card *card)
{
    short *ptrSource, *ptrTarget;
    unsigned long i, len1, len2;
    len1=len2=0;

    if (card->pcmout_swptr < card->pcmout_hwptr)
    {
        len1 = ((card->pcmout_hwptr-card->pcmout_swptr)>>1)/6;
    }
    else
    {
        len1 = ((PCMOUT_DMABUF_SIZE-card->pcmout_swptr)>>1)/6;
        len2 = ((card->pcmout_hwptr)>>1)/6;
    }

    if (output_channels==2) // Front Left/Right
    {
        ptrSource=(short *)(card->pcmout_rawbuf+card->pcmout_swptr);
        ptrTarget=(short *)(card->dmaout_rawbuf+card->pcmout_swptr/3);
        for(i=0; i<len1; i++)
        {
            *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
            *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
            ptrSource++;ptrSource++;ptrSource++;ptrSource++;
        }

        ptrSource=(short *)(card->pcmout_rawbuf);
        ptrTarget=(short *)(card->dmaout_rawbuf);
        for(i=0; i<len2; i++)
        {
            *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
            *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
            ptrSource++;ptrSource++;ptrSource++;ptrSource++;
        }
    }

    if (output_channels==4) // Front Left/Right, Rear Left/Right
    {
        ptrSource=(short *)(card->pcmout_rawbuf+card->pcmout_swptr);
        ptrTarget=(short *)(card->dmaout_rawbuf+card->pcmout_swptr*2/3);
        if (card->XearMode)
        {
            for(i=0; i<len1; i++)
            {
                *(ptrTarget+2)=*ptrSource; ptrSource++; 
                *(ptrTarget+3)=*ptrSource; ptrSource++;
                *ptrTarget=*ptrSource; ptrSource++;
                *(ptrTarget+1)=*ptrSource; ptrSource++;
                ptrTarget++;ptrTarget++;ptrTarget++;ptrTarget++;
                ptrSource++;ptrSource++;
            }
        }
        else
        {
            for(i=0; i<len1; i++)
            {
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++;
            }
        }

        ptrSource=(short *)(card->pcmout_rawbuf);
        ptrTarget=(short *)(card->dmaout_rawbuf);
        if (card->XearMode)
        {
            for(i=0; i<len2; i++)
            {
                *(ptrTarget+2)=*ptrSource; ptrSource++; 
                *(ptrTarget+3)=*ptrSource; ptrSource++;
                *ptrTarget=*ptrSource; ptrSource++;
                *(ptrTarget+1)=*ptrSource; ptrSource++;
                ptrTarget++;ptrTarget++;ptrTarget++;ptrTarget++;
                ptrSource++;ptrSource++;
            }
        }
        else
        {
            for(i=0; i<len2; i++)
            {
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                *ptrTarget=*ptrSource; ptrSource++; ptrTarget++;
                ptrSource++;ptrSource++;
            }
        }
    }

    if (output_channels==6) // Front Left/Right, Rear Left/Right, Center/Bass
    {
        ptrSource=(short *)(card->pcmout_rawbuf+card->pcmout_swptr);
        ptrTarget=(short *)(card->dmaout_rawbuf+card->pcmout_swptr);
        if (card->XearMode)
        {
            for(i=0; i<len1; i++)
            {
                *(ptrTarget+4)=*ptrSource; ptrSource++; 
                *(ptrTarget+5)=*ptrSource; ptrSource++;
                *(ptrTarget+0)=*ptrSource; ptrSource++;
                *(ptrTarget+1)=*ptrSource; ptrSource++;
                *(ptrTarget+2)=*ptrSource; ptrSource++;
                *(ptrTarget+3)=*ptrSource; ptrSource++;
                ptrTarget+=6;
            }
        }
        else
        {
            for(i=0; i<len1; i++)
            {
                *(ptrTarget+0)=*ptrSource; ptrSource++; 
                *(ptrTarget+1)=*ptrSource; ptrSource++;
                *(ptrTarget+4)=*ptrSource; ptrSource++;
                *(ptrTarget+5)=*ptrSource; ptrSource++;
                *(ptrTarget+2)=*ptrSource; ptrSource++;
                *(ptrTarget+3)=*ptrSource; ptrSource++;
                ptrTarget+=6;
            }
        }

        ptrSource=(short *)(card->pcmout_rawbuf);
        ptrTarget=(short *)(card->dmaout_rawbuf);
        if (card->XearMode)
        {
            for(i=0; i<len2; i++)
            {
                *(ptrTarget+4)=*ptrSource; ptrSource++; 
                *(ptrTarget+5)=*ptrSource; ptrSource++;
                *(ptrTarget+0)=*ptrSource; ptrSource++;
                *(ptrTarget+1)=*ptrSource; ptrSource++;
                *(ptrTarget+2)=*ptrSource; ptrSource++;
                *(ptrTarget+3)=*ptrSource; ptrSource++;
                ptrTarget+=6;
            }
        }
        else
        {
            for(i=0; i<len2; i++)
            {
                *(ptrTarget+0)=*ptrSource; ptrSource++; 
                *(ptrTarget+1)=*ptrSource; ptrSource++;
                *(ptrTarget+4)=*ptrSource; ptrSource++;
                *(ptrTarget+5)=*ptrSource; ptrSource++;
                *(ptrTarget+2)=*ptrSource; ptrSource++;
                *(ptrTarget+3)=*ptrSource; ptrSource++;
                ptrTarget+=6;
            }
        }
    }
}
