#include "cmaudio.h"

/* table to scale scale from OSS mixer value to AC97 mixer register value */	
static struct ac97_mixer_hw ac97_hw_cmi9739[SOUND_MIXER_NRDEVICES]= 
{
	[SOUND_MIXER_VOLUME]  = {AC97_MASTER_VOL_STEREO,32},
	[SOUND_MIXER_PCM]     = {AC97_PCMOUT_VOL,       32},
	[SOUND_MIXER_LINE]    = {AC97_LINEIN_VOL,       32},
	[SOUND_MIXER_MIC]     = {AC97_MIC_VOL,          32},
	[SOUND_MIXER_CD]      = {AC97_CD_VOL,           32},
	[SOUND_MIXER_LINE1]   = {AC97_AUX_VOL,          32},
};

/* this table has default mixer values for all OSS mixers. */
static struct mixer_defaults_cmi9739 {
	int mixer;
	unsigned int value;
} mixer_defaults_cmi9739[SOUND_MIXER_NRDEVICES] = {
	/* all values 0 -> 100 in bytes */
	{SOUND_MIXER_VOLUME,    0x4949},
	{SOUND_MIXER_PCM,	0x4949},
	{SOUND_MIXER_LINE,	0x4949},
	{SOUND_MIXER_MIC,	0x4949},
	{SOUND_MIXER_CD,	0x4949},
	{SOUND_MIXER_LINE1,	0x4949},
	{-1,0}
};

void InitialMixerCMI9739(struct cmedia_card *card)
{
    unsigned int val;
    int i;
    unsigned long flags;

    spin_lock_irqsave(&card->lock, flags);
    card->supported_mixers = SOUND_MASK_VOLUME|SOUND_MASK_PCM|SOUND_MASK_LINE|SOUND_MASK_MIC|
                             SOUND_MASK_CD|SOUND_MASK_LINE1|SOUND_MASK_DIGITAL1;
    card->stereo_mixers=card->supported_mixers&(~SOUND_MASK_MIC);
    card->record_sources=card->supported_mixers&(~SOUND_MASK_PCM);
    spin_unlock_irqrestore(&card->lock, flags);

    /* initialize mixer channel volumes */
    for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) 
    {
        struct mixer_defaults_cmi9739 *md = &mixer_defaults_cmi9739[i];
        if (md->mixer == -1) break;

        if (!card_supported_mixer(card, md->mixer)) continue;

        Set_Mixer_CMI9739(card, md->mixer, md->value);
    }

    // Open Microphone
    card->WriteRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL, 0x3000);

    // Set Recording Gain
    card->WriteRegisterWord(card, 0x1c, 0x0000);

    // Set Surround Mixer Volume
    card->WriteRegisterWord(card, 0x38, 0x0000);

    // Set Center/Base Mixer Volume
    card->WriteRegisterWord(card, 0x36, 0x0000);

    // Set Recording Select
    val = card->RecMask_io(card, 1, 0);
    card->RecMask_io(card, 0, val);

    // Enable S/PDIF out
    val = card->ReadRegisterWord(card, AC97REG_EXT_AUDIO_CTRL);
    card->WriteRegisterWord(card , AC97REG_EXT_AUDIO_CTRL, 0x05c4);
    val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
    card->WriteRegisterWord(card , AC97REG_SPDIF_FUNCTION, 0x0009);
    card->bMonitorSPDIF=1;

    // Enable Mix2Surround
    val = card->ReadRegisterWord(card, AC97REG_MULTI_CHANNEL_CTRL);
    card->WriteRegisterWord(card , AC97REG_MULTI_CHANNEL_CTRL, val|0x0200);
    card->Mix2Surround=1;
}

int RecMask_io_CMI9739(struct cmedia_card *card, int rw, int mask) 
{
    unsigned int val, item;
    struct ac97_mixer_hw *mh;

    if (rw) 
    {
        // check spdif in first
        val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
        if (val&0x10) return (1 << SOUND_MIXER_DIGITAL1);

        // read it from the card
        val = card->ReadRegisterWord(card, AC97_RECORD_SELECT);
        return (1 << ac97_rm2oss[val & 0x07]);
    }

    /* else, write the first set in the mask as the output */	
    /* clear out current set value first (AC97 supports only 1 input!) */

    // check spdif in first
    val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
    if (val&0x10) val = (1 << SOUND_MIXER_DIGITAL1);
    else
    {
        val = card->ReadRegisterWord(card, AC97_RECORD_SELECT) & 0x07;
        val = (1 << ac97_rm2oss[val]);
    }

    if (mask != val) mask &= ~val;

    if (mask == (1 << SOUND_MIXER_DIGITAL1))
    {
        val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
        card->WriteRegisterWord(card, AC97REG_SPDIF_FUNCTION, val|0x10);

        val = 0x0707;
        card->WriteRegisterWord(card, AC97_RECORD_SELECT, val);

        val = 0x8000;
        card->WriteRegisterWord(card , AC97REG_RECORD_GAIN, val);
        return 0;
    }
    else
    {
        val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
        card->WriteRegisterWord(card, AC97REG_SPDIF_FUNCTION, val&~(0x10));
    }

    val = ffs(mask); 
    val = ac97_oss_rm[val-1];
    val |= val << 8;  /* set both channels */

    card->WriteRegisterWord(card, AC97_RECORD_SELECT, val);

    item = ac97_rm2oss[val&0x07];
    mh = &ac97_hw_cmi9739[item];
    val = card->ReadRegisterWord(card , mh->offset);
    switch(item)
    {
    case SOUND_MIXER_MIC:
        val = ((val&0x001f)<<8)|(val&0x801f);
        break;
    default:
        val &= 0x9f1f;
        break;
    }
    val = (val&0x8000)|((~((val&0x1f1f)>>1))&0x0f0f);
    card->WriteRegisterWord(card , AC97REG_RECORD_GAIN, val);

    return 0;
};

/* write the OSS encoded volume to the given OSS encoded mixer, again caller's job to
   make sure all is well in arg land, call with spinlock held */
void write_mixer_9739(struct cmedia_card *card, int oss_channel, unsigned int left, unsigned int right)
{
    u16 val = 0, item;
    struct ac97_mixer_hw *mh=NULL;

    if (oss_channel==SOUND_MIXER_DIGITAL1)
    {
        val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
        if (val&0x10)
        {
            val = 0x8000;
            card->WriteRegisterWord(card , AC97REG_RECORD_GAIN, val);
        }

        if (left == 0 && right == 0) 
        {
            card->bMonitorSPDIF = 0;
            val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
            card->WriteRegisterWord(card , AC97REG_SPDIF_FUNCTION, val&(~0x08));
        }
        else 
        {
            card->bMonitorSPDIF = 1;
            if (card->dma_state & DAC_RUNNING)
            {
                val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
                card->WriteRegisterWord(card , AC97REG_SPDIF_FUNCTION, val&(~0x08));
            }
            else
            {
                val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
                card->WriteRegisterWord(card , AC97REG_SPDIF_FUNCTION, val|0x08);
            }
        }

        return;
    }

    mh = &ac97_hw_cmi9739[oss_channel];

    if(left == 0) 
    {
        val = AC97_MUTE;
    }
    else if (oss_channel == SOUND_MIXER_MIC) 
    {
        val = card->ReadRegisterWord(card , mh->offset) & ~0x801f;
        left = ((100 - left) * mh->scale) / 100;
        if (left >= mh->scale) left = mh->scale-1;
        val |= left;
        /*  the low bit is optional in the tone sliders and masking
            it lets us avoid the 0xf 'bypass'.. */
    }
    else
    {
        right = ((100 - right) * mh->scale) / 100;
        left = ((100 - left) * mh->scale) / 100;
        if (right >= mh->scale) right = mh->scale-1;
        if (left >= mh->scale) left = mh->scale-1;

        val = (left << 8) | right;
    } 

    card->WriteRegisterWord(card, mh->offset, val);

    // check spdif in first
    val = card->ReadRegisterWord(card, AC97REG_SPDIF_FUNCTION);
    if (val&0x10)
    {
        val = 0x8000;
        card->WriteRegisterWord(card , AC97REG_RECORD_GAIN, val);
        return;
    }
    
    // Setting Software Recording Gain
    val = card->ReadRegisterWord(card, AC97_RECORD_SELECT);
    item = ac97_rm2oss[val&0x07];
    mh = &ac97_hw_cmi9739[item];
    val = card->ReadRegisterWord(card , mh->offset);
    switch(item)
    {
    case SOUND_MIXER_MIC:
        if (val&0x8000) val = ((val&0x001f)<<8)|(val&0x001f);
        break;
    default:
        val &= 0x9f1f;
        break;
    }
    val = (val&0x8000)|((~((val&0x1f1f)>>1))&0x0f0f);
    card->WriteRegisterWord(card , AC97REG_RECORD_GAIN, val);
}

/* a thin wrapper for write_mixer */
void Set_Mixer_CMI9739(struct cmedia_card *card, unsigned int oss_mixer, unsigned int val ) 
{
    unsigned int left,right;

    /* cleanse input a little */
    right = ((val >> 8)  & 0xff) ;
    left = (val  & 0xff) ;

    if (right > 100) right = 100;
    if (left > 100) left = 100;

    card->mixer_state[oss_mixer] = (right << 8) | left;
    write_mixer_9739(card, oss_mixer, left, right);
}

