/* ***** BEGIN LICENSE BLOCK ***** 
 * Version: RCSL 1.0/RPSL 1.0 
 *  
 * Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved. 
 *      
 * The contents of this file, and the files included with this file, are 
 * subject to the current version of the RealNetworks Public Source License 
 * Version 1.0 (the "RPSL") available at 
 * http://www.helixcommunity.org/content/rpsl unless you have licensed 
 * the file under the RealNetworks Community Source License Version 1.0 
 * (the "RCSL") available at http://www.helixcommunity.org/content/rcsl, 
 * in which case the RCSL will apply. You may also obtain the license terms 
 * directly from RealNetworks.  You may not use this file except in 
 * compliance with the RPSL or, if you have a valid RCSL with RealNetworks 
 * applicable to this file, the RCSL.  Please see the applicable RPSL or 
 * RCSL for the rights, obligations and limitations governing use of the 
 * contents of the file.  
 *  
 * This file is part of the Helix DNA Technology. RealNetworks is the 
 * developer of the Original Code and owns the copyrights in the portions 
 * it created. 
 *  
 * This file, and the files included with this file, is distributed and made 
 * available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER 
 * EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES, 
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS 
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. 
 * 
 * Technology Compatibility Kit Test Suite(s) Location: 
 *    http://www.helixcommunity.org/content/tck 
 * 
 * Contributor(s): 
 *  
 * ***** END LICENSE BLOCK ***** */ 



// what about error bag
// what about logging

#ifndef _HXTCONFIGAGENTHELPER_H
#define _HXTCONFIGAGENTHELPER_H


#include "ihxtbase.h"
#include "ihxtpropertybag.h"

class CHXTConfigurationAgentHelper: 
	public IHXTConfigurationAgent
{
public:
	
    CHXTConfigurationAgentHelper() 
		: m_pPropertyBag( NULL )
		, m_pFactory( NULL )
		, m_bInitDone( FALSE )
		, m_bAllowChanges(TRUE)
	{
	}

	~CHXTConfigurationAgentHelper()
	{
		HX_RELEASE( m_pPropertyBag );
		HX_RELEASE( m_pFactory );
	}

	STDMETHODIMP Initialize( IHXTPropertyBag* pPropBag, IHXTPropertyBag** ppErrorBag )
	{
		HX_RESULT res = HXR_OK;

		// Allow successful initialization at most once
		if (m_bInitDone  || !m_pFactory || !m_pPropertyBag)
		{
			return HXR_ENC_IMPROPER_STATE;
		}

		// Create new bag to hold props consumed during filter initialization
		IHXTPropertyBag *pConsumedProps = NULL; 
		res = m_pFactory->CreateInstance( IID_IHXTPropertyBag, (void **) &pConsumedProps );

		// Create new bag to hold properties that failed validation
		IHXTPropertyBag *pErrorBag = NULL;
		if (SUCCEEDED(res))
			res = m_pFactory->CreateInstance( IID_IHXTPropertyBag, (void **) &pErrorBag );

		if (SUCCEEDED(res))
			res = OnInitialize( pPropBag, pConsumedProps, pErrorBag ); 

		// Add consumed props to internal prop bag
		if (SUCCEEDED(res)) 
		{						
			IHXTPropertyEnumerator *pEnum = NULL;
			res = pConsumedProps->GetPropertyEnumerator( &pEnum );

			IHXTProperty*  pProp  = NULL;
			HX_RESULT resEnum = HXR_OK;

			if (SUCCEEDED(res))			
				resEnum = pEnum->First( &pProp );

			while (SUCCEEDED(resEnum) && resEnum != HXR_ELEMENT_NOT_FOUND && SUCCEEDED(res))
			{
				res = m_pPropertyBag->SetProperty( pProp );
				HX_RELEASE( pProp );

				if (SUCCEEDED(res))
					resEnum = pEnum->Next( &pProp );
			}
			HX_RELEASE( pEnum );
			HX_RELEASE( pProp );
		}

		// Enumerate through the unconsumed props and dispatch them appropriately
		if (SUCCEEDED(res) && pPropBag) 
		{						
			IHXTPropertyEnumerator *pEnum = NULL;
			res = pPropBag->GetPropertyEnumerator( &pEnum );

			IHXTProperty*  pProp  = NULL;
			HX_RESULT resEnum = HXR_OK;

			if (SUCCEEDED(res))			
				resEnum = pEnum->First( &pProp );

			while (SUCCEEDED(resEnum) && resEnum != HXR_ELEMENT_NOT_FOUND && SUCCEEDED(res))
			{
				// If the property wasn't already consumed, call the overrideable SetProp function
				IHXTProperty *pConsumedProp = NULL;
				HX_RESULT resConsumed = pConsumedProps->GetProperty( pProp->GetKey(), &pConsumedProp );

				if (FAILED(resConsumed))
				{
					switch ( pProp->GetType())
					{
					case eTypeBOOL:
					{
						BOOL bValue = 0;
						res = pPropBag->GetBool( pProp->GetKey(), &bValue );
							
						if (SUCCEEDED(res))
							res = SetBool( pProp->GetKey(), bValue );
						break;
					}				
					case eTypeINT32:
					{
						INT32 lValue = 0;
						res = pPropBag->GetInt( pProp->GetKey(), &lValue);

						if (SUCCEEDED(res))
							res = SetInt( pProp->GetKey(), lValue);
						break;
					}
					case eTypeUINT32:
					{
						UINT32 ulValue = 0;
						res = pPropBag->GetUint(pProp->GetKey(), &ulValue);

						if (SUCCEEDED(res))
							res = SetUint(pProp->GetKey(), ulValue);
						break;
					}

					case eTypeINT64:
					{
						INT64 llValue = 0;
						res = pPropBag->GetInt64(pProp->GetKey(), &llValue);

						if (SUCCEEDED(res))
							res = SetInt64(pProp->GetKey(), llValue);
						break;
					}

					case eTypeDouble:
					{
						double dValue = 0;
						res = pPropBag->GetDouble(pProp->GetKey(), &dValue);

						if (SUCCEEDED(res))
							res = SetDouble(pProp->GetKey(), dValue);
						break;
					}

					case eTypeCString:
					{
						const char* szValue = NULL;
						res = pPropBag->GetString(pProp->GetKey(), &szValue);

						if (SUCCEEDED(res))
							res = SetString( pProp->GetKey(), szValue );
						break;
					}

					case eTypeComposite:
					{
						IHXTPropertyBag*  pTempBag;
						res = pPropBag->GetPropertyBag( pProp->GetKey(), &pTempBag );

						if (SUCCEEDED(res))
							res = SetPropertyBag( pProp->GetKey(), pTempBag );

						HX_RELEASE( pTempBag );
						break;
					}

					case eTypeUnknown:
					{
						IUnknown* pUnknown = NULL;
						res = pPropBag->GetUnknown(pProp->GetKey(), &pUnknown );

						if (SUCCEEDED(res))
							res = SetUnknown(pProp->GetKey(), pUnknown );
						
						HX_RELEASE( pUnknown );
						break;
					}

					// Revisit when more types are added
					default:
						res = SetProperty(pProp);
					};
					
				}
				// Add any properties that failed validation to the error bag
				if (FAILED(res))
					HX_RESULT resAppend = pErrorBag->SetProperty(pProp);

				HX_RELEASE( pConsumedProp );
				resEnum = pEnum->Next( &pProp );
			}
			HX_RELEASE( pEnum );
			HX_RELEASE( pProp );
		}

		// If everything was successful, mark initialization as complete
		if (SUCCEEDED(res))
			m_bInitDone = TRUE;

		if (ppErrorBag)
		{
			*ppErrorBag = pErrorBag;
			(*ppErrorBag)->AddRef();
		}

		HX_RELEASE( pErrorBag );
		HX_RELEASE( pConsumedProps );

		return res;
	}
	

	STDMETHOD_(UINT32, GetCount)()
	{
		return m_pPropertyBag->GetCount();
	}
	
	STDMETHODIMP GetProperty( const char* szName, IHXTProperty** ppProperty ) 
	{
		if (!szName || !ppProperty)
		{
			return HXR_POINTER;
		}
		HX_RESULT res = m_pPropertyBag->GetProperty(szName, ppProperty);
		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to get property: %s", szName);
		}
		return res;
	}

	STDMETHODIMP SetProperty( IHXTProperty *pProperty ) 
	{
		HX_RESULT res = HXR_OK;
		
		if (!pProperty)
		{
			return HXR_POINTER;
		}

		// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}

		if (SUCCEEDED(res))
			res = OnSetProperty(pProperty); 
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetProperty(pProperty); 

		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to set property: %s", pProperty->GetKey());
		}
		return res;
	}


	STDMETHODIMP GetBool( const char* szName, BOOL *pbValue) 
	{
		if (!szName || !pbValue)
		{
			return HXR_POINTER;
		}
		HX_RESULT res = m_pPropertyBag->GetBool(szName, pbValue);
		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to get bool: %s", szName);
		}
		return res; 
	}

	STDMETHODIMP SetBool( const char* szName, BOOL bValue) 
	{
		HX_RESULT res = HXR_OK;
	
		if (!szName)
		{
			return HXR_POINTER;
		}
		
		// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}
		
		if (SUCCEEDED(res)) 
			res = OnSetBool(szName, bValue); 
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetBool(szName, bValue); 

		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to set bool: %s", szName);
		}
		return res;
	}

	STDMETHODIMP GetInt( const char* szName, INT32 *puValue) 
	{
		if (!szName || !puValue)
		{
			return HXR_POINTER;
		}
		HX_RESULT res = m_pPropertyBag->GetInt(szName, puValue);
		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to get int: %s", szName);
		}
		return res; 
	}

	STDMETHODIMP SetInt( const char* szName, INT32 uValue) 
	{
		HX_RESULT res = HXR_OK;
	
		if (!szName)
		{
			return HXR_POINTER;
		}
		
		// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}

		if (SUCCEEDED(res)) 
			res = OnSetInt(szName, uValue); 
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetInt(szName, uValue); 

		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to set int: %s", szName);
		}
		return res;
	}

	STDMETHODIMP GetUint( const char* szName, UINT32* pulValue) 
	{
		if (!szName || !pulValue)
		{
			return HXR_POINTER;
		}
		HX_RESULT res = m_pPropertyBag->GetUint(szName, pulValue);
		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to get uint: %s", szName);
		}
		return res; 
	}

	STDMETHODIMP SetUint( const char* szName, UINT32 uValue) 
	{
		HX_RESULT res = HXR_OK;
	
		if (!szName)
		{
			return HXR_POINTER;
		}
		
		// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}

		if (SUCCEEDED(res)) 
			res = OnSetUint(szName, uValue); 
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetUint(szName, uValue); 

		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to set uint: %s", szName);
		}
		return res;
	}


	STDMETHODIMP GetInt64( const char* szName, INT64 *puValue) 
	{
		if (!szName || !puValue)
		{
			return HXR_POINTER;
		}
		HX_RESULT res = m_pPropertyBag->GetInt64(szName, puValue);
		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to get int64: %s", szName);
		}
		return res;
	}

	STDMETHODIMP SetInt64( const char* szName, INT64 uValue) 
	{
		HX_RESULT res = HXR_OK;

		if (!szName)
		{
			return HXR_POINTER;
		}
				// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}
		
		if (SUCCEEDED(res)) 
			res = OnSetInt64(szName, uValue); 
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetInt64(szName, uValue); 

		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to set int64: %s", szName);
		}
		return res;
	}


	STDMETHODIMP GetDouble( const char* szName, double *puValue) 
	{
		if (!szName || !puValue)
		{
			return HXR_POINTER;
		}
		HX_RESULT res = m_pPropertyBag->GetDouble(szName, puValue);
		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to set double: %s", szName);
		}	
		return res;
	}

	STDMETHODIMP SetDouble( const char* szName, double uValue) 
	{
		HX_RESULT res = HXR_OK;
	
		if (!szName)
		{
			return HXR_POINTER;
		}
		
		// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}

		if (SUCCEEDED(res)) 
			res = OnSetDouble(szName, uValue); 
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetDouble(szName, uValue); 

		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to set double: %s", szName);
		}
		return res;
	}


	STDMETHODIMP GetString( const char* szName, const CHAR **pcszValue ) 
	{
		if (!szName || !pcszValue)
		{
			return HXR_POINTER;
		}
		HX_RESULT res = m_pPropertyBag->GetString(szName, pcszValue);
		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to get string: %s", szName);
		}
		return res; 
	}

	STDMETHODIMP SetString( const char* szName, const CHAR *cszValue ) 
	{
		HX_RESULT res = HXR_OK;
	
		if (!szName || !cszValue)
		{
			return HXR_POINTER;
		}
		
		// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}

		if (SUCCEEDED(res)) 
			res = OnSetString(szName, cszValue);

		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetString(szName, cszValue);

		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to set string: %s", szName);
		}
		return res;
	}

	STDMETHODIMP GetPropertyBag( const char* szName, IHXTPropertyBag **ppValue) 
	{
		if (!szName || !ppValue)
		{
			return HXR_POINTER;
		}
		HX_RESULT res = m_pPropertyBag->GetPropertyBag(szName, ppValue);
		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to get property bag: %s", szName);
		}
		return res; 
	}

	STDMETHODIMP SetPropertyBag( const char* szName, IHXTPropertyBag *pValue) 
	{
		HX_RESULT res = HXR_OK;
	
		if (!szName || !pValue)
		{
			return HXR_POINTER;
		}
		
		// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}

		if (SUCCEEDED(res)) 
			res = OnSetPropertyBag(szName, pValue);

		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetPropertyBag(szName, pValue);

		if (FAILED(res))
		{
			//HXTLOG(LC_SDK_WARN, FA_GEN_FILTER, "Failed to set property bag: %s", szName);
		}
		return res;
	}

	STDMETHODIMP GetIntList( const char* szName, IHXTIntList **ppValue) 
	{
		return m_pPropertyBag->GetIntList(szName, ppValue);
	}

	STDMETHODIMP SetIntList( const char* szName, IHXTIntList *pValue) 
	{	
		HX_RESULT res = OnSetIntList(szName, pValue);
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetIntList(szName, pValue);
			
		return res;
	} 

	STDMETHODIMP GetUintList( const char* szName, IHXTUintList **ppValue) 
	{
		return m_pPropertyBag->GetUintList(szName, ppValue);
	}

	STDMETHODIMP SetUintList( const char* szName, IHXTUintList *pValue) 
	{	
		HX_RESULT res = OnSetUintList(szName, pValue);
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetUintList(szName, pValue);
			
		return res;
	} 


	STDMETHODIMP GetInt64List( const char* szName, IHXTInt64List **ppValue)
	{
		return m_pPropertyBag->GetInt64List(szName, ppValue);
	}

	STDMETHODIMP SetInt64List( const char* szName, IHXTInt64List *pValue) 
	{	
		HX_RESULT res = OnSetInt64List(szName, pValue);

		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetInt64List(szName, pValue);
			
		return res;
	} 

	STDMETHODIMP GetDoubleList( const char* szName, IHXTDoubleList **ppValue) 
	{
		return m_pPropertyBag->GetDoubleList(szName, ppValue);
	}

	STDMETHODIMP SetDoubleList( const char* szName, IHXTDoubleList *pValue) 
	{	
		HX_RESULT res = OnSetDoubleList(szName, pValue);
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetDoubleList(szName, pValue);
			
		return res;
	} 

	STDMETHODIMP GetIntRange( const char* szName, IHXTIntRange **ppValue) 
	{
		return m_pPropertyBag->GetIntRange(szName, ppValue);
	}


	STDMETHODIMP SetIntRange( const char* szName, IHXTIntRange *pValue) 
	{
		HX_RESULT res = OnSetIntRange(szName, pValue);
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetIntRange(szName, pValue);

		return res;
	} 

	STDMETHODIMP GetUintRange( const char* szName, IHXTUintRange **ppValue) 
	{
		return m_pPropertyBag->GetUintRange(szName, ppValue);
	}

	STDMETHODIMP SetUintRange( const char* szName, IHXTUintRange *pValue) 
	{
		HX_RESULT res = OnSetUintRange(szName, pValue);
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetUintRange(szName, pValue);

		return res;
	} 

	STDMETHODIMP GetInt64Range( const char* szName, IHXTInt64Range **ppValue)
	{
		return m_pPropertyBag->GetInt64Range(szName, ppValue);
	}

	STDMETHODIMP SetInt64Range( const char* szName, IHXTInt64Range *pValue) 
	{	
		HX_RESULT res = OnSetInt64Range(szName, pValue);
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetInt64Range(szName, pValue);
			
		return res;		
	} 

	STDMETHODIMP GetDoubleRange( const char* szName, IHXTDoubleRange **ppValue) 
	{
		return m_pPropertyBag->GetDoubleRange(szName, ppValue);
	}

	STDMETHODIMP SetDoubleRange( const char* szName, IHXTDoubleRange *pValue) 
	{	
		HX_RESULT res = OnSetDoubleRange(szName, pValue);
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetDoubleRange(szName, pValue);
			
		return res;	
	} 

	STDMETHODIMP GetUnknown( const char* szName, IUnknown **ppType )
	{
		return m_pPropertyBag->GetUnknown( szName, ppType );
	}

	STDMETHODIMP SetUnknown( const char* szName, IUnknown *pValue )
	{
		HX_RESULT res = HXR_OK;

		if (!szName || !pValue)
		{
			return HXR_POINTER;
		}

		// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}
		
		if (SUCCEEDED(res)) 
			res = OnSetUnknown(szName, pValue);
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->SetUnknown(szName, pValue);

		//if (FAILED(res))
		//	HXTLOG(LC_SDK_DIAG, FA_GEN_FILTER, "Failed to set IUnknown: %s", szName);

		return res;
	}

	STDMETHODIMP Remove( const char* szName )
	{
		HX_RESULT res = HXR_OK;
		// Validate params

		if (!szName)
		{
			return HXR_POINTER;
		}

		// Check if changes are allowed
		if (!m_bAllowChanges)
		{
			res = HXR_ENC_IMPROPER_STATE;
		}

		if (SUCCEEDED(res)) 
			res = OnRemove(szName);
		
		if (SUCCEEDED(res)) 
			res = m_pPropertyBag->Remove( szName );
		
		//if (FAILED(res))
		//	HXTLOG(LC_SDK_DIAG, FA_GEN_FILTER, "Failed to remove IUnknown: %s", szName);
		
		return res;
	}


	STDMETHODIMP GetPropertyEnumerator( IHXTPropertyEnumerator **ppEnumerator ) 
	{
		return m_pPropertyBag->GetPropertyEnumerator(ppEnumerator);
	}

	STDMETHODIMP GetPropertyBagEnumerator( IHXTPropertyEnumerator **ppEnumerator ) 
	{
		return m_pPropertyBag->GetPropertyBagEnumerator(ppEnumerator);
	}

protected:

	// Override this function to handle special properties that can only be 
	// set at init time.  Any properties that are consumed should be placed in pConsumedPropBag.
	// Any properties that are not consumed will be propagated via the usual OnSetXXX call.
	// Any invalid properties should be placed in pErrorBag.
	STDMETHOD(OnInitialize)( IHXTPropertyBag* pPropBag, IHXTPropertyBag* pConsumedPropBag, IHXTPropertyBag* pErrorBag ) {return HXR_OK;}

	// Override these functions to verify/take action on properties -- they will only be added to the property bag 
	// if the overriding function returns SUCCESS.
    STDMETHOD(OnSetBool)( const char* szName, BOOL bValue) {return HXR_FAIL;}
    STDMETHOD(OnSetInt)( const char* szName, INT32 uValue) {return HXR_FAIL;}
    STDMETHOD(OnSetUint)( const char* szName, UINT32 uValue) {return HXR_FAIL;}
    STDMETHOD(OnSetInt64)( const char* szName, INT64 uValue) {return HXR_FAIL;}
    STDMETHOD(OnSetDouble)( const char* szName, double uValue) {return HXR_FAIL;}

    STDMETHOD(OnSetIntList)( const char* szName, IHXTIntList *pValue) {return HXR_FAIL;}
    STDMETHOD(OnSetUintList)( const char* szName, IHXTUintList *pValue) {return HXR_FAIL;}
    STDMETHOD(OnSetInt64List)( const char* szName, IHXTInt64List *pValue) {return HXR_FAIL;}
    STDMETHOD(OnSetDoubleList)( const char* szName, IHXTDoubleList *pValue) {return HXR_FAIL;}

    STDMETHOD(OnSetIntRange)( const char* szName, IHXTIntRange *pValue) {return HXR_FAIL;}
    STDMETHOD(OnSetUintRange)( const char* szName, IHXTUintRange *pValue) {return HXR_FAIL;}
    STDMETHOD(OnSetInt64Range)( const char* szName, IHXTInt64Range *pValue) {return HXR_FAIL;}
    STDMETHOD(OnSetDoubleRange)( const char* szName, IHXTDoubleRange *pValue) {return HXR_FAIL;}
    
    STDMETHOD(OnSetString)( const char* szName, const CHAR *cszValue ) {return HXR_FAIL;}

	STDMETHOD(OnSetUnknown)( const char* szName, IUnknown *pObject ) {return HXR_FAIL;}
	STDMETHOD(OnSetProperty)( IHXTProperty *pProperty ) {return HXR_FAIL;}
    STDMETHOD(OnSetPropertyBag)( const char* szName, IHXTPropertyBag *pValue) {return HXR_FAIL;}

	STDMETHOD(OnRemove)( const char* szName ) {return HXR_FAIL;}

    // Note: This is not ref-counted since caller should never hang on to it
    IHXTPropertyBag* GetActualPropertyBag() { return m_pPropertyBag;}  

	void setAllowPropertyChanges(BOOL bAllowChanges) {m_bAllowChanges = bAllowChanges;}

	HX_RESULT init( IHXCommonClassFactory* pFactory )
	{
		if (!pFactory)
			return HXR_POINTER;		

		HX_RELEASE( m_pFactory );
		m_pFactory = pFactory;
		m_pFactory->AddRef();

		HX_RESULT res = HXR_OK;
		// Create internal property bag -- used to hold filter properties
		if (!m_pPropertyBag)			
			res = m_pFactory->CreateInstance( IID_IHXTPropertyBag, (void **) &m_pPropertyBag );

		return res;
	}

	IHXTPropertyBag *m_pPropertyBag;		

protected:

	inline BOOL isInitDone() {return m_bInitDone;}

	IHXCommonClassFactory *m_pFactory;		

private:

	BOOL m_bInitDone;
	BOOL m_bAllowChanges;
};


#endif // _HXTCONFIGAGENTHELPER_H

