#include <GL/gl.h>
#include <GL/glu.h>

#include <cstdlib>
#include <cmath>
#include <iostream>

#include "font.hpp"
#include "graphics.hpp"

/** Font coordinates */
int CHARMAP_DEFAULT[] = {
  'a', 34, 304, 100, 364,
  'b', 136, 303, 195, 364,
  'c', 233, 302, 292, 365,
  'd', 332, 303, 392, 364,
  'e', 432, 302, 492, 365,
  'f', 534, 303, 594, 363, 
  'g', 638, 303, 697, 363,
  'h', 738, 304, 797, 364,
  'i', 837, 306, 895, 364,
  'j', 932, 304, 986, 364,
};
int N_CHARS_DEFAULT = 'j' - 'a';
Font* fontDefault = NULL;

char* CHARS_12X12 = "abcdefghijklmnopqrstuvwxyz0123456789?!.# ";
int N_CHARS_12X12 = strlen(CHARS_12X12);
Font* font12x12 = NULL;

char* CHARS_14X16 = "abcdefghijklmnopqrstuvwxyz-+23456789?!.# ";
int N_CHARS_14X16 = strlen(CHARS_12X12);
Font* font14x16 = NULL;

#define X0 18
#define Y0 60
#define X1 62
#define Y1 15
#define WIDTH 56
int CHARMAP_DANCE[] = {
  'a', X0, Y0, X1, Y1,
  'b', X0 + WIDTH, Y0, X1 + WIDTH, Y1,
  'c', X0 + WIDTH + WIDTH, Y0, X1 + WIDTH + WIDTH, Y1,
};
int N_CHARS_DANCE = 3;
Font* fontDance = NULL;

Font* getFontDance() {
  if (fontDance == NULL) {
    fontDance = new Font("dance.png", N_CHARS_DANCE, CHARMAP_DANCE, 256, 256);
  }
  return fontDance;
}

Font* getFontDefault() {
  if (fontDefault == NULL) {
    fontDefault = new Font("font.png", N_CHARS_DEFAULT, CHARMAP_DEFAULT, 1024, 1024);
  }
  return fontDefault;
}

/**
 * Builds the 12x12 font from a 256x256 pixel picture
 * in which 3 lines of 13x13 pixels are used for characters.
 */
Font* buildFont12x12() {
  int* charMap = new int[5 * N_CHARS_12X12];
  int ofs = 0;
  for (int i = 0; i < N_CHARS_12X12; i++) {
    int line = (i / 19);
    charMap[ofs++] = CHARS_12X12[i];
    int x0 = 13 * (i - line * 19);
    int y0 = 12 + line * 13;
    int x1 = 13 * (i - line * 19) + 12;
    int y1 = 0 + line * 13;
    charMap[ofs++] = x0;
    charMap[ofs++] = y0;
    charMap[ofs++] = x1;
    charMap[ofs++] = y1;
  }
  font12x12 = new Font("font_12x12.png", N_CHARS_12X12, charMap, 256, 256);
  delete charMap;
  return font12x12;
}

Font* getFont12x12() {
  if (font12x12 == NULL) {
    font12x12 = buildFont12x12();
  }
  return font12x12;
}

/**
 * Builds the 14x16 font from a 256x256 pixel picture
 * in which 4 lines of 15x17 pixels are used for characters.
 */
Font* buildFont14x16() {
  int* charMap = new int[5 * N_CHARS_14X16];
  int ofs = 0;
  for (int i = 0; i < N_CHARS_14X16; i++) {
    int line = (i / 12);
    charMap[ofs++] = CHARS_14X16[i];
    int xofs = 23;
    int yofs = 11;
    int x0 = 16 * (i - line * 12);
    int y0 = 17 + line * 18;
    int x1 = 16 * (i - line * 12) + 14;
    int y1 = 0 + line * 18;
    charMap[ofs++] = x0 + xofs;
    charMap[ofs++] = y0 + yofs;
    charMap[ofs++] = x1 + xofs;
    charMap[ofs++] = y1 + yofs;
  }
  font14x16 = new Font("font_14x16.png", N_CHARS_14X16, charMap, 256, 256);
  delete charMap;
  return font14x16;
}

Font* getFont14x16() {
  if (font14x16 == NULL) {
    font14x16 = buildFont14x16();
  }
  return font14x16;
}

Font::Font(char* filename, int nChars, int coordMap[], int width, int height) {
  _textureId = load_texture_from_file(filename, 1);
  buildCoords(nChars, coordMap, width, height);
}

void Font::buildCoords(int nChars, int coordMap[], int width, int height) {
  int index = 0;
  for (int i = 0; i < nChars; i++) {
    char c = coordMap[index++];
    int x0 = coordMap[index++];
    int y0 = coordMap[index++];
    int x1 = coordMap[index++];
    int y1 = coordMap[index++];
    _ofs0[(int) c].x = (double) x0 / width;
    _ofs0[(int) c].y = (double) y0 / height;
    _ofs1[(int) c].x = (double) x1 / width;
    _ofs1[(int) c].y = (double) y1 / height;
  }
}

void Font::paint(Vector3 pos, char c) {
  Vector2* ofs0 = getCharOffset0(c);
  Vector2* ofs1 = getCharOffset1(c);

  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, _textureId);
  glDisable(GL_LIGHTING);

  glBegin(GL_QUADS);

  glTexCoord2f(ofs0->x, ofs0->y);
  glVertex3f(pos.x - 0.5, pos.y - 0.5, pos.z);

  glTexCoord2f(ofs1->x, ofs0->y);
  glVertex3f(pos.x + 0.5, pos.y - 0.5, pos.z);

  glTexCoord2f(ofs1->x, ofs1->y);
  glVertex3f(pos.x + 0.5, pos.y + 0.5, pos.z);

  glTexCoord2f(ofs0->x, ofs1->y);
  glVertex3f(pos.x - 0.5, pos.y + 0.5, pos.z);

  glEnd();
  glDisable(GL_TEXTURE_2D);
}

void Font::paint(int x0, int y0, int x1, int y1, char c) {
  Vector2* ofs0 = getCharOffset0(c);
  Vector2* ofs1 = getCharOffset1(c);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();

  glLoadIdentity();
  glOrtho(0.0, 640.0, 480.0, 0.0, 0.0, 1000);

  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, _textureId);
  glDisable(GL_LIGHTING);

  glBegin(GL_QUADS);
  glTexCoord2f(ofs0->x, ofs0->y);
  glVertex3f(x0, y0, 0.0);

  glTexCoord2f(ofs1->x, ofs0->y);
  glVertex3f(x1, y0, 0.0);

  glTexCoord2f(ofs1->x, ofs1->y);
  glVertex3f(x1, y1, 0.0);

  glTexCoord2f(ofs0->x, ofs1->y);
  glVertex3f(x0, y1, 0.0);
  glEnd();

  glDisable(GL_TEXTURE_2D);

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();

  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}

Vector2* Font::getCharOffset0(char c) {
  return &_ofs0[(int) c];
}

Vector2* Font::getCharOffset1(char c) {
  return &_ofs1[(int) c];
}
