#include <math.h>

#include "feedback.hpp"
#include "types.hpp"

#include <GL/glext.h>

#include <SDL/SDL.h>
#include <SDL/SDL_image.h>

#define BLOB_WIDTH 256.0
#define BLOB_HEIGHT 64.0

#define WIDTH 512.0
#define HEIGHT 512.0

GLuint create_texture(int width, int height);
GLuint create_blot(int width, int height);

double _time;

Feedback::Feedback() {
  _texture = create_texture((int) WIDTH, (int) HEIGHT);
  _blot = create_blot(32, 32);
}

int counter = 0;
double local_time;
void Feedback::effect(double time) {
  counter++;
  local_time = time;
  _time = local_time;
  draw();
}

void blot_blob(double x0, double y0, double width, double height) {
  x0 -= width / 2.0;
  y0 -= height / 2.0;

  double x1 = x0 + width;
  double y1 = y0 + height;

  glBegin(GL_QUADS);

  glTexCoord2f(0.0, 0.0);
  glVertex3f(x0, y0, 0.0);

  glTexCoord2f(1.0, 0.0);
  glVertex3f(x1, y0, 0.0);

  glTexCoord2f(1.0, 1.0);
  glVertex3f(x1, y1, 0.0);

  glTexCoord2f(0.0, 1.0);
  glVertex3f(x0, y1, 0.0);

  glEnd();
}

void Feedback::blot() {
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  glTranslatef(256.0, 256.0, 0.0);
  glRotatef(15.0 * sin(_time * 4.0), 0.0, 0.0, 1.0);

  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, _blot);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  // Red
  glColor3f(0.0, 0.7, 0.0);
  blot_blob(0.0, 32.0, BLOB_WIDTH, BLOB_HEIGHT);

  // White
  glColor3f(1.0, 1.0, 1.0);
  blot_blob(0.0, 0.0, BLOB_WIDTH, BLOB_HEIGHT);

  // Blue
  glColor3f(1.0, 0.0, 0.0);
  blot_blob(0.0, -32.0, BLOB_WIDTH, BLOB_HEIGHT);

  // Big overlay
  glColor4f(0.2, 0.2, 0.2, 0.05);
  blot_blob(0.0, 0.0, 512.0, 512.0);

  glColor4f(1.0, 1.0, 1.0, 1.0);
  glPopMatrix();

  glDisable(GL_BLEND);
}


void Feedback::draw() {
//   glBlendEquation(GL_MAX);
  glDisable(GL_DEPTH_TEST);

  // Setup the proper drawing mode
  glDisable(GL_LIGHTING);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(0, 640, 0, 480, -1, 1);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  glBlendEquation(GL_MAX);

  double z = 0.0;

  // Perform the feedback thing

  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, _texture);

  // X loop
  glBegin(GL_QUADS);
  glColor4f(1.0, 1.0, 1.0, 1.0);
  for (int i = 0; i < WIDTH; i++) {
    double x0 = (double) i;
    double x1 = (double) i + 1.0;
    double tx0 = x0 / WIDTH;
    double tx1 = x1 / WIDTH;

    double y_offset = (sin(local_time * 1.0 + 2.0 * sin(local_time * 2.0 - i / 520.0) * cos(local_time * 3.0 + i / 90.0) + i / 100.0)) / 100.0;
    double ty0 = 0.0 + y_offset;
    double ty1 = 1.0 + y_offset;

    glTexCoord2f(tx0, ty0);
    glVertex3f(x0, 0.0, z);

    glTexCoord2f(tx1, ty0);
    glVertex3f(x1, 0.0, z);

    glTexCoord2f(tx1, ty1);
    glVertex3f(x1, HEIGHT, z);

    glTexCoord2f(tx0, ty1);
    glVertex3f(x0, HEIGHT, z);
  }
  glEnd();
  // Copy the screen back to the texture
  glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, 0, 0, (int) WIDTH, (int) HEIGHT, 0);

  // Y loop
  glBegin(GL_QUADS);
  glColor4f(1.0, 1.0, 1.0, 1.0);
  for (int i = 0; i < HEIGHT; i++) {
    double y0 = (double) i;
    double y1 = (double) i + 1.0;
    double ty0 = y0 / HEIGHT;
    double ty1 = y1 / HEIGHT;

    double x_offset = (sin(local_time * 1.6 + 1.7 * sin(local_time * 1.9 - i / 337.0) * + i / 193.0)) / 100.0;
    double tx0 = 0.0 + x_offset;
    double tx1 = 1.0 + x_offset;

    glTexCoord2f(tx0, ty0);
    glVertex3f(0.0, y0, z);

    glTexCoord2f(tx1, ty0);
    glVertex3f(WIDTH, y0, z);

    glTexCoord2f(tx1, ty1);
    glVertex3f(WIDTH, y1, z);

    glTexCoord2f(tx0, ty1);
    glVertex3f(0.0, y1, z);
  }
  glEnd();

  blot();

  // Copy the screen back to the texture
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, _texture);
  glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, 0, 0, (int) WIDTH, (int) HEIGHT, 0);

  // Reset the drawing mode
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}

GLuint create_texture(int width, int height) {
  GLuint texture;
  uint32* buffer = (uint32*) new GLuint[width * height * 4 * sizeof(uint32)];

  for (int i = 0; i < height; i++) {
    for (int j = 0; j < width; j++) {
      int offset = i * width + j;
      double x = (double) j / height;
      double y = (double) i / width;
      double cx = 0.5;
      double cy = 0.5;
      double dist = 1.0 - sqrt((x - cx) * (x - cx) + (y - cy) * (y - cy)) * sqrt(4.0);
      if (dist < 0.0) {
        dist = 0.0;
      }
      int r = (int) (0xff * dist) & 0xff;
      int g = (int) (0xff * dist) & 0xff;
      int b = (int) (0xff * dist) & 0xff;
      int alpha = (int) (0xff * i) & 0xff;
      buffer[offset] = 0x01000000 * alpha | 0x00010000 * b | 0x00000001 * r | 0x00000100 * g;
//      buffer[offset] = 0xff000000; // black
//       buffer[offset] = 0xff7f7f7f; // gray
    }
  }

  glGenTextures(1, &texture);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexImage2D(GL_TEXTURE_2D, 0, 4, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, buffer);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

  delete buffer;
  return texture;
}

GLuint create_blot(int width, int height) {
  GLuint texture;
  uint32* buffer = (uint32*) new GLuint[width * height * 4 * sizeof(uint32)];

  for (int i = 0; i < height; i++) {
    for (int j = 0; j < width; j++) {
      int offset = i * width + j;
      double x = (double) j / height;
      double y = (double) i / width;
      double cx = 0.5;
      double cy = 0.5;
      double dist = 1.0 - sqrt((x - cx) * (x - cx) + (y - cy) * (y - cy)) * sqrt(4.0);
      if (dist < 0.0) {
        dist = 0.0;
      }
      int r = (int) (0xff * dist) & 0xff;
      int g = (int) (0xff * dist) & 0xff;
      int b = (int) (0xff * dist) & 0xff;
      int alpha = (int) (0xff * dist) & 0xff;
      buffer[offset] = 0x01000000 * alpha | 0x00010000 * b | 0x00000001 * r | 0x00000100 * g;
    }
  }

  glGenTextures(1, &texture);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexImage2D(GL_TEXTURE_2D, 0, 4, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, buffer);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  delete buffer;
  return texture;
}
