#include <GL/gl.h>
#include <GL/glu.h>

#include <cstdlib>
#include <cmath>
#include <iostream>

#include "cubes.hpp"
#include "utils.hpp"
#include "graphics.hpp"

char* font = {
  ".1111111."
  "111111111"
  "11.....11"
  "11.....11"
  "111111111"
  "11111111."
  "11......."
  "111111111"
  ".11111111"

  ".11111111"
  "111111111"
  "11......."
  "11......."
  "11......."
  "11......."
  "11......."
  "111111111"
  ".11111111"

  ".11111111"
  "111111111"
  "11......."
  "11......."
  "1111111.."
  "1111111.."
  "11......."
  "11......."
  "11......."

  "11......."
  "11......."
  "11......."
  "11......."
  "11111111."
  "111111111"
  "11.....11"
  "11.....11"
  "11.....11"

  "11111111."
  "111111111"
  "11.....11"
  "11.....11"
  "11......."
  "11......."
  "11......."
  "11......."
  "11......."

  "11.....11"
  "11.....11"
  "11.....11"
  "11.....11"
  "11.....11"
  "11.....11"
  "11.....11"
  "111111111"
  ".11111111"

  "11......."
  "11......."
  "11......."
  "11......."
  "11......."
  "11......."
  "11......."
  "111111111"
  ".11111111"

  ".1111111."
  "111111111"
  "11.....11"
  "11.....11"
  "111111111"
  "11111111."
  "11......."
  "111111111"
  ".11111111"

  ".11111111"
  "111111111"
  "11......."
  "11......."
  "11111111."
  ".11111111"
  ".......11"
  "111111111"
  "11111111."

};


void sort_faces(Face4* faces, int n_faces);

/**
 * Constructor
 * @param n_width number of cubes in the x direction
 * @param n_height number of cubes in the y direction
 * @param n_depth number of cubes in the z direction
 */
Cubes::Cubes(int n_width, int n_height, int n_depth) {
  _n_width = n_width;
  _n_height = n_height;
  _n_depth = n_depth;
  _cubes = new Cube[n_width * n_height * n_depth];
  _prev_stage = -1;
  _t0 = -1.0;
}

double ALL_CUBE_SIZE = 3.0;
double MIN_CUBE_SIZE = 0.1;

void Cubes::effect(double time, int stage, int noot) {
  if (_prev_stage != stage) {
    _prev_stage = stage;
    _t0 = time;
  }
  double local_time = time - _t0;
  _time = local_time;

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  glLoadIdentity();

  glTranslatef(-1.0f, -0.1f, -6.0f);

//   if (stage > 0)
    glTranslatef(0.0f, (((noot+4) % 8) == 0) * 0.15, 0.0);
  if (stage > 1)
    glTranslatef(0.0, 0.0, - 0.1 * noot);

  glRotatef(time * 40.0, 1.0, 0.0, 0.0);
  glRotatef(sin(time) * 40.0, 0.0, 1.0, 0.0);
  glRotatef(180.0, 0.0, 0.0, 1.0);
  glRotatef(180.0, 1.0, 0.0, 0.0);

  float d0 = 0.0;
  float d1 = 0.0;

  dprintf("%d\n", stage);
//   if (stage == 3) {
//     _peel_time = time - _t0;
//   } else {
     _peel_time = 0.0;
//   }

//   dprintf("stage = %d\n", stage);
//   dprintf("local time = %f\n", local_time);

  double d0_shrink = 0.0;
  double d1_shrink = 0.0;

  if (stage == 1) {
    if (local_time < 1.0) {
      d0_shrink = local_time;
    } else {
      d0_shrink = 1.0;
    }
  }

  if (stage >= 2) {
    d0_shrink = 1.0;
    if (local_time < 1.0) {
      d1_shrink = local_time;
    } else {
      d1_shrink = 1.0;
    }
  }

  d0 = MIN_CUBE_SIZE * d0_shrink * 1.0;
  d1 = MIN_CUBE_SIZE * d1_shrink * 1.0;

  float u0 = ALL_CUBE_SIZE;
  float u1 = (u0 - 2.0 * d0) / 3.0;
  float u2 = (u1 - 2.0 * d1) / 3.0;

  setCoords9(d0, d1, u0, u2);
    draw_antialiased();
    //draw();

  glPopMatrix();
}

/**
 * Calculate cube coordinates for the specific case where there are 9x9x9 cubes.
 */
void Cubes::setCoords9(double dist0, double dist1, double all_size, double cube_size) {
  assert(_n_width == _n_height);
  assert(_n_height == _n_depth);
  assert(_n_depth == 9);

  double dist[9];
  for (int i = 0; i < 9; i++) {
    if ((i % 3) == 2) {
      dist[i] = dist0;
    } else {
      dist[i] = dist1;
    }
  }

  double char_offsets[9];
  for (int i = 0; i < 9; i++) {
    char_offsets[i] = i * _peel_time;
  }

  setCoords(dist, char_offsets, all_size, cube_size);
}

/**
 * Sets the cube coordinates.
 * @param dist array of inter-cube distances, one for each cube in a specific direction
 * @param char_offsets offsets for each character
 * @param all_size the size of the total cube grid in each direction
 * @param cube_size the size of an individual cube in each direction
 */
void Cubes::setCoords(double dist[], double char_offsets[], double all_size, double cube_size) {
  double sum_dist = 0.0;
  for (int i = 0; i < _n_width; i++) {
    sum_dist += dist[i];
  }

  double x0 = -0.5 * all_size;
  double y0 = -0.5 * all_size;
  double z0 = -0.5 * all_size;

  int n = 0;
  double x = x0;
  for (int i = 0; i < _n_width; i++) {
    double y = y0;
    for (int j = 0; j < _n_height; j++) {
      double z = z0;
      for (int k = 0; k < _n_depth; k++) {
	double x_offset = char_offsets[k];
	_cubes[n].set(x + x_offset, y, z,
		      x + cube_size + x_offset, y + cube_size, z + cube_size);
	// Set color based on bitmap ("ecfh rules")
	char ch = font[j * 9 + i + k * 81];
	double color = (ch == '1') * 0.5 + 0.5;
	_cubes[n].set_color(color, color, color);
	z += dist[k] + cube_size;
	n++;
      }
      y += dist[j] + cube_size;
    }
    x += dist[i] + cube_size;
  }
}

/**
 * Draws all cubes sorted front to back, so that anti-aliasing can
 * be performed.
 */
void Cubes::draw_antialiased() {
  set_gl_polygon_smoothing();
  Face4 faces[_n_width * _n_height * _n_depth * 6];

  int n_faces = 0;
  for (int i = 0; i < _n_width * _n_height * _n_depth; i++) {
    int added_faces = _cubes[i].add_faces(&faces[n_faces]);
    n_faces += added_faces;
  }

  sort_faces(faces, n_faces);

  glEnable(GL_COLOR_MATERIAL);
  glBegin(GL_QUADS);
  for (int i = n_faces - 1; i > 0; i--) {
    //    glColor4f(faces[i].r, faces[i].g, faces[i].b, faces[i].b);
    double meuk = sin(_time + i/1000.0);
    glColor4f(faces[i].r * meuk, faces[i].g, faces[i].b, meuk);
    glNormal3f(faces[i].normal.x, faces[i].normal.y, faces[i].normal.z);
    glVertex3f(faces[i].p0.x, faces[i].p0.y, faces[i].p0.z);
    glVertex3f(faces[i].p1.x, faces[i].p1.y, faces[i].p1.z);
    glVertex3f(faces[i].p2.x, faces[i].p2.y, faces[i].p2.z);
    glVertex3f(faces[i].p3.x, faces[i].p3.y, faces[i].p3.z);
  }
  glEnd();

  glDisable(GL_COLOR_MATERIAL);
  glColor4f(1.0, 1.0, 1.0, 0.0);
}

double calc_distance(Vector3 v) {
  GLdouble modelview[16];
  glGetDoublev(GL_MODELVIEW_MATRIX, modelview);
  
  /*
    The 4x4 matrix is transposed.  We require the z coordinate of the transformed
    vector.
       0  4  8 12
       1  5  9 13
    -> 2  6 10 14
       3  7 11 15
  */
  double distance = modelview[2] * v.x + modelview[6] * v.y + modelview[10] * v.z;
  return distance;
}

double max(double x, double y) {
  return (x > y) ? x : y;
}

void update_distance(Face4* face) {
  Vector3 sum(face->p0.x + face->p1.x + face->p2.x + face->p3.x,
	      face->p0.y + face->p1.y + face->p2.y + face->p3.y,
	      face->p0.z + face->p1.z + face->p2.z + face->p3.z);
  face->distance = calc_distance(sum);
}

void update_distances(Face4* faces, int n_faces) {
  for (int i = 0; i < n_faces; i++) {
    update_distance(&faces[i]);
  }
}

int compare_faces(const void* face0_ptr, const void* face1_ptr) {
  Face4* face0 = (Face4*) face0_ptr;
  Face4* face1 = (Face4*) face1_ptr;
  if (face0->distance == face1->distance) {
    // nonsense, must use epsilon
    return 0;
  }
  if (face0->distance < face1->distance) {
    return -1;
  } else {
    return 1;
  }
}

void sort_faces(Face4* faces, int n_faces) {
  update_distances(faces, n_faces);
  qsort(faces, n_faces, sizeof (Face4), compare_faces);
}

/*
 * The default draw routine (without sorting) looks like this:
 */
void Cubes::draw() {
  for (int i = 0; i < _n_width * _n_height * _n_depth; i++) {
    _cubes[i].draw();
  }
}
