#include <GL/gl.h>

#include <cmath>

#include "cubebitmap.hpp"
#include "cube.hpp"
#include "utils.hpp"

const double WIDTH = 5.5;
const double HEIGHT = 4.1;

const double X_OFFSET = -2.6;
const double Y_OFFSET = -1.92;
const double Z_OFFSET = -5.0;

// const double WIDTH = 3.2;
// const double HEIGHT = 2.8;

// const double X_OFFSET = -2.2;
// const double Y_OFFSET = -1.8;
// const double Z_OFFSET = -5.0;

CubeBitmap::CubeBitmap(int n_width, int n_height) {
  _n_width = n_width;
  _n_height = n_height;
  _pixmap0 = new double[_n_width * _n_height];
  _pixmap1 = new double[_n_width * _n_height];
  _pixmap2 = new double[_n_width * _n_height];
  _scale =0.0;
  _rotate =0.0;
  _randomize = 0.0;
  _time = 0.0;
}

/**
 * Initializes the values of the _pixmap to a simple plasma.
 * @param t the time
 */
void CubeBitmap::set_demo_bitmap(double t) {
  for (int i = 0; i < _n_height; i++) {
    for (int j = 0; j < _n_width; j++) {
      int offset = i * _n_width + j;

      double u = (double) i / _n_height;
      double v = (double) j / _n_width;
      
      double value0 = 
	+ (sin(u * 1.3 * cos(v * 9.1 - t * 0.7 - 3.1) + t * 1.0 - 7.3)
	   * cos(v * 2.5 * sin(u * 13.1 + t * 0.1 - 5.1) + t * 1.1 - 3.2))
	+ (cos(u * 17.3 * sin(v * 3.2 - t * 0.35 + 0.7) + t * 1.2 - 2.1)
	   * sin(v * 1.4 * cos(u * 2.1 + t * 0.3 - 0.7) + t * 1.3 + 0.8));
      double value1 = 
	+ (sin(u * 2.3 * cos(v * 3.1 - t * 0.3 - 3.1) + t * 1.0 - 1.3)
	   * cos(v * 1.5 * sin(u * 2.1 + t * 0.2 - 5.1) + t * 1.1 - 3.2))
	+ (cos(u * 0.7 * sin(v * 7.3 - t * 0.25 + 0.7) + t * 1.2 - 2.1)
	   * sin(v * 2.4 * cos(u * 3.1 + t * 0.7 - 0.7) + t * 1.3 + 0.8));
      double value2 = 
	+ (sin(u * 4.3 * cos(v * 3.1 - t * 0.7 - 3.1) + t * 1.0 - 0.3)
	   * cos(v * 1.5 * sin(u * 12.1 + t * 0.1 - 5.1) + t * 1.1 - 3.2))
	+ (cos(u * 15.9 * sin(v * 5.2 - t * 0.35 + 0.7) + t * 1.2 - 0.1)
	   * sin(v * 2.4 * cos(u * 9.1 + t * 0.3 - 0.7) + t * 1.3 + 0.8));
       _pixmap0[offset] = value0;
       _pixmap1[offset] = value1;
       _pixmap2[offset] = value2;
    }
  }
}

double t0, t1;

void CubeBitmap::effect(double time, int stage) {
  _time = time;
  set_demo_bitmap(time);

  switch (stage) {
  case 0:
    _scale = 0.0;
    _rotate = 0.0;
    t0 = time;
    break;
  case 1:
    _scale = time - t0;
    _rotate = time - t1;
    t1 = time;
    break;
  case 2:
    _randomize = time - t1;
    break;
  default:
    dprintf("GORT\n");
  }
    
  if (_scale > 1.0) {
    _scale = 1.0;
  }
  if (_rotate > 1.0) {
    _rotate = 1.0;
  }
  if (_randomize > 1.0) {
    _randomize = 1.0;
  }

  draw();
}

void CubeBitmap::draw() {
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  glEnable(GL_LIGHTING);
  glEnable(GL_POLYGON_SMOOTH);

  Cube cube;
  double size = 2.0 / _n_width;
  cube.set(-size / 2.0, -size / 2.0, -size / 2.0, size / 2.0, size / 2.0, size / 2.0);

  double x;
  double y;

  for (int i = 0; i < _n_height; i++) {
    x = WIDTH / _n_height * i;
    for (int j = 0; j < _n_width; j++) {
      y = HEIGHT / _n_width * j;
      int offset = i * _n_width + j;
      //      double angle = _pixmap1[offset] * 90.0;
      float scale = _pixmap0[offset] * _pixmap0[offset] + 0.5;

      glPushMatrix();
      glLoadIdentity();

      // Lighting
      double lx0 = -cos(_time * 2.0) * 0.1;
      double ly0 = -sin(_time * 1.0) * 0.1;
      double lz0 = -2.0;

      double lx1 = 0.0;
      double ly1 = 0.0;
      double lz1 = -3.0;

      GLfloat position[] = { lx0, ly0, lz0, 1.0 };
      GLfloat direction[] = { lx1, ly1, lz1, 1.0 };
      glLightfv(GL_LIGHT1, GL_POSITION, position);
      glLightfv(GL_LIGHT1, GL_SPOT_DIRECTION, direction);

//       glBegin(GL_LINES);
//       glColor3f(1.0, 0.0, 0.0);
//       glVertex3f(lx0, ly0, lz0);
//       glColor3f(0.0, 1.0, 0.0);
//       glVertex3f(lx1, ly1, lz1);
//       glEnd();

      double r = _pixmap0[offset];
      double g = _pixmap1[offset];
      double b = _pixmap2[offset];

      //      GLfloat lightSpecular[] = { r * r + 0.3, g * g + 0.2, b * b + 0.5, 1.0f };
      GLfloat lightDiffuse[] = { r * r + 0.3, g * g + 0.2, b * b + 0.5, 1.0f };
       GLfloat lightSpecular[] = { 0.5, 1.0, 0.2, 1.0 };
//       GLfloat lightDiffuse[] = { 0.5, 1.0, 0.2, 1.0 };
//      GLfloat lightAmbient[] = { 0.3, 0.9, 0.9, 1.0 };
      GLfloat lightAmbient[] = { 0.1, 0.1, 0.1, 1.0 };
      
      glLightfv(GL_LIGHT1, GL_AMBIENT, lightAmbient);
      glLightfv(GL_LIGHT1, GL_DIFFUSE, lightDiffuse);
      glLightfv(GL_LIGHT1, GL_SPECULAR, lightSpecular);

//       glLightf(GL_LIGHT1, GL_SPOT_EXPONENT, 3.0);
//       glLightf(GL_LIGHT1, GL_SPOT_CUTOFF, 50.0);
//       glLightf(GL_LIGHT1, GL_CONSTANT_ATTENUATION, 0.0);
//       glLightf(GL_LIGHT1, GL_LINEAR_ATTENUATION, 0.1);
//       glLightf(GL_LIGHT1, GL_QUADRATIC_ATTENUATION, 0.1);

//       (GL_DEPTH_TEST); 
      glDisable(GL_BLEND);
//       glShadeModel(GL_SMOOTH);
//       glEnable(GL_POLYGON_SMOOTH);

//       glMaterialf(GL_FRONT, GL_SHININESS, 10.0);
      
      // Rotation and scaling
      glTranslatef(x + X_OFFSET, y + Y_OFFSET, Z_OFFSET); // - angle / 500.0);

      glScalef(scale, scale, scale);
      glRotatef(_pixmap1[offset] * 90.0, 1, 1, 1);

      if (_randomize) {
	glRotatef(_randomize * _pixmap2[offset] * 90.0, 1, 0, 0);
	glRotatef(_randomize * _pixmap1[offset] * 90.0, 0, 1, 0);
	glRotatef(_randomize * _pixmap0[offset] * 90.0, 0, 0, 1);
      }

      // Draw cubes
      cube.draw();

      glPopMatrix();
    }
  }

  glPopMatrix();

}
