#!/bin/bash

BLACK=$'\033'"[0;40m"
RED=$'\033'"[0;41m"
GREEN=$'\033'"[0;42m"
YELLOW=$'\033'"[0;43m"
BLUE=$'\033'"[0;44m"
PURPLE=$'\033'"[0;45m"
CYAN=$'\033'"[0;46m"
WHITE=$'\033'"[0;47m"
OFF=$'\033'"[0m"
ESC=$'\033'
CURSHOME=$'\033'"[1;1H"
CURSOFF=$'\033'"[?25l"
CURSON=$'\033'"[?25h"
CURSPOSSAVE=$'\033'"[s"
CURSPOSLOAD=$'\033'"[u"

BG=$BLUE  # Set startup background color

let LINES=$(stty size|cut -d " " -f 1)
let HALFLINES=${LINES}/2
let COLUMNS=$(stty size|cut -d " " -f 2)
let HALFCOLS=$COLUMNS/2
let COLMIN=60
let LINEMIN=20
DEMOPID=$$
export DEMOPID

TEMPDIR="/tmp/demo"
SYNCDATA="${TEMPDIR}/syncdata.inc"
MUSIC="./sunshine.mp3"
MUSIC_PRECALC="./precalc.mp3"
MUSICPLAYER="$(which mpg321) -v"
DEBUGFILE="${TEMPDIR}/debug"
SYNCFILE="${TEMPDIR}/synctime"
BCIN="${TEMPDIR}/bc-$DEMOPID.in"
BCOUT="${TEMPDIR}/bc-$DEMOPID.out"

rm $SYNCDATA &>/dev/null

if [ "$MUSICPLAYER" == " -v" ]
then
  echo "mpg321 not found. Please install it."
  exit 1
fi
if [ $COLUMNS -lt $COLMIN ]
then
  echo "The demo requires a minimum of $COLMIN columns."
  exit 1
fi
if [ $LINES -lt $LINEMIN ]
then
  echo "The demo requires a minimum of $LINEMIN rows."
  exit 1
fi
if [ $LINES -gt 26 ] || [ $COLUMNS -gt 82 ]
then
  echo "The demo is optimized for standard tty resolution (80x24)."
  echo "It will scale, but it might look different from what was intended."
  sleep 4
fi

declare -a fb
declare -a posarray1
declare -a posarray2


# Traps

function clean_exit {
  kill "${EFFECT[@]}" &>/dev/null
  kill $SYNCPID &>/dev/null
  kill $tmppid &>/dev/null
  # Exit bc
  echo "exit" >&7
  exec 7>&-
  exec 7<&-
  exec 8>&-
  exec 8<&-
  rm $BCIN
  rm $BCOUT

  rm ${TEMPDIR}/* &>/dev/null
  rmdir $TEMPDIR &>/dev/null

  echo "$OFF"
  echo "$CURSPOSLOAD"
  echo
  echo "Quitted.."
  echo "$CURSON"
  exit 0
} 2>/dev/null

function killed {
  exit 0
}

function child_finished {
  :
}

function continue {
  kill -9 $sleeper &>/dev/null
}

trap "clean_exit" SIGINT
trap "child_finished" SIGCHLD
trap "continue" SIGCONT
trap "killed" SIGKILL


# Start bc with fifo sockets

function bc_die {
  echo "$1" >&2
  if [ -e /proc/self/fd/7 ]; then
    echo "exit" >&7 #Stop bc if running.
    exec 7>&-  #Close file descriptor 7
    exec 7<&-
  fi
  if [ -e /proc/self/fd/8 ] ;then
    exec 8>&- #Close file descriptor 8
    exec 8<&-
  fi
  [ -e $BCIN ] && rm $BCIN
  [ -e $BCOUT ] && rm $BCOUT
  kill -SIGINT $DEMOPID
}


function sync {
  stop=$(cat $SYNCFILE)
}


function fb_setup {
  for pos in $(seq 0 $(($LINES*$COLUMNS)))
  do
    fb[$pos]="$BG"
  done
}

function fb_clear {
  fb=("${fb[@]//*/}")
  fb[0]="$BG"
}

function fb_clear_from {
  # Params x y
  let index=$(($2-1))*$COLUMNS+$(($1-1))
  fb_tmp=("${fb[@]:${index}}")
  fb=("${fb[@]:0:$index}" "${fb_tmp[@]//*/}")
  fb[$index]="$BG"
}

function fb_put {
  # Params: x y color
  let index=$(($2-1))*$COLUMNS+$(($1-1))
  [ $index -ge 0 ] && [ $index -lt ${#fb[@]} ] && fb[$index]="$3"
  let index+=1
  [ $index -ge 0 ] && [ $index -lt ${#fb[@]} ] && fb[$index]="${fb[$index]/#/$BG}"
}

function fb_put_index {
  # Params: index color topx topy
  let top=$(($4-1))*$COLUMNS+$(($3-1))
  if [ $1 -ge $top ] && [ $1 -le $(($LINES*$COLUMNS)) ]
  then
    fb[$1]="$2"
    if [ $(($1+1)) -lt ${#fb[@]} ]
    then
      fb[$(($1+1))]="${fb[$(($1+1))]/#/$BG}"
    fi
  fi
}

function refresh {
  echo -n "${CURSHOME}${fb[@]}"
}

function fb_square {
  # Params: x0 y0 x1 y1 color
  let x0=$1
  let y0=$2
  let x1=$3
  let y1=$4

  for i in $(seq $y0 $y1)
  do
    for j in $(seq $x0 $x1)
    do
      let index=$(($i-1))*$COLUMNS+$(($j-1))
      fb[$index]="$5"
    done
    if [ $index -lt $(($LINES*$COLUMNS)) ]
    then
      fb[$(($index+1))]="$BG${fb[$(($index+1))]}"
    fi
  done
}

function fb_line {
  # Params: x0 y0 x1 y1 color
  let x0=$1
  let y0=$2
  let x1=$3
  let y1=$4

  let y=$(($y1-$y0))
  let y=${y/-/}   # abs()
  let x=$(($x1-$x0))
  let x=${x/-/}   # abs()

  if [ $y -gt $x ]
  then
    steep=2
  else
    steep=1
  fi

  if [ $steep -eq 2 ]
  then
    let a=$y0
    let y0=$x0
    let x0=$a

    let a=$y1
    let y1=$x1
    let x1=$a
  fi

  if [ $x0 -gt $x1 ]
  then
    let a=$x1
    let x1=$x0
    let x0=$a

    let a=$y1
    let y1=$y0
    let y0=$a
  fi


  let deltax=$x1-$x0
  let deltay=$(($y1-$y0))
  let deltay=${deltay/-/}  # abs()
  let error=0
  let ystep=0
  let y=$y0
  if [ $y0 -lt $y1 ]
  then
    let ystep=1
  else
    let ystep=-1
  fi
  for x in $(seq $x0 $x1)
  do
    if [ $steep -eq 2 ]
    then
      fb_put $y $x $5
    else
      fb_put $x $y $5
    fi

    let error+=$deltay
    if [ $(($error<<1)) -ge $deltax ]
    then
      let y+=ystep
      let error-=deltax
    fi
  done
}

function fb_circle {
  # Params xCenter yCenter radius color topline bottomline # for clipping
  let x=0
  let y=$3
  echo "scale=0; d=3-(2*$3);d" >&7
  read -t1 d <&8
  echo "x=0; y=$3" >&7

  while [ $x -le $y ]
  do
    [ $(($2+$y)) -ge $5 ] && [ $(($2+$y)) -le $6 ] && [ $(($1+$x)) -gt 0 ] && [ $(($1+$x)) -le $COLUMNS ] && \
        if [ "$7" != "" ]; then for i in $(seq 0 $7); do fb_put $(($1+$x+$i)) $(($2+$y)) $4; done; else fb_put $(($1+$x)) $(($2+$y)) $4; fi
    [ $(($2+$x)) -ge $5 ] && [ $(($2+$x)) -le $6 ] && [ $(($1+$y)) -gt 0 ] && [ $(($1+$y)) -le $COLUMNS ] && \
        if [ "$7" != "" ]; then for i in $(seq 0 $7); do fb_put $(($1+$y+$i)) $(($2+$x)) $4; done; else fb_put $(($1+$y)) $(($2+$x)) $4; fi
    [ $(($2+$y)) -ge $5 ] && [ $(($2+$y)) -le $6 ] && [ $(($1-$x)) -gt 0 ] && [ $(($1-$x)) -le $COLUMNS ] && \
        if [ "$7" != "" ]; then for i in $(seq 0 $7); do fb_put $(($1-$x+$i)) $(($2+$y)) $4; done; else fb_put $(($1-$x)) $(($2+$y)) $4; fi
    [ $(($2-$x)) -ge $5 ] && [ $(($2-$x)) -le $6 ] && [ $(($1+$y)) -gt 0 ] && [ $(($1+$y)) -le $COLUMNS ] && \
        if [ "$7" != "" ]; then for i in $(seq 0 $7); do fb_put $(($1+$y+$i)) $(($2-$x)) $4; done; else fb_put $(($1+$y)) $(($2-$x)) $4; fi
    [ $(($2-$y)) -ge $5 ] && [ $(($2-$y)) -le $6 ] && [ $(($1-$x)) -gt 0 ] && [ $(($1-$x)) -le $COLUMNS ] && \
        if [ "$7" != "" ]; then for i in $(seq 0 $7); do fb_put $(($1-$x+$i)) $(($2-$y)) $4; done; else fb_put $(($1-$x)) $(($2-$y)) $4; fi
    [ $(($2-$x)) -ge $5 ] && [ $(($2-$x)) -le $6 ] && [ $(($1-$y)) -gt 0 ] && [ $(($1-$y)) -le $COLUMNS ] && \
        if [ "$7" != "" ]; then for i in $(seq 0 $7); do fb_put $(($1-$y+$i)) $(($2-$x)) $4; done; else fb_put $(($1-$y)) $(($2-$x)) $4; fi
    [ $(($2-$y)) -ge $5 ] && [ $(($2-$y)) -le $6 ] && [ $(($1+$x)) -gt 0 ] && [ $(($1+$x)) -le $COLUMNS ] && \
        if [ "$7" != "" ]; then for i in $(seq 0 $7); do fb_put $(($1+$x+$i)) $(($2-$y)) $4; done; else fb_put $(($1+$x)) $(($2-$y)) $4; fi
    [ $(($2+$x)) -ge $5 ] && [ $(($2+$x)) -le $6 ] && [ $(($1-$y)) -gt 0 ] && [ $(($1-$y)) -le $COLUMNS ] && \
        if [ "$7" != "" ]; then for i in $(seq 0 $7); do fb_put $(($1-$y+$i)) $(($2+$x)) $4; done; else fb_put $(($1-$y)) $(($2+$x)) $4; fi

    if [ $d -lt 0 ]
    then
      echo "d+=$x*4+6;d" >&7
      read -t1 d <&8
    else
      echo "d+=($x-$y)*4+10;d" >&7
      read -t1 d <&8
      let y-=1
    fi
    let x+=1
  done
}

function draw_donut {
  # Params xCenter yCenter xHeight yHeight color front

  if [ "$6" == "1" ]
  then
    donut_back $1 $2 $3 $4 $5 $6
  else
    donut_front $1 $2 $3 $4 $5 $6
  fi

  bzh_logo $((HALFLINES-3)) $WHITE

  if [ "$6" == "1" ]
  then
    donut_front $1 $2 $3 $4 $5 $6
  else
    donut_back $1 $2 $3 $4 $5 $6
  fi

}

function donut_back {
  let lx=$3
  for i in $(seq 0 $4)
  do
    echo "scale=5;x=(sqrt(($3*$3)*(($4*$4)-($i*$i))))/$4;scale=0;x/1" >&7
    read -t1 xx <&8

    for j in $(seq $(($1+$xx)) $(($lx+$1+5)))
    do
      fb_put $j $(($2+$i)) $5
    done

    for j in $(seq $(($1+$xx)) $(($lx+$1+5)))
    do
      fb_put $j $(($2-$i)) $5
    done
    let lx=$xx
  done
}


function donut_front {
  let lx=$3
  for i in $(seq 0 $4)
  do
    echo "scale=5;x=(sqrt(($3*$3)*(($4*$4)-($i*$i))))/$4;scale=0;x/1" >&7
    read -t1 xx <&8

    for j in $(seq $((-$lx+$1-5)) $(($1-$xx)))
    do
      fb_put $j $(($2+$i)) $5
    done

    for j in $(seq $((-$lx+$1-5)) $(($1-$xx)))
    do
      fb_put $j $(($2-$i)) $5
    done

    let lx=$xx
  done
}

function fb_circle_pos {
  # Params xCenter yCenter radius topline bottomline # for clipping
  let x=0
  let y=$3
  echo "scale=0; d=3-(2*$3);d" >&7
  read -t1 d <&8
  echo "x=0; y=$3" >&7

  while [ $x -le $y ]
  do
    [ $(($2+$y)) -ge $4 ] && [ $(($2+$y)) -le $5 ] && [ $(($1+$x)) -gt 0 ] && [ $(($1+$x)) -le $COLUMNS ] && \
        echo -n "$((($2+$y-1)*$COLUMNS+($1+$x-1))) "
    [ $(($2+$x)) -ge $4 ] && [ $(($2+$x)) -le $5 ] && [ $(($1+$y)) -gt 0 ] && [ $(($1+$y)) -le $COLUMNS ] && \
        echo -n "$((($2+$x-1)*$COLUMNS+($1+$y-1))) "
    [ $(($2+$y)) -ge $4 ] && [ $(($2+$y)) -le $5 ] && [ $(($1-$x)) -gt 0 ] && [ $(($1-$x)) -le $COLUMNS ] && \
        echo -n "$((($2+$y-1)*$COLUMNS+($1-$x-1))) "
    [ $(($2-$x)) -ge $4 ] && [ $(($2-$x)) -le $5 ] && [ $(($1+$y)) -gt 0 ] && [ $(($1+$y)) -le $COLUMNS ] && \
        echo -n "$((($2-$x-1)*$COLUMNS+($1+$y-1))) "
    [ $(($2-$y)) -ge $4 ] && [ $(($2-$y)) -le $5 ] && [ $(($1-$x)) -gt 0 ] && [ $(($1-$x)) -le $COLUMNS ] && \
        echo -n "$((($2-$y-1)*$COLUMNS+($1-$x-1))) "
    [ $(($2-$x)) -ge $4 ] && [ $(($2-$x)) -le $5 ] && [ $(($1-$y)) -gt 0 ] && [ $(($1-$y)) -le $COLUMNS ] && \
        echo -n "$((($2-$x-1)*$COLUMNS+($1-$y-1))) "
    [ $(($2-$y)) -ge $4 ] && [ $(($2-$y)) -le $5 ] && [ $(($1+$x)) -gt 0 ] && [ $(($1+$x)) -le $COLUMNS ] && \
        echo -n "$((($2-$y-1)*$COLUMNS+($1+$x-1))) "
    [ $(($2+$x)) -ge $4 ] && [ $(($2+$x)) -le $5 ] && [ $(($1-$y)) -gt 0 ] && [ $(($1-$y)) -le $COLUMNS ] && \
        echo -n "$((($2+$x-1)*$COLUMNS+($1-$y-1))) "

    if [ $d -lt 0 ]
    then
      echo "d+=$x*4+6;d" >&7
      read -t1 d <&8
    else
      echo "d+=($x-$y)*4+10;d" >&7
      read -t1 d <&8
      let y-=1
    fi
    let x+=1
  done
}



function line_rotator {
  # Params x0 y0 x1 y1 step
  if [ $5 -gt 1 ]
  then
    let step2=$5-1
  else
    let step2=$5
  fi
  while true
  do
    color=("$BLUE" "$RED" "$GREEN" "$YELLOW" "$PURPLE" "$CYAN" "$WHITE")
    for i in $(seq $1 $5 $3)
    do
      fb_clear_from 1 8
      fb_line $i $2 $(($3-$i)) $4 ${color[$cs]}
      refresh
      let cs+=1
      if [ $cs -gt 6 ]; then let cs=0; fi
    done
    for i in $(seq $2 $step2 $4)
    do
      fb_clear_from 1 8
      fb_line $1 $(($4+$2-$i)) $3 $i ${color[$cs]}
      refresh
      let cs+=1
      if [ $cs -gt 6 ]; then let cs=0; fi
    done
  done
}

function calc_circle1 {
  posarray1=()
  for j in $(seq 1 2 $(($HALFCOLS+2)))
  do
    posarray1=("${posarray1[@]}" $(fb_circle_pos $HALFCOLS $(($HALFLINES)) $j 1 $LINES))
  done
}

function calc_circle2 {
  posarray2=()
  for j in $(seq 1 2 $(($HALFCOLS+2)))
  do
    posarray2=("${posarray2[@]}" $(fb_circle_pos $HALFCOLS $(($HALFLINES)) $j 0 $(($LINES+2))))
  done
}

function moire_circle {
  while :
  do
    fb=("${moire_frame0[@]}")
    refresh
    fb=("${moire_frame1[@]}")
    refresh
    fb=("${moire_frame2[@]}")
    refresh
    fb=("${moire_frame3[@]}")
    refresh
    fb=("${moire_frame4[@]}")
    refresh
    fb=("${moire_frame5[@]}")
    refresh
    fb=("${moire_frame6[@]}")
    refresh
    fb=("${moire_frame7[@]}")
    refresh
  done
}

function calc_moire_circle {
    BG="$BLACK"

    fb_clear
    for posindex in "${posarray1[@]}"; do fb_put_index $posindex $1 1 1; done
    for posindex in "${posarray2[@]}"; do fb_put_index $((posindex-2)) $2 1 1; done
    moire_frame0=("${fb[@]}")

    fb_clear
    for posindex in "${posarray1[@]}"; do fb_put_index $posindex $1 1 1; done
    for posindex in "${posarray2[@]}"; do fb_put_index $((posindex-$COLUMNS-1)) $2 1 1; done
    moire_frame1=("${fb[@]}")

    fb_clear
    for posindex in "${posarray1[@]}"; do fb_put_index $posindex $1 1 1; done
    for posindex in "${posarray2[@]}"; do fb_put_index $((posindex-($COLUMNS*2))) $2 1 1; done
    moire_frame2=("${fb[@]}")

    fb_clear
    for posindex in "${posarray1[@]}"; do fb_put_index $posindex $1 1 1; done
    for posindex in "${posarray2[@]}"; do fb_put_index $((posindex-$COLUMNS+1)) $2 1 1; done
    moire_frame3=("${fb[@]}")

    fb_clear
    for posindex in "${posarray1[@]}"; do fb_put_index $posindex $1 1 1; done
    for posindex in "${posarray2[@]}"; do fb_put_index $((posindex+2)) $2 1 1; done
    moire_frame4=("${fb[@]}")

    fb_clear
    for posindex in "${posarray1[@]}"; do fb_put_index $posindex $1 1 1; done
    for posindex in "${posarray2[@]}"; do fb_put_index $((posindex+$COLUMNS+1)) $2 1 1; done
    moire_frame5=("${fb[@]}")

    fb_clear
    for posindex in "${posarray1[@]}"; do fb_put_index $posindex $1 1 1; done
    for posindex in "${posarray2[@]}"; do fb_put_index $((posindex+($COLUMNS*2))) $2 1 1; done
    moire_frame6=("${fb[@]}")

    fb_clear
    for posindex in "${posarray1[@]}"; do fb_put_index $posindex $1 1 1; done
    for posindex in "${posarray2[@]}"; do fb_put_index $((posindex+$COLUMNS-1)) $2 1 1; done
    moire_frame7=("${fb[@]}")

    BG="$BLUE"
    fb_clear
}

function cube {
  # Define cube
  let cube_x[0]=-10
  let cube_y[0]=10
  let cube_z[0]=10

  let cube_x[1]=10
  let cube_y[1]=10
  let cube_z[1]=10

  let cube_x[2]=10
  let cube_y[2]=-10
  let cube_z[2]=10

  let cube_x[3]=-10
  let cube_y[3]=-10
  let cube_z[3]=10

  let cube_x[4]=-10
  let cube_y[4]=10
  let cube_z[4]=-10

  let cube_x[5]=10
  let cube_y[5]=10
  let cube_z[5]=-10

  let cube_x[6]=10
  let cube_y[6]=-10
  let cube_z[6]=-10

  let cube_x[7]=-10
  let cube_y[7]=-10
  let cube_z[7]=-10

  let angle=0
  let zoom=2000
  sign="-"
  init_scroller
  while :
  do
    render_cube $angle $((360-$angle)) 0
    let angle+=20
    if [ $angle -eq 360 ]
    then
      let angle=0
    fi
  done
}

function render_cube {
  let xa=$1
  let ya=$2
  let za=$3
  let zoom=$zoom${sign}100
  if [ $zoom -lt 900 ] || [ $zoom -gt 2000 ]
  then
    if [ $sign == "+" ]
    then
      sign="-"
    else
      sign="+"
    fi
  fi

  echo "scale=4" >&7
  echo "xdeg=$xa*pi/180; ydeg=$ya*pi/180; zdeg=$za*pi/180" >&7

  echo "sx=s(xdeg); sy=s(ydeg); sz=s(zdeg)" >&7
  echo "cx=c(xdeg); cy=c(ydeg); cz=c(zdeg)" >&7

  echo "mat00=cx*cz+sx*sy*sz" >&7
  echo "mat10=-cx*sz+cz*sx*sy" >&7
  echo "mat20=cy*sx" >&7
  echo "mat01=cy*sz" >&7
  echo "mat11=cy*cz" >&7
  echo "mat21=-sy" >&7
  echo "mat02=-cz*sx+cx*sy*sz" >&7
  echo "mat12=sx*sz+cx*cz*sy" >&7
  echo "mat22=cx*cy" >&7

  for i in $(seq 0 7)
  do
    echo "rx${i}=${cube_x[$i]}*mat00+${cube_y[$i]}*mat10+${cube_z[$i]}*mat20-150" >&7
    echo "ry${i}=${cube_x[$i]}*mat01+${cube_y[$i]}*mat11+${cube_z[$i]}*mat21" >&7
    echo "rz${i}=${cube_x[$i]}*mat02+${cube_y[$i]}*mat12+${cube_z[$i]}*mat22+$zoom" >&7

    echo "scale=0" >&7
    echo "(rx${i}*500)/rz${i}+$HALFCOLS" >&7
    read -t1 scrx[$i] <&8
    echo "(ry${i}*500)/rz${i}+$HALFLINES" >&7
    read -t1 scry[$i] <&8

    echo "scale=4" >&7
  done
  fb_clear
  for i in $(seq 0 3)
  do
    fb_line ${scrx[$i]} ${scry[$i]} ${scrx[$(($i+4))]} ${scry[$(($i+4))]} $BLACK
    fb_line ${scrx[$i]} ${scry[$i]} ${scrx[$((($i+1)%4))]} ${scry[$((($i+1)%4))]} $BLACK
    fb_line ${scrx[$(($i+4))]} ${scry[$(($i+4))]} ${scrx[$(((($i+1)%4)+4))]} ${scry[$(((($i+1)%4)+4))]} $BLACK
  done
  refresh
  show_scroller
}

function init_scroller {
  for i in $(seq 1 $COLUMNS)
  do
    scrolltext="$scrolltext "
  done
  scrolltext="${scrolltext}Bashoholics - Presented at Kindergarden 2006. Code: Arcane, Music: Herb Alpert and Westbam & Dr.Motte.."
  for i in $(seq 1 $COLUMNS)
  do
    scrolltext="$scrolltext "
  done
  tmp_scroll="$scrolltext"
  if [ ${#tmp_scroll} -lt 10 ]
  then
    tmp_scroll="$tmp_scroll$scrolltext"
  fi
}

function show_scroller {
  tmp_scroll=${tmp_scroll:1}
  echo -n "$CYAN$ESC[30m$ESC[$((LINES-1));1H${tmp_scroll:0:$COLUMNS}"
}

function fade_in_strobe {
  
  for i in 1 2
  do
    BG=$WHITE
    fb_clear_from 1 1
    refresh
    sleep 0.1
    BG=$BLACK
    fb_clear_from 1 1
    refresh
    sleep 1.48
  done

  for i in $(seq 1 4)
  do
    BG=$WHITE
    fb_clear_from 1 1
    refresh
    sleep 0.1
    BG=$BLACK
    fb_clear_from 1 1
    refresh
    sleep 0.76
  done

  for i in $(seq 1 4)
  do
    BG=$WHITE
    fb_clear_from 1 1
    refresh
    sleep 0.1
    BG=$BLACK
    fb_clear_from 1 1
    refresh
    sleep 0.37
  done

  for i in $(seq 1 4)
  do
    BG=$WHITE
    fb_clear_from 1 1
    refresh
    sleep 0.1
    BG=$BLACK
    fb_clear_from 1 1
    refresh
    sleep 0.185
  done

  while :
  do
    BG=$WHITE
    fb_clear_from 1 1
    refresh
    sleep 0.1
    BG=$BLACK
    fb_clear_from 1 1
    refresh
    sleep 0.1
  done

}

function calc_voxel {
  PALETTE=("$WHITE" "$CYAN" "$GREEN" "$YELLOW" "$BLUE")

  BG="$BLACK"
  fb_setup

  let depth=8   # Even number
  let width=30   # Even number
  let x_origin=18
  let y_origin=28
  let max_height=20
  let initial_height=10

  declare -a mapx
  declare -a mapy
  declare -a mapz
  declare -a mapcol
  declare -a displayx
  declare -a displayy
  declare -a displayz
  declare -a displaycol

  let period=300
  let granularity=3
  let theta_incr=16

  echo "scale=10" >&7
  echo "cos_10=c(10*pi/180)" >&7
  echo "sin_10=s(10*pi/180)" >&7

  voxel_gen_map

  let framecount=0
  fb_clear
  let loopnum=0
  let y_origin+=5
  while [ $loopnum -lt 180 ]
  do
    echo "scale=5;$loopnum*pi/180" >&7
    read -t1 theta <&8

    if [ $loopnum -lt 90 ]
    then
      let y_origin+=2
    else
      let y_origin-=2
    fi

    voxel_rotate_map $theta

    voxel_sort
    fb_setup
    voxel_display

    case "$framecount" in
      0) voxelframe0=("${fb[@]}") ;;
      1) voxelframe1=("${fb[@]}"); voxelframe21=("${fb[@]}") ;;
      2) voxelframe2=("${fb[@]}"); voxelframe20=("${fb[@]}") ;;
      3) voxelframe3=("${fb[@]}"); voxelframe19=("${fb[@]}") ;;
      4) voxelframe4=("${fb[@]}"); voxelframe18=("${fb[@]}") ;;
      5) voxelframe5=("${fb[@]}"); voxelframe17=("${fb[@]}") ;;
      6) voxelframe6=("${fb[@]}"); voxelframe16=("${fb[@]}") ;;
      7) voxelframe7=("${fb[@]}"); voxelframe15=("${fb[@]}") ;;
      8) voxelframe8=("${fb[@]}"); voxelframe14=("${fb[@]}") ;;
      9) voxelframe9=("${fb[@]}"); voxelframe13=("${fb[@]}") ;;
      10) voxelframe10=("${fb[@]}"); voxelframe12=("${fb[@]}") ;;
      11) voxelframe11=("${fb[@]}") ;;
    esac
    let framecount+=1
    let loopnum+=$theta_incr
  done
  let voxelcount=$framecount+$framecount-3
  BG="$BLUE"
}

function voxel {
  while :
  do
    for i in $(seq 0 $voxelcount)
    do
      fb_clear
      case "$i" in
        0) fb=("${voxelframe0[@]}") ;;
        1) fb=("${voxelframe1[@]}") ;;
        2) fb=("${voxelframe2[@]}") ;;
        3) fb=("${voxelframe3[@]}") ;;
        4) fb=("${voxelframe4[@]}") ;;
        5) fb=("${voxelframe5[@]}") ;;
        6) fb=("${voxelframe6[@]}") ;;
        7) fb=("${voxelframe7[@]}") ;;
        8) fb=("${voxelframe8[@]}") ;;
        9) fb=("${voxelframe9[@]}") ;;
        10) fb=("${voxelframe10[@]}") ;;
        11) fb=("${voxelframe11[@]}") ;;
        12) fb=("${voxelframe12[@]}") ;;
        13) fb=("${voxelframe13[@]}") ;;
        14) fb=("${voxelframe14[@]}") ;;
        15) fb=("${voxelframe15[@]}") ;;
        16) fb=("${voxelframe16[@]}") ;;
        17) fb=("${voxelframe17[@]}") ;;
        18) fb=("${voxelframe18[@]}") ;;
        19) fb=("${voxelframe19[@]}") ;;
        20) fb=("${voxelframe20[@]}") ;;
        21) fb=("${voxelframe21[@]}") ;;
      esac
      refresh
      sleep 0.2
    done
  done
}

function voxel_sort {
  let j=0
  while read line
  do
    tmp=($(echo $line))
    displayx[$j]=${tmp[0]}
    displayy[$j]=${tmp[1]}
    displayz[$j]=${tmp[2]}
    displaycol[$j]=${tmp[3]}
    let j+=1
  done << EOF
  $(for i in $(seq 0 $((${#displayz[@]}-1))); do echo "${displayx[$i]} ${displayy[$i]} ${displayz[$i]} ${displaycol[$i]}"; done | sort -k 3 -n)
EOF
}

function voxel_display {
  for i in $(seq 0 $((${#displaycol[@]}-1))|sort -n -r)
  do
    let x1=${displayx[$i]}-$granularity
    let y1=${displayy[$i]}-$granularity
    let x2=${displayx[$i]}+$granularity

    draw_voxel $x1 $y1 $x2 ${displaycol[$i]}
  done
}

function draw_voxel {
  let x=$1

  while [ $x -lt $3 ]
  do
    fb_put $x $2 ${PALETTE[$4]}         #Length of voxels
    fb_put $x $(($2+1)) ${PALETTE[$4]}
    fb_put $x $(($2+2)) ${PALETTE[$4]}
    fb_put $x $(($2+3)) ${PALETTE[$4]}
    fb_put $x $(($2+4)) ${PALETTE[$4]}
    fb_put $x $(($2+5)) ${PALETTE[$4]}
    fb_put $x $(($2+6)) ${PALETTE[$4]}
    fb_put $x $(($2+7)) ${PALETTE[$4]}
    let x+=1
  done
}

function voxel_rotate_map {
  echo "scale=10" >&7
  echo "theta=$1; x=0; y=0; z=0; tx=0; ty=0; tz=0; cos_=c(theta); sin_=s(theta)" >&7
  let i=0
  let j=0
  let k=0


  for i in $(seq 1 $depth)
  do
    for j in $(seq 1 $width)
    do
      echo "scale=0" >&7
      echo "($i-1)*$width+($j-1)" >&7
      read -t1 mapindex <&8

      echo "scale=10" >&7

      # Rotate y-axis
      echo "x=${mapx[$mapindex]}*cos_ + ${mapz[$mapindex]}*sin_" >&7
      echo "z=${mapx[$mapindex]}*sin_ + ${mapz[$mapindex]}*cos_" >&7
      echo "tx=x" >&7

      # Rotate z-axis
      echo "x=tx*cos_10 - ${mapy[$mapindex]}*sin_10" >&7
#      echo "y=tx*sin_10 + ${mapy[$mapindex]}*cos_10" >&7
      echo "y=${mapy[$mapindex]}" >&7

      echo "ty=y; tz=z" >&7

      # Rotate x-axis
#      echo "y=ty*cos_10 - tz*sin_10" >&7
      echo "z=ty*sin_10 + tz*cos_10" >&7

      # Fetch x,y,z values
      echo "scale=0;x/1" >&7
      read -t1 x <&8
      echo "y/1" >&7
      read -t1 y <&8
      echo "z/1" >&7
      read -t1 z <&8

      let displayx[$k]=$x+$x_origin
      let displayy[$k]=$y_origin-$y
      let displayz[$k]=$z+50
      let displaycol[$k]=mapcol[$mapindex]
      let k+=1
    done
  done
}

function voxel_gen_map {
  let i=0
  let j=0
  let xincr=0
  let zincr=0
  echo "scale=0" >&7
  echo "$max_height/($depth/2)" >&7
  read -t1 height_incr <&8
  echo "scale=5;$period/$width" >&7
  read -t1 theta_inc <&8
  let height=$initial_height
  let theta=0
  echo "$depth*$width-1" >&7
  read -t1 last_index <&8
  echo "$granularity*$depth" >&7
  read -t1 max_z <&8

  for i in $(seq 1 $(($depth/2)))
  do
    for j in $(seq 1 $width)
    do
      echo "($i-1)*$width+($j-1)" >&7
      read -t1 mapindex <&8
      echo "$last_index-$mapindex" >&7
      read -t1 lastmapindex <&8

      let mapx[$mapindex]=$xincr+$granularity
      let xincr+=$granularity
      let mapz[$mapindex]=$zincr
      echo "scale=5;tmp=$height*s($theta*pi/180);scale=0;tmp/1" >&7
      read -t1 mapy[$mapindex] <&8
      echo "scale=5;tmp=($y_origin - ${mapy[$mapindex]})*5/$LINES;scale=0;tmp/1" >&7
      read -t1 mapcol[$mapindex] <&8
      if [ ${mapcol[$mapindex]} -lt 0 ]; then let mapcol[$mapindex]=0; fi

      let mapx[$lastmapindex]=$xincr+$granularity
      let xincr+=$granularity
      let mapz[$lastmapindex]=$max_z
      echo "scale=5;tmp=$height*s($theta*pi/180);scale=0;tmp/1" >&7
      read -t1 mapy[$lastmapindex] <&8
      echo "scale=5;tmp=($y_origin - ${mapy[$lastmapindex]})*5/$LINES;scale=0;tmp/1" >&7
      read -t1 mapcol[$lastmapindex] <&8
      if [ ${mapcol[$lastmapindex]} -lt 0 ]; then let mapcol[$lastmapindex]=0; fi

      echo "scale=5;$theta+$theta_inc" >&7
      read -t1 theta <&8
    done
    let xincr=0
    let zincr+=$granularity
    let max_z-=$granularity
    let height+=$height_incr
    let theta=0
  done
}

function bash_logo {
  # Param: row color parts
  startx=$((($COLUMNS>>1)-29))

  if [ $3 -ge 1 ]
  then
    fb_put $((startx+2)) $1 $2
    fb_put $((startx+4)) $1 $2
    fb_put $((startx+8)) $1 $2

    fb_put $((startx+2)) $(($1+1)) $2
    fb_put $((startx+4)) $(($1+1)) $2
    fb_put $((startx+8)) $(($1+1)) $2

    fb_put $startx $(($1+2)) $2
    fb_put $((startx+1)) $(($1+2)) $2
    fb_put $((startx+2)) $(($1+2)) $2
    fb_put $((startx+3)) $(($1+2)) $2
    fb_put $((startx+4)) $(($1+2)) $2
    fb_put $((startx+5)) $(($1+2)) $2
    fb_put $((startx+6)) $(($1+2)) $2
    fb_put $((startx+8)) $(($1+2)) $2

    fb_put $((startx+2)) $(($1+3)) $2
    fb_put $((startx+4)) $(($1+3)) $2
    fb_put $((startx+8)) $(($1+3)) $2

    fb_put $startx $(($1+4)) $2
    fb_put $((startx+1)) $(($1+4)) $2
    fb_put $((startx+2)) $(($1+4)) $2
    fb_put $((startx+3)) $(($1+4)) $2
    fb_put $((startx+4)) $(($1+4)) $2
    fb_put $((startx+5)) $(($1+4)) $2
    fb_put $((startx+6)) $(($1+4)) $2
    fb_put $((startx+8)) $(($1+4)) $2

    fb_put $((startx+2)) $(($1+5)) $2
    fb_put $((startx+4)) $(($1+5)) $2

    fb_put $((startx+2)) $(($1+6)) $2
    fb_put $((startx+4)) $(($1+6)) $2
    fb_put $((startx+8)) $(($1+6)) $2
  fi

  if [ $3 -ge 2 ]
  then
    fb_put $(($startx+18)) $1 $2
    fb_put $(($startx+23)) $1 $2
    fb_put $((startx+16)) $1 $2

    fb_put $((startx+15)) $(($1+1)) $2
    fb_put $(($startx+18)) $(($1+1)) $2

    fb_put $((startx+14)) $(($1+2)) $2
    fb_put $(($startx+18)) $(($1+2)) $2
    fb_put $(($startx+19)) $(($1+2)) $2
    fb_put $(($startx+20)) $(($1+2)) $2
    fb_put $(($startx+23)) $(($1+2)) $2
    fb_put $(($startx+25)) $(($1+2)) $2
    fb_put $(($startx+26)) $(($1+2)) $2
    fb_put $(($startx+27)) $(($1+2)) $2

    fb_put $((startx+13)) $(($1+3)) $2
    fb_put $(($startx+18)) $(($1+3)) $2
    fb_put $(($startx+21)) $(($1+3)) $2
    fb_put $(($startx+23)) $(($1+3)) $2
    fb_put $(($startx+25)) $(($1+3)) $2
    fb_put $(($startx+28)) $(($1+3)) $2

    fb_put $((startx+12)) $(($1+4)) $2
    fb_put $(($startx+18)) $(($1+4)) $2
    fb_put $(($startx+21)) $(($1+4)) $2
    fb_put $(($startx+23)) $(($1+4)) $2
    fb_put $(($startx+25)) $(($1+4)) $2
    fb_put $(($startx+28)) $(($1+4)) $2

    fb_put $((startx+11)) $(($1+5)) $2
    fb_put $(($startx+18)) $(($1+5)) $2
    fb_put $(($startx+21)) $(($1+5)) $2
    fb_put $(($startx+23)) $(($1+5)) $2
    fb_put $(($startx+25)) $(($1+5)) $2
    fb_put $(($startx+28)) $(($1+5)) $2

    fb_put $((startx+10)) $(($1+6)) $2
    fb_put $(($startx+18)) $(($1+6)) $2
    fb_put $(($startx+19)) $(($1+6)) $2
    fb_put $(($startx+20)) $(($1+6)) $2
    fb_put $(($startx+23)) $(($1+6)) $2
    fb_put $(($startx+25)) $(($1+6)) $2
    fb_put $(($startx+28)) $(($1+6)) $2
  fi

  if [ $3 -ge 3 ]
  then
    fb_put $(($startx+36)) $1 $2
    fb_put $(($startx+38)) $1 $2
    fb_put $(($startx+52)) $1 $2

    fb_put $(($startx+35)) $(($1+1)) $2
    fb_put $(($startx+38)) $(($1+1)) $2
    fb_put $(($startx+52)) $(($1+1)) $2

    fb_put $(($startx+34)) $(($1+2)) $2
    fb_put $(($startx+38)) $(($1+2)) $2
    fb_put $(($startx+39)) $(($1+2)) $2
    fb_put $(($startx+40)) $(($1+2)) $2
    fb_put $(($startx+44)) $(($1+2)) $2
    fb_put $(($startx+45)) $(($1+2)) $2
    fb_put $(($startx+49)) $(($1+2)) $2
    fb_put $(($startx+50)) $(($1+2)) $2
    fb_put $(($startx+52)) $(($1+2)) $2
    fb_put $(($startx+53)) $(($1+2)) $2
    fb_put $(($startx+54)) $(($1+2)) $2

    fb_put $(($startx+33)) $(($1+3)) $2
    fb_put $(($startx+38)) $(($1+3)) $2
    fb_put $(($startx+41)) $(($1+3)) $2
    fb_put $(($startx+46)) $(($1+3)) $2
    fb_put $(($startx+48)) $(($1+3)) $2
    fb_put $(($startx+52)) $(($1+3)) $2
    fb_put $(($startx+55)) $(($1+3)) $2

    fb_put $(($startx+32)) $(($1+4)) $2
    fb_put $(($startx+38)) $(($1+4)) $2
    fb_put $(($startx+41)) $(($1+4)) $2
    fb_put $(($startx+44)) $(($1+4)) $2
    fb_put $(($startx+45)) $(($1+4)) $2
    fb_put $(($startx+46)) $(($1+4)) $2
    fb_put $(($startx+49)) $(($1+4)) $2
    fb_put $(($startx+52)) $(($1+4)) $2
    fb_put $(($startx+55)) $(($1+4)) $2

    fb_put $(($startx+31)) $(($1+5)) $2
    fb_put $(($startx+38)) $(($1+5)) $2
    fb_put $(($startx+41)) $(($1+5)) $2
    fb_put $(($startx+43)) $(($1+5)) $2
    fb_put $(($startx+46)) $(($1+5)) $2
    fb_put $(($startx+50)) $(($1+5)) $2
    fb_put $(($startx+52)) $(($1+5)) $2
    fb_put $(($startx+55)) $(($1+5)) $2

    fb_put $(($startx+30)) $(($1+6)) $2
    fb_put $(($startx+38)) $(($1+6)) $2
    fb_put $(($startx+39)) $(($1+6)) $2
    fb_put $(($startx+40)) $(($1+6)) $2
    fb_put $(($startx+44)) $(($1+6)) $2
    fb_put $(($startx+45)) $(($1+6)) $2
    fb_put $(($startx+46)) $(($1+6)) $2
    fb_put $(($startx+48)) $(($1+6)) $2
    fb_put $(($startx+49)) $(($1+6)) $2
    fb_put $(($startx+52)) $(($1+6)) $2
    fb_put $(($startx+55)) $(($1+6)) $2
  fi
}


function bsh_logo {
  # Param: row color
  startx=$((($COLUMNS>>1)-24))
  if [ "$3" != "" ]; then startx=$3; fi

  fb_put $startx $1 $2
  fb_put $(($startx+14)) $1 $2
  fb_put $(($startx+24)) $1 $2
  fb_put $(($startx+34)) $1 $2
  fb_put $(($startx+36)) $1 $2

  fb_put $startx $(($1+1)) $2
  fb_put $(($startx+14)) $(($1+1)) $2
  fb_put $(($startx+24)) $(($1+1)) $2
  fb_put $(($startx+34)) $(($1+1)) $2

  fb_put $startx $(($1+2)) $2
  fb_put $(($startx+1)) $(($1+2)) $2
  fb_put $(($startx+2)) $(($1+2)) $2
  fb_put $(($startx+6)) $(($1+2)) $2
  fb_put $(($startx+7)) $(($1+2)) $2
  fb_put $(($startx+11)) $(($1+2)) $2
  fb_put $(($startx+12)) $(($1+2)) $2
  fb_put $(($startx+14)) $(($1+2)) $2
  fb_put $(($startx+15)) $(($1+2)) $2
  fb_put $(($startx+16)) $(($1+2)) $2
  fb_put $(($startx+20)) $(($1+2)) $2
  fb_put $(($startx+21)) $(($1+2)) $2
  fb_put $(($startx+24)) $(($1+2)) $2
  fb_put $(($startx+25)) $(($1+2)) $2
  fb_put $(($startx+26)) $(($1+2)) $2
  fb_put $(($startx+30)) $(($1+2)) $2
  fb_put $(($startx+31)) $(($1+2)) $2
  fb_put $(($startx+34)) $(($1+2)) $2
  fb_put $(($startx+36)) $(($1+2)) $2
  fb_put $(($startx+39)) $(($1+2)) $2
  fb_put $(($startx+40)) $(($1+2)) $2
  fb_put $(($startx+44)) $(($1+2)) $2
  fb_put $(($startx+45)) $(($1+2)) $2

  fb_put $startx $(($1+3)) $2
  fb_put $(($startx+3)) $(($1+3)) $2
  fb_put $(($startx+8)) $(($1+3)) $2
  fb_put $(($startx+10)) $(($1+3)) $2
  fb_put $(($startx+14)) $(($1+3)) $2
  fb_put $(($startx+17)) $(($1+3)) $2
  fb_put $(($startx+19)) $(($1+3)) $2
  fb_put $(($startx+22)) $(($1+3)) $2
  fb_put $(($startx+24)) $(($1+3)) $2
  fb_put $(($startx+27)) $(($1+3)) $2
  fb_put $(($startx+29)) $(($1+3)) $2
  fb_put $(($startx+32)) $(($1+3)) $2
  fb_put $(($startx+34)) $(($1+3)) $2
  fb_put $(($startx+36)) $(($1+3)) $2
  fb_put $(($startx+38)) $(($1+3)) $2
  fb_put $(($startx+41)) $(($1+3)) $2
  fb_put $(($startx+43)) $(($1+3)) $2

  fb_put $startx $(($1+4)) $2
  fb_put $(($startx+3)) $(($1+4)) $2
  fb_put $(($startx+6)) $(($1+4)) $2
  fb_put $(($startx+7)) $(($1+4)) $2
  fb_put $(($startx+8)) $(($1+4)) $2
  fb_put $(($startx+11)) $(($1+4)) $2
  fb_put $(($startx+14)) $(($1+4)) $2
  fb_put $(($startx+17)) $(($1+4)) $2
  fb_put $(($startx+19)) $(($1+4)) $2
  fb_put $(($startx+22)) $(($1+4)) $2
  fb_put $(($startx+24)) $(($1+4)) $2
  fb_put $(($startx+27)) $(($1+4)) $2
  fb_put $(($startx+29)) $(($1+4)) $2
  fb_put $(($startx+32)) $(($1+4)) $2
  fb_put $(($startx+34)) $(($1+4)) $2
  fb_put $(($startx+36)) $(($1+4)) $2
  fb_put $(($startx+38)) $(($1+4)) $2
  fb_put $(($startx+44)) $(($1+4)) $2

  fb_put $startx $(($1+5)) $2
  fb_put $(($startx+3)) $(($1+5)) $2
  fb_put $(($startx+5)) $(($1+5)) $2
  fb_put $(($startx+8)) $(($1+5)) $2
  fb_put $(($startx+12)) $(($1+5)) $2
  fb_put $(($startx+14)) $(($1+5)) $2
  fb_put $(($startx+17)) $(($1+5)) $2
  fb_put $(($startx+19)) $(($1+5)) $2
  fb_put $(($startx+22)) $(($1+5)) $2
  fb_put $(($startx+24)) $(($1+5)) $2
  fb_put $(($startx+27)) $(($1+5)) $2
  fb_put $(($startx+29)) $(($1+5)) $2
  fb_put $(($startx+32)) $(($1+5)) $2
  fb_put $(($startx+34)) $(($1+5)) $2
  fb_put $(($startx+36)) $(($1+5)) $2
  fb_put $(($startx+38)) $(($1+5)) $2
  fb_put $(($startx+41)) $(($1+5)) $2
  fb_put $(($startx+45)) $(($1+5)) $2

  fb_put $startx $(($1+6)) $2
  fb_put $(($startx+1)) $(($1+6)) $2
  fb_put $(($startx+2)) $(($1+6)) $2
  fb_put $(($startx+6)) $(($1+6)) $2
  fb_put $(($startx+7)) $(($1+6)) $2
  fb_put $(($startx+8)) $(($1+6)) $2
  fb_put $(($startx+10)) $(($1+6)) $2
  fb_put $(($startx+11)) $(($1+6)) $2
  fb_put $(($startx+14)) $(($1+6)) $2
  fb_put $(($startx+17)) $(($1+6)) $2
  fb_put $(($startx+20)) $(($1+6)) $2
  fb_put $(($startx+21)) $(($1+6)) $2
  fb_put $(($startx+24)) $(($1+6)) $2
  fb_put $(($startx+27)) $(($1+6)) $2
  fb_put $(($startx+30)) $(($1+6)) $2
  fb_put $(($startx+31)) $(($1+6)) $2
  fb_put $(($startx+34)) $(($1+6)) $2
  fb_put $(($startx+36)) $(($1+6)) $2
  fb_put $(($startx+39)) $(($1+6)) $2
  fb_put $(($startx+40)) $(($1+6)) $2
  fb_put $(($startx+43)) $(($1+6)) $2
  fb_put $(($startx+44)) $(($1+6)) $2

}

function bzh_logo {
  # Param: row color
  startx=$((($COLUMNS>>1)-24))
  if [ "$3" != "" ]; then startx=$3; fi
  fb_put $startx $1 $2
  fb_put $(($startx+25)) $1 $2
  fb_put $(($startx+35)) $1 $2
  fb_put $(($startx+37)) $1 $2
  fb_put $startx $(($1+1)) $2
  fb_put $(($startx+25)) $(($1+1)) $2
  fb_put $(($startx+35)) $(($1+1)) $2
  fb_put $startx $(($1+2)) $2
  fb_put $(($startx+1)) $(($1+2)) $2
  fb_put $(($startx+2)) $(($1+2)) $2
  fb_put $(($startx+6)) $(($1+2)) $2
  fb_put $(($startx+7)) $(($1+2)) $2
  fb_put $(($startx+11)) $(($1+2)) $2
  fb_put $(($startx+12)) $(($1+2)) $2
  fb_put $(($startx+15)) $(($1+2)) $2
  fb_put $(($startx+16)) $(($1+2)) $2
  fb_put $(($startx+17)) $(($1+2)) $2
  fb_put $(($startx+18)) $(($1+2)) $2
  fb_put $(($startx+21)) $(($1+2)) $2
  fb_put $(($startx+22)) $(($1+2)) $2
  fb_put $(($startx+25)) $(($1+2)) $2
  fb_put $(($startx+26)) $(($1+2)) $2
  fb_put $(($startx+27)) $(($1+2)) $2
  fb_put $(($startx+31)) $(($1+2)) $2
  fb_put $(($startx+32)) $(($1+2)) $2
  fb_put $(($startx+35)) $(($1+2)) $2
  fb_put $(($startx+37)) $(($1+2)) $2
  fb_put $(($startx+40)) $(($1+2)) $2
  fb_put $(($startx+41)) $(($1+2)) $2
  fb_put $(($startx+44)) $(($1+2)) $2
  fb_put $(($startx+45)) $(($1+2)) $2
  fb_put $(($startx+46)) $(($1+2)) $2
  fb_put $startx $(($1+3)) $2
  fb_put $(($startx+3)) $(($1+3)) $2
  fb_put $(($startx+5)) $(($1+3)) $2
  fb_put $(($startx+8)) $(($1+3)) $2
  fb_put $(($startx+10)) $(($1+3)) $2
  fb_put $(($startx+13)) $(($1+3)) $2
  fb_put $(($startx+17)) $(($1+3)) $2
  fb_put $(($startx+20)) $(($1+3)) $2
  fb_put $(($startx+23)) $(($1+3)) $2
  fb_put $(($startx+25)) $(($1+3)) $2
  fb_put $(($startx+28)) $(($1+3)) $2
  fb_put $(($startx+30)) $(($1+3)) $2
  fb_put $(($startx+33)) $(($1+3)) $2
  fb_put $(($startx+35)) $(($1+3)) $2
  fb_put $(($startx+37)) $(($1+3)) $2
  fb_put $(($startx+39)) $(($1+3)) $2
  fb_put $(($startx+43)) $(($1+3)) $2
  fb_put $(($startx+44)) $(($1+3)) $2
  fb_put $startx $(($1+4)) $2
  fb_put $(($startx+3)) $(($1+4)) $2
  fb_put $(($startx+5)) $(($1+4)) $2
  fb_put $(($startx+8)) $(($1+4)) $2
  fb_put $(($startx+10)) $(($1+4)) $2
  fb_put $(($startx+13)) $(($1+4)) $2
  fb_put $(($startx+16)) $(($1+4)) $2
  fb_put $(($startx+20)) $(($1+4)) $2
  fb_put $(($startx+23)) $(($1+4)) $2
  fb_put $(($startx+25)) $(($1+4)) $2
  fb_put $(($startx+28)) $(($1+4)) $2
  fb_put $(($startx+30)) $(($1+4)) $2
  fb_put $(($startx+33)) $(($1+4)) $2
  fb_put $(($startx+35)) $(($1+4)) $2
  fb_put $(($startx+37)) $(($1+4)) $2
  fb_put $(($startx+39)) $(($1+4)) $2
  fb_put $(($startx+45)) $(($1+4)) $2
  fb_put $(($startx+46)) $(($1+4)) $2
  fb_put $startx $(($1+5)) $2
  fb_put $(($startx+1)) $(($1+5)) $2
  fb_put $(($startx+2)) $(($1+5)) $2
  fb_put $(($startx+6)) $(($1+5)) $2
  fb_put $(($startx+7)) $(($1+5)) $2
  fb_put $(($startx+11)) $(($1+5)) $2
  fb_put $(($startx+12)) $(($1+5)) $2
  fb_put $(($startx+15)) $(($1+5)) $2
  fb_put $(($startx+16)) $(($1+5)) $2
  fb_put $(($startx+17)) $(($1+5)) $2
  fb_put $(($startx+18)) $(($1+5)) $2
  fb_put $(($startx+21)) $(($1+5)) $2
  fb_put $(($startx+22)) $(($1+5)) $2
  fb_put $(($startx+25)) $(($1+5)) $2
  fb_put $(($startx+28)) $(($1+5)) $2
  fb_put $(($startx+31)) $(($1+5)) $2
  fb_put $(($startx+32)) $(($1+5)) $2
  fb_put $(($startx+35)) $(($1+5)) $2
  fb_put $(($startx+37)) $(($1+5)) $2
  fb_put $(($startx+40)) $(($1+5)) $2
  fb_put $(($startx+41)) $(($1+5)) $2
  fb_put $(($startx+43)) $(($1+5)) $2
  fb_put $(($startx+44)) $(($1+5)) $2
  fb_put $(($startx+45)) $(($1+5)) $2
}

function sine_move {
  # Params: amplitude1 amplitude2 anglefreq1 anglefreq2 phase1 phase2 function
  echo "amp2=$2" >&7
  echo "phix=$5*pi/180" >&7
  echo "phiy=$6*pi/180" >&7

  while :
  do
    let t=0
    while [ $t -lt 700 ]
    do
      echo "scale=5;x=($1*s($3*($t/70)+phix))*10;scale=0;(x/1)+40" >&7
      read -t1 x <&8
      echo "scale=5;y=(amp2*s($4*($t/70)+phiy))*10;scale=0;(y/1)+7" >&7
      read -t1 y <&8

      fb_clear
      ${7} $x $y

      refresh
      let t+=1
    done
  done
}

function draw_smurf4 {
  # Params: x y
  let x=$1
  let y=$2

  fb_put $(($x-1)) $(($y-1)) $BLACK
  fb_put $x $(($y-1)) $BLACK
  fb_put $(($x+1)) $(($y-1)) $BLACK
  fb_put $(($x+2)) $(($y-1)) $BLACK
  fb_put $(($x+3)) $(($y-1)) $BLACK
  fb_put $(($x+4)) $(($y-1)) $BLACK

  fb_put $(($x-3)) $y $BLACK
  fb_put $(($x-2)) $y $BLACK
  fb_put $(($x-1)) $y $WHITE
  fb_put $x $y $WHITE
  fb_put $(($x+1)) $y $WHITE
  fb_put $(($x+2)) $y $WHITE
  fb_put $(($x+3)) $y $WHITE
  fb_put $(($x+4)) $y $WHITE
  fb_put $(($x+5)) $y $BLACK
  fb_put $(($x+6)) $y $BLACK

  fb_put $(($x-5)) $(($y+1)) $BLACK
  fb_put $(($x-4)) $(($y+1)) $BLACK
  fb_put $(($x-3)) $(($y+1)) $WHITE
  fb_put $(($x-2)) $(($y+1)) $WHITE
  fb_put $(($x-1)) $(($y+1)) $WHITE
  fb_put $x $(($y+1)) $WHITE
  fb_put $(($x+1)) $(($y+1)) $WHITE
  fb_put $(($x+2)) $(($y+1)) $WHITE
  fb_put $(($x+3)) $(($y+1)) $WHITE
  fb_put $(($x+4)) $(($y+1)) $WHITE
  fb_put $(($x+5)) $(($y+1)) $WHITE
  fb_put $(($x+6)) $(($y+1)) $WHITE
  fb_put $(($x+7)) $(($y+1)) $BLACK

  fb_put $(($x-6)) $(($y+2)) $BLACK
  fb_put $(($x-5)) $(($y+2)) $WHITE
  fb_put $(($x-4)) $(($y+2)) $WHITE
  fb_put $(($x-3)) $(($y+2)) $WHITE
  fb_put $(($x-2)) $(($y+2)) $WHITE
  fb_put $(($x-1)) $(($y+2)) $WHITE
  fb_put $x $(($y+2)) $WHITE
  fb_put $(($x+1)) $(($y+2)) $BLACK
  fb_put $(($x+2)) $(($y+2)) $BLACK
  fb_put $(($x+3)) $(($y+2)) $WHITE
  fb_put $(($x+4)) $(($y+2)) $WHITE
  fb_put $(($x+5)) $(($y+2)) $WHITE
  fb_put $(($x+6)) $(($y+2)) $WHITE
  fb_put $(($x+7)) $(($y+2)) $BLACK

  fb_put $(($x-7)) $(($y+3)) $BLACK
  fb_put $(($x-6)) $(($y+3)) $BLACK
  fb_put $(($x-5)) $(($y+3)) $WHITE
  fb_put $(($x-4)) $(($y+3)) $WHITE
  fb_put $(($x-3)) $(($y+3)) $WHITE
  fb_put $(($x-2)) $(($y+3)) $WHITE
  fb_put $(($x-1)) $(($y+3)) $WHITE
  fb_put $x $(($y+3)) $WHITE
  fb_put $(($x+1)) $(($y+3)) $WHITE
  fb_put $(($x+2)) $(($y+3)) $WHITE
  fb_put $(($x+3)) $(($y+3)) $BLACK
  fb_put $(($x+4)) $(($y+3)) $BLACK
  fb_put $(($x+5)) $(($y+3)) $BLACK
  fb_put $(($x+6)) $(($y+3)) $BLACK

  fb_put $(($x-8)) $(($y+4)) $BLACK
  fb_put $(($x-7)) $(($y+4)) $WHITE
  fb_put $(($x-6)) $(($y+4)) $WHITE
  fb_put $(($x-5)) $(($y+4)) $WHITE
  fb_put $(($x-4)) $(($y+4)) $WHITE
  fb_put $(($x-3)) $(($y+4)) $WHITE
  fb_put $(($x-2)) $(($y+4)) $WHITE
  fb_put $(($x-1)) $(($y+4)) $WHITE
  fb_put $x $(($y+4)) $WHITE
  fb_put $(($x+1)) $(($y+4)) $WHITE
  fb_put $(($x+2)) $(($y+4)) $WHITE
  fb_put $(($x+3)) $(($y+4)) $WHITE
  fb_put $(($x+4)) $(($y+4)) $WHITE
  fb_put $(($x+5)) $(($y+4)) $BLACK
  fb_put $(($x+6)) $(($y+4)) $BLACK

  fb_put $(($x-8)) $(($y+5)) $BLACK
  fb_put $(($x-7)) $(($y+5)) $WHITE
  fb_put $(($x-6)) $(($y+5)) $WHITE
  fb_put $(($x-5)) $(($y+5)) $WHITE
  fb_put $(($x-4)) $(($y+5)) $WHITE
  fb_put $(($x-3)) $(($y+5)) $WHITE
  fb_put $(($x-2)) $(($y+5)) $WHITE
  fb_put $(($x-1)) $(($y+5)) $WHITE
  fb_put $x $(($y+5)) $WHITE
  fb_put $(($x+1)) $(($y+5)) $WHITE
  fb_put $(($x+2)) $(($y+5)) $WHITE
  fb_put $(($x+3)) $(($y+5)) $WHITE
  fb_put $(($x+4)) $(($y+5)) $WHITE
  fb_put $(($x+5)) $(($y+5)) $WHITE
  fb_put $(($x+6)) $(($y+5)) $WHITE
  fb_put $(($x+7)) $(($y+5)) $BLACK

  fb_put $(($x-7)) $(($y+6)) $BLACK
  fb_put $(($x-6)) $(($y+6)) $BLACK
  fb_put $(($x-5)) $(($y+6)) $BLACK
  fb_put $(($x-4)) $(($y+6)) $BLACK
  fb_put $(($x-3)) $(($y+6)) $BLUE
  fb_put $(($x-2)) $(($y+6)) $BLUE
  fb_put $(($x-1)) $(($y+6)) $BLUE
  fb_put $x $(($y+6)) $BLUE
  fb_put $(($x+1)) $(($y+6)) $WHITE
  fb_put $(($x+2)) $(($y+6)) $WHITE
  fb_put $(($x+3)) $(($y+6)) $BLUE
  fb_put $(($x+4)) $(($y+6)) $BLUE
  fb_put $(($x+5)) $(($y+6)) $BLACK
  fb_put $(($x+6)) $(($y+6)) $BLACK
  fb_put $(($x+7)) $(($y+6)) $BLACK
  fb_put $(($x+8)) $(($y+6)) $BLACK

  fb_put $(($x-8)) $(($y+7)) $BLACK
  fb_put $(($x-7)) $(($y+7)) $BLUE
  fb_put $(($x-6)) $(($y+7)) $BLUE
  fb_put $(($x-5)) $(($y+7)) $BLUE
  fb_put $(($x-4)) $(($y+7)) $BLUE
  fb_put $(($x-3)) $(($y+7)) $BLUE
  fb_put $(($x-2)) $(($y+7)) $BLUE
  fb_put $(($x-1)) $(($y+7)) $BLUE
  fb_put $x $(($y+7)) $BLUE
  fb_put $(($x+1)) $(($y+7)) $WHITE
  fb_put $(($x+2)) $(($y+7)) $BLACK
  fb_put $(($x+3)) $(($y+7)) $BLUE
  fb_put $(($x+4)) $(($y+7)) $BLUE
  fb_put $(($x+5)) $(($y+7)) $BLUE
  fb_put $(($x+6)) $(($y+7)) $BLUE
  fb_put $(($x+7)) $(($y+7)) $BLUE
  fb_put $(($x+8)) $(($y+7)) $BLUE
  fb_put $(($x+9)) $(($y+7)) $BLACK

  fb_put $(($x-8)) $(($y+8)) $BLACK
  fb_put $(($x-7)) $(($y+8)) $BLUE
  fb_put $(($x-6)) $(($y+8)) $BLUE
  fb_put $(($x-5)) $(($y+8)) $BLUE
  fb_put $(($x-4)) $(($y+8)) $BLUE
  fb_put $(($x-3)) $(($y+8)) $BLUE
  fb_put $(($x-2)) $(($y+8)) $BLUE
  fb_put $(($x-1)) $(($y+8)) $BLUE
  fb_put $x $(($y+8)) $BLUE
  fb_put $(($x+1)) $(($y+8)) $BLUE
  fb_put $(($x+2)) $(($y+8)) $BLUE
  fb_put $(($x+3)) $(($y+8)) $BLUE
  fb_put $(($x+4)) $(($y+8)) $BLUE
  fb_put $(($x+5)) $(($y+8)) $BLUE
  fb_put $(($x+6)) $(($y+8)) $BLUE
  fb_put $(($x+7)) $(($y+8)) $BLUE
  fb_put $(($x+8)) $(($y+8)) $BLUE
  fb_put $(($x+9)) $(($y+8)) $BLACK

  fb_put $(($x-8)) $(($y+9)) $BLACK
  fb_put $(($x-7)) $(($y+9)) $BLUE
  fb_put $(($x-6)) $(($y+9)) $BLUE
  fb_put $(($x-5)) $(($y+9)) $BLUE
  fb_put $(($x-4)) $(($y+9)) $BLUE
  fb_put $(($x-3)) $(($y+9)) $BLUE
  fb_put $(($x-2)) $(($y+9)) $BLUE
  fb_put $(($x-1)) $(($y+9)) $BLACK
  fb_put $x $(($y+9)) $BLACK
  fb_put $(($x+1)) $(($y+9)) $BLUE
  fb_put $(($x+2)) $(($y+9)) $BLUE
  fb_put $(($x+3)) $(($y+9)) $BLUE
  fb_put $(($x+4)) $(($y+9)) $BLUE
  fb_put $(($x+5)) $(($y+9)) $BLUE
  fb_put $(($x+6)) $(($y+9)) $BLUE
  fb_put $(($x+7)) $(($y+9)) $BLACK
  fb_put $(($x+8)) $(($y+9)) $BLACK

  fb_put $(($x-7)) $(($y+10)) $BLACK
  fb_put $(($x-6)) $(($y+10)) $BLACK
  fb_put $(($x-5)) $(($y+10)) $BLACK
  fb_put $(($x-4)) $(($y+10)) $BLACK
  fb_put $(($x-3)) $(($y+10)) $BLUE
  fb_put $(($x-2)) $(($y+10)) $BLUE
  fb_put $(($x-1)) $(($y+10)) $BLUE
  fb_put $x $(($y+10)) $BLUE
  fb_put $(($x+1)) $(($y+10)) $BLACK
  fb_put $(($x+2)) $(($y+10)) $BLACK
  fb_put $(($x+3)) $(($y+10)) $BLACK
  fb_put $(($x+4)) $(($y+10)) $BLACK
  fb_put $(($x+5)) $(($y+10)) $BLACK
  fb_put $(($x+6)) $(($y+10)) $BLACK

  fb_put $(($x-3)) $(($y+11)) $BLACK
  fb_put $(($x-2)) $(($y+11)) $BLACK
  fb_put $(($x-1)) $(($y+11)) $BLUE
  fb_put $x $(($y+11)) $BLUE
  fb_put $(($x+1)) $(($y+11)) $BLUE
  fb_put $(($x+2)) $(($y+11)) $BLUE
  fb_put $(($x+3)) $(($y+11)) $BLUE
  fb_put $(($x+4)) $(($y+11)) $BLUE
  fb_put $(($x+5)) $(($y+11)) $BLACK

  fb_put $(($x-3)) $(($y+12)) $BLACK
  fb_put $(($x-2)) $(($y+12)) $BLACK
  fb_put $(($x-1)) $(($y+12)) $BLUE
  fb_put $x $(($y+12)) $BLUE
  fb_put $(($x+1)) $(($y+12)) $BLUE
  fb_put $(($x+2)) $(($y+12)) $BLUE
  fb_put $(($x+3)) $(($y+12)) $BLACK
  fb_put $(($x+4)) $(($y+12)) $BLACK

  fb_put $(($x-5)) $(($y+13)) $BLACK
  fb_put $(($x-4)) $(($y+13)) $BLACK
  fb_put $(($x-3)) $(($y+13)) $BLUE
  fb_put $(($x-2)) $(($y+13)) $BLUE
  fb_put $(($x-1)) $(($y+13)) $BLUE
  fb_put $x $(($y+13)) $BLUE
  fb_put $(($x+1)) $(($y+13)) $BLUE
  fb_put $(($x+2)) $(($y+13)) $BLUE
  fb_put $(($x+3)) $(($y+13)) $BLUE
  fb_put $(($x+4)) $(($y+13)) $BLUE
  fb_put $(($x+5)) $(($y+13)) $BLACK

  fb_put $(($x-7)) $(($y+14)) $BLACK
  fb_put $(($x-6)) $(($y+14)) $BLACK
  fb_put $(($x-5)) $(($y+14)) $BLUE
  fb_put $(($x-4)) $(($y+14)) $BLUE
  fb_put $(($x-3)) $(($y+14)) $BLUE
  fb_put $(($x-2)) $(($y+14)) $BLUE
  fb_put $(($x-1)) $(($y+14)) $BLUE
  fb_put $x $(($y+14)) $BLUE
  fb_put $(($x+1)) $(($y+14)) $BLUE
  fb_put $(($x+2)) $(($y+14)) $BLUE
  fb_put $(($x+3)) $(($y+14)) $BLUE
  fb_put $(($x+4)) $(($y+14)) $BLUE
  fb_put $(($x+5)) $(($y+14)) $BLACK
  fb_put $(($x+6)) $(($y+14)) $BLACK

  fb_put $(($x-8)) $(($y+15)) $BLACK
  fb_put $(($x-7)) $(($y+15)) $BLUE
  fb_put $(($x-6)) $(($y+15)) $BLUE
  fb_put $(($x-5)) $(($y+15)) $BLUE
  fb_put $(($x-4)) $(($y+15)) $BLUE
  fb_put $(($x-3)) $(($y+15)) $BLUE
  fb_put $(($x-2)) $(($y+15)) $BLUE
  fb_put $(($x-1)) $(($y+15)) $BLUE
  fb_put $x $(($y+15)) $BLUE
  fb_put $(($x+1)) $(($y+15)) $BLUE
  fb_put $(($x+2)) $(($y+15)) $BLUE
  fb_put $(($x+3)) $(($y+15)) $BLUE
  fb_put $(($x+4)) $(($y+15)) $BLUE
  fb_put $(($x+5)) $(($y+15)) $BLUE
  fb_put $(($x+6)) $(($y+15)) $BLUE
  fb_put $(($x+7)) $(($y+15)) $BLACK

  fb_put $(($x-8)) $(($y+16)) $BLACK
  fb_put $(($x-7)) $(($y+16)) $BLUE
  fb_put $(($x-6)) $(($y+16)) $BLUE
  fb_put $(($x-5)) $(($y+16)) $BLUE
  fb_put $(($x-4)) $(($y+16)) $BLUE
  fb_put $(($x-3)) $(($y+16)) $BLUE
  fb_put $(($x-2)) $(($y+16)) $BLUE
  fb_put $(($x-1)) $(($y+16)) $BLUE
  fb_put $x $(($y+16)) $BLUE
  fb_put $(($x+1)) $(($y+16)) $BLUE
  fb_put $(($x+2)) $(($y+16)) $BLUE
  fb_put $(($x+3)) $(($y+16)) $BLUE
  fb_put $(($x+4)) $(($y+16)) $BLUE
  fb_put $(($x+5)) $(($y+16)) $BLUE
  fb_put $(($x+6)) $(($y+16)) $BLUE
  fb_put $(($x+7)) $(($y+16)) $BLACK

  fb_put $(($x-7)) $(($y+17)) $BLACK
  fb_put $(($x-6)) $(($y+17)) $BLACK
  fb_put $(($x-5)) $(($y+17)) $BLUE
  fb_put $(($x-4)) $(($y+17)) $BLUE
  fb_put $(($x-3)) $(($y+17)) $BLUE
  fb_put $(($x-2)) $(($y+17)) $BLUE
  fb_put $(($x-1)) $(($y+17)) $BLUE
  fb_put $x $(($y+17)) $BLUE
  fb_put $(($x+1)) $(($y+17)) $BLUE
  fb_put $(($x+2)) $(($y+17)) $BLUE
  fb_put $(($x+3)) $(($y+17)) $BLUE
  fb_put $(($x+4)) $(($y+17)) $BLUE
  fb_put $(($x+5)) $(($y+17)) $BLACK
  fb_put $(($x+6)) $(($y+17)) $BLACK

  fb_put $(($x-5)) $(($y+18)) $BLACK
  fb_put $(($x-4)) $(($y+18)) $BLACK
  fb_put $(($x-3)) $(($y+18)) $WHITE
  fb_put $(($x-2)) $(($y+18)) $WHITE
  fb_put $(($x-1)) $(($y+18)) $WHITE
  fb_put $x $(($y+18)) $WHITE
  fb_put $(($x+1)) $(($y+18)) $WHITE
  fb_put $(($x+2)) $(($y+18)) $WHITE
  fb_put $(($x+3)) $(($y+18)) $WHITE
  fb_put $(($x+4)) $(($y+18)) $WHITE
  fb_put $(($x+5)) $(($y+18)) $BLACK

  fb_put $(($x-4)) $(($y+19)) $BLACK
  fb_put $(($x-3)) $(($y+19)) $WHITE
  fb_put $(($x-2)) $(($y+19)) $WHITE
  fb_put $(($x-1)) $(($y+19)) $WHITE
  fb_put $x $(($y+19)) $WHITE
  fb_put $(($x+1)) $(($y+19)) $WHITE
  fb_put $(($x+2)) $(($y+19)) $WHITE
  fb_put $(($x+3)) $(($y+19)) $WHITE
  fb_put $(($x+4)) $(($y+19)) $WHITE
  fb_put $(($x+5)) $(($y+19)) $BLACK
  fb_put $(($x+6)) $(($y+19)) $BLACK

  fb_put $(($x-5)) $(($y+20)) $BLACK
  fb_put $(($x-4)) $(($y+20)) $BLACK
  fb_put $(($x-3)) $(($y+20)) $WHITE
  fb_put $(($x-2)) $(($y+20)) $WHITE
  fb_put $(($x-1)) $(($y+20)) $WHITE
  fb_put $x $(($y+20)) $WHITE
  fb_put $(($x+1)) $(($y+20)) $BLACK
  fb_put $(($x+2)) $(($y+20)) $BLACK
  fb_put $(($x+3)) $(($y+20)) $WHITE
  fb_put $(($x+4)) $(($y+20)) $WHITE
  fb_put $(($x+5)) $(($y+20)) $WHITE
  fb_put $(($x+6)) $(($y+20)) $WHITE
  fb_put $(($x+7)) $(($y+20)) $BLACK

  fb_put $(($x-6)) $(($y+21)) $BLACK
  fb_put $(($x-5)) $(($y+21)) $WHITE
  fb_put $(($x-4)) $(($y+21)) $WHITE
  fb_put $(($x-3)) $(($y+21)) $WHITE
  fb_put $(($x-2)) $(($y+21)) $WHITE
  fb_put $(($x-1)) $(($y+21)) $BLACK
  fb_put $x $(($y+21)) $BLACK
  fb_put $(($x+1)) $(($y+21)) $BLACK
  fb_put $(($x+2)) $(($y+21)) $BLACK
  fb_put $(($x+3)) $(($y+21)) $WHITE
  fb_put $(($x+4)) $(($y+21)) $WHITE
  fb_put $(($x+5)) $(($y+21)) $WHITE
  fb_put $(($x+6)) $(($y+21)) $WHITE
  fb_put $(($x+7)) $(($y+21)) $BLACK

  fb_put $(($x-7)) $(($y+22)) $BLACK
  fb_put $(($x-6)) $(($y+22)) $BLACK
  fb_put $(($x-5)) $(($y+22)) $WHITE
  fb_put $(($x-4)) $(($y+22)) $WHITE
  fb_put $(($x-3)) $(($y+22)) $WHITE
  fb_put $(($x-2)) $(($y+22)) $WHITE
  fb_put $(($x-1)) $(($y+22)) $BLACK
  fb_put $x $(($y+22)) $BLACK
  fb_put $(($x+1)) $(($y+22)) $WHITE
  fb_put $(($x+2)) $(($y+22)) $WHITE
  fb_put $(($x+3)) $(($y+22)) $WHITE
  fb_put $(($x+4)) $(($y+22)) $WHITE
  fb_put $(($x+5)) $(($y+22)) $BLACK
  fb_put $(($x+6)) $(($y+22)) $BLACK
  fb_put $(($x+7)) $(($y+22)) $BLACK
  fb_put $(($x+8)) $(($y+22)) $BLACK

  fb_put $(($x-8)) $(($y+23)) $BLACK
  fb_put $(($x-7)) $(($y+23)) $WHITE
  fb_put $(($x-6)) $(($y+23)) $WHITE
  fb_put $(($x-5)) $(($y+23)) $WHITE
  fb_put $(($x-4)) $(($y+23)) $WHITE
  fb_put $(($x-3)) $(($y+23)) $BLACK
  fb_put $(($x-2)) $(($y+23)) $BLACK
  fb_put $(($x-1)) $(($y+23)) $BLACK
  fb_put $x $(($y+23)) $BLACK
  fb_put $(($x+1)) $(($y+23)) $WHITE
  fb_put $(($x+2)) $(($y+23)) $WHITE
  fb_put $(($x+3)) $(($y+23)) $WHITE
  fb_put $(($x+4)) $(($y+23)) $WHITE
  fb_put $(($x+5)) $(($y+23)) $WHITE
  fb_put $(($x+6)) $(($y+23)) $WHITE
  fb_put $(($x+7)) $(($y+23)) $WHITE
  fb_put $(($x+8)) $(($y+23)) $WHITE
  fb_put $(($x+9)) $(($y+23)) $BLACK

  fb_put $(($x-8)) $(($y+24)) $BLACK
  fb_put $(($x-7)) $(($y+24)) $WHITE
  fb_put $(($x-6)) $(($y+24)) $WHITE
  fb_put $(($x-5)) $(($y+24)) $WHITE
  fb_put $(($x-4)) $(($y+24)) $WHITE
  fb_put $(($x-3)) $(($y+24)) $WHITE
  fb_put $(($x-2)) $(($y+24)) $WHITE
  fb_put $(($x-1)) $(($y+24)) $WHITE
  fb_put $x $(($y+24)) $WHITE
  fb_put $(($x+1)) $(($y+24)) $BLACK
  fb_put $(($x+2)) $(($y+24)) $BLACK
  fb_put $(($x+3)) $(($y+24)) $WHITE
  fb_put $(($x+4)) $(($y+24)) $WHITE
  fb_put $(($x+5)) $(($y+24)) $WHITE
  fb_put $(($x+6)) $(($y+24)) $WHITE
  fb_put $(($x+7)) $(($y+24)) $WHITE
  fb_put $(($x+8)) $(($y+24)) $WHITE
  fb_put $(($x+9)) $(($y+24)) $BLACK

  fb_put $(($x-7)) $(($y+25)) $BLACK
  fb_put $(($x-6)) $(($y+25)) $BLACK
  fb_put $(($x-5)) $(($y+25)) $WHITE
  fb_put $(($x-4)) $(($y+25)) $WHITE
  fb_put $(($x-3)) $(($y+25)) $WHITE
  fb_put $(($x-2)) $(($y+25)) $WHITE
  fb_put $(($x-1)) $(($y+25)) $WHITE
  fb_put $x $(($y+25)) $WHITE
  fb_put $(($x+1)) $(($y+25)) $BLACK
  fb_put $(($x+3)) $(($y+25)) $BLACK
  fb_put $(($x+4)) $(($y+25)) $BLACK
  fb_put $(($x+5)) $(($y+25)) $BLACK
  fb_put $(($x+6)) $(($y+25)) $BLACK
  fb_put $(($x+7)) $(($y+25)) $BLACK
  fb_put $(($x+8)) $(($y+25)) $BLACK

  fb_put $(($x-5)) $(($y+26)) $BLACK
  fb_put $(($x-4)) $(($y+26)) $BLACK
  fb_put $(($x-3)) $(($y+26)) $BLACK
  fb_put $(($x-2)) $(($y+26)) $BLACK
  fb_put $(($x-1)) $(($y+26)) $BLACK
  fb_put $x $(($y+26)) $BLACK
}

function draw_smurf3 {
  # Params: x y
  let x=$1
  let y=$2

  fb_put $(($x-1)) $y $BLACK
  fb_put $x $(($y+1)) $BLACK
  fb_put $(($x+1)) $y $BLACK
  fb_put $(($x+2)) $y $BLACK
  fb_put $(($x+3)) $y $BLACK
  fb_put $(($x+4)) $y $BLACK

  fb_put $(($x-3)) $(($y+1)) $BLACK
  fb_put $(($x-2)) $(($y+1)) $BLACK
  fb_put $(($x-1)) $(($y+1)) $WHITE
  fb_put $x $(($y+1)) $WHITE
  fb_put $(($x+1)) $(($y+1)) $WHITE
  fb_put $(($x+2)) $(($y+1)) $WHITE
  fb_put $(($x+3)) $(($y+1)) $WHITE
  fb_put $(($x+4)) $(($y+1)) $WHITE
  fb_put $(($x+5)) $(($y+1)) $BLACK
  fb_put $(($x+6)) $(($y+1)) $BLACK

  fb_put $(($x-5)) $(($y+2)) $BLACK
  fb_put $(($x-4)) $(($y+2)) $BLACK
  fb_put $(($x-3)) $(($y+2)) $WHITE
  fb_put $(($x-2)) $(($y+2)) $WHITE
  fb_put $(($x-1)) $(($y+2)) $WHITE
  fb_put $x $(($y+2)) $WHITE
  fb_put $(($x+1)) $(($y+2)) $WHITE
  fb_put $(($x+2)) $(($y+2)) $WHITE
  fb_put $(($x+3)) $(($y+2)) $WHITE
  fb_put $(($x+4)) $(($y+2)) $WHITE
  fb_put $(($x+5)) $(($y+2)) $WHITE
  fb_put $(($x+6)) $(($y+2)) $WHITE
  fb_put $(($x+7)) $(($y+2)) $BLACK

  fb_put $(($x-6)) $(($y+3)) $BLACK
  fb_put $(($x-5)) $(($y+3)) $WHITE
  fb_put $(($x-4)) $(($y+3)) $WHITE
  fb_put $(($x-3)) $(($y+3)) $WHITE
  fb_put $(($x-2)) $(($y+3)) $WHITE
  fb_put $(($x-1)) $(($y+3)) $WHITE
  fb_put $x $(($y+3)) $WHITE
  fb_put $(($x+1)) $(($y+3)) $BLACK
  fb_put $(($x+2)) $(($y+3)) $BLACK
  fb_put $(($x+3)) $(($y+3)) $WHITE
  fb_put $(($x+4)) $(($y+3)) $WHITE
  fb_put $(($x+5)) $(($y+3)) $WHITE
  fb_put $(($x+6)) $(($y+3)) $WHITE
  fb_put $(($x+7)) $(($y+3)) $BLACK

  fb_put $(($x-7)) $(($y+4)) $BLACK
  fb_put $(($x-6)) $(($y+4)) $BLACK
  fb_put $(($x-5)) $(($y+4)) $WHITE
  fb_put $(($x-4)) $(($y+4)) $WHITE
  fb_put $(($x-3)) $(($y+4)) $WHITE
  fb_put $(($x-2)) $(($y+4)) $WHITE
  fb_put $(($x-1)) $(($y+4)) $WHITE
  fb_put $x $(($y+4)) $WHITE
  fb_put $(($x+1)) $(($y+4)) $WHITE
  fb_put $(($x+2)) $(($y+4)) $WHITE
  fb_put $(($x+3)) $(($y+4)) $BLACK
  fb_put $(($x+4)) $(($y+4)) $BLACK
  fb_put $(($x+5)) $(($y+4)) $BLACK
  fb_put $(($x+6)) $(($y+4)) $BLACK

  fb_put $(($x-8)) $(($y+5)) $BLACK
  fb_put $(($x-7)) $(($y+5)) $WHITE
  fb_put $(($x-6)) $(($y+5)) $WHITE
  fb_put $(($x-5)) $(($y+5)) $WHITE
  fb_put $(($x-4)) $(($y+5)) $WHITE
  fb_put $(($x-3)) $(($y+5)) $WHITE
  fb_put $(($x-2)) $(($y+5)) $WHITE
  fb_put $(($x-1)) $(($y+5)) $WHITE
  fb_put $x $(($y+5)) $WHITE
  fb_put $(($x+1)) $(($y+5)) $WHITE
  fb_put $(($x+2)) $(($y+5)) $WHITE
  fb_put $(($x+3)) $(($y+5)) $WHITE
  fb_put $(($x+4)) $(($y+5)) $WHITE
  fb_put $(($x+5)) $(($y+5)) $BLACK
  fb_put $(($x+6)) $(($y+5)) $BLACK

  fb_put $(($x-8)) $(($y+6)) $BLACK
  fb_put $(($x-7)) $(($y+6)) $WHITE
  fb_put $(($x-6)) $(($y+6)) $WHITE
  fb_put $(($x-5)) $(($y+6)) $WHITE
  fb_put $(($x-4)) $(($y+6)) $WHITE
  fb_put $(($x-3)) $(($y+6)) $WHITE
  fb_put $(($x-2)) $(($y+6)) $WHITE
  fb_put $(($x-1)) $(($y+6)) $WHITE
  fb_put $x $(($y+6)) $WHITE
  fb_put $(($x+1)) $(($y+6)) $WHITE
  fb_put $(($x+2)) $(($y+6)) $WHITE
  fb_put $(($x+3)) $(($y+6)) $WHITE
  fb_put $(($x+4)) $(($y+6)) $WHITE
  fb_put $(($x+5)) $(($y+6)) $WHITE
  fb_put $(($x+6)) $(($y+6)) $WHITE
  fb_put $(($x+7)) $(($y+6)) $BLACK

  fb_put $(($x-7)) $(($y+7)) $BLACK
  fb_put $(($x-6)) $(($y+7)) $BLACK
  fb_put $(($x-5)) $(($y+7)) $BLACK
  fb_put $(($x-4)) $(($y+7)) $BLACK
  fb_put $(($x-3)) $(($y+7)) $BLUE
  fb_put $(($x-2)) $(($y+7)) $BLUE
  fb_put $(($x-1)) $(($y+7)) $BLUE
  fb_put $x $(($y+7)) $BLUE
  fb_put $(($x+1)) $(($y+7)) $WHITE
  fb_put $(($x+2)) $(($y+7)) $WHITE
  fb_put $(($x+3)) $(($y+7)) $BLUE
  fb_put $(($x+4)) $(($y+7)) $BLUE
  fb_put $(($x+5)) $(($y+7)) $BLACK
  fb_put $(($x+6)) $(($y+7)) $BLACK
  fb_put $(($x+7)) $(($y+7)) $BLACK
  fb_put $(($x+8)) $(($y+7)) $BLACK

  fb_put $(($x-8)) $(($y+8)) $BLACK
  fb_put $(($x-7)) $(($y+8)) $BLUE
  fb_put $(($x-6)) $(($y+8)) $BLUE
  fb_put $(($x-5)) $(($y+8)) $BLUE
  fb_put $(($x-4)) $(($y+8)) $BLUE
  fb_put $(($x-3)) $(($y+8)) $BLUE
  fb_put $(($x-2)) $(($y+8)) $BLUE
  fb_put $(($x-1)) $(($y+8)) $BLUE
  fb_put $x $(($y+8)) $BLUE
  fb_put $(($x+1)) $(($y+8)) $WHITE
  fb_put $(($x+2)) $(($y+8)) $BLACK
  fb_put $(($x+3)) $(($y+8)) $BLUE
  fb_put $(($x+4)) $(($y+8)) $BLUE
  fb_put $(($x+5)) $(($y+8)) $BLUE
  fb_put $(($x+6)) $(($y+8)) $BLUE
  fb_put $(($x+7)) $(($y+8)) $BLUE
  fb_put $(($x+8)) $(($y+8)) $BLUE
  fb_put $(($x+9)) $(($y+8)) $BLACK

  fb_put $(($x-8)) $(($y+9)) $BLACK
  fb_put $(($x-7)) $(($y+9)) $BLUE
  fb_put $(($x-6)) $(($y+9)) $BLUE
  fb_put $(($x-5)) $(($y+9)) $BLUE
  fb_put $(($x-4)) $(($y+9)) $BLUE
  fb_put $(($x-3)) $(($y+9)) $BLUE
  fb_put $(($x-2)) $(($y+9)) $BLUE
  fb_put $(($x-1)) $(($y+9)) $BLUE
  fb_put $x $(($y+9)) $BLUE
  fb_put $(($x+1)) $(($y+9)) $BLUE
  fb_put $(($x+2)) $(($y+9)) $BLUE
  fb_put $(($x+3)) $(($y+9)) $BLUE
  fb_put $(($x+4)) $(($y+9)) $BLUE
  fb_put $(($x+5)) $(($y+9)) $BLUE
  fb_put $(($x+6)) $(($y+9)) $BLUE
  fb_put $(($x+7)) $(($y+9)) $BLUE
  fb_put $(($x+8)) $(($y+9)) $BLUE
  fb_put $(($x+9)) $(($y+9)) $BLACK

  fb_put $(($x-8)) $(($y+10)) $BLACK
  fb_put $(($x-7)) $(($y+10)) $BLUE
  fb_put $(($x-6)) $(($y+10)) $BLUE
  fb_put $(($x-5)) $(($y+10)) $BLUE
  fb_put $(($x-4)) $(($y+10)) $BLUE
  fb_put $(($x-3)) $(($y+10)) $BLUE
  fb_put $(($x-2)) $(($y+10)) $BLUE
  fb_put $(($x-1)) $(($y+10)) $BLACK
  fb_put $x $(($y+10)) $BLACK
  fb_put $(($x+1)) $(($y+10)) $BLUE
  fb_put $(($x+2)) $(($y+10)) $BLUE
  fb_put $(($x+3)) $(($y+10)) $BLUE
  fb_put $(($x+4)) $(($y+10)) $BLUE
  fb_put $(($x+5)) $(($y+10)) $BLUE
  fb_put $(($x+6)) $(($y+10)) $BLUE
  fb_put $(($x+7)) $(($y+10)) $BLACK
  fb_put $(($x+8)) $(($y+10)) $BLACK

  fb_put $(($x-7)) $(($y+11)) $BLACK
  fb_put $(($x-6)) $(($y+11)) $BLACK
  fb_put $(($x-5)) $(($y+11)) $BLACK
  fb_put $(($x-4)) $(($y+11)) $BLACK
  fb_put $(($x-3)) $(($y+11)) $BLUE
  fb_put $(($x-2)) $(($y+11)) $BLUE
  fb_put $(($x-1)) $(($y+11)) $BLUE
  fb_put $x $(($y+11)) $BLUE
  fb_put $(($x+1)) $(($y+11)) $BLACK
  fb_put $(($x+2)) $(($y+11)) $BLACK
  fb_put $(($x+3)) $(($y+11)) $BLACK
  fb_put $(($x+4)) $(($y+11)) $BLACK
  fb_put $(($x+5)) $(($y+11)) $BLACK
  fb_put $(($x+6)) $(($y+11)) $BLACK

  fb_put $(($x-3)) $(($y+12)) $BLACK
  fb_put $(($x-2)) $(($y+12)) $BLACK
  fb_put $(($x-1)) $(($y+12)) $BLUE
  fb_put $x $(($y+12)) $BLUE
  fb_put $(($x+1)) $(($y+12)) $BLUE
  fb_put $(($x+2)) $(($y+12)) $BLUE
  fb_put $(($x+3)) $(($y+12)) $BLUE
  fb_put $(($x+4)) $(($y+12)) $BLUE
  fb_put $(($x+5)) $(($y+12)) $BLACK

  fb_put $(($x-3)) $(($y+13)) $BLACK
  fb_put $(($x-2)) $(($y+13)) $BLACK
  fb_put $(($x-1)) $(($y+13)) $BLUE
  fb_put $x $(($y+13)) $BLUE
  fb_put $(($x+1)) $(($y+13)) $BLUE
  fb_put $(($x+2)) $(($y+13)) $BLUE
  fb_put $(($x+3)) $(($y+13)) $BLACK
  fb_put $(($x+4)) $(($y+13)) $BLACK

  fb_put $(($x-4)) $(($y+14)) $BLACK
  fb_put $(($x-3)) $(($y+14)) $BLUE
  fb_put $(($x-2)) $(($y+14)) $BLUE
  fb_put $(($x-1)) $(($y+14)) $BLUE
  fb_put $x $(($y+14)) $BLUE
  fb_put $(($x+1)) $(($y+14)) $BLUE
  fb_put $(($x+2)) $(($y+14)) $BLUE
  fb_put $(($x+3)) $(($y+14)) $BLUE
  fb_put $(($x+4)) $(($y+14)) $BLUE
  fb_put $(($x+5)) $(($y+14)) $BLACK

  fb_put $(($x-4)) $(($y+15)) $BLACK
  fb_put $(($x-3)) $(($y+15)) $BLUE
  fb_put $(($x-2)) $(($y+15)) $BLUE
  fb_put $(($x-1)) $(($y+15)) $BLUE
  fb_put $x $(($y+15)) $BLUE
  fb_put $(($x+1)) $(($y+15)) $BLUE
  fb_put $(($x+2)) $(($y+15)) $BLUE
  fb_put $(($x+3)) $(($y+15)) $BLUE
  fb_put $(($x+4)) $(($y+15)) $BLUE
  fb_put $(($x+5)) $(($y+15)) $BLACK

  fb_put $(($x-4)) $(($y+16)) $BLACK
  fb_put $(($x-3)) $(($y+16)) $BLUE
  fb_put $(($x-2)) $(($y+16)) $BLUE
  fb_put $(($x-1)) $(($y+16)) $BLUE
  fb_put $x $(($y+16)) $BLUE
  fb_put $(($x+1)) $(($y+16)) $BLUE
  fb_put $(($x+2)) $(($y+16)) $BLUE
  fb_put $(($x+3)) $(($y+16)) $BLUE
  fb_put $(($x+4)) $(($y+16)) $BLUE
  fb_put $(($x+5)) $(($y+16)) $BLACK

  fb_put $(($x-4)) $(($y+17)) $BLACK
  fb_put $(($x-3)) $(($y+17)) $BLUE
  fb_put $(($x-2)) $(($y+17)) $BLUE
  fb_put $(($x-1)) $(($y+17)) $BLUE
  fb_put $x $(($y+17)) $BLUE
  fb_put $(($x+1)) $(($y+17)) $BLUE
  fb_put $(($x+2)) $(($y+17)) $BLUE
  fb_put $(($x+3)) $(($y+17)) $BLUE
  fb_put $(($x+4)) $(($y+17)) $BLUE
  fb_put $(($x+5)) $(($y+17)) $BLACK

  fb_put $(($x-4)) $(($y+18)) $BLACK
  fb_put $(($x-3)) $(($y+18)) $BLUE
  fb_put $(($x-2)) $(($y+18)) $BLUE
  fb_put $(($x-1)) $(($y+18)) $BLUE
  fb_put $x $(($y+18)) $BLUE
  fb_put $(($x+1)) $(($y+18)) $BLUE
  fb_put $(($x+2)) $(($y+18)) $BLUE
  fb_put $(($x+3)) $(($y+18)) $BLUE
  fb_put $(($x+4)) $(($y+18)) $BLUE
  fb_put $(($x+5)) $(($y+18)) $BLACK

  fb_put $(($x-4)) $(($y+19)) $BLACK
  fb_put $(($x-3)) $(($y+19)) $WHITE
  fb_put $(($x-2)) $(($y+19)) $WHITE
  fb_put $(($x-1)) $(($y+19)) $WHITE
  fb_put $x $(($y+19)) $WHITE
  fb_put $(($x+1)) $(($y+19)) $WHITE
  fb_put $(($x+2)) $(($y+19)) $WHITE
  fb_put $(($x+3)) $(($y+19)) $WHITE
  fb_put $(($x+4)) $(($y+19)) $WHITE
  fb_put $(($x+5)) $(($y+19)) $BLACK
  fb_put $(($x+6)) $(($y+19)) $BLACK

  fb_put $(($x-4)) $(($y+20)) $BLACK
  fb_put $(($x-3)) $(($y+20)) $WHITE
  fb_put $(($x-2)) $(($y+20)) $WHITE
  fb_put $(($x-1)) $(($y+20)) $WHITE
  fb_put $x $(($y+20)) $WHITE
  fb_put $(($x+1)) $(($y+20)) $WHITE
  fb_put $(($x+2)) $(($y+20)) $WHITE
  fb_put $(($x+3)) $(($y+20)) $WHITE
  fb_put $(($x+4)) $(($y+20)) $WHITE
  fb_put $(($x+5)) $(($y+20)) $WHITE
  fb_put $(($x+6)) $(($y+20)) $WHITE
  fb_put $(($x+7)) $(($y+20)) $BLACK

  fb_put $(($x-3)) $(($y+21)) $BLACK
  fb_put $(($x-2)) $(($y+21)) $BLACK
  fb_put $(($x-1)) $(($y+21)) $BLACK
  fb_put $x $(($y+21)) $BLACK
  fb_put $(($x+1)) $(($y+21)) $WHITE
  fb_put $(($x+2)) $(($y+21)) $WHITE
  fb_put $(($x+3)) $(($y+21)) $WHITE
  fb_put $(($x+4)) $(($y+21)) $WHITE
  fb_put $(($x+5)) $(($y+21)) $WHITE
  fb_put $(($x+6)) $(($y+21)) $WHITE
  fb_put $(($x+7)) $(($y+21)) $BLACK

  fb_put $(($x-3)) $(($y+22)) $BLACK
  fb_put $(($x-2)) $(($y+22)) $BLACK
  fb_put $(($x-1)) $(($y+22)) $WHITE
  fb_put $x $(($y+22)) $WHITE
  fb_put $(($x+1)) $(($y+22)) $WHITE
  fb_put $(($x+2)) $(($y+22)) $WHITE
  fb_put $(($x+3)) $(($y+22)) $WHITE
  fb_put $(($x+4)) $(($y+22)) $WHITE
  fb_put $(($x+5)) $(($y+22)) $BLACK
  fb_put $(($x+6)) $(($y+22)) $BLACK

  fb_put $(($x-5)) $(($y+23)) $BLACK
  fb_put $(($x-4)) $(($y+23)) $BLACK
  fb_put $(($x-3)) $(($y+23)) $WHITE
  fb_put $(($x-2)) $(($y+23)) $WHITE
  fb_put $(($x-1)) $(($y+23)) $WHITE
  fb_put $x $(($y+23)) $WHITE
  fb_put $(($x+1)) $(($y+23)) $WHITE
  fb_put $(($x+2)) $(($y+23)) $WHITE
  fb_put $(($x+3)) $(($y+23)) $WHITE
  fb_put $(($x+4)) $(($y+23)) $WHITE
  fb_put $(($x+5)) $(($y+23)) $WHITE
  fb_put $(($x+6)) $(($y+23)) $WHITE
  fb_put $(($x+7)) $(($y+23)) $BLACK

  fb_put $(($x-6)) $(($y+24)) $BLACK
  fb_put $(($x-5)) $(($y+24)) $WHITE
  fb_put $(($x-4)) $(($y+24)) $WHITE
  fb_put $(($x-3)) $(($y+24)) $WHITE
  fb_put $(($x-2)) $(($y+24)) $WHITE
  fb_put $(($x-1)) $(($y+24)) $BLACK
  fb_put $x $(($y+24)) $BLACK
  fb_put $(($x+1)) $(($y+24)) $WHITE
  fb_put $(($x+2)) $(($y+24)) $WHITE
  fb_put $(($x+3)) $(($y+24)) $WHITE
  fb_put $(($x+4)) $(($y+24)) $WHITE
  fb_put $(($x+5)) $(($y+24)) $WHITE
  fb_put $(($x+6)) $(($y+24)) $WHITE
  fb_put $(($x+7)) $(($y+24)) $BLACK

  fb_put $(($x-6)) $(($y+25)) $BLACK
  fb_put $(($x-5)) $(($y+25)) $WHITE
  fb_put $(($x-4)) $(($y+25)) $WHITE
  fb_put $(($x-3)) $(($y+25)) $WHITE
  fb_put $(($x-2)) $(($y+25)) $WHITE
  fb_put $(($x-1)) $(($y+25)) $WHITE
  fb_put $x $(($y+25)) $WHITE
  fb_put $(($x+1)) $(($y+25)) $WHITE
  fb_put $(($x+2)) $(($y+25)) $WHITE
  fb_put $(($x+3)) $(($y+25)) $BLACK
  fb_put $(($x+4)) $(($y+25)) $BLACK
  fb_put $(($x+5)) $(($y+25)) $BLACK
  fb_put $(($x+6)) $(($y+25)) $BLACK

  fb_put $(($x-5)) $(($y+26)) $BLACK
  fb_put $(($x-4)) $(($y+26)) $BLACK
  fb_put $(($x-3)) $(($y+26)) $BLACK
  fb_put $(($x-2)) $(($y+26)) $BLACK
  fb_put $(($x-1)) $(($y+26)) $BLACK
  fb_put $x $(($y+26)) $BLACK
  fb_put $(($x+1)) $(($y+26)) $BLACK
  fb_put $(($x+2)) $(($y+26)) $BLACK
}

function draw_smurf2 {
  # Params: x y
  let x=$1
  let y=$2

  fb_put $(($x-1)) $(($y+1)) $BLACK
  fb_put $x $(($y+1)) $BLACK
  fb_put $(($x+1)) $(($y+1)) $BLACK
  fb_put $(($x+2)) $(($y+1)) $BLACK
  fb_put $(($x+3)) $(($y+1)) $BLACK
  fb_put $(($x+4)) $(($y+1)) $BLACK

  fb_put $(($x-3)) $(($y+2)) $BLACK
  fb_put $(($x-2)) $(($y+2)) $BLACK
  fb_put $(($x-1)) $(($y+2)) $WHITE
  fb_put $x $(($y+2)) $WHITE
  fb_put $(($x+1)) $(($y+2)) $WHITE
  fb_put $(($x+2)) $(($y+2)) $WHITE
  fb_put $(($x+3)) $(($y+2)) $WHITE
  fb_put $(($x+4)) $(($y+2)) $WHITE
  fb_put $(($x+5)) $(($y+2)) $BLACK
  fb_put $(($x+6)) $(($y+2)) $BLACK

  fb_put $(($x-5)) $(($y+3)) $BLACK
  fb_put $(($x-4)) $(($y+3)) $BLACK
  fb_put $(($x-3)) $(($y+3)) $WHITE
  fb_put $(($x-2)) $(($y+3)) $WHITE
  fb_put $(($x-1)) $(($y+3)) $WHITE
  fb_put $x $(($y+3)) $WHITE
  fb_put $(($x+1)) $(($y+3)) $WHITE
  fb_put $(($x+2)) $(($y+3)) $WHITE
  fb_put $(($x+3)) $(($y+3)) $WHITE
  fb_put $(($x+4)) $(($y+3)) $WHITE
  fb_put $(($x+5)) $(($y+3)) $WHITE
  fb_put $(($x+6)) $(($y+3)) $WHITE
  fb_put $(($x+7)) $(($y+3)) $BLACK

  fb_put $(($x-6)) $(($y+4)) $BLACK
  fb_put $(($x-5)) $(($y+4)) $WHITE
  fb_put $(($x-4)) $(($y+4)) $WHITE
  fb_put $(($x-3)) $(($y+4)) $WHITE
  fb_put $(($x-2)) $(($y+4)) $WHITE
  fb_put $(($x-1)) $(($y+4)) $WHITE
  fb_put $x $(($y+4)) $WHITE
  fb_put $(($x+1)) $(($y+4)) $BLACK
  fb_put $(($x+2)) $(($y+4)) $BLACK
  fb_put $(($x+3)) $(($y+4)) $WHITE
  fb_put $(($x+4)) $(($y+4)) $WHITE
  fb_put $(($x+5)) $(($y+4)) $WHITE
  fb_put $(($x+6)) $(($y+4)) $WHITE
  fb_put $(($x+7)) $(($y+4)) $BLACK

  fb_put $(($x-7)) $(($y+5)) $BLACK
  fb_put $(($x-6)) $(($y+5)) $BLACK
  fb_put $(($x-5)) $(($y+5)) $WHITE
  fb_put $(($x-4)) $(($y+5)) $WHITE
  fb_put $(($x-3)) $(($y+5)) $WHITE
  fb_put $(($x-2)) $(($y+5)) $WHITE
  fb_put $(($x-1)) $(($y+5)) $WHITE
  fb_put $x $(($y+5)) $WHITE
  fb_put $(($x+1)) $(($y+5)) $WHITE
  fb_put $(($x+2)) $(($y+5)) $WHITE
  fb_put $(($x+3)) $(($y+5)) $BLACK
  fb_put $(($x+4)) $(($y+5)) $BLACK
  fb_put $(($x+5)) $(($y+5)) $BLACK
  fb_put $(($x+6)) $(($y+5)) $BLACK

  fb_put $(($x-8)) $(($y+6)) $BLACK
  fb_put $(($x-7)) $(($y+6)) $WHITE
  fb_put $(($x-6)) $(($y+6)) $WHITE
  fb_put $(($x-5)) $(($y+6)) $WHITE
  fb_put $(($x-4)) $(($y+6)) $WHITE
  fb_put $(($x-3)) $(($y+6)) $WHITE
  fb_put $(($x-2)) $(($y+6)) $WHITE
  fb_put $(($x-1)) $(($y+6)) $WHITE
  fb_put $x $(($y+6)) $WHITE
  fb_put $(($x+1)) $(($y+6)) $WHITE
  fb_put $(($x+2)) $(($y+6)) $WHITE
  fb_put $(($x+3)) $(($y+6)) $WHITE
  fb_put $(($x+4)) $(($y+6)) $WHITE
  fb_put $(($x+5)) $(($y+6)) $BLACK
  fb_put $(($x+6)) $(($y+6)) $BLACK

  fb_put $(($x-8)) $(($y+7)) $BLACK
  fb_put $(($x-7)) $(($y+7)) $WHITE
  fb_put $(($x-6)) $(($y+7)) $WHITE
  fb_put $(($x-5)) $(($y+7)) $WHITE
  fb_put $(($x-4)) $(($y+7)) $WHITE
  fb_put $(($x-3)) $(($y+7)) $WHITE
  fb_put $(($x-2)) $(($y+7)) $WHITE
  fb_put $(($x-1)) $(($y+7)) $WHITE
  fb_put $x $(($y+7)) $WHITE
  fb_put $(($x+1)) $(($y+7)) $WHITE
  fb_put $(($x+2)) $(($y+7)) $WHITE
  fb_put $(($x+3)) $(($y+7)) $WHITE
  fb_put $(($x+4)) $(($y+7)) $WHITE
  fb_put $(($x+5)) $(($y+7)) $WHITE
  fb_put $(($x+6)) $(($y+7)) $WHITE
  fb_put $(($x+7)) $(($y+7)) $BLACK

  fb_put $(($x-7)) $(($y+8)) $BLACK
  fb_put $(($x-6)) $(($y+8)) $BLACK
  fb_put $(($x-5)) $(($y+8)) $BLUE
  fb_put $(($x-4)) $(($y+8)) $BLUE
  fb_put $(($x-3)) $(($y+8)) $BLUE
  fb_put $(($x-2)) $(($y+8)) $BLUE
  fb_put $(($x-1)) $(($y+8)) $BLUE
  fb_put $x $(($y+8)) $BLUE
  fb_put $(($x+1)) $(($y+8)) $WHITE
  fb_put $(($x+2)) $(($y+8)) $WHITE
  fb_put $(($x+3)) $(($y+8)) $BLUE
  fb_put $(($x+4)) $(($y+8)) $BLUE
  fb_put $(($x+5)) $(($y+8)) $BLACK
  fb_put $(($x+6)) $(($y+8)) $BLACK
  fb_put $(($x+7)) $(($y+8)) $BLACK
  fb_put $(($x+8)) $(($y+8)) $BLACK

  fb_put $(($x-8)) $(($y+9)) $BLACK
  fb_put $(($x-7)) $(($y+9)) $BLUE
  fb_put $(($x-6)) $(($y+9)) $BLUE
  fb_put $(($x-5)) $(($y+9)) $BLUE
  fb_put $(($x-4)) $(($y+9)) $BLUE
  fb_put $(($x-3)) $(($y+9)) $BLUE
  fb_put $(($x-2)) $(($y+9)) $BLUE
  fb_put $(($x-1)) $(($y+9)) $BLUE
  fb_put $x $(($y+9)) $BLUE
  fb_put $(($x+1)) $(($y+9)) $WHITE
  fb_put $(($x+2)) $(($y+9)) $BLACK
  fb_put $(($x+3)) $(($y+9)) $BLUE
  fb_put $(($x+4)) $(($y+9)) $BLUE
  fb_put $(($x+5)) $(($y+9)) $BLUE
  fb_put $(($x+6)) $(($y+9)) $BLUE
  fb_put $(($x+7)) $(($y+9)) $BLUE
  fb_put $(($x+8)) $(($y+9)) $BLUE
  fb_put $(($x+9)) $(($y+9)) $BLACK

  fb_put $(($x-8)) $(($y+10)) $BLACK
  fb_put $(($x-7)) $(($y+10)) $BLUE
  fb_put $(($x-6)) $(($y+10)) $BLUE
  fb_put $(($x-5)) $(($y+10)) $BLUE
  fb_put $(($x-4)) $(($y+10)) $BLUE
  fb_put $(($x-3)) $(($y+10)) $BLUE
  fb_put $(($x-2)) $(($y+10)) $BLUE
  fb_put $(($x-1)) $(($y+10)) $BLUE
  fb_put $x $(($y+10)) $BLUE
  fb_put $(($x+1)) $(($y+10)) $BLUE
  fb_put $(($x+2)) $(($y+10)) $BLUE
  fb_put $(($x+3)) $(($y+10)) $BLUE
  fb_put $(($x+4)) $(($y+10)) $BLUE
  fb_put $(($x+5)) $(($y+10)) $BLUE
  fb_put $(($x+6)) $(($y+10)) $BLUE
  fb_put $(($x+7)) $(($y+10)) $BLUE
  fb_put $(($x+8)) $(($y+10)) $BLUE
  fb_put $(($x+9)) $(($y+10)) $BLACK

  fb_put $(($x-8)) $(($y+11)) $BLACK
  fb_put $(($x-7)) $(($y+11)) $BLUE
  fb_put $(($x-6)) $(($y+11)) $BLUE
  fb_put $(($x-5)) $(($y+11)) $BLUE
  fb_put $(($x-4)) $(($y+11)) $BLUE
  fb_put $(($x-3)) $(($y+11)) $BLUE
  fb_put $(($x-2)) $(($y+11)) $BLUE
  fb_put $(($x-1)) $(($y+11)) $BLACK
  fb_put $x $(($y+11)) $BLACK
  fb_put $(($x+1)) $(($y+11)) $BLUE
  fb_put $(($x+2)) $(($y+11)) $BLUE
  fb_put $(($x+3)) $(($y+11)) $BLUE
  fb_put $(($x+4)) $(($y+11)) $BLUE
  fb_put $(($x+5)) $(($y+11)) $BLUE
  fb_put $(($x+6)) $(($y+11)) $BLUE
  fb_put $(($x+7)) $(($y+11)) $BLACK
  fb_put $(($x+8)) $(($y+11)) $BLACK

  fb_put $(($x-7)) $(($y+12)) $BLACK
  fb_put $(($x-6)) $(($y+12)) $BLACK
  fb_put $(($x-5)) $(($y+12)) $BLACK
  fb_put $(($x-4)) $(($y+12)) $BLACK
  fb_put $(($x-3)) $(($y+12)) $BLUE
  fb_put $(($x-2)) $(($y+12)) $BLUE
  fb_put $(($x-1)) $(($y+12)) $BLUE
  fb_put $x $(($y+12)) $BLUE
  fb_put $(($x+1)) $(($y+12)) $BLACK
  fb_put $(($x+2)) $(($y+12)) $BLACK
  fb_put $(($x+3)) $(($y+12)) $BLACK
  fb_put $(($x+4)) $(($y+12)) $BLACK
  fb_put $(($x+5)) $(($y+12)) $BLACK
  fb_put $(($x+6)) $(($y+12)) $BLACK

  fb_put $(($x-3)) $(($y+13)) $BLACK
  fb_put $(($x-2)) $(($y+13)) $BLACK
  fb_put $(($x-1)) $(($y+13)) $BLUE
  fb_put $x $(($y+13)) $BLUE
  fb_put $(($x+1)) $(($y+13)) $BLUE
  fb_put $(($x+2)) $(($y+13)) $BLUE
  fb_put $(($x+3)) $(($y+13)) $BLUE
  fb_put $(($x+4)) $(($y+13)) $BLUE
  fb_put $(($x+5)) $(($y+13)) $BLACK

  fb_put $(($x-3)) $(($y+14)) $BLACK
  fb_put $(($x-2)) $(($y+14)) $BLACK
  fb_put $(($x-1)) $(($y+14)) $BLUE
  fb_put $x $(($y+14)) $BLUE
  fb_put $(($x+1)) $(($y+14)) $BLUE
  fb_put $(($x+2)) $(($y+14)) $BLUE
  fb_put $(($x+3)) $(($y+14)) $BLACK
  fb_put $(($x+4)) $(($y+14)) $BLACK

  fb_put $(($x-5)) $(($y+15)) $BLACK
  fb_put $(($x-4)) $(($y+15)) $BLACK
  fb_put $(($x-3)) $(($y+15)) $BLUE
  fb_put $(($x-2)) $(($y+15)) $BLUE
  fb_put $(($x-1)) $(($y+15)) $BLUE
  fb_put $x $(($y+15)) $BLUE
  fb_put $(($x+1)) $(($y+15)) $BLUE
  fb_put $(($x+2)) $(($y+15)) $BLUE
  fb_put $(($x+3)) $(($y+15)) $BLUE
  fb_put $(($x+4)) $(($y+15)) $BLUE
  fb_put $(($x+5)) $(($y+15)) $BLACK
  fb_put $(($x+6)) $(($y+15)) $BLACK

  fb_put $(($x-7)) $(($y+16)) $BLACK
  fb_put $(($x-6)) $(($y+16)) $BLACK
  fb_put $(($x-5)) $(($y+16)) $BLUE
  fb_put $(($x-4)) $(($y+16)) $BLUE
  fb_put $(($x-3)) $(($y+16)) $BLUE
  fb_put $(($x-2)) $(($y+16)) $BLUE
  fb_put $(($x-1)) $(($y+16)) $BLUE
  fb_put $x $(($y+16)) $BLUE
  fb_put $(($x+1)) $(($y+16)) $BLUE
  fb_put $(($x+2)) $(($y+16)) $BLUE
  fb_put $(($x+3)) $(($y+16)) $BLUE
  fb_put $(($x+4)) $(($y+16)) $BLUE
  fb_put $(($x+5)) $(($y+16)) $BLUE
  fb_put $(($x+6)) $(($y+16)) $BLUE
  fb_put $(($x+7)) $(($y+16)) $BLACK

  fb_put $(($x-8)) $(($y+17)) $BLACK
  fb_put $(($x-7)) $(($y+17)) $BLUE
  fb_put $(($x-6)) $(($y+17)) $BLUE
  fb_put $(($x-5)) $(($y+17)) $BLUE
  fb_put $(($x-4)) $(($y+17)) $BLUE
  fb_put $(($x-3)) $(($y+17)) $BLUE
  fb_put $(($x-2)) $(($y+17)) $BLUE
  fb_put $(($x-1)) $(($y+17)) $BLUE
  fb_put $x $(($y+17)) $BLUE
  fb_put $(($x+1)) $(($y+17)) $BLUE
  fb_put $(($x+2)) $(($y+17)) $BLUE
  fb_put $(($x+3)) $(($y+17)) $BLUE
  fb_put $(($x+4)) $(($y+17)) $BLUE
  fb_put $(($x+5)) $(($y+17)) $BLUE
  fb_put $(($x+6)) $(($y+17)) $BLUE
  fb_put $(($x+7)) $(($y+17)) $BLACK

  fb_put $(($x-8)) $(($y+18)) $BLACK
  fb_put $(($x-7)) $(($y+18)) $BLUE
  fb_put $(($x-6)) $(($y+18)) $BLUE
  fb_put $(($x-5)) $(($y+18)) $BLUE
  fb_put $(($x-4)) $(($y+18)) $BLUE
  fb_put $(($x-3)) $(($y+18)) $BLUE
  fb_put $(($x-2)) $(($y+18)) $BLUE
  fb_put $(($x-1)) $(($y+18)) $BLUE
  fb_put $x $(($y+18)) $BLUE
  fb_put $(($x+1)) $(($y+18)) $BLUE
  fb_put $(($x+2)) $(($y+18)) $BLUE
  fb_put $(($x+3)) $(($y+18)) $BLUE
  fb_put $(($x+4)) $(($y+18)) $BLUE
  fb_put $(($x+5)) $(($y+18)) $BLACK
  fb_put $(($x+6)) $(($y+18)) $BLACK

  fb_put $(($x-7)) $(($y+19)) $BLACK
  fb_put $(($x-6)) $(($y+19)) $BLACK
  fb_put $(($x-5)) $(($y+19)) $BLUE
  fb_put $(($x-4)) $(($y+19)) $BLUE
  fb_put $(($x-3)) $(($y+19)) $BLUE
  fb_put $(($x-2)) $(($y+19)) $BLUE
  fb_put $(($x-1)) $(($y+19)) $BLUE
  fb_put $x $(($y+19)) $BLUE
  fb_put $(($x+1)) $(($y+19)) $BLUE
  fb_put $(($x+2)) $(($y+19)) $BLUE
  fb_put $(($x+3)) $(($y+19)) $BLUE
  fb_put $(($x+4)) $(($y+19)) $BLUE
  fb_put $(($x+5)) $(($y+19)) $BLACK

  fb_put $(($x-5)) $(($y+20)) $BLACK
  fb_put $(($x-4)) $(($y+20)) $BLACK
  fb_put $(($x-3)) $(($y+20)) $WHITE
  fb_put $(($x-2)) $(($y+20)) $WHITE
  fb_put $(($x-1)) $(($y+20)) $WHITE
  fb_put $x $(($y+20)) $WHITE
  fb_put $(($x+1)) $(($y+20)) $WHITE
  fb_put $(($x+2)) $(($y+20)) $WHITE
  fb_put $(($x+3)) $(($y+20)) $WHITE
  fb_put $(($x+4)) $(($y+20)) $WHITE
  fb_put $(($x+5)) $(($y+20)) $BLACK

  fb_put $(($x-6)) $(($y+21)) $BLACK
  fb_put $(($x-5)) $(($y+21)) $WHITE
  fb_put $(($x-4)) $(($y+21)) $WHITE
  fb_put $(($x-3)) $(($y+21)) $WHITE
  fb_put $(($x-2)) $(($y+21)) $WHITE
  fb_put $(($x-1)) $(($y+21)) $WHITE
  fb_put $x $(($y+21)) $WHITE
  fb_put $(($x+1)) $(($y+21)) $WHITE
  fb_put $(($x+2)) $(($y+21)) $WHITE
  fb_put $(($x+3)) $(($y+21)) $WHITE
  fb_put $(($x+4)) $(($y+21)) $WHITE
  fb_put $(($x+5)) $(($y+21)) $BLACK

  fb_put $(($x-6)) $(($y+22)) $BLACK
  fb_put $(($x-5)) $(($y+22)) $WHITE
  fb_put $(($x-4)) $(($y+22)) $WHITE
  fb_put $(($x-3)) $(($y+22)) $WHITE
  fb_put $(($x-2)) $(($y+22)) $WHITE
  fb_put $(($x-1)) $(($y+22)) $WHITE
  fb_put $x $(($y+22)) $WHITE
  fb_put $(($x+1)) $(($y+22)) $BLACK
  fb_put $(($x+2)) $(($y+22)) $BLACK
  fb_put $(($x+3)) $(($y+22)) $BLACK
  fb_put $(($x+4)) $(($y+22)) $BLACK

  fb_put $(($x-5)) $(($y+23)) $BLACK
  fb_put $(($x-4)) $(($y+23)) $BLACK
  fb_put $(($x-3)) $(($y+23)) $WHITE
  fb_put $(($x-2)) $(($y+23)) $WHITE
  fb_put $(($x-1)) $(($y+23)) $WHITE
  fb_put $x $(($y+23)) $WHITE
  fb_put $(($x+1)) $(($y+23)) $WHITE
  fb_put $(($x+2)) $(($y+23)) $WHITE
  fb_put $(($x+3)) $(($y+23)) $BLACK
  fb_put $(($x+4)) $(($y+23)) $BLACK

  fb_put $(($x-4)) $(($y+24)) $BLACK
  fb_put $(($x-3)) $(($y+24)) $WHITE
  fb_put $(($x-2)) $(($y+24)) $WHITE
  fb_put $(($x-1)) $(($y+24)) $WHITE
  fb_put $x $(($y+24)) $WHITE
  fb_put $(($x+1)) $(($y+24)) $WHITE
  fb_put $(($x+2)) $(($y+24)) $WHITE
  fb_put $(($x+3)) $(($y+24)) $WHITE
  fb_put $(($x+4)) $(($y+24)) $WHITE
  fb_put $(($x+5)) $(($y+24)) $BLACK

  fb_put $(($x-4)) $(($y+25)) $BLACK
  fb_put $(($x-3)) $(($y+25)) $WHITE
  fb_put $(($x-2)) $(($y+25)) $WHITE
  fb_put $(($x-1)) $(($y+25)) $WHITE
  fb_put $x $(($y+25)) $WHITE
  fb_put $(($x+1)) $(($y+25)) $WHITE
  fb_put $(($x+2)) $(($y+25)) $WHITE
  fb_put $(($x+3)) $(($y+25)) $WHITE
  fb_put $(($x+4)) $(($y+25)) $WHITE
  fb_put $(($x+5)) $(($y+25)) $BLACK

  fb_put $(($x-3)) $(($y+26)) $BLACK
  fb_put $(($x-2)) $(($y+26)) $BLACK
  fb_put $(($x-1)) $(($y+26)) $BLACK
  fb_put $x $(($y+26)) $BLACK
  fb_put $(($x+1)) $(($y+26)) $BLACK
  fb_put $(($x+2)) $(($y+26)) $BLACK
  fb_put $(($x+3)) $(($y+26)) $BLACK
  fb_put $(($x+4)) $(($y+26)) $BLACK
}

function draw_smurf1 {
  # Params: x y
  let x=$1
  let y=$2

  fb_put $(($x-1)) $y $BLACK
  fb_put $x $y $BLACK
  fb_put $(($x+1)) $y $BLACK
  fb_put $(($x+2)) $y $BLACK
  fb_put $(($x+3)) $y $BLACK
  fb_put $(($x+4)) $y $BLACK

  fb_put $(($x-3)) $(($y+1)) $BLACK
  fb_put $(($x-2)) $(($y+1)) $BLACK
  fb_put $(($x-1)) $(($y+1)) $WHITE
  fb_put $x $(($y+1)) $WHITE
  fb_put $(($x+1)) $(($y+1)) $WHITE
  fb_put $(($x+2)) $(($y+1)) $WHITE
  fb_put $(($x+3)) $(($y+1)) $WHITE
  fb_put $(($x+4)) $(($y+1)) $WHITE
  fb_put $(($x+5)) $(($y+1)) $BLACK
  fb_put $(($x+6)) $(($y+1)) $BLACK

  fb_put $(($x-5)) $(($y+2)) $BLACK
  fb_put $(($x-4)) $(($y+2)) $BLACK
  fb_put $(($x-3)) $(($y+2)) $WHITE
  fb_put $(($x-2)) $(($y+2)) $WHITE
  fb_put $(($x-1)) $(($y+2)) $WHITE
  fb_put $x $(($y+2)) $WHITE
  fb_put $(($x+1)) $(($y+2)) $WHITE
  fb_put $(($x+2)) $(($y+2)) $WHITE
  fb_put $(($x+3)) $(($y+2)) $WHITE
  fb_put $(($x+4)) $(($y+2)) $WHITE
  fb_put $(($x+5)) $(($y+2)) $WHITE
  fb_put $(($x+6)) $(($y+2)) $WHITE
  fb_put $(($x+7)) $(($y+2)) $BLACK

  fb_put $(($x-6)) $(($y+3)) $BLACK
  fb_put $(($x-5)) $(($y+3)) $WHITE
  fb_put $(($x-4)) $(($y+3)) $WHITE
  fb_put $(($x-3)) $(($y+3)) $WHITE
  fb_put $(($x-2)) $(($y+3)) $WHITE
  fb_put $(($x-1)) $(($y+3)) $WHITE
  fb_put $x $(($y+3)) $WHITE
  fb_put $(($x+1)) $(($y+3)) $BLACK
  fb_put $(($x+2)) $(($y+3)) $BLACK
  fb_put $(($x+3)) $(($y+3)) $WHITE
  fb_put $(($x+4)) $(($y+3)) $WHITE
  fb_put $(($x+5)) $(($y+3)) $WHITE
  fb_put $(($x+6)) $(($y+3)) $WHITE
  fb_put $(($x+7)) $(($y+3)) $BLACK

  fb_put $(($x-7)) $(($y+4)) $BLACK
  fb_put $(($x-6)) $(($y+4)) $BLACK
  fb_put $(($x-5)) $(($y+4)) $WHITE
  fb_put $(($x-4)) $(($y+4)) $WHITE
  fb_put $(($x-3)) $(($y+4)) $WHITE
  fb_put $(($x-2)) $(($y+4)) $WHITE
  fb_put $(($x-1)) $(($y+4)) $WHITE
  fb_put $x $(($y+4)) $WHITE
  fb_put $(($x+1)) $(($y+4)) $WHITE
  fb_put $(($x+2)) $(($y+4)) $WHITE
  fb_put $(($x+3)) $(($y+4)) $BLACK
  fb_put $(($x+4)) $(($y+4)) $BLACK
  fb_put $(($x+5)) $(($y+4)) $BLACK
  fb_put $(($x+6)) $(($y+4)) $BLACK

  fb_put $(($x-8)) $(($y+5)) $BLACK
  fb_put $(($x-7)) $(($y+5)) $WHITE
  fb_put $(($x-6)) $(($y+5)) $WHITE
  fb_put $(($x-5)) $(($y+5)) $WHITE
  fb_put $(($x-4)) $(($y+5)) $WHITE
  fb_put $(($x-3)) $(($y+5)) $WHITE
  fb_put $(($x-2)) $(($y+5)) $WHITE
  fb_put $(($x-1)) $(($y+5)) $WHITE
  fb_put $x $(($y+5)) $WHITE
  fb_put $(($x+1)) $(($y+5)) $WHITE
  fb_put $(($x+2)) $(($y+5)) $WHITE
  fb_put $(($x+3)) $(($y+5)) $WHITE
  fb_put $(($x+4)) $(($y+5)) $WHITE
  fb_put $(($x+5)) $(($y+5)) $BLACK
  fb_put $(($x+6)) $(($y+5)) $BLACK

  fb_put $(($x-8)) $(($y+6)) $BLACK
  fb_put $(($x-7)) $(($y+6)) $WHITE
  fb_put $(($x-6)) $(($y+6)) $WHITE
  fb_put $(($x-5)) $(($y+6)) $WHITE
  fb_put $(($x-4)) $(($y+6)) $WHITE
  fb_put $(($x-3)) $(($y+6)) $WHITE
  fb_put $(($x-2)) $(($y+6)) $WHITE
  fb_put $(($x-1)) $(($y+6)) $WHITE
  fb_put $x $(($y+6)) $WHITE
  fb_put $(($x+1)) $(($y+6)) $WHITE
  fb_put $(($x+2)) $(($y+6)) $WHITE
  fb_put $(($x+3)) $(($y+6)) $WHITE
  fb_put $(($x+4)) $(($y+6)) $WHITE
  fb_put $(($x+5)) $(($y+6)) $WHITE
  fb_put $(($x+6)) $(($y+6)) $WHITE
  fb_put $(($x+7)) $(($y+6)) $BLACK

  fb_put $(($x-7)) $(($y+7)) $BLACK
  fb_put $(($x-6)) $(($y+7)) $BLACK
  fb_put $(($x-5)) $(($y+7)) $BLACK
  fb_put $(($x-4)) $(($y+7)) $BLACK
  fb_put $(($x-3)) $(($y+7)) $BLUE
  fb_put $(($x-2)) $(($y+7)) $BLUE
  fb_put $(($x-1)) $(($y+7)) $BLUE
  fb_put $x $(($y+7)) $BLUE
  fb_put $(($x+1)) $(($y+7)) $WHITE
  fb_put $(($x+2)) $(($y+7)) $WHITE
  fb_put $(($x+3)) $(($y+7)) $BLUE
  fb_put $(($x+4)) $(($y+7)) $BLUE
  fb_put $(($x+5)) $(($y+7)) $BLACK
  fb_put $(($x+6)) $(($y+7)) $BLACK
  fb_put $(($x+7)) $(($y+7)) $BLACK
  fb_put $(($x+8)) $(($y+7)) $BLACK

  fb_put $(($x-8)) $(($y+8)) $BLACK
  fb_put $(($x-7)) $(($y+8)) $BLUE
  fb_put $(($x-6)) $(($y+8)) $BLUE
  fb_put $(($x-5)) $(($y+8)) $BLUE
  fb_put $(($x-4)) $(($y+8)) $BLUE
  fb_put $(($x-3)) $(($y+8)) $BLUE
  fb_put $(($x-2)) $(($y+8)) $BLUE
  fb_put $(($x-1)) $(($y+8)) $BLUE
  fb_put $x $(($y+8)) $BLUE
  fb_put $(($x+1)) $(($y+8)) $WHITE
  fb_put $(($x+2)) $(($y+8)) $BLACK
  fb_put $(($x+3)) $(($y+8)) $BLUE
  fb_put $(($x+4)) $(($y+8)) $BLUE
  fb_put $(($x+5)) $(($y+8)) $BLUE
  fb_put $(($x+6)) $(($y+8)) $BLUE
  fb_put $(($x+7)) $(($y+8)) $BLUE
  fb_put $(($x+8)) $(($y+8)) $BLUE
  fb_put $(($x+9)) $(($y+8)) $BLACK

  fb_put $(($x-8)) $(($y+9)) $BLACK
  fb_put $(($x-7)) $(($y+9)) $BLUE
  fb_put $(($x-6)) $(($y+9)) $BLUE
  fb_put $(($x-5)) $(($y+9)) $BLUE
  fb_put $(($x-4)) $(($y+9)) $BLUE
  fb_put $(($x-3)) $(($y+9)) $BLUE
  fb_put $(($x-2)) $(($y+9)) $BLUE
  fb_put $(($x-1)) $(($y+9)) $BLUE
  fb_put $x $(($y+9)) $BLUE
  fb_put $(($x+1)) $(($y+9)) $BLUE
  fb_put $(($x+2)) $(($y+9)) $BLUE
  fb_put $(($x+3)) $(($y+9)) $BLUE
  fb_put $(($x+4)) $(($y+9)) $BLUE
  fb_put $(($x+5)) $(($y+9)) $BLUE
  fb_put $(($x+6)) $(($y+9)) $BLUE
  fb_put $(($x+7)) $(($y+9)) $BLUE
  fb_put $(($x+8)) $(($y+9)) $BLUE
  fb_put $(($x+9)) $(($y+9)) $BLACK

  fb_put $(($x-8)) $(($y+10)) $BLACK
  fb_put $(($x-7)) $(($y+10)) $BLUE
  fb_put $(($x-6)) $(($y+10)) $BLUE
  fb_put $(($x-5)) $(($y+10)) $BLUE
  fb_put $(($x-4)) $(($y+10)) $BLUE
  fb_put $(($x-3)) $(($y+10)) $BLUE
  fb_put $(($x-2)) $(($y+10)) $BLUE
  fb_put $(($x-1)) $(($y+10)) $BLACK
  fb_put $x $(($y+10)) $BLACK
  fb_put $(($x+1)) $(($y+10)) $BLUE
  fb_put $(($x+2)) $(($y+10)) $BLUE
  fb_put $(($x+3)) $(($y+10)) $BLUE
  fb_put $(($x+4)) $(($y+10)) $BLUE
  fb_put $(($x+5)) $(($y+10)) $BLUE
  fb_put $(($x+6)) $(($y+10)) $BLUE
  fb_put $(($x+7)) $(($y+10)) $BLACK
  fb_put $(($x+8)) $(($y+10)) $BLACK

  fb_put $(($x-7)) $(($y+11)) $BLACK
  fb_put $(($x-6)) $(($y+11)) $BLACK
  fb_put $(($x-5)) $(($y+11)) $BLACK
  fb_put $(($x-4)) $(($y+11)) $BLACK
  fb_put $(($x-3)) $(($y+11)) $BLUE
  fb_put $(($x-2)) $(($y+11)) $BLUE
  fb_put $(($x-1)) $(($y+11)) $BLUE
  fb_put $x $(($y+11)) $BLUE
  fb_put $(($x+1)) $(($y+11)) $BLACK
  fb_put $(($x+2)) $(($y+11)) $BLACK
  fb_put $(($x+3)) $(($y+11)) $BLACK
  fb_put $(($x+4)) $(($y+11)) $BLACK
  fb_put $(($x+5)) $(($y+11)) $BLACK
  fb_put $(($x+6)) $(($y+11)) $BLACK
  fb_put $(($x+7)) $(($y+11)) $BLACK
  fb_put $(($x+8)) $(($y+11)) $BLACK

  fb_put $(($x-3)) $(($y+12)) $BLACK
  fb_put $(($x-2)) $(($y+12)) $BLACK
  fb_put $(($x-1)) $(($y+12)) $BLUE
  fb_put $x $(($y+12)) $BLUE
  fb_put $(($x+1)) $(($y+12)) $BLUE
  fb_put $(($x+2)) $(($y+12)) $BLUE
  fb_put $(($x+3)) $(($y+12)) $BLUE
  fb_put $(($x+4)) $(($y+12)) $BLUE
  fb_put $(($x+5)) $(($y+12)) $BLACK
  fb_put $(($x+6)) $(($y+12)) $BLACK
  fb_put $(($x+7)) $(($y+12)) $BLUE
  fb_put $(($x+8)) $(($y+12)) $BLUE
  fb_put $(($x+9)) $(($y+12)) $BLACK

  fb_put $(($x-7)) $(($y+13)) $BLACK
  fb_put $(($x-6)) $(($y+13)) $BLACK
  fb_put $(($x-5)) $(($y+13)) $BLACK
  fb_put $(($x-4)) $(($y+13)) $BLACK
  fb_put $(($x-3)) $(($y+13)) $BLACK
  fb_put $(($x-2)) $(($y+13)) $BLACK
  fb_put $(($x-1)) $(($y+13)) $BLUE
  fb_put $x $(($y+13)) $BLUE
  fb_put $(($x+1)) $(($y+13)) $BLUE
  fb_put $(($x+2)) $(($y+13)) $BLUE
  fb_put $(($x+3)) $(($y+13)) $BLACK
  fb_put $(($x+4)) $(($y+13)) $BLACK
  fb_put $(($x+5)) $(($y+13)) $BLUE
  fb_put $(($x+6)) $(($y+13)) $BLUE
  fb_put $(($x+7)) $(($y+13)) $BLUE
  fb_put $(($x+8)) $(($y+13)) $BLUE
  fb_put $(($x+9)) $(($y+13)) $BLACK

  fb_put $(($x-8)) $(($y+14)) $BLACK
  fb_put $(($x-7)) $(($y+14)) $BLUE
  fb_put $(($x-6)) $(($y+14)) $BLUE
  fb_put $(($x-5)) $(($y+14)) $BLUE
  fb_put $(($x-4)) $(($y+14)) $BLUE
  fb_put $(($x-3)) $(($y+14)) $BLUE
  fb_put $(($x-2)) $(($y+14)) $BLUE
  fb_put $(($x-1)) $(($y+14)) $BLUE
  fb_put $x $(($y+14)) $BLUE
  fb_put $(($x+1)) $(($y+14)) $BLUE
  fb_put $(($x+2)) $(($y+14)) $BLUE
  fb_put $(($x+3)) $(($y+14)) $BLUE
  fb_put $(($x+4)) $(($y+14)) $BLUE
  fb_put $(($x+5)) $(($y+14)) $BLACK
  fb_put $(($x+6)) $(($y+14)) $BLACK
  fb_put $(($x+7)) $(($y+14)) $BLUE
  fb_put $(($x+8)) $(($y+14)) $BLUE
  fb_put $(($x+9)) $(($y+14)) $BLACK

  fb_put $(($x-8)) $(($y+15)) $BLACK
  fb_put $(($x-7)) $(($y+15)) $BLUE
  fb_put $(($x-6)) $(($y+15)) $BLUE
  fb_put $(($x-5)) $(($y+15)) $BLUE
  fb_put $(($x-4)) $(($y+15)) $BLUE
  fb_put $(($x-3)) $(($y+15)) $BLUE
  fb_put $(($x-2)) $(($y+15)) $BLUE
  fb_put $(($x-1)) $(($y+15)) $BLUE
  fb_put $x $(($y+15)) $BLUE
  fb_put $(($x+1)) $(($y+15)) $BLUE
  fb_put $(($x+2)) $(($y+15)) $BLUE
  fb_put $(($x+3)) $(($y+15)) $BLUE
  fb_put $(($x+4)) $(($y+15)) $BLUE
  fb_put $(($x+5)) $(($y+15)) $BLUE
  fb_put $(($x+6)) $(($y+15)) $BLUE
  fb_put $(($x+7)) $(($y+15)) $BLUE
  fb_put $(($x+8)) $(($y+15)) $BLUE
  fb_put $(($x+9)) $(($y+15)) $BLACK

  fb_put $(($x-8)) $(($y+16)) $BLACK
  fb_put $(($x-7)) $(($y+16)) $BLUE
  fb_put $(($x-6)) $(($y+16)) $BLUE
  fb_put $(($x-5)) $(($y+16)) $BLACK
  fb_put $(($x-4)) $(($y+16)) $BLACK
  fb_put $(($x-3)) $(($y+16)) $BLUE
  fb_put $(($x-2)) $(($y+16)) $BLUE
  fb_put $(($x-1)) $(($y+16)) $BLUE
  fb_put $x $(($y+16)) $BLUE
  fb_put $(($x+1)) $(($y+16)) $BLUE
  fb_put $(($x+2)) $(($y+16)) $BLUE
  fb_put $(($x+3)) $(($y+16)) $BLUE
  fb_put $(($x+4)) $(($y+16)) $BLUE
  fb_put $(($x+5)) $(($y+16)) $BLUE
  fb_put $(($x+6)) $(($y+16)) $BLUE
  fb_put $(($x+7)) $(($y+16)) $BLUE
  fb_put $(($x+8)) $(($y+16)) $BLUE
  fb_put $(($x+9)) $(($y+16)) $BLACK

  fb_put $(($x-8)) $(($y+17)) $BLACK
  fb_put $(($x-7)) $(($y+17)) $BLUE
  fb_put $(($x-6)) $(($y+17)) $BLUE
  fb_put $(($x-5)) $(($y+17)) $BLUE
  fb_put $(($x-4)) $(($y+17)) $BLUE
  fb_put $(($x-3)) $(($y+17)) $BLACK
  fb_put $(($x-2)) $(($y+17)) $BLACK
  fb_put $(($x-1)) $(($y+17)) $BLUE
  fb_put $x $(($y+17)) $BLUE
  fb_put $(($x+1)) $(($y+17)) $BLUE
  fb_put $(($x+2)) $(($y+17)) $BLUE
  fb_put $(($x+3)) $(($y+17)) $BLUE
  fb_put $(($x+4)) $(($y+17)) $BLUE
  fb_put $(($x+5)) $(($y+17)) $BLACK
  fb_put $(($x+6)) $(($y+17)) $BLACK
  fb_put $(($x+7)) $(($y+17)) $BLACK
  fb_put $(($x+8)) $(($y+17)) $BLACK

  fb_put $(($x-8)) $(($y+18)) $BLACK
  fb_put $(($x-7)) $(($y+18)) $BLUE
  fb_put $(($x-6)) $(($y+18)) $BLUE
  fb_put $(($x-5)) $(($y+18)) $BLACK
  fb_put $(($x-4)) $(($y+18)) $BLACK
  fb_put $(($x-3)) $(($y+18)) $BLUE
  fb_put $(($x-2)) $(($y+18)) $BLUE
  fb_put $(($x-1)) $(($y+18)) $BLUE
  fb_put $x $(($y+18)) $BLUE
  fb_put $(($x+1)) $(($y+18)) $BLUE
  fb_put $(($x+2)) $(($y+18)) $BLUE
  fb_put $(($x+3)) $(($y+18)) $BLUE
  fb_put $(($x+4)) $(($y+18)) $BLUE
  fb_put $(($x+5)) $(($y+18)) $BLACK

  fb_put $(($x-7)) $(($y+19)) $BLACK
  fb_put $(($x-6)) $(($y+19)) $BLACK
  fb_put $(($x-4)) $(($y+19)) $BLACK
  fb_put $(($x-3)) $(($y+19)) $WHITE
  fb_put $(($x-2)) $(($y+19)) $WHITE
  fb_put $(($x-1)) $(($y+19)) $WHITE
  fb_put $x $(($y+19)) $WHITE
  fb_put $(($x+1)) $(($y+19)) $WHITE
  fb_put $(($x+2)) $(($y+19)) $WHITE
  fb_put $(($x+3)) $(($y+19)) $WHITE
  fb_put $(($x+4)) $(($y+19)) $WHITE
  fb_put $(($x+5)) $(($y+19)) $BLACK

  fb_put $(($x-4)) $(($y+20)) $BLACK
  fb_put $(($x-3)) $(($y+20)) $WHITE
  fb_put $(($x-2)) $(($y+20)) $WHITE
  fb_put $(($x-1)) $(($y+20)) $WHITE
  fb_put $x $(($y+20)) $WHITE
  fb_put $(($x+1)) $(($y+20)) $WHITE
  fb_put $(($x+2)) $(($y+20)) $WHITE
  fb_put $(($x+3)) $(($y+20)) $WHITE
  fb_put $(($x+4)) $(($y+20)) $WHITE
  fb_put $(($x+5)) $(($y+20)) $BLACK

  fb_put $(($x-7)) $(($y+21)) $BLACK
  fb_put $(($x-6)) $(($y+21)) $BLACK
  fb_put $(($x-5)) $(($y+21)) $BLACK
  fb_put $(($x-4)) $(($y+21)) $BLACK
  fb_put $(($x-3)) $(($y+21)) $WHITE
  fb_put $(($x-2)) $(($y+21)) $WHITE
  fb_put $(($x-1)) $(($y+21)) $WHITE
  fb_put $x $(($y+21)) $WHITE
  fb_put $(($x+1)) $(($y+21)) $BLACK
  fb_put $(($x+2)) $(($y+21)) $BLACK
  fb_put $(($x+3)) $(($y+21)) $WHITE
  fb_put $(($x+4)) $(($y+21)) $WHITE
  fb_put $(($x+5)) $(($y+21)) $BLACK

  fb_put $(($x-8)) $(($y+22)) $BLACK
  fb_put $(($x-7)) $(($y+22)) $WHITE
  fb_put $(($x-6)) $(($y+22)) $WHITE
  fb_put $(($x-5)) $(($y+22)) $WHITE
  fb_put $(($x-4)) $(($y+22)) $WHITE
  fb_put $(($x-3)) $(($y+22)) $WHITE
  fb_put $(($x-2)) $(($y+22)) $WHITE
  fb_put $(($x-1)) $(($y+22)) $WHITE
  fb_put $x $(($y+22)) $WHITE
  fb_put $(($x+1)) $(($y+22)) $BLACK
  fb_put $(($x+2)) $(($y+22)) $BLACK
  fb_put $(($x+3)) $(($y+22)) $WHITE
  fb_put $(($x+4)) $(($y+22)) $WHITE
  fb_put $(($x+5)) $(($y+22)) $BLACK
  fb_put $(($x+6)) $(($y+22)) $BLACK
  fb_put $(($x+7)) $(($y+22)) $BLACK
  fb_put $(($x+8)) $(($y+22)) $BLACK

  fb_put $(($x-8)) $(($y+23)) $BLACK
  fb_put $(($x-7)) $(($y+23)) $WHITE
  fb_put $(($x-6)) $(($y+23)) $WHITE
  fb_put $(($x-5)) $(($y+23)) $WHITE
  fb_put $(($x-4)) $(($y+23)) $WHITE
  fb_put $(($x-3)) $(($y+23)) $BLACK
  fb_put $(($x-2)) $(($y+23)) $BLACK
  fb_put $(($x-1)) $(($y+23)) $BLACK
  fb_put $x $(($y+23)) $BLACK
  fb_put $(($x+2)) $(($y+23)) $BLACK
  fb_put $(($x+3)) $(($y+23)) $WHITE
  fb_put $(($x+4)) $(($y+23)) $WHITE
  fb_put $(($x+5)) $(($y+23)) $WHITE
  fb_put $(($x+6)) $(($y+23)) $WHITE
  fb_put $(($x+7)) $(($y+23)) $WHITE
  fb_put $(($x+8)) $(($y+23)) $WHITE
  fb_put $(($x+9)) $(($y+23)) $BLACK

  fb_put $(($x-7)) $(($y+24)) $BLACK
  fb_put $(($x-6)) $(($y+24)) $BLACK
  fb_put $(($x-5)) $(($y+24)) $WHITE
  fb_put $(($x-4)) $(($y+24)) $WHITE
  fb_put $(($x-3)) $(($y+24)) $WHITE
  fb_put $(($x-2)) $(($y+24)) $WHITE
  fb_put $(($x-1)) $(($y+24)) $BLACK
  fb_put $(($x+2)) $(($y+24)) $BLACK
  fb_put $(($x+3)) $(($y+24)) $WHITE
  fb_put $(($x+4)) $(($y+24)) $WHITE
  fb_put $(($x+5)) $(($y+24)) $WHITE
  fb_put $(($x+6)) $(($y+24)) $WHITE
  fb_put $(($x+7)) $(($y+24)) $WHITE
  fb_put $(($x+8)) $(($y+24)) $WHITE
  fb_put $(($x+9)) $(($y+24)) $BLACK

  fb_put $(($x-6)) $(($y+25)) $BLACK
  fb_put $(($x-5)) $(($y+25)) $WHITE
  fb_put $(($x-4)) $(($y+25)) $WHITE
  fb_put $(($x-3)) $(($y+25)) $WHITE
  fb_put $(($x-2)) $(($y+25)) $WHITE
  fb_put $(($x-1)) $(($y+25)) $BLACK
  fb_put $(($x+2)) $(($y+25)) $BLACK
  fb_put $(($x+3)) $(($y+25)) $WHITE
  fb_put $(($x+4)) $(($y+25)) $WHITE
  fb_put $(($x+5)) $(($y+25)) $WHITE
  fb_put $(($x+6)) $(($y+25)) $WHITE
  fb_put $(($x+7)) $(($y+25)) $BLACK
  fb_put $(($x+8)) $(($y+25)) $BLACK

  fb_put $(($x-5)) $(($y+26)) $BLACK
  fb_put $(($x-4)) $(($y+26)) $BLACK
  fb_put $(($x-3)) $(($y+26)) $BLACK
  fb_put $(($x-2)) $(($y+26)) $BLACK
  fb_put $(($x+3)) $(($y+26)) $BLACK
  fb_put $(($x+4)) $(($y+26)) $BLACK
  fb_put $(($x+5)) $(($y+26)) $BLACK
  fb_put $(($x+6)) $(($y+26)) $BLACK
}

function draw_bubble {
  # Params: x y
  let x=$1
  let y=$2

  fb_put $x $y $BLACK

  fb_put $(($x-1)) $(($y+1)) $BLACK
  fb_put $x $(($y+1)) $WHITE
  fb_put $(($x+1)) $(($y+1)) $BLACK

  fb_put $(($x-4)) $(($y+2)) $BLACK
  fb_put $(($x-3)) $(($y+2)) $BLACK
  fb_put $(($x-2)) $(($y+2)) $BLACK
  fb_put $(($x-1)) $(($y+2)) $WHITE
  fb_put $x $(($y+2)) $WHITE
  fb_put $(($x+1)) $(($y+2)) $WHITE
  fb_put $(($x+2)) $(($y+2)) $BLACK
  fb_put $(($x+3)) $(($y+2)) $BLACK

  fb_put $(($x-5)) $(($y+3)) $BLACK
  fb_put $(($x-4)) $(($y+3)) $WHITE
  fb_put $(($x-3)) $(($y+3)) $WHITE
  fb_put $(($x-2)) $(($y+3)) $WHITE
  fb_put $(($x-1)) $(($y+3)) $GREEN
  fb_put $x $(($y+3)) $GREEN
  fb_put $(($x+1)) $(($y+3)) $GREEN
  fb_put $(($x+2)) $(($y+3)) $GREEN
  fb_put $(($x+3)) $(($y+3)) $GREEN
  fb_put $(($x+4)) $(($y+3)) $BLACK

  fb_put $(($x-5)) $(($y+4)) $BLACK
  fb_put $(($x-4)) $(($y+4)) $WHITE
  fb_put $(($x-3)) $(($y+4)) $WHITE
  fb_put $(($x-2)) $(($y+4)) $WHITE
  fb_put $(($x-1)) $(($y+4)) $GREEN
  fb_put $x $(($y+4)) $GREEN
  fb_put $(($x+1)) $(($y+4)) $GREEN
  fb_put $(($x+2)) $(($y+4)) $GREEN
  fb_put $(($x+3)) $(($y+4)) $GREEN
  fb_put $(($x+4)) $(($y+4)) $BLACK

  fb_put $(($x-4)) $(($y+5)) $BLACK
  fb_put $(($x-3)) $(($y+5)) $WHITE
  fb_put $(($x-2)) $(($y+5)) $GREEN
  fb_put $(($x-1)) $(($y+5)) $GREEN
  fb_put $x $(($y+5)) $WHITE
  fb_put $(($x+1)) $(($y+5)) $WHITE
  fb_put $(($x+2)) $(($y+5)) $GREEN
  fb_put $(($x+3)) $(($y+5)) $WHITE
  fb_put $(($x+4)) $(($y+5)) $WHITE
  fb_put $(($x+5)) $(($y+5)) $BLACK

  fb_put $(($x-5)) $(($y+6)) $BLACK
  fb_put $(($x-4)) $(($y+6)) $BLACK
  fb_put $(($x-3)) $(($y+6)) $GREEN
  fb_put $(($x-2)) $(($y+6)) $GREEN
  fb_put $(($x-1)) $(($y+6)) $WHITE
  fb_put $x $(($y+6)) $WHITE
  fb_put $(($x+1)) $(($y+6)) $BLACK
  fb_put $(($x+2)) $(($y+6)) $GREEN
  fb_put $(($x+3)) $(($y+6)) $BLACK
  fb_put $(($x+4)) $(($y+6)) $WHITE
  fb_put $(($x+5)) $(($y+6)) $GREEN
  fb_put $(($x+6)) $(($y+6)) $BLACK

  fb_put $(($x-6)) $(($y+7)) $BLACK
  fb_put $(($x-5)) $(($y+7)) $WHITE
  fb_put $(($x-4)) $(($y+7)) $WHITE
  fb_put $(($x-3)) $(($y+7)) $GREEN
  fb_put $(($x-2)) $(($y+7)) $GREEN
  fb_put $(($x-1)) $(($y+7)) $WHITE
  fb_put $x $(($y+7)) $WHITE
  fb_put $(($x+1)) $(($y+7)) $BLACK
  fb_put $(($x+2)) $(($y+7)) $GREEN
  fb_put $(($x+3)) $(($y+7)) $BLACK
  fb_put $(($x+4)) $(($y+7)) $WHITE
  fb_put $(($x+5)) $(($y+7)) $WHITE
  fb_put $(($x+6)) $(($y+7)) $BLACK

  fb_put $(($x-5)) $(($y+8)) $BLACK
  fb_put $(($x-4)) $(($y+8)) $WHITE
  fb_put $(($x-3)) $(($y+8)) $GREEN
  fb_put $(($x-2)) $(($y+8)) $GREEN
  fb_put $(($x-1)) $(($y+8)) $WHITE
  fb_put $x $(($y+8)) $WHITE
  fb_put $(($x+1)) $(($y+8)) $WHITE
  fb_put $(($x+2)) $(($y+8)) $GREEN
  fb_put $(($x+3)) $(($y+8)) $WHITE
  fb_put $(($x+4)) $(($y+8)) $WHITE
  fb_put $(($x+5)) $(($y+8)) $WHITE
  fb_put $(($x+6)) $(($y+8)) $BLACK

  fb_put $(($x-5)) $(($y+9)) $BLACK
  fb_put $(($x-4)) $(($y+9)) $WHITE
  fb_put $(($x-3)) $(($y+9)) $GREEN
  fb_put $(($x-2)) $(($y+9)) $GREEN
  fb_put $(($x-1)) $(($y+9)) $WHITE
  fb_put $x $(($y+9)) $WHITE
  fb_put $(($x+1)) $(($y+9)) $WHITE
  fb_put $(($x+2)) $(($y+9)) $GREEN
  fb_put $(($x+3)) $(($y+9)) $WHITE
  fb_put $(($x+4)) $(($y+9)) $WHITE
  fb_put $(($x+5)) $(($y+9)) $WHITE
  fb_put $(($x+6)) $(($y+9)) $BLACK

  fb_put $(($x-5)) $(($y+10)) $BLACK
  fb_put $(($x-4)) $(($y+10)) $WHITE
  fb_put $(($x-3)) $(($y+10)) $GREEN
  fb_put $(($x-2)) $(($y+10)) $GREEN
  fb_put $(($x-1)) $(($y+10)) $GREEN
  fb_put $x $(($y+10)) $GREEN
  fb_put $(($x+1)) $(($y+10)) $GREEN
  fb_put $(($x+2)) $(($y+10)) $GREEN
  fb_put $(($x+3)) $(($y+10)) $GREEN
  fb_put $(($x+4)) $(($y+10)) $GREEN
  fb_put $(($x+5)) $(($y+10)) $GREEN
  fb_put $(($x+6)) $(($y+10)) $GREEN
  fb_put $(($x+7)) $(($y+10)) $BLACK

  fb_put $(($x-5)) $(($y+11)) $BLACK
  fb_put $(($x-4)) $(($y+11)) $BLACK
  fb_put $(($x-3)) $(($y+11)) $GREEN
  fb_put $(($x-2)) $(($y+11)) $GREEN
  fb_put $(($x-1)) $(($y+11)) $BLACK
  fb_put $x $(($y+11)) $BLACK
  fb_put $(($x+1)) $(($y+11)) $BLACK
  fb_put $(($x+2)) $(($y+11)) $WHITE
  fb_put $(($x+3)) $(($y+11)) $WHITE
  fb_put $(($x+4)) $(($y+11)) $BLACK
  fb_put $(($x+5)) $(($y+11)) $GREEN
  fb_put $(($x+6)) $(($y+11)) $BLACK

  fb_put $(($x-5)) $(($y+12)) $BLACK
  fb_put $(($x-4)) $(($y+12)) $GREEN
  fb_put $(($x-3)) $(($y+12)) $GREEN
  fb_put $(($x-2)) $(($y+12)) $GREEN
  fb_put $(($x-1)) $(($y+12)) $GREEN
  fb_put $x $(($y+12)) $GREEN
  fb_put $(($x+1)) $(($y+12)) $GREEN
  fb_put $(($x+2)) $(($y+12)) $BLACK
  fb_put $(($x+3)) $(($y+12)) $BLACK
  fb_put $(($x+4)) $(($y+12)) $GREEN
  fb_put $(($x+5)) $(($y+12)) $BLACK

  fb_put $(($x-5)) $(($y+13)) $BLACK
  fb_put $(($x-4)) $(($y+13)) $GREEN
  fb_put $(($x-3)) $(($y+13)) $GREEN
  fb_put $(($x-2)) $(($y+13)) $GREEN
  fb_put $(($x-1)) $(($y+13)) $GREEN
  fb_put $x $(($y+13)) $GREEN
  fb_put $(($x+1)) $(($y+13)) $GREEN
  fb_put $(($x+2)) $(($y+13)) $BLACK
  fb_put $(($x+3)) $(($y+13)) $BLACK
  fb_put $(($x+4)) $(($y+13)) $GREEN
  fb_put $(($x+5)) $(($y+13)) $BLACK

  fb_put $(($x-5)) $(($y+14)) $BLACK
  fb_put $(($x-4)) $(($y+14)) $GREEN
  fb_put $(($x-3)) $(($y+14)) $GREEN
  fb_put $(($x-2)) $(($y+14)) $BLACK
  fb_put $(($x-1)) $(($y+14)) $GREEN
  fb_put $x $(($y+14)) $WHITE
  fb_put $(($x+1)) $(($y+14)) $WHITE
  fb_put $(($x+2)) $(($y+14)) $WHITE
  fb_put $(($x+3)) $(($y+14)) $WHITE
  fb_put $(($x+4)) $(($y+14)) $WHITE
  fb_put $(($x+5)) $(($y+14)) $GREEN
  fb_put $(($x+6)) $(($y+14)) $BLACK

  fb_put $(($x-5)) $(($y+15)) $BLACK
  fb_put $(($x-4)) $(($y+15)) $WHITE
  fb_put $(($x-3)) $(($y+15)) $BLACK
  fb_put $(($x-2)) $(($y+15)) $GREEN
  fb_put $(($x-1)) $(($y+15)) $GREEN
  fb_put $x $(($y+15)) $WHITE
  fb_put $(($x+1)) $(($y+15)) $WHITE
  fb_put $(($x+2)) $(($y+15)) $WHITE
  fb_put $(($x+3)) $(($y+15)) $WHITE
  fb_put $(($x+4)) $(($y+15)) $WHITE
  fb_put $(($x+5)) $(($y+15)) $BLACK
  fb_put $(($x+6)) $(($y+15)) $WHITE
  fb_put $(($x+7)) $(($y+15)) $BLACK

  fb_put $(($x-6)) $(($y+16)) $BLACK
  fb_put $(($x-5)) $(($y+16)) $GREEN
  fb_put $(($x-4)) $(($y+16)) $BLACK
  fb_put $(($x-3)) $(($y+16)) $BLACK
  fb_put $(($x-2)) $(($y+16)) $BLACK
  fb_put $(($x-1)) $(($y+16)) $BLACK
  fb_put $x $(($y+16)) $BLACK
  fb_put $(($x+1)) $(($y+16)) $WHITE
  fb_put $(($x+2)) $(($y+16)) $WHITE
  fb_put $(($x+3)) $(($y+16)) $WHITE
  fb_put $(($x+4)) $(($y+16)) $BLACK
  fb_put $(($x+5)) $(($y+16)) $BLACK
  fb_put $(($x+6)) $(($y+16)) $BLACK

  fb_put $(($x-6)) $(($y+17)) $BLACK
  fb_put $(($x-5)) $(($y+17)) $GREEN
  fb_put $(($x-4)) $(($y+17)) $BLACK
  fb_put $(($x-3)) $(($y+17)) $BLACK
  fb_put $(($x-2)) $(($y+17)) $BLACK
  fb_put $(($x-1)) $(($y+17)) $BLACK
  fb_put $x $(($y+17)) $BLACK
  fb_put $(($x+1)) $(($y+17)) $WHITE
  fb_put $(($x+2)) $(($y+17)) $WHITE
  fb_put $(($x+3)) $(($y+17)) $WHITE
  fb_put $(($x+4)) $(($y+17)) $BLACK
  fb_put $(($x+5)) $(($y+17)) $BLACK
  fb_put $(($x+6)) $(($y+17)) $BLACK

  fb_put $(($x-7)) $(($y+18)) $BLACK
  fb_put $(($x-6)) $(($y+18)) $GREEN
  fb_put $(($x-5)) $(($y+18)) $GREEN
  fb_put $(($x-4)) $(($y+18)) $BLACK
  fb_put $(($x-3)) $(($y+18)) $WHITE
  fb_put $(($x-2)) $(($y+18)) $GREEN
  fb_put $(($x-1)) $(($y+18)) $WHITE
  fb_put $x $(($y+18)) $GREEN
  fb_put $(($x+1)) $(($y+18)) $BLACK
  fb_put $(($x+2)) $(($y+18)) $WHITE
  fb_put $(($x+3)) $(($y+18)) $BLACK
  fb_put $(($x+4)) $(($y+18)) $GREEN
  fb_put $(($x+5)) $(($y+18)) $GREEN
  fb_put $(($x+6)) $(($y+18)) $WHITE
  fb_put $(($x+7)) $(($y+18)) $BLACK
}

function donut {
  while :
  do
    for x_radius in $(seq 1 8 35)
    do
      fb_clear
      draw_donut $HALFCOLS $HALFLINES $x_radius $(($HALFLINES-4)) $BLACK 0
      refresh
    done
    for x_radius in $(seq 1 8 35|sort -r -n)
    do
      fb_clear
      draw_donut $HALFCOLS $HALFLINES $x_radius $(($HALFLINES-4)) $BLACK 1
      refresh
    done
  done
}

function calc_cycle {
  # Params: col1 col2 col3 col4 tmpcol

  COLOR[1]=$1
  COLOR[2]=$2
  COLOR[3]=$3
  COLOR[4]=$4

  # Setup screen
  for i in $(seq 1 4)
  do
    for x in $(seq $i 4 $COLUMNS)
    do
      for y in $(seq 1 $LINES)
      do
        fb_put $x $y ${COLOR[$i]}
      done
    done
  done
  cycle1=("${fb[@]}")

  COLOR[5]=${COLOR[4]}
  COLOR[4]=${COLOR[3]}
  COLOR[3]=${COLOR[2]}
  COLOR[2]=${COLOR[1]}
  COLOR[1]=${COLOR[5]}
  fb=("${fb[@]//${COLOR[5]}/$5}")
  fb=("${fb[@]//${COLOR[4]}/${COLOR[5]}}")
  fb=("${fb[@]//${COLOR[3]}/${COLOR[4]}}")
  fb=("${fb[@]//${COLOR[2]}/${COLOR[3]}}")
  fb=("${fb[@]//$5/${COLOR[2]}}")
  cycle2=("${fb[@]}")

  COLOR[5]=${COLOR[4]}
  COLOR[4]=${COLOR[3]}
  COLOR[3]=${COLOR[2]}
  COLOR[2]=${COLOR[1]}
  COLOR[1]=${COLOR[5]}
  fb=("${fb[@]//${COLOR[5]}/$5}")
  fb=("${fb[@]//${COLOR[4]}/${COLOR[5]}}")
  fb=("${fb[@]//${COLOR[3]}/${COLOR[4]}}")
  fb=("${fb[@]//${COLOR[2]}/${COLOR[3]}}")
  fb=("${fb[@]//$5/${COLOR[2]}}")
  cycle3=("${fb[@]}")

  COLOR[5]=${COLOR[4]}
  COLOR[4]=${COLOR[3]}
  COLOR[3]=${COLOR[2]}
  COLOR[2]=${COLOR[1]}
  COLOR[1]=${COLOR[5]}
  fb=("${fb[@]//${COLOR[5]}/$5}")
  fb=("${fb[@]//${COLOR[4]}/${COLOR[5]}}")
  fb=("${fb[@]//${COLOR[3]}/${COLOR[4]}}")
  fb=("${fb[@]//${COLOR[2]}/${COLOR[3]}}")
  fb=("${fb[@]//$5/${COLOR[2]}}")
  cycle4=("${fb[@]}")
}

function run_cycle {
  while :
  do
    fb=("${cycle1[@]}")
    draw_smurf1 $(($HALFCOLS-8)) $(($HALFLINES-13))
    refresh

    fb=("${cycle2[@]}")
    draw_smurf2 $(($HALFCOLS-8)) $(($HALFLINES-13))
    refresh

    fb=("${cycle3[@]}")
    draw_smurf3 $(($HALFCOLS-8)) $(($HALFLINES-13))
    refresh

    fb=("${cycle4[@]}")
    draw_smurf4 $(($HALFCOLS-8)) $(($HALFLINES-13))
    refresh
  done
}

function out_grid_slide {
  # Params: topy bottomy
  for j in $(seq 1 8)
  do
    for i in $(seq $(($1-1)) $(($2-1)))
    do
      if [ $(($i%2)) -eq 0 ]
      then
        fb_shift $i $((COLUMNS/8))
      else
        fb_shift $i -$(($COLUMNS/8))
      fi
    done
    refresh
  done
}

function fb_shift {
  # Params: line change
  let index=$1*$COLUMNS
  let step=${2/-/}
  let shiftpos=$index+$step
  tmparr=()
  for i in $(seq 1 $step)
  do
    tmparr[${#tmparr[@]}]="$BG"
  done
  if [ $2 -gt 0 ]
  then
    fb=("${fb[@]:0:$index}" "${tmparr[@]}" "${fb[@]:$index:$(($COLUMNS-$step))}" "${fb[@]:$(($index+$COLUMNS))}")
  else
    fb=("${fb[@]:0:$index}" "${fb[@]:$shiftpos:$(($COLUMNS-$step))}" "${tmparr[@]}" "${fb[@]:$(($index+$COLUMNS))}")
  fi
}

function effect_at {
  let stoptime=$1
  shift
  echo "$stoptime" > $SYNCFILE
  kill -10 $SYNCPID
  sleep 10000 &
  let sleeper=$!
  wait $sleeper
  $@
}

function stop_effect {
  echo "$1" > $SYNCFILE
  kill -10 $SYNCPID &>/dev/null
  sleep 10000 &
  let sleeper=$!
  wait $sleeper
  kill -9 $2 &>/dev/null
}

function sync_sniffer {
  trap "sync" 10
  let curpos=0
  let stop=100000   #Arbitrary high stoptime
  while read -d "
" cur
  do
    # Read curpos from musicplayer output, if garble set curpos to 0
    let curpos=$(echo ${cur:28:8}|sed -e 's/[:.]//g' -e 's/^[0]*//g') 2>/dev/null || let curpos=0  

    if [ $curpos -ge $stop ]
    then
      kill -SIGCONT $DEMOPID
    fi
  done < <($MUSICPLAYER -q $MUSIC 2>&1)
}


# Initialization

rm "${TEMPDIR}/*" 2>/dev/null
mkdir -p "$TEMPDIR"
touch $SYNCFILE

mkfifo $BCIN $BCOUT || die "Cannot create FIFO's in $TEMPDIR"
exec 7<>$BCIN || die "Could not attach to $BCIN"
exec 8<>$BCOUT || die "Could not attach to $BCOUT"
bc -l 0<&7 1>&8 2>&8 &

# Calc PI for later
echo "scale=10; pi=4*a(1)" >&7

echo "$CURSOFF"
echo -n "$ESC[${LINES};0H"
echo "$CURSPOSSAVE"
BG="$BLUE"  # Temp voxel thingie
fb_setup
refresh

# Precalcing
$MUSICPLAYER $MUSIC_PRECALC $MUSIC_PRECALC $MUSIC_PRECALC $MUSIC_PRECALC &>/dev/null &
tmppid=$!
echo -n "$ESC[$HALFLINES;$((HALFCOLS-4))H$ESC[37mLoading.."

calc_circle1
calc_circle2
calc_moire_circle $WHITE $RED
calc_cycle $BLACK $YELLOW $GREEN $CYAN $PURPLE
calc_voxel

fb_setup
refresh
kill -9 $tmppid &>/dev/null
echo -n "$ESC[$HALFLINES;$((HALFCOLS-4))H         "


# Start the demo

sync_sniffer &  #Run music and syncing
SYNCPID=$!
renice +20 $SYNCPID &>/dev/null

effect_at "415" bzh_logo 3 "$WHITE"
refresh

effect_at "504" bzh_logo 3 "$BLUE"
refresh
effect_at "580" bzh_logo 10 "$WHITE"
refresh
effect_at "666" bzh_logo 10 "$BLUE"
refresh
effect_at "752" bzh_logo 1 "$WHITE"
refresh
effect_at "838" bzh_logo 1 "$BLUE"
refresh
effect_at "924" bzh_logo $(($LINES-7)) "$WHITE"
refresh
effect_at "1010" bzh_logo $(($LINES-7)) "$BLUE"
refresh
effect_at "1096" bzh_logo 1 "$WHITE"
refresh
effect_at "1182" bzh_logo 1 "$BLUE"
refresh
effect_at "1268" bzh_logo $(($HALFLINES-2)) "$WHITE"
refresh
effect_at "1354" bzh_logo $(($HALFLINES-2)) "$BLUE"
refresh
effect_at "1440" bzh_logo 1 "$WHITE"
refresh
effect_at "1526" bzh_logo 1 "$BLUE"
refresh
effect_at "1612" bzh_logo $(($LINES-5)) "$WHITE"
refresh
effect_at "1698" bzh_logo $(($LINES-5)) "$BLUE"
refresh

effect_at "1784" bzh_logo 1 "$GREEN"
BG="$BLACK"
fb_line 1 7 $COLUMNS 7 $CYAN
fb_clear_from 1 8
line_rotator 1 8 $COLUMNS $LINES 5 &
EFFECT[1]=$!
stop_effect "3053" ${EFFECT[1]}

out_grid_slide 1 7

effect_at "3223" ":"
moire_circle &
EFFECT[2]=$!
stop_effect "10000" ${EFFECT[2]}

BG=$WHITE
fb_clear_from 1 1
refresh
BG=$BLACK
fb_clear_from 1 1
bsh_logo $(($HALFLINES-3)) $WHITE
refresh

effect_at "10360" ":"
BG=$WHITE
fb_clear_from 1 1
refresh
BG=$BLACK
fb_clear_from 1 1
bash_logo $(($HALFLINES-3)) $WHITE 1
refresh

effect_at "10705" ":"
BG=$WHITE
fb_clear_from 1 1
refresh
BG=$BLACK
fb_clear_from 1 1
bash_logo $(($HALFLINES-3)) $WHITE 2
refresh

effect_at "11050" ":"
BG=$WHITE
fb_clear_from 1 1
refresh
BG=$BLACK
fb_clear_from 1 1
bash_logo $(($HALFLINES-3)) $WHITE 3
refresh

effect_at "11400" ":"
BG=$WHITE
fb_clear_from 1 1
refresh
sleep 0.1
BG=$BLACK
fb_clear_from 1 1
refresh

effect_at "11473" ":"
BG=$WHITE
fb_clear_from 1 1
refresh
sleep 0.1
BG=$BLACK
fb_clear_from 1 1
refresh

effect_at "11580" ":"
BG=$WHITE
fb_clear_from 1 1
refresh
sleep 0.1
BG=$BLACK
fb_clear_from 1 1
refresh

effect_at "11653" ":"
BG=$WHITE
fb_clear_from 1 1
refresh
sleep 0.1
BG=$BLACK
fb_clear_from 1 1
refresh

effect_at "11726" ":"
fb_square 1 1 $((COLUMNS/3)) $(($LINES/3)) $CYAN
refresh

effect_at "11799" ":"
fb_square $((COLUMNS/3)) 1 $((COLUMNS/3*2)) $((LINES/3)) $CYAN
refresh

effect_at "11908" ":"
fb_square $((COLUMNS/3*2)) 1 $((COLUMNS)) $((LINES/3)) $CYAN
refresh

effect_at "11981" ":"
fb_square 1 $((LINES/3)) $((COLUMNS/3)) $(($LINES/3*2)) $CYAN
refresh

effect_at "12090" ":"
fb_square $((COLUMNS/3)) $((LINES/3)) $((COLUMNS/3*2)) $(($LINES/3*2)) $CYAN
refresh

effect_at "12163" ":"
fb_square $((COLUMNS/3*2)) $((LINES/3)) $((COLUMNS)) $(($LINES/3*2)) $CYAN
refresh

effect_at "12272" ":"
fb_square 1 $((LINES/3*2)) $((COLUMNS/3)) $(($LINES)) $CYAN
refresh

effect_at "12345" ":"
fb_square $((COLUMNS/3)) $((LINES/3*2)) $((COLUMNS/3*2)) $(($LINES)) $CYAN
refresh

effect_at "12418" ":"
fb_square $((COLUMNS/3*2)) $((LINES/3*2)) $((COLUMNS)) $(($LINES)) $CYAN
refresh

effect_at "12783" ":"
BG=$CYAN
fb_clear
refresh
cube &
EFFECT[3]=$!
stop_effect "15557" ${EFFECT[3]}

sine_move 3 0.8 4 12 30 45 draw_bubble &
EFFECT[4]=$!
stop_effect "20980" ${EFFECT[4]}

BG=$WHITE
fb_clear_from 1 1
refresh
BG=$BLUE
fb_clear_from 1 1
bzh_logo $(($HALFLINES-3)) $WHITE
refresh

effect_at "21126" ":"
donut &
EFFECT[5]=$!
stop_effect "23754" ${EFFECT[5]}

run_cycle &
EFFECT[5]=$!
stop_effect "25871" ${EFFECT[5]}

fade_in_strobe &
EFFECT[6]=$!
stop_effect "31261" ${EFFECT[6]}

voxel &
EFFECT[7]=$!
stop_effect "34028" ${EFFECT[7]}

BG=$WHITE
fb_clear_from 1 1
refresh
BG=$BLACK
fb_clear_from 1 1
refresh
echo "$ESC[$HALFLINES;$(($HALFCOLS-2))H$ESC[37mPhew!"


sleep 12


# Clean up at the end

kill $SYNCPID &>/dev/null
kill "${EFFECT[@]}" &>/dev/null
# Exit bc
echo "exit" >&7
exec 7>&-
exec 7<&-
exec 8>&-
exec 8<&-
rm $BCIN
rm $BCOUT

rm ${TEMPDIR}/* &>/dev/null
rmdir $TEMPDIR &>/dev/null

reset
echo "$CURSPOSLOAD"
echo "$OFF"
echo "$CURSON"
echo "\o/"
exit 0
