///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef SHARED_THMEMORY_H
#define SHARED_THMEMORY_H
///////////////////////////////////////////////////////////////////////////////////////////////////

template <typename T>
class ThPtr
{
public:
// constructors
	inline														ThPtr(void);
	inline														ThPtr(T* object);
	inline														ThPtr(const ThPtr<T>& source);
	inline														~ThPtr(void);
// methods
	inline T*													detach(void);
	inline void												release(void);
// operators
	inline T*													operator -> (void);
	inline const T*										operator -> (void) const;
	inline														operator T* (void);
	inline														operator const T* (void) const;
	inline ThPtr<T>&									operator = (T* object);
	inline ThPtr<T>&									operator = (const ThPtr<T>& source);
// attributes
	inline T*													getObject(void);
	inline const T*										getObject(void) const;
private:
// data
	T*																m_object;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThRef
{
public:
// constructors
	inline														ThRef(void);
	inline														ThRef(const ThRef<T>& source);
	inline														~ThRef(void);
// methods
	inline void												release(void);
// operators
	inline T*													operator -> (void) const;
	inline														operator T* (void) const;
	inline ThRef<T>&									operator = (T* object);
	inline ThRef<T>&									operator = (const ThRef<T>& source);
// attributes
	inline T*													getObject(void) const;
private:
// data
	T*																m_object;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThRefObject
{
// friends
	friend ThRef<T>;
public:
// constructors
	inline														ThRefObject(void);
	inline														~ThRefObject(void);
protected:
// methods
	inline void												reference(void);
	inline bool												release(void);
private:
// data
	unsigned int											m_count;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThBlock
{
public:
// constructors
	inline														ThBlock(void);
	inline explicit										ThBlock(unsigned int count);
	inline														ThBlock(const ThBlock<T>& source);
	inline														~ThBlock(void);
// methods
	inline void												allocate(unsigned int count);
	inline void												resize(unsigned int count);
	inline void												release(void);
// operators
	inline														operator T* (void);
	inline														operator const T* (void) const;
	inline ThBlock<T>&								operator = (const ThBlock<T>& source);
// attributes
	inline T*													getData(void);
	inline const T*										getData(void) const;
	inline unsigned int								getSize(void) const;
	inline unsigned int								getCount(void) const;
private:
// data
	T*																m_data;
	unsigned int											m_count;
};

//-------------------------------------------------------------------------------------------------

typedef ThBlock<char> ThByteBlock;

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThListItem
{
// friends
	friend class ThItemList<T>;
public:
// constructors
	inline														ThListItem(void);
	inline														ThListItem(const ThListItem<T>& source);
	inline														~ThListItem(void);
// methods
	inline void												detach(void);
// operators
	inline ThListItem<T>&							operator = (const ThListItem<T>& source);
// attributes
	inline bool												isAttached(void) const;
	inline T*													getPrev(void);
	inline const T*										getPrev(void) const;
	inline T*													getNext(void);
	inline const T*										getNext(void) const;
private:
// data
	T*																m_prev;
	T*																m_next;
	ThItemList<T>*										m_list;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThItemList
{
// friends
	friend class ThListItem<T>;
public:
// constructors
	inline														ThItemList(void);
	inline														ThItemList(const ThItemList<T>& source);
	inline														~ThItemList(void);
// methods
	inline void												attachFirst(T* item);
	inline void												attachLast(T* item);
	inline void												attachAt(T* item, unsigned int index);
	inline ThIterator<T>							iterate(void);
	inline ThConstIterator<T>					iterate(void) const;
	inline void												release(void);
// operators
	inline ThItemList<T>&							operator = (const ThItemList<T>& source);
// attributes
	inline T*													getFirst(void);
	inline const T*										getFirst(void) const;
	inline T*													getLast(void);
	inline const T*										getLast(void) const;
	inline T*													getAt(unsigned int index);
	inline const T*										getAt(unsigned int index) const;
	inline unsigned int								getSize(void) const;
	inline unsigned int								getCount(void) const;
private:
// data
	T*																m_head;
	unsigned int											m_count;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThHashItem : public ThListItem<T>
{
public:
// constructors
	inline														ThHashItem(unsigned int hash = 0);
	inline														ThHashItem(const ThHashItem<T>& source);
// operators
	inline ThHashItem<T>&							operator = (const ThHashItem<T>& source);
// attributes
	inline unsigned int								getHash(void) const;
	inline void												setHash(unsigned int hash);
private:
// data
	unsigned int											m_hash;
};

//-------------------------------------------------------------------------------------------------

template <typename T, unsigned int N = 10>
class ThHashList
{
public:
// constructors
	inline														ThHashList(void);
	inline														ThHashList(const ThHashList<T,N>& source);
// methods
	inline void												attach(T* item);
	inline void												release(void);
// operators
	inline ThHashList<T,N>&						operator = (const ThHashList<T,N>& source);
// attributes
	inline T*													getItem(unsigned int hash);
	inline const T*										getItem(unsigned int hash) const;
private:
// data
	ThBlock<ThItemList<T> >						m_root;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThIterator
{
public:
// constructors
	inline														ThIterator(void);
	inline														ThIterator(T* item);
	inline														ThIterator(ThItemList<T>& list);
// methods
	inline T*													next(void);
	inline T*													detach(void);
	inline void												release(void);
// operators
	inline T*													operator -> (void) const;
	inline														operator T* (void) const;
	inline ThIterator<T>&							operator = (T* item);
// attributes
	inline T*													getItem(void);
	inline const T*										getItem(void) const;
private:
// data
	T*																m_item;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThConstIterator
{
public:
// constructors
	inline														ThConstIterator(void);
	inline														ThConstIterator(const T* item);
	inline														ThConstIterator(const ThItemList<T>& list);
// methods
	inline const T*										next(void);
	inline const T*										detach(void);
// operators
	inline const T*										operator -> (void) const;
	inline														operator const T* (void) const;
	inline ThConstIterator<T>&				operator = (const T* item);
// attributes
	inline const T*										getItem(void) const;
private:
// data
	const T*													m_item;
};

//-------------------------------------------------------------------------------------------------

template <typename T>
class ThSingleton
{
// friends
	friend T;
public:
// operators
	inline T*													operator -> (void);
	inline														operator T* (void);
// methods
	inline void												release(void); // NOTE: this is a design flaw!
private:
// methods
	inline void												attach(T* object);
// data
	ThPtr<T>													m_object;
};

///////////////////////////////////////////////////////////////////////////////////////////////////

interface IThStaticBlock
{
// constructors
	virtual														~IThStaticBlock(void);
// methods
	virtual void*											lock(void) = 0;
	virtual void											unlock(void) = 0;
// attributes
	virtual unsigned int							getSize(void) const = 0;
// static methods
	static IThStaticBlock*						createInstance(unsigned int size);
};

//-------------------------------------------------------------------------------------------------

interface IThDynamicBlock : public IThStaticBlock
{
// constructors
	virtual														~IThDynamicBlock(void);
// methods
	virtual void											allocate(unsigned int size) = 0;
	virtual void											resize(unsigned int size) = 0;
	virtual void											release(void) = 0;
// static methods
	static IThDynamicBlock*						createInstance(unsigned int size);
};

///////////////////////////////////////////////////////////////////////////////////////////////////

#include "ThMemory.inl"

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* SHARED_THMEMORY_H */
///////////////////////////////////////////////////////////////////////////////////////////////////
