///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#define _USE_MATH_DEFINES

#include <cmath>
#include <cstdlib>

///////////////////////////////////////////////////////////////////////////////////////////////////

inline float random(void)
{
	return (float) rand() / (float) RAND_MAX;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThRange constructors ---------------------------------------------------------------------------

template <typename T>
inline ThRange<T>::ThRange(void):
	m_min(0),
	m_max(0)
{
}

template <typename T>
inline ThRange<T>::ThRange(T min, T max):
	m_min(min),
	m_max(max)
{
}

// ThRange methods --------------------------------------------------------------------------------

template <typename T>
inline T ThRange<T>::generate(void) const
{
	return m_min + ((m_max - m_min) * (T) rand()) / (T) RAND_MAX;
}

template <typename T>
inline void ThRange<T>::reset(void)
{
	m_min = 0;
	m_max = 0;
}

// ThRange attributes -----------------------------------------------------------------------------

template <typename T>
inline void ThRange<T>::set(T min, T max)
{
	m_min = min;
	m_max = max;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSpace constructors ---------------------------------------------------------------------------

template <typename T>
inline ThSpace<T>::ThSpace(void)
{
	m_min.reset();
	m_max.reset();
}

template <typename T>
inline ThSpace<T>::ThSpace(T min, T max):
	m_min(min),
	m_max(max)
{
}

// ThSpace methods --------------------------------------------------------------------------------

template <typename T>
inline T ThSpace<T>::generate(void) const
{
	return m_min + ((m_max - m_min) * (float) rand()) / (float) RAND_MAX;
}

template <typename T>
inline void ThSpace<T>::reset(void)
{
	m_min.reset();
	m_max.reset();
}

// ThSpace attributes -----------------------------------------------------------------------------

template <typename T>
inline void ThSpace<T>::set(T min, T max)
{
	m_min = min;
	m_max = max;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThVector2 constructors -------------------------------------------------------------------------

inline ThVector2::ThVector2(void):
	x(0.f),
	y(0.f)
{
}

inline ThVector2::ThVector2(float sx, float sy):
	x(sx),
	y(sy)
{
}

// ThVector2 methods ------------------------------------------------------------------------------

inline float ThVector2::length(void) const
{
	return (float) sqrtf(float(x * x + y * y));
}

inline float ThVector2::dotProduct(const ThVector2& vector) const
{
	return x * x + y * y;
}

inline void ThVector2::scaleTo(float len)
{
	const float scale = len / length();

	x *= scale;
	y *= scale;
}

inline void ThVector2::normalize(void)
{
	const float scale = 1.f / length();

	x *= scale;
	y *= scale;
}

inline void ThVector2::reset(void)
{
	x = 0.f;
	y = 0.f;
}

// ThVector2 operators ----------------------------------------------------------------------------

inline ThVector2::operator float* (void)
{
	return &(x);
}

inline ThVector2::operator const float* (void) const
{
	return &(x);
}

inline ThVector2 ThVector2::operator - (void) const
{
	return ThVector2(-x, -y);
}

inline ThVector2 ThVector2::operator + (float value) const
{
	return ThVector2(x + value, y + value);
}

inline ThVector2 ThVector2::operator - (float value) const
{
	return ThVector2(x - value, y - value);
}

inline ThVector2 ThVector2::operator * (float value) const
{
	return ThVector2(x * value, y * value);
}

inline ThVector2 ThVector2::operator / (float value) const
{
	return ThVector2(x / value, y / value);
}

inline ThVector2 ThVector2::operator += (float value)
{
	x += value;
	y += value;
	return *this;
}

inline ThVector2& ThVector2::operator -= (float value)
{
	x -= value;
	y -= value;
	return *this;
}

inline ThVector2& ThVector2::operator *= (float value)
{
	x *= value;
	y *= value;
	return *this;
}

inline ThVector2& ThVector2::operator /= (float value)
{
	x /= value;
	y /= value;
	return *this;
}

inline ThVector2 ThVector2::operator + (const ThVector2& vector) const
{
	return ThVector2(x + vector.x, y + vector.y);
}

inline ThVector2 ThVector2::operator - (const ThVector2& vector) const
{
	return ThVector2(x - vector.x, y - vector.y);
}

inline ThVector2 ThVector2::operator * (const ThVector2& vector) const
{
	return ThVector2(x * vector.x, y * vector.y);
}

inline ThVector2 ThVector2::operator / (const ThVector2& vector) const
{
	return ThVector2(x / vector.x, y / vector.y);
}

inline ThVector2 ThVector2::operator += (const ThVector2& vector)
{
	x += vector.x;
	y += vector.y;
	return *this;
}

inline ThVector2& ThVector2::operator -= (const ThVector2& vector)
{
	x -= vector.x;
	y -= vector.y;
	return *this;
}

inline ThVector2& ThVector2::operator *= (const ThVector2& vector)
{
	x *= vector.x;
	y *= vector.y;
	return *this;
}

inline ThVector2& ThVector2::operator /= (const ThVector2& vector)
{
	x /= vector.x;
	y /= vector.y;
	return *this;
}

inline bool ThVector2::operator == (const ThVector2& vector) const
{
	return x == vector.x && y == vector.y;
}

inline bool ThVector2::operator != (const ThVector2& vector) const
{
	return x != vector.x || y != vector.y;
}

// ThVector2 attributes ---------------------------------------------------------------------------

inline void ThVector2::set(float sx, float sy)
{
	x = sx;
	y = sy;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThVector3 constructors -------------------------------------------------------------------------

inline ThVector3::ThVector3(void):
	x(0.f),
	y(0.f),
	z(0.f)
{
}

inline ThVector3::ThVector3(float sx, float sy, float sz):
	x(sx),
	y(sy),
	z(sz)
{
}

// ThVector3 methods ------------------------------------------------------------------------------

inline float ThVector3::length(void) const
{
	return sqrtf(x * x + y * y + z * z);
}

inline float ThVector3::dotProduct(const ThVector3& vector) const
{
	return x * vector.x + y * vector.y + z * vector.z;
}

inline ThVector3 ThVector3::crossProduct(const ThVector3& vector) const
{
	return ThVector3(vector.z * y - vector.y * z, vector.x * z - vector.z * x, vector.y * x - vector.x * y);
}

inline void ThVector3::scaleTo(float len)
{
	const float scale = len / length();

	x *= scale;
	y *= scale;
	z *= scale;
}

inline void ThVector3::normalize(void)
{
	const float scale = 1.f / length();

	x *= scale;
	y *= scale;
	z *= scale;
}

inline void ThVector3::reset(void)
{
	x = 0.f;
	y = 0.f;
	z = 0.f;
}

// ThVector3 operators ----------------------------------------------------------------------------

inline ThVector3 ThVector3::operator - (void) const
{
	return ThVector3(-x, -y, -z);
}

inline ThVector3 ThVector3::operator + (float value) const
{
	return ThVector3(x + value, y + value, z + value);
}

inline ThVector3 ThVector3::operator - (float value) const
{
	return ThVector3(x - value, y - value, z - value);
}

inline ThVector3 ThVector3::operator * (float value) const
{
	return ThVector3(x * value, y * value, z * value);
}

inline ThVector3 ThVector3::operator / (float value) const
{
	return ThVector3(x / value, y / value, z / value);
}

inline ThVector3 ThVector3::operator += (float value)
{
	x += value;
	y += value;
	z += value;
	return *this;
}

inline ThVector3& ThVector3::operator -= (float value)
{
	x -= value;
	y -= value;
	z -= value;
	return *this;
}

inline ThVector3& ThVector3::operator *= (float value)
{
	x *= value;
	y *= value;
	z *= value;
	return *this;
}

inline ThVector3& ThVector3::operator /= (float value)
{
	x /= value;
	y /= value;
	z /= value;
	return *this;
}

inline ThVector3 ThVector3::operator + (const ThVector3& vector) const
{
	return ThVector3(x + vector.x, y + vector.y, z + vector.z);
}

inline ThVector3 ThVector3::operator - (const ThVector3& vector) const
{
	return ThVector3(x - vector.x, y - vector.y, z - vector.z);
}

inline ThVector3 ThVector3::operator * (const ThVector3& vector) const
{
	return ThVector3(x * vector.x, y * vector.y, z * vector.z);
}

inline ThVector3 ThVector3::operator / (const ThVector3& vector) const
{
	return ThVector3(x / vector.x, y / vector.y, z / vector.z);
}

inline ThVector3 ThVector3::operator += (const ThVector3& vector)
{
	x += vector.x;
	y += vector.y;
	z += vector.z;
	return *this;
}

inline ThVector3& ThVector3::operator -= (const ThVector3& vector)
{
	x -= vector.x;
	y -= vector.y;
	z -= vector.z;
	return *this;
}

inline ThVector3& ThVector3::operator *= (const ThVector3& vector)
{
	x *= vector.x;
	y *= vector.y;
	z *= vector.z;
	return *this;
}

inline ThVector3& ThVector3::operator /= (const ThVector3& vector)
{
	x /= vector.x;
	y /= vector.y;
	z /= vector.z;
	return *this;
}

inline bool ThVector3::operator == (const ThVector3& vector) const
{
	return x == vector.x && y == vector.y && z == vector.z;
}

inline bool ThVector3::operator != (const ThVector3& vector) const
{
	return x != vector.x || y != vector.y || z != vector.z;
}

// ThVector3 attributes ---------------------------------------------------------------------------

inline void ThVector3::set(float sx, float sy, float sz)
{
	x = sx;
	y = sy;
	z = sz;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThVector4 constructors -------------------------------------------------------------------------

inline ThVector4::ThVector4(void):
	x(0),
	y(0),
	z(0),
	w(0)
{
}

inline ThVector4::ThVector4(float sx, float sy, float sz, float sw):
	x(sx),
	y(sy),
	z(sz),
	w(sw)
{
}

// ThVector4 methods ------------------------------------------------------------------------------

inline float ThVector4::length(void) const
{
	return sqrtf(x * x + y * y + z * z + w * w);
}

inline void ThVector4::scaleTo(float len)
{
	const float scale = len / length();

	x *= scale;
	y *= scale;
	z *= scale;
	w *= scale;
}

inline void ThVector4::normalize(void)
{
	const float scale = 1.f / length();

	x *= scale;
	y *= scale;
	z *= scale;
	w *= scale;
}

inline void ThVector4::reset(void)
{
	x = 0.f;
	y = 0.f;
	z = 0.f;
	w = 0.f;
}

// ThVector4 operators ----------------------------------------------------------------------------

inline ThVector4::operator float* (void)
{
	return &(x);
}

inline ThVector4::operator const float* (void) const
{
	return &(x);
}

inline ThVector4 ThVector4::operator - (void) const
{
	return ThVector4(-x, -y, -z, -w);
}

inline ThVector4 ThVector4::operator + (float value) const
{
	return ThVector4(x + value, y + value, z + value, w + value);
}

inline ThVector4 ThVector4::operator - (float value) const
{
	return ThVector4(x - value, y - value, z - value, w - value);
}

inline ThVector4 ThVector4::operator * (float value) const
{
	return ThVector4(x * value, y * value, z * value, w * value);
}

inline ThVector4 ThVector4::operator / (float value) const
{
	value = 1.f / value;

	return ThVector4(x * value, y * value, z * value, w * value);
}

inline ThVector4 ThVector4::operator += (float value)
{
	x += value;
	y += value;
	z += value;
	w += value;
	return *this;
}

inline ThVector4& ThVector4::operator -= (float value)
{
	x -= value;
	y -= value;
	z -= value;
	w -= value;
	return *this;
}

inline ThVector4& ThVector4::operator *= (float value)
{
	x *= value;
	y *= value;
	z *= value;
	w *= value;
	return *this;
}

inline ThVector4& ThVector4::operator /= (float value)
{
	const float scale = 1.f / value;

	x *= scale;
	y *= scale;
	z *= scale;
	w *= scale;
	return *this;
}

inline ThVector4 ThVector4::operator + (const ThVector4& vector) const
{
	return ThVector4(x + vector.x, y + vector.y, z + vector.z, w + vector.w);
}

inline ThVector4 ThVector4::operator - (const ThVector4& vector) const
{
	return ThVector4(x - vector.x, y - vector.y, z - vector.z, w - vector.w);
}

inline ThVector4 ThVector4::operator * (const ThVector4& vector) const
{
	return ThVector4(x * vector.x, y * vector.y, z * vector.z, w * vector.w);
}

inline ThVector4 ThVector4::operator / (const ThVector4& vector) const
{
	return ThVector4(x / vector.x, y / vector.y, z / vector.z, w / vector.w);
}

inline ThVector4 ThVector4::operator += (const ThVector4& vector)
{
	x += vector.x;
	y += vector.y;
	z += vector.z;
	w += vector.w;
	return *this;
}

inline ThVector4& ThVector4::operator -= (const ThVector4& vector)
{
	x -= vector.x;
	y -= vector.y;
	z -= vector.z;
	w -= vector.w;
	return *this;
}

inline ThVector4& ThVector4::operator *= (const ThVector4& vector)
{
	x *= vector.x;
	y *= vector.y;
	z *= vector.z;
	w *= vector.w;
	return *this;
}

inline ThVector4& ThVector4::operator /= (const ThVector4& vector)
{
	x /= vector.x;
	y /= vector.y;
	z /= vector.z;
	w /= vector.w;
	return *this;
}

inline bool ThVector4::operator == (const ThVector4& vector) const
{
	return x == vector.x && y == vector.y && z == vector.z && w == vector.w;
}

inline bool ThVector4::operator != (const ThVector4& vector) const
{
	return x != vector.x || y != vector.y || z != vector.z || w != vector.w;
}

// ThVector4 attributes ---------------------------------------------------------------------------

inline void ThVector4::set(float sx, float sy, float sz, float sw)
{
	x = sx;
	y = sy;
	z = sz;
	w = sw;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMatrix3 constructors -------------------------------------------------------------------------

inline ThMatrix3::ThMatrix3(void)
{
}

inline ThMatrix3::ThMatrix3(const ThVector3& sx, const ThVector3& sy, const ThVector3& sz):
	x(sx),
	y(sy),
	z(sz)
{
}

// ThMatrix3 methods ------------------------------------------------------------------------------

inline void ThMatrix3::invert(void)
{
	// TODO: implement!
}

inline void ThMatrix3::transpose(void)
{
	float temp;

	temp = x.y; x.y = y.x; y.x = temp;
	temp = x.z; x.z = z.x; z.x = temp;
	temp = y.z; y.z = z.y; z.y = temp;
}

inline void ThMatrix3::normalize(void)
{
	// TODO: implement!
}

inline void ThMatrix3::reset(void)
{
	x.set(1.f, 0.f, 0.f);
	y.set(0.f, 1.f, 0.f);
	z.set(0.f, 0.f, 1.f);
}

// ThMatrix3 operators ----------------------------------------------------------------------------

inline ThMatrix3::operator float* (void)
{
	return &(x.x);
}

inline ThMatrix3::operator const float* (void) const
{
	return &(x.x);
}

inline ThMatrix3 ThMatrix3::operator + (const ThMatrix3& matrix) const
{
	// TODO: implement!
}

inline ThMatrix3 ThMatrix3::operator - (const ThMatrix3& matrix) const
{
	// TODO: implement!
}

inline ThMatrix3 ThMatrix3::operator * (const ThMatrix3& matrix) const
{
	// TODO: implement!
}

inline ThMatrix3& ThMatrix3::operator += (const ThMatrix3& matrix)
{
	// TODO: implement!
}

inline ThMatrix3& ThMatrix3::operator -= (const ThMatrix3& matrix)
{
	// TODO: implement!
}

inline ThMatrix3& ThMatrix3::operator *= (const ThMatrix3& matrix)
{
	// TODO: implement!
}

// ThMatrix3 vector methods -----------------------------------------------------------------------

inline void ThMatrix3::rotateVector(ThVector3& vector) const
{
	ThVector3 temp;

	temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x;
	temp.y = vector.y * x.y + vector.y * y.y + vector.z * z.y;
	temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z;

	vector = temp;
}

// ThMatrix3 attributes ---------------------------------------------------------------------------

inline void ThMatrix3::setAngles(const ThVector3& angles)
{
	ThVector3 sin, cos;

	sin.set(sinf(angles.x), sinf(angles.y), sinf(angles.z));
	cos.set(cosf(angles.x), cosf(angles.y), cosf(angles.z));

	x.x = cos.y * cos.z;
	x.y = sin.x * sin.y * cos.z - cos.x * sin.z;
	x.z = cos.x * sin.y * cos.z + sin.x * sin.z;

	y.x = cos.y * sin.z;
	y.y = cos.x * cos.z + sin.x * sin.y * sin.z;
	y.z = cos.x * sin.y * sin.z - sin.x * cos.z;

	z.x = -sin.y;
	z.y = sin.x * cos.y;
	z.z = cos.x * cos.y;
}

inline void ThMatrix3::setVector(const ThVector3& vector, float angle)
{
	const float sinAngle = sinf(angle);
	const float cosAngle = cosf(angle);

  x.x = vector.x * vector.x * (1.f - cosAngle) + cosAngle;
  x.y = vector.x * vector.y * (1.f - cosAngle) - vector.z * sinAngle;
  x.z = vector.x * vector.z * (1.f - cosAngle) + vector.y * sinAngle;
  y.x = vector.x * vector.y * (1.f - cosAngle) + vector.z * sinAngle;
  y.y = vector.y * vector.y * (1.f - cosAngle) + cosAngle;
  y.z = vector.y * vector.z * (1.f - cosAngle) - vector.x * sinAngle;
  z.x = vector.x * vector.z * (1.f - cosAngle) - vector.y * sinAngle;
  z.y = vector.y * vector.z * (1.f - cosAngle) + vector.x * sinAngle;
  z.z = vector.z * vector.z * (1.f - cosAngle) + cosAngle;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMatrix4 constructors -------------------------------------------------------------------------

inline ThMatrix4::ThMatrix4(void)
{
}

inline ThMatrix4::ThMatrix4(const ThVector4& sx, const ThVector4& sy, const ThVector4& sz, const ThVector4& sw):
	x(sx),
	y(sy),
	z(sz),
	w(sw)
{
}

// ThMatrix4 methods ------------------------------------------------------------------------------

inline void ThMatrix4::invert(void)
{
	ThVector3 position;

	position.x = w.x;
	position.y = w.y;
	position.z = w.z;

	float temp;

	temp = x.y; x.y = y.x; y.x = temp;
	temp = x.z; x.z = z.x; z.x = temp;
	temp = y.z; y.z = z.y; z.y = temp;

	// TODO: implement!
}

inline void ThMatrix4::transpose(void)
{
	float temp;

	temp = x.y; x.y = y.x; y.x = temp;
	temp = x.z; x.z = z.x; z.x = temp;
	temp = x.w; x.w = w.x; w.x = temp;
	temp = y.z; y.z = z.y; z.y = temp;
	temp = y.w; y.w = w.y; w.y = temp;
	temp = z.w; z.w = w.z; w.z = temp;
}

inline void ThMatrix4::normalize(void)
{
	// TODO: implement!
}

inline void ThMatrix4::reset(void)
{
	x.set(1.f, 0.f, 0.f, 0.f);
	y.set(0.f, 1.f, 0.f, 0.f);
	z.set(0.f, 0.f, 1.f, 0.f);
	w.set(0.f, 0.f, 0.f, 1.f);
}

// ThMatrix4 vector methods -----------------------------------------------------------------------

inline void ThMatrix4::transformVector(ThVector3& vector) const
{
	ThVector3 temp;

	temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x + w.x;
	temp.y = vector.y * x.y + vector.y * y.y + vector.z * z.y + w.y;
	temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z + w.z;

	vector = temp;
}

inline void ThMatrix4::transformVector(ThVector4& vector) const
{
	ThVector4 temp;

	temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x + vector.w * w.x;
	temp.y = vector.y * x.y + vector.y * y.y + vector.z * z.y + vector.w * w.y;
	temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z + vector.w * w.z;
	temp.w = vector.x * x.w + vector.y * y.w + vector.z * z.w + vector.w * w.w;

	vector = temp;
}

inline void ThMatrix4::translateVector(ThVector3& vector) const
{
	vector.x += w.x;
	vector.y += w.y;
	vector.z += w.z;
}

inline void ThMatrix4::rotateVector(ThVector3& vector) const
{
	ThVector3 temp;

	temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x;
	temp.y = vector.y * x.y + vector.y * y.y + vector.z * z.y;
	temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z;

	vector = temp;
}

// ThMatrix4 operators ----------------------------------------------------------------------------

inline ThMatrix4::operator float* (void)
{
	return &(x.x);
}

inline ThMatrix4::operator const float* (void) const
{
	return &(x.x);
}

inline ThMatrix4 ThMatrix4::operator + (const ThMatrix4& matrix) const
{
	// TODO: implement!
}

inline ThMatrix4 ThMatrix4::operator - (const ThMatrix4& matrix) const
{
	// TODO: implement!
}

inline ThMatrix4 ThMatrix4::operator * (const ThMatrix4& matrix) const
{
	// TODO: implement!
}

inline ThMatrix4& ThMatrix4::operator += (const ThMatrix4& matrix)
{
	// TODO: implement!
}

inline ThMatrix4& ThMatrix4::operator -= (const ThMatrix4& matrix)
{
	// TODO: implement!
}

inline ThMatrix4& ThMatrix4::operator *= (const ThMatrix4& matrix)
{
	// TODO: implement!
}

// ThMatrix4 attributes ---------------------------------------------------------------------------

inline void ThMatrix4::setAngles(const ThVector3& angles)
{
	ThVector3 sin, cos;

	sin.set(sinf(angles.x), sinf(angles.y), sinf(angles.z));
	cos.set(cosf(angles.x), cosf(angles.y), cosf(angles.z));

	x.x = cos.y * cos.z;
	x.y = sin.x * sin.y * cos.z - cos.x * sin.z;
	x.z = cos.x * sin.y * cos.z + sin.x * sin.z;

	y.x = cos.y * sin.z;
	y.y = cos.x * cos.z + sin.x * sin.y * sin.z;
	y.z = cos.x * sin.y * sin.z - sin.x * cos.z;

	z.x = -sin.y;
	z.y = sin.x * cos.y;
	z.z = cos.x * cos.y;
}

inline void ThMatrix4::setVector(const ThVector3& vector, float tilt)
{
	ThVector3 angles;

	angles.x = (float) atan2f(vector.y, sqrtf(vector.x * vector.x + vector.z * vector.z));
	angles.y = (float) atan2f(vector.x, vector.z);
	angles.z = tilt;

	setAngles(angles);
}

inline void ThMatrix4::setScaling(const ThVector3& vector)
{
	x.x = vector.x;
	y.y = vector.y;
	z.z = vector.z;
}

inline void ThMatrix4::setPosition(const ThVector3& vector)
{
	w.x = vector.x;
	w.y = vector.y;
	w.z = vector.z;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThArea constructors ----------------------------------------------------------------------------

inline ThArea::ThArea(void)
{
}

inline ThArea::ThArea(float x, float y, float width, float height):
	m_pos(x, y),
	m_size(width, height)
{
}

// ThArea methods ---------------------------------------------------------------------------------

inline bool ThArea::contains(const ThArea& area) const
{
	if (!contains(area.m_pos))
		return false;

	ThVector2 offset = area.m_pos - m_pos;

	if (offset.x + area.m_size.x > m_size.x || offset.y + area.m_size.y > m_size.y)
		return false;

	return true;
}

inline bool ThArea::contains(const ThVector2& vector) const
{
	if (vector.x < m_pos.x || vector.y < m_pos.y)
		return false;

	if (vector.x > m_pos.x + m_size.x || vector.y > m_pos.y + m_size.y)
		return false;

	return true;
}

inline bool ThArea::intersects(const ThArea& area) const
{
	if (m_pos.x + m_size.x < area.m_pos.x || m_pos.y + m_size.y < area.m_pos.y)
		return false;

	if (area.m_pos.x + area.m_size.x < m_pos.x || area.m_pos.y + area.m_size.y < m_pos.y)
		return false;

	return true;
}

inline void ThArea::reset(void)
{
	m_pos.reset();
	m_size.reset();
}

// ThArea operators -------------------------------------------------------------------------------

inline bool ThArea::operator == (const ThArea& area) const
{
	return m_pos == area.m_pos && m_size == area.m_size;
}

inline bool ThArea::operator != (const ThArea& area) const
{
	return m_pos != area.m_pos || m_size != area.m_size;
}

// ThArea attributes ------------------------------------------------------------------------------

inline void ThArea::set(float x, float y, float width, float height)
{
	m_pos.set(x, y);
	m_size.set(width, height);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSphere constructors --------------------------------------------------------------------------

inline ThSphere::ThSphere(void)
{
}

inline ThSphere::ThSphere(const ThVector3& center, float radius):
	m_center(center),
	m_radius(radius)
{
}

// ThSphere methods -------------------------------------------------------------------------------

inline bool ThSphere::contains(const ThVector3& position) const
{
	ThVector3 vector = position - m_center;

	return vector.dotProduct(vector) < m_radius * m_radius;
}

inline bool ThSphere::contains(const ThSphere& sphere) const
{
	if (m_radius < sphere.m_radius)
		return false;

	ThVector3 vector = sphere.m_center - m_center;

	return vector.dotProduct(vector) < (m_radius - sphere.m_radius) * (m_radius - sphere.m_radius);
}

inline bool ThSphere::intersects(const ThSphere& sphere) const
{
	ThVector3 vector = sphere.m_center - m_center;

	return vector.dotProduct(vector) < (m_radius + sphere.m_radius) * (m_radius + sphere.m_radius);
}

inline void ThSphere::reset(void)
{
	m_center.reset();
	m_radius = 0.f;
}

// ThSphere operators -----------------------------------------------------------------------------

inline bool ThSphere::operator == (const ThSphere& sphere) const
{
	return m_center == sphere.m_center && m_radius == sphere.m_radius;
}

inline bool ThSphere::operator != (const ThSphere& sphere) const
{
	return m_center != sphere.m_center || m_radius != sphere.m_radius;
}

// ThSphere attributes ----------------------------------------------------------------------------

inline void ThSphere::set(const ThVector3& center, float radius)
{
	m_center = center;
	m_radius = radius;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
