///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThString.h>

#include <cstdio>
#include <cctype>
#include <cstring>
#include <cstdarg>
#include <cstdlib>
#include <cmath>

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThString constructors --------------------------------------------------------------------------

ThString::ThString(void)
{
}

ThString::ThString(unsigned int size)
{
	ThString();

	allocate(size);
}

ThString::ThString(const char* string)
{
	ThString();

	operator = (string);
}

ThString::ThString(const ThString& string)
{
	ThString();

	operator = (string);
}

ThString::~ThString(void)
{
	release();
}

// ThString methods -------------------------------------------------------------------------------

/*! Allocates new a buffer of the specified size, discarding any existing string data.
 *	\param size [in] The requested size of the buffer.
 */
void ThString::allocate(unsigned int size)
{
	if (size)
	{
		// allocate only if larger size was requested
		if (m_data.getSize() < size)
			m_data.allocate(size);

		empty();
	}
	else
		release();
}

/*! Resizes an existing buffer to specified size. Any existing character data is preserved, up to the size of the new buffer.
 *	\param size [in] The requested size of the buffer.
 *	\remarks If no buffer is currently allocated, this is identical to ThString::allocate.
 */
void ThString::resize(unsigned int size)
{
	if (m_data && size)
	{
		// resize only if larger size was requested
		if (m_data.getSize() < size)
			m_data.resize(size);
		else
			m_data[size - 1] = '\0';
	}
	else
		allocate(size);
}

/*! Releases any currently allocated buffer.
 */
void ThString::release(void)
{
	m_data.release();
}

void ThString::releaseExtra(void)
{
	if (!m_data)
		return;

	const unsigned int size = strlen(m_data) + 1;
	if (m_data.getSize() > size)
		m_data.resize(size);
}

/*! Copies the specified string into this string, resizing the buffer as needed.
 *	\param string [in] The string to be copied into this string.
 */
void ThString::copy(const char* string)
{
	if (string)
	{
		allocate(strlen(string) + 1);

		strcpy(m_data, string);
	}
	else
		release();
}

const char* ThString::copy(const char* string, unsigned int length)
{
	// copy nothing if given nothing
	if (!string)
	{
		release();
		return NULL;
	}

	allocate(length + 1);

	return copy(m_data, string, length);
}

/*! Copies a line of text into the string, resizing as needed.
 *	\param string [in] The string to be copied from.
 *	\return The beginnning of the next line, or \c NULL of there are no more lines.
 *	\bug This method will skip one 'platform-independent' line. This means that any character combination matching a Unix, DOS, Windows or MacOS line break will be considered a valid match.
 */
const char* ThString::copyLine(const char* string)
{
	// copy nothing if given nothing
	if (*string == '\0')
	{
		release();
		return NULL;
	}

	// calculate length of line
	unsigned int length = 0;
	while (string[length] != '\0' && string[length] != '\r' && string[length] != '\n')
		length++;

	// copy line data
	string = copy(string, length);

	// skip exactly one 'platform independent' line
	if (*string == '\r')
	{
		string++;

		if (*string == '\n')
			string++;
	}
	else
	{
		if (*string == '\n')
			string++;
	}

	// return start of next line
	return string;
}

/*! Copies a token, as defined by the set of separator characters, into the string, resizing as needed.
 *	\param string [in] The string to be copied from.
 *	\param separators [in] A string containing the characters to be considered separator.
 *	\return The beginnning of the next token, or \c NULL of there are no more tokens.
 *	\remarks If no separators are specified, a set of default whitespace characters are used.
 */
const char* ThString::copyToken(const char* string, const char* separators)
{
	// copy nothing if given nothing
	if (*string == '\0')
	{
		release();
		return NULL;
	}

	// default separators, basically any whitespace
	if (!separators || *separators == '\0')
		separators = " \t\n\r";

	// skip leading separator characters
	while (*string != '\0' && strchr(separators, *string))
		string++;

	// calculate length of token
	unsigned int length = 0;
	while (string[length] != '\0' && !strchr(separators, string[length]))
		length++;

	// this means a string full of separators
	if (!length)
	{
		release();
		return NULL;
	}

	// copy token data
	string = copy(string, length);

	// skip tailing separator characters
	while (*string != '\0' && strchr(separators, *string))
		string++;

	// return start of next token
	return string;
}

/*! Appends the specified string to this string.
 *	\param string [in] The string to be appended.
 */
void ThString::append(const char* string)
{
	if (m_data)
	{
		const unsigned int length = strlen(m_data);

		resize(length + strlen(string) + 1);

		// strcpy is faster than strcat
		strcpy(m_data + length, string);
	}
	else
		copy(string);
}

void ThString::format(const char* format, ...)
{
	THSAFEFORMAT(m_data, m_data.getSize(), format);
}

void ThString::empty(void)
{
	empty(m_data);
}

unsigned int ThString::length(void) const
{
	return length(m_data);
}

unsigned int ThString::hash(unsigned int length) const
{
	return hash(m_data, length);
}

unsigned int ThString::hashNoCase(unsigned int length) const
{
	return hashNoCase(m_data, length);
}

bool ThString::equals(const char* string) const
{
	return equals(m_data, string);
}

int ThString::compare(const char* string) const
{
	return compare(m_data, string);
}

bool ThString::equalsNoCase(const char* string) const
{
	return equalsNoCase(m_data, string);
}

int ThString::compareNoCase(const char* string) const
{
	return compareNoCase(m_data, string);
}

int ThString::convertToInt(void) const
{
	return convertToInt(m_data);
}

float ThString::convertToFloat(void) const
{
	return convertToFloat(m_data);
}

char* ThString::find(char c)
{
	return find(m_data, c);
}

const char* ThString::find(char c) const
{
	return find(m_data, c);
}

char* ThString::find(const char* string)
{
	return find(m_data, string);
}

const char* ThString::find(const char* string) const
{
	return find(m_data, string);
}

char* ThString::reverseFind(char c)
{
	return reverseFind(m_data, c);
}

const char* ThString::reverseFind(char c) const
{
	return reverseFind(m_data, c);
}

void ThString::makeUpper(void)
{
	makeUpper(m_data);
}

void ThString::makeLower(void)
{
	makeLower(m_data);
}

void ThString::reverse(void)
{
	reverse(m_data);
}

void ThString::trimLeft(const char* pattern)
{
	trimLeft(m_data, pattern);
}

void ThString::trimRight(const char* pattern)
{
	trimRight(m_data, pattern);
}

// ThString static methods ------------------------------------------------------------------------

/*! Appends the specified string.
 *	\param data [in/out] The string to be appended to.
 *	\paran string [in] The string to append.
 */
void ThString::append(char* data, const char* string)
{
	strcat(data, string);
}

void ThString::formatS(char* data, const char* format, ...)
{
	THFORMAT(data, format);
}

void ThString::formatS(char* data, unsigned int size, const char* format, ...)
{
	THSAFEFORMAT(data, size, format);
}

/*! Empties the specified string.
 *	\param data [in/out] The string to be emtpied.
 */
void ThString::empty(char* data)
{
	*data = '\0';
}

/*! Returns the length of the specified string.
 *	\param data [in] The string.
 *	\return The length of the specified string, not including the null-character.
 */
unsigned int ThString::length(const char* data)
{
	return strlen(data);
}

/*! Calculates a 'unique' hash value from the specified string.
 *	\param data [in] The string to be hashed.
 *	\return The calculated hash value.
 *	\remarks Zero is not a valid hash result.
 *	\remarks This algorithm is an adaption of the ELF format hash.
 */
unsigned int ThString::hash(const char* data, unsigned int length)
{
	unsigned char i = 1;
	unsigned int hash = 0;
	unsigned int temp;

  while (*data != '\0')
  {
		const char c = *data++;

		hash = (hash << 4) + c;

		if (temp = hash & 0xF0000000)
			hash ^= temp >> 24;

		hash &= ~temp;

		if (length && !(--length))
			break;
	}

	// 1 is considered 'the least likely result of hashing'
	return hash ? hash : 1;
}

/*! Calculates a 'unique' hash value from a lower-case version of the specified string.
 *	\param data [in] The string to be hashed.
 *	\return The calculated hash value.
 *	\remarks Zero is not a valid hash result.
 *	\remarks This algorithm is an adaption of the ELF format hash.
 */
unsigned int ThString::hashNoCase(const char* data, unsigned int length)
{
	unsigned char i = 1;
	unsigned int hash = 0;
	unsigned int temp;

  while (*data != '\0')
  {
		const char c = tolower(*data++);

		hash = (hash << 4) + c;

		if (temp = hash & 0xF0000000)
			hash ^= temp >> 24;

		hash &= ~temp;

		if (length && !(--length))
			break;
	}

	// 1 is considered 'the least likely result of hashing'
	return hash ? hash : 1;
}

/*! Copies the specified string into the destination buffer.
 *	\param data [in/out] The destination buffer.
 *	\param string [in] The string to be copied.
 *	\bug No bounds checking is done on the destination buffer. It is the responsibility of the caller to provide valid input parameters.
 */
void ThString::copy(char* data, const char* string)
{
	strcpy(data, string);
}

/*! Copies the specified number of character into the destination buffer.
 *	\param data [in/out] The destination buffer.
 *	\param string [in] The string to be copied from.
 *	\param length [in] The number of characters to be copied.
 *	\return The first character of \c string following the copied section.
 *	\bug No bounds checking is done on either the source or destination buffers. It is the responsibility of the caller to provide valid input parameters.
 */
const char* ThString::copy(char* data, const char* string, unsigned int length)
{
	memcpy(data, string, length);
	data[length] = '\0';

	return string + length;
}

/*! Copies a line of text into the specified buffer.
 *	\param data [in/out] The destination buffer.
 *	\param string [in] The string to be copied from.
 *	\return The beginnning of the next line, or \c NULL of there are no more lines.
 *	\bug No bounds checking is done on the destination buffer. It is the responsibility of the caller to provide valid input parameters.
 *	\bug This method will skip one 'platform-independent' line. This means that any character combination matching a Unix, DOS, Windows or MacOS line break will be considered a valid match.
 */
const char* ThString::copyLine(char* data, const char* string)
{
	// copy nothing if given nothing
	if (*string == '\0')
		return NULL;

	// calculate length of line
	unsigned int length = 0;
	while (string[length] != '\0' && string[length] != '\r' && string[length] != '\n')
		length++;

	// copy line data
	string = copy(data, string, length);

	// skip exactly one 'platorm-independent' line
	if (*string == '\r')
	{
		string++;

		if (*string == '\n')
			string++;
	}
	else
	{
		if (*string == '\n')
			string++;
	}

	// return start of next line
	return string;
}

/*! Copies a token, as defined by the set of separator characters, into the specified buffer.
 *	\param data [in/out] The destination buffer.
 *	\param string [in] The string to be copied from.
 *	\param separators [in] A string containing the characters to be considered separator.
 *	\return The beginnning of the next token, or \c NULL of there are no more tokens.
 *	\remarks If no separators are specified, a set of default whitespace characters are used.
 *	\bug No bounds checking is done on the destination buffer. It is the responsibility of the caller to provide valid input parameters.
 */
const char* ThString::copyToken(char* data, const char* string, const char* separators)
{
	// copy nothing if given nothing
	if (*string == '\0')
		return NULL;

	// default separators, basically any whitespace
	if (!separators || *separators == '\0')
		separators = " \t\n\r";

	// skip leading separator characters
	while (*string != '\0' && strchr(separators, *string))
		string++;

	// calculate length of token
	unsigned int length = 0;
	while (string[length] != '\0' && !strchr(separators, string[length]))
		length++;

	// this means a string full of separators
	if (!length)
		return NULL;

	// copy token data
	string = copy(data, string, length);

	// skip tailing separator characters
	while (*string != '\0' && strchr(separators, *string))
		string++;

	// return start of next token
	return string;
}

/*! Compares the two specified strings.
 *	\param data [in] The first string to be compared.
 *	\param string [in] The second string to be compared.
 *	\return \c true if the strings are equal, or \c false if they are not.
 */
bool ThString::equals(const char* data, const char* string)
{
	return strcmp(data, string) == 0;
}

/*! Compares the two specified strings, in C style.
 *	\param data [in] The first string to be compared.
 *	\param string [in] The second string to be compared.
 *	\return The lexographic relation between the strings.
 */
int ThString::compare(const char* data, const char* string)
{
	return strcmp(data, string);
}

/*! Compares the two specified strings, ignoring case.
 *	\param data [in] The first string to be compared.
 *	\param string [in] The second string to be compared.
 *	\return \c true if the strings are equal, or \c false if they are not.
 */
bool ThString::equalsNoCase(const char* data, const char* string)
{
	return stricmp(data, string) == 0;
}

/*! Compares the two specified strings, in C style, ignoring case.
 *	\param data [in] The first string to be compared.
 *	\param string [in] The second string to be compared.
 *	\return The lexographic relation between the strings.
 */
int ThString::compareNoCase(const char* data, const char* string)
{
	return stricmp(data, string);
}

/*! Converts the specied string to an integer value.
 *	\param data [in] The string to be converted.
 *	\return The converted integer value.
 */
int ThString::convertToInt(const char* data)
{
	return (int) strtol(data, NULL, 0);
}

/*! Converts the specied string to a floating-point value.
 *	\param data [in] The string to be converted.
 *	\return The converted floating-point value.
 */
float ThString::convertToFloat(const char* data)
{
	return (float) atof(data);
}

/*! Finds the first occurrence of the specified character in the specified string.
 *	\param data [in/out] The string to be searched.
 *	\param c [in] The character to be located.
 *	\return The address of the found character, or \c NULL if it is not found.
 */
char* ThString::find(char* data, char c)
{
	return strchr(data, c);
}

/*! Finds the first occurrence of the specified character in the specified string.
 *	\param data [in/out] The string to be searched.
 *	\param c [in] The character to be located.
 *	\return The address of the found character, or \c NULL if it is not found.
 */
const char* ThString::find(const char* data, char c)
{
	return strchr(data, c);
}

char* ThString::find(char* data, const char* string)
{
	return strstr(data, string);
}

const char* ThString::find(const char* data, const char* string)
{
	return strstr(data, string);
}

/*! Finds the last occurrence of the specified character in the specified string.
 *	\param data [in/out] The string to be searched.
 *	\param c [in] The character to be located.
 *	\return The address of the found character, or \c NULL if it is not found.
 */
char* ThString::reverseFind(char* data, char c)
{
	return strrchr(data, c);
}

/*! Finds the last occurrence of the specified character in the specified string.
 *	\param data [in/out] The string to be searched.
 *	\param c [in] The character to be located.
 *	\return The address of the found character, or \c NULL if it is not found.
 */
const char* ThString::reverseFind(const char* data, char c)
{
	return strrchr(data, c);
}

/*! Makes the specified string upper-case.
 *	\param data [in/out] The string to be converted.
 */
void ThString::makeUpper(char* data)
{
	strupr(data);
}

/*! Makes the specified string lower-case.
 *	\param data [in/out] The string to be converted.
 */
void ThString::makeLower(char* data)
{
	strlwr(data);
}

/*! Reverses the specified string.
 *	\param data [in/out] The string to be reversed.
 */
void ThString::reverse(char* data)
{
	strrev(data);
}

void ThString::trimLeft(char* data, const char* pattern)
{
	unsigned int count = 0;

	while (*data != '\0' && strchr(pattern, data[count]))
		count++;

	while (data[0] = data[count])
		data++;
}

void ThString::trimRight(char* data, const char* pattern)
{
	unsigned int length = 0;

	while (*data != '\0')
	{
		if (strchr(pattern, *data))
			length++;
		else
			length = 0;

		data++;
	}

	*(data - length) = '\0';
}

// ThString operators -----------------------------------------------------------------------------

ThString::operator char* (void)
{
	return m_data;
}

ThString::operator const char* (void) const
{
	return m_data;
}

ThString& ThString::operator = (const char* string)
{
	copy(string);

	return *this;
}

ThString& ThString::operator = (const ThString& string)
{
	copy(string);

	return *this;
}

// ThString attributes ----------------------------------------------------------------------------

bool ThString::isEmpty(void) const
{
	return m_data[0] != '\0';
}

char* ThString::getData(void)
{
	return m_data;
}

const char* ThString::getData(void) const
{
	return m_data;
}

unsigned int ThString::getSize(void) const
{
	return m_data.getCount();
}

// ThString static data ---------------------------------------------------------------------------

const unsigned int ThString::m_primes[16] = { 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59 };

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThStringItem constructors ----------------------------------------------------------------------

ThStringItem::ThStringItem(void)
{
}

ThStringItem::ThStringItem(unsigned int size):
	ThString(size)
{
}

ThStringItem::ThStringItem(const char* string):
	ThString(string)
{
}

// ThStringItem operators -------------------------------------------------------------------------

ThStringItem& ThStringItem::operator = (const char* string)
{
	ThString::operator = (string);

	return *this;
}

ThStringItem& ThStringItem::operator = (const ThString& string)
{
	ThString::operator = (string);

	return *this;
}

ThStringItem& ThStringItem::operator = (const ThStringItem& string)
{
	ThString::operator = (string);

	return *this;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThStringList methods ---------------------------------------------------------------------------

ThStringItem* ThStringList::find(const char* pattern)
{
	for (ThIterator<ThStringItem> item(getFirst());  item;  item.next())
	{
		if (item->find(pattern))
			return item;
	}

	return NULL;
}

const ThStringItem* ThStringList::find(const char* pattern) const
{
	for (ThConstIterator<ThStringItem> item(getFirst());  item;  item.next())
	{
		if (item->find(pattern))
			return item;
	}

	return NULL;
}

ThStringItem* ThStringList::findExact(const char* string)
{
	for (ThIterator<ThStringItem> item(getFirst());  item;  item.next())
	{
		if (item->equals(string))
			return item;
	}

	return NULL;
}

const ThStringItem* ThStringList::findExact(const char* string) const
{
	for (ThConstIterator<ThStringItem> item(getFirst());  item;  item.next())
	{
		if (item->equals(string))
			return item;
	}

	return NULL;
}

ThStringItem* ThStringList::findNoCase(const char* string)
{
	for (ThIterator<ThStringItem> item(getFirst());  item;  item.next())
	{
		if (item->equalsNoCase(string))
			return item;
	}

	return NULL;
}

const ThStringItem* ThStringList::findNoCase(const char* string) const
{
	for (ThConstIterator<ThStringItem> item(getFirst());  item;  item.next())
	{
		if (item->equalsNoCase(string))
			return item;
	}

	return NULL;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
