///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <ThCore.h>
#include <ThMemory.h>
#include <ThString.h>
#include <ThError.h>
#include <ThStream.h>
#include <ThChunk.h>
#include <ThStorage.h>

#include <windows.h>

#include "ThStorage.h"

///////////////////////////////////////////////////////////////////////////////////////////////////

ThSingleton<IThStorage> Storage;

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThStorage constructors ------------------------------------------------------------------------

IThStorage::~IThStorage(void)
{
}

// IThStorage static methods ----------------------------------------------------------------------

bool IThStorage::create(const char* fileName)
{
	if (Storage)
		return true;

	if (fileName)
	{
		// attempt to open storage file

		ThPtr<ThChunkStorage> storage = new ThChunkStorage();

		if (storage->open(fileName))
		{
			Storage.attach(storage.detach());
			return true;
		}
	}
	else
	{
		// attempt to create root storage

		ThPtr<ThRootStorage> storage = new ThRootStorage();

		if (storage->open())
		{
			Storage.attach(storage.detach());
			return true;
		}
	}
	
	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThFileStream constructors ----------------------------------------------------------------------

ThFileStream::ThFileStream(HANDLE file, const char* name):
	m_file(file),
	m_name(name)
{
}

ThFileStream::~ThFileStream(void)
{
	CloseHandle(m_file);
}

// ThFileStream interface methods -----------------------------------------------------------------

unsigned int ThFileStream::read(void* data, unsigned int size)
{
	DWORD bytesRead;

	if (!ReadFile(m_file, data, size, &bytesRead, NULL))
	{
		THDEBUG(Error->write("Storage", "Unable to read %u bytes from file '%s'.", size, m_name.getData()));
		return 0;
	}

	return bytesRead;
}

unsigned int ThFileStream::write(const void* data, unsigned int size)
{
	DWORD bytesWritten;

	if (!WriteFile(m_file, data, size, &bytesWritten, NULL))
	{
		THDEBUG(Error->write("Storage", "Unable to write %u bytes to file '%s'.", size, m_name.getData()));
		return 0;
	}

	return bytesWritten;
}

unsigned int ThFileStream::skip(unsigned int size)
{
	const unsigned int streamSize = getSize();
	const unsigned int position = getPosition();

	if (position + size <= streamSize)
		size = streamSize - position - 1;

	if (!seek(position + size))
	{
		THDEBUG(Error->write("Storage", "Unable to skip %u bytes in file '%s'.", size, m_name.getData()));
		return 0;
	}

	return size;
}

bool ThFileStream::seek(unsigned int position)
{
	const DWORD result = SetFilePointer(m_file, position, NULL, FILE_BEGIN);

	// NOTE: this is actually missing in older Platform SDKs.
#ifdef INVALID_SET_FILE_POINTER
	if (result == INVALID_SET_FILE_POINTER)
	{
		THDEBUG(Error->write("Storage", "Unable to set position to %u in file '%s'.", position, m_name.getData()));
		return false;
	}
#endif

	return true;
}

bool ThFileStream::truncate(void)
{
	if (!SetEndOfFile(m_file))
	{
		THDEBUG(Error->write("Storage", "Unable to truncate file '%s'.", m_name.getData()));
		return false;
	}

	return true;
}

// ThFileStream interface attributes --------------------------------------------------------------

bool ThFileStream::isEOF(void) const
{
	if (getPosition() < getSize())
		return false;

	return true;
}

unsigned int ThFileStream::getSize(void) const
{
	const DWORD size = GetFileSize(m_file, NULL);
	if (size == -1)
	{
		THDEBUG(Error->write("Storage", "Unable to retrieve size for file '%s'.", m_name.getData()));
		return 0;
	}

	return size;
}

unsigned int ThFileStream::getPosition(void) const
{
	const DWORD position = SetFilePointer(m_file, 0, NULL, FILE_CURRENT);

	// NOTE: this is actually missing in older Platform SDKs.
#ifdef INVALID_SET_FILE_POINTER
	if (position == INVALID_SET_FILE_POINTER)
	{
		THDEBUG(Error->write("Storage", "Unable to retrieve position for file '%s'.", m_name.getData()));
		return 0;
	}
#endif

	return position;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThRootStorage constructors ---------------------------------------------------------------------

ThRootStorage::ThRootStorage(void)
{
}

ThRootStorage::~ThRootStorage(void)
{
	close();
}

// ThRootStorage methods --------------------------------------------------------------------------

bool ThRootStorage::open(void)
{
	close();

	return true;
}

void ThRootStorage::close(void)
{
	m_files.release();
}

// ThRootStorage storage methods ------------------------------------------------------------------

IThStorage* ThRootStorage::createStorage(const char* name)
{
	ThPtr<IThStream> file = createFile(name);
	if (!file)
		return NULL;

	ThPtr<ThChunkStorage> storage = new ThChunkStorage();

	if (!storage->open(file.detach()))
		return NULL;

	return storage.detach();
}

IThStorage* ThRootStorage::openStorage(const char* name)
{
	ThPtr<IThStream> file = openFile(name);
	if (!file)
		return NULL;

	ThPtr<ThChunkStorage> storage = new ThChunkStorage();

	if (!storage->open(file.detach()))
		return NULL;

	return storage.detach();
}

// ThRootStorage folder methods -------------------------------------------------------------------

bool ThRootStorage::createFolder(const char* name)
{
	if (!CreateDirectory(name, NULL))
	{
		THDEBUG(Error->write("Storage", "Unable to create folder '%s'.", name));
		return false;
	}

	return true;
}

bool ThRootStorage::destroyFolder(const char* name)
{
	if (!RemoveDirectory(name))
	{
		THDEBUG(Error->write("Storage", "Unable to destroy folder '%s'.", name));
		return false;
	}

	return true;
}

bool ThRootStorage::listFolders(const char* folderName, ThStringList& folders)
{
	HANDLE handle;
	WIN32_FIND_DATA wfd;

	char pattern[MAX_PATH];

	ThString::formatS(pattern, sizeof(pattern), "%s/*.*", folderName);

	handle = FindFirstFile(pattern, &wfd);
	if (handle == INVALID_HANDLE_VALUE)
	{
		THDEBUG(Error->write("Storage", "Unable to find any folders in folder '%s'.", folderName));
		return false;
	}

	do
	{
		if (wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			folders.attachFirst(new ThStringItem(wfd.cFileName));
	}
	while (FindNextFile(handle, &wfd));

	FindClose(handle);
	handle = NULL;

	return true;
}

bool ThRootStorage::listFiles(const char* folderName, ThStringList& files)
{
	HANDLE handle;
	WIN32_FIND_DATA wfd;

	char pattern[MAX_PATH];

	ThString::formatS(pattern, sizeof(pattern), "%s/*.*", folderName);

	handle = FindFirstFile(pattern, &wfd);
	if (handle == INVALID_HANDLE_VALUE)
	{
		THDEBUG(Error->write("Storage", "Unable to find any files in folder '%s'.", folderName));
		return false;
	}

	do
	{
		if (wfd.dwFileAttributes ^ FILE_ATTRIBUTE_DIRECTORY)
			files.attachFirst(new ThStringItem(wfd.cFileName));
	}
	while (FindNextFile(handle, &wfd));

	FindClose(handle);
	handle = NULL;

	return true;
}

// ThRootStorage file methods ---------------------------------------------------------------------

IThStream* ThRootStorage::createFile(const char* name)
{
	HANDLE file = CreateFile(name, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ, NULL, CREATE_ALWAYS, FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (file == INVALID_HANDLE_VALUE)
	{
		THDEBUG(Error->write("Storage", "Unable to create file '%s'.", name));
		return NULL;
	}

	return new ThFileStream(file, name);
}

IThStream* ThRootStorage::openFile(const char* name)
{
	HANDLE file = CreateFile(name, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (file == INVALID_HANDLE_VALUE)
	{
		THDEBUG(Error->write("Storage", "Unable to open file '%s'.", name));
		return NULL;
	}

	return new ThFileStream(file, name);
}

bool ThRootStorage::requestFile(ThString& name, bool exist)
{
	char fileName[MAX_PATH];

	ThString::empty(fileName);

	OPENFILENAME ofn;

	ZeroMemory(&ofn, sizeof(ofn));
	ofn.lStructSize = sizeof(ofn);
	ofn.hwndOwner   = GetActiveWindow();
	ofn.lpstrFilter = "All files\0*.*\0";
	ofn.lpstrFile   = fileName;
	ofn.nMaxFile    = sizeof(fileName);
	ofn.Flags       = OFN_ENABLESIZING | OFN_EXPLORER | OFN_HIDEREADONLY | OFN_NOREADONLYRETURN | OFN_NOCHANGEDIR;

	if (exist)
		ofn.Flags |= OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST;
	else
		ofn.Flags |= OFN_OVERWRITEPROMPT;

	if (exist)
	{
		if (!GetOpenFileName(&ofn))
			return false;
	}
	else
	{
		if (!GetSaveFileName(&ofn))
			return false;
	}

	name = fileName;
	return true;
}

bool ThRootStorage::destroyFile(const char* name)
{
	if (!DeleteFile(name))
	{
		THDEBUG(Error->write("Storage", "Unable to destroy file '%s'.", name));
		return false;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThChunkStorage constructors --------------------------------------------------------------------

ThChunkStorage::ThChunkStorage(void)
{
}

ThChunkStorage::~ThChunkStorage(void)
{
	close();
}

// ThChunkStorage methods -------------------------------------------------------------------------

bool ThChunkStorage::open(const char* name)
{
	HANDLE file = CreateFile(name, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (file == INVALID_HANDLE_VALUE)
	{
		THDEBUG(Error->write("Storage", "Unable to open storage file '%s'.", name));
		return NULL;
	}

	IThStream* stream = new ThFileStream(file, name);

	return open(stream);
}

bool ThChunkStorage::open(IThStream* stream)
{
	close();

	m_stream = stream;

	if (m_stream->getSize())
	{
		// fail on load failure
		if (!m_root.load(m_stream))
		{
			close();
			return false;
		}

		// fail on invalid hierarchy
		if (m_root.getType() != THCHUNK_STORAGE)
		{
			close();
			return false;
		}
	}
	else
		m_root.setType(THCHUNK_STORAGE);

	return true;
}

void ThChunkStorage::close(void)
{
	if (m_stream)
	{
		m_stream->seek(0);
		m_stream->truncate();

		m_root.save(m_stream);
		m_root.release();

		m_stream.release();
	}
}

// ThChunkStorage storage methods -----------------------------------------------------------------

IThStorage* ThChunkStorage::createStorage(const char* name)
{
	// NOTE: this could be implemented, to allow nested storages.

	return NULL;
}

IThStorage* ThChunkStorage::openStorage(const char* name)
{
	// NOTE: this could be implemented, to allow nested storages.

	return NULL;
}

// ThChunkStorage folder methods ------------------------------------------------------------------

bool ThChunkStorage::createFolder(const char* name)
{
	ThChunk* folder = findChild(name, false);

	if (!folder || folder->getType() != THCHUNK_STGFOLDER)
		return false;

	ThChunk* child = findChild(folder, name);

	if (child)
	{
		if (child->getType() != THCHUNK_STGFOLDER)
			return false;
	}
	else
	{
		child = new ThChunk(folder);
		child->setType(THCHUNK_STGFOLDER);
		child->setName(name);
	}

	return true;
}

bool ThChunkStorage::destroyFolder(const char* name)
{
	ThChunk* folder = findChild(name, true);

	if (!folder || folder->getType() != THCHUNK_STGFOLDER)
		return false;

	// cannot remove non-empty folders ( for root storage compability )
	if (folder->getFirstChild())
		return false;

	delete folder;
	folder = NULL;

	return true;
}

bool ThChunkStorage::listFolders(const char* folderName, ThStringList& folders)
{
	ThChunk* folder = findChild(folderName, true);

	if (!folder || folder->getType() != THCHUNK_STGFOLDER)
		return false;

	for (ThConstIterator<ThChunk> child(folder->getFirstChild());  child;  child.next())
	{
		if (child->getType() == THCHUNK_STGFOLDER)
		{
			ThStringItem* item = new ThStringItem(child->getName());

			folders.attachFirst(item);
		}
	}

	return true;
}

bool ThChunkStorage::listFiles(const char* folderName, ThStringList& files)
{
	ThChunk* folder = findChild(folderName, true);

	if (!folder || folder->getType() != THCHUNK_STGFOLDER)
		return false;

	for (ThConstIterator<ThChunk> child(folder->getFirstChild());  child;  child.next())
	{
		if (child->getType() == THCHUNK_STGFILE)
		{
			ThStringItem* item = new ThStringItem(child->getName());

			files.attachFirst(item);
		}
	}

	return true;
}

// ThChunkStorage file methods --------------------------------------------------------------------

IThStream* ThChunkStorage::createFile(const char* name)
{
	ThChunk* folder = findChild(name, false);

	if (!folder || folder->getType() != THCHUNK_STGFOLDER)
		return NULL;

	ThChunk* file = findChild(folder, name);

	if (file)
	{
		if (file->getType() != THCHUNK_STGFILE)
			return NULL;

		// delete existing file
		file->release();
	}
	else
		file = new ThChunk(folder);

	// setup file data
	file->setType(THCHUNK_STGFILE);
	file->setName(name);

	return CreateAggregateStream(file->getStream());
}

IThStream* ThChunkStorage::openFile(const char* name)
{
	ThChunk* file = findChild(name, true);

	if (!file || file->getType() != THCHUNK_STGFILE)
		return NULL;

	return CreateAggregateStream(file->getStream());
}

bool ThChunkStorage::requestFile(ThString& name, bool exist)
{
	return false;
}

bool ThChunkStorage::destroyFile(const char* name)
{
	ThChunk* file = findChild(name, true);

	if (!file || file->getType() != THCHUNK_STGFILE)
		return false;

	delete file;
	file = NULL;

	return true;
}

// ThChunkStorage methods -------------------------------------------------------------------------

ThChunk* ThChunkStorage::findChild(ThChunk* parent, const char* name)
{
	const unsigned int hash = ThString::hashNoCase(name);

	for (ThIterator<ThChunk> child(parent->getFirstChild());  child;  child.next())
	{
		if (child->getHash() == hash)
			return child;
	}

	return NULL;
}

ThChunk* ThChunkStorage::findChild(const char*& name, bool complete)
{
	ThString folderName;
	ThChunk* chunk = &m_root;

	// traverse folders
	while (ThString::find(name, '\\'))
	{
		name = folderName.copyToken(name, "\\");

		if (folderName.equals(".") || folderName.equals(".."))
			return NULL;

		chunk = findChild(chunk, folderName);
		if (!chunk || chunk->getType() != THCHUNK_STGFOLDER)
			return NULL;
	}

	if (complete)
	{
		// find child
		chunk = findChild(chunk, name);
		if (!chunk)
			return NULL;
	}

	return chunk;
}

bool ThChunkStorage::validatePath(const char* path)
{
	if (ThString::find(path, ':'))
		return false;

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
