
/*
 *
 *      L O A D I N G
 *
 */

#include "ray.h"

static const char *pattern_header =
"viewport\n\t%d\t%d\t%d\t%f\n"
"camera\n\t%f\t%f\t%f\n\t%f\t%f\t%f\n\t%f\n"
"light\n\t%f\t%f\t%f\n";

static const char *pattern_chunkid =
"%s\n";

static unsigned char cmap[768];
static unsigned char pmap[65536];

/*
 *	T E X T U R E   L O A D E R
 */

RGB *load_texture( const char *filename )
{
	FILE *fd;

	struct {
		char id[4];     /* "RIX3" */
		short width, height, a_mysterious_one;
	} hdr;

	RGB *rgb;
	int i;

	if ( !(fd = fopen( filename, "rb" )) ) {
		perror( "fopen: " );
		return NULL;
	}

	fread( &hdr, 10, 1, fd );

	if ( (hdr.width != 256) || (hdr.height != 256) ) {
		fclose( fd );
		return NULL;
	}

	fread( cmap, 768, 1, fd );
	fread( pmap, 65536, 1, fd );

	if ( !(rgb = (RGB *)malloc( 256*256*sizeof(RGB) )) ) {
		fclose( fd );
		return NULL;
	}

	for ( i = 0; i < 65536; i++ ) {
		rgb[i].r = (float)(cmap[(int)(pmap[i])*3])/64.0f;
		rgb[i].g = (float)(cmap[(int)(pmap[i])*3+1])/64.0f;
		rgb[i].b = (float)(cmap[(int)(pmap[i])*3+2])/64.0f;
	}

	fclose( fd );

	return rgb;
}

/*
 *	P L A N E   L O A D E R
 */

static MODEL_PLANE *load_plane( FILE **fd )
{
	MODEL_PLANE *obj;

	if ( !(obj = (MODEL_PLANE *)malloc( sizeof(MODEL_PLANE) )) )
		return NULL;

	if ( fscanf( *fd, "\t%f\t%f\t%f\t%f\n\t%f\t%f\t%f\n\t%f\n",
	  	     &obj->A, &obj->B, &obj->C, &obj->D,
		     &obj->color.r, &obj->color.g, &obj->color.b,
		     &obj->reflect ) != 8 ) {
		free( obj );
		return NULL; 
	}

	return obj;
}

/*
 *	S P H E R E   L O A D E R
 */

static MODEL_SPHERE *load_sphere( FILE **fd )
{
	MODEL_SPHERE *obj;

	if ( !(obj = (MODEL_SPHERE *)malloc( sizeof(MODEL_SPHERE) )) )
		return NULL;

	if ( fscanf( *fd, "\t%f\t%f\t%f\t%f\n\t%f\t%f\t%f\n\t%f\t",
	  	     &obj->loc.x, &obj->loc.y, &obj->loc.z, &obj->rad,
		     &obj->color.r, &obj->color.g, &obj->color.b,
		     &obj->reflect ) != 8 ) {
		free( obj );
		return NULL; 
	}

	return obj;
}

/*
 *	E L L I P S O I D   L O A D E R
 */

static MODEL_ELLIPSOID *load_ellipsoid( FILE **fd )
{
	MODEL_ELLIPSOID *obj;

	if ( !(obj = (MODEL_ELLIPSOID *)malloc( sizeof(MODEL_ELLIPSOID) )) )
		return NULL;

	if ( fscanf( *fd, "\t%f\t%f\t%f\n\t%f\t%f\t%f\n\t%f\n\t%f\t%f\t%f\n\t%f\n",
	  	     &obj->loc.x, &obj->loc.y, &obj->loc.z, 
	  	     &obj->exc.x, &obj->exc.y, &obj->exc.z, 
		     &obj->rad,
		     &obj->color.r, &obj->color.g, &obj->color.b,
		     &obj->reflect ) != 11 ) {
		free( obj );
		return NULL; 
	}

	return obj;
}

/*
 *	T E X T U R E D   P L A N E   L O A D E R
 */

static MODEL_PLANE_TEXT *load_plane_text( FILE **fd )
{
	MODEL_PLANE_TEXT *obj;
	char filename[256];

	if ( !(obj = (MODEL_PLANE_TEXT *)malloc( sizeof(MODEL_PLANE_TEXT) )) )
		return NULL;

	if ( fscanf( *fd, "\t%f\t%f\t%f\t%f\n\t%s\n\t%f\t",
	  	     &obj->A, &obj->B, &obj->C, &obj->D, filename, &obj->reflect ) != 6 ) {
		free( obj );
		return NULL; 
	}

	if ( !(obj->texmap = load_texture( filename )) ) {
		free( obj );
		return NULL;
	}

	return obj;
}

/*
 *	T E X T U R E D   S P H E R E   L O A D E R
 */

static MODEL_SPHERE_TEXT *load_sphere_text( FILE **fd )
{
	MODEL_SPHERE_TEXT *obj;
	char filename[256];

	if ( !(obj = (MODEL_SPHERE_TEXT *)malloc( sizeof(MODEL_SPHERE_TEXT) )) )
		return NULL;

	if ( fscanf( *fd, "\t%f\t%f\t%f\t%f\n\t%s\n\t%f\n",
	  	     &obj->loc.x, &obj->loc.y, &obj->loc.z, &obj->rad, filename, &obj->reflect ) != 6 ) {
		free( obj );
		return NULL; 
	}

	if ( !(obj->texmap = load_texture( filename )) ) {
		free( obj );
		return NULL;
	}

	return obj;
}

/*
 *	T E X T U R E D   E L L I P S O I D   L O A D E R
 */

static MODEL_ELLIPSOID_TEXT *load_ellipsoid_text( FILE **fd )
{
	MODEL_ELLIPSOID_TEXT *obj;
	char filename[256];

	if ( !(obj = (MODEL_ELLIPSOID_TEXT *)malloc( sizeof(MODEL_ELLIPSOID_TEXT) )) )
		return NULL;

	if ( fscanf( *fd, "\t%f\t%f\t%f\n\t%f\t%f\t%f\n\t%f\n\t%s\n\t%f\n",
	  	     &obj->loc.x, &obj->loc.y, &obj->loc.z, 
	  	     &obj->exc.x, &obj->exc.y, &obj->exc.z, 
		     &obj->rad,	filename, &obj->reflect ) != 9 ) {
		free( obj );
		return NULL; 
	}

	if ( !(obj->texmap = load_texture( filename )) ) {
		free( obj );
		return NULL;
	}

	return obj;
}

/*
 *	S C E N E   L O A D E R
 */

int load_scene( const char *filename, VIEWPORT *vp, SCENE *scn )
{
	FILE *fd;
	static char id[64];
	int n;

	if ( !(fd = fopen( filename, "rt" )) )
		return 0;
       
	fscanf( fd, pattern_header, 
		&vp->width, &vp->height, &vp->depth, &vp->viewer_dist,
		&scn->cam.rvp.x, &scn->cam.rvp.y, &scn->cam.rvp.z,
		&scn->cam.lap.x, &scn->cam.lap.y, &scn->cam.lap.z,
		&scn->cam.roll,
		&scn->light.x, &scn->light.y, &scn->light.z );

	/* no one will be as crazy as to use more than 10 objects for the moment :) */
	scn->objects = (MODEL *)malloc( sizeof(MODEL)*10 );
	n = 0;

	while ( !feof( fd ) ) {
		fscanf( fd, pattern_chunkid, id );

		if ( !strcmp( id, "plane" ) ) {
			scn->objects[n].id = MDL_PLANE;
			if ( !(scn->objects[n].model_info = load_plane( &fd )) )
				fprintf( stderr, "RAYTR: error loading plane\n" );
			else
				n++;
		} else if ( !strcmp( id, "sphere" ) ) {
			scn->objects[n].id = MDL_SPHERE;
			if ( !(scn->objects[n].model_info = load_sphere( &fd )) )
				fprintf( stderr, "RAYTR: error loading sphere\n" );
			else
				n++;
		} else if ( !strcmp( id, "ellipsoid" ) ) {
			scn->objects[n].id = MDL_ELLIPSOID;
			if ( !(scn->objects[n].model_info = load_ellipsoid( &fd )) )
				fprintf( stderr, "RAYTR: error loading ellipsoid\n" );
			else
				n++;
		} else if ( !strcmp( id, "plane_text" ) ) {
			scn->objects[n].id = MDL_PLANE_TEXT;
			if ( !(scn->objects[n].model_info = load_plane_text( &fd )) )
				fprintf( stderr, "RAYTR: error loading texture mapped plane\n" );
			else
				n++;
		} else if ( !strcmp( id, "sphere_text" ) ) {
			scn->objects[n].id = MDL_SPHERE_TEXT;
			if ( !(scn->objects[n].model_info = load_sphere_text( &fd )) )
				fprintf( stderr, "RAYTR: error loading texture mapped sphere\n" );
			else
				n++;
		} else if ( !strcmp( id, "ellipsoid_text" ) ) {
			scn->objects[n].id = MDL_ELLIPSOID_TEXT;
			if ( !(scn->objects[n].model_info = load_ellipsoid_text( &fd )) )
				fprintf( stderr, "RAYTR: error loading texture mapped ellipsoid\n" );
			else
				n++;
		} else
			fprintf( stderr, "RAYTR: warning - unknown object \"%s\"\n", id );
	}

	scn->num_objects = n;

	fprintf( stderr, "RAYTR: %d object(s) loaded.\n", n );

	fclose( fd );

	return 1;
}

