﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using OpenTK;
using OpenTK.Graphics.OpenGL;
using MouseEventArgs = OpenTK.Input.MouseEventArgs;

namespace Pixels3d
{
    /// <summary>
    /// Class for handling the GL surface of the GL control, takes care of sizing and setup etc
    /// </summary>
    public class GlSurface
    {
        public static GlSurface Instance = new GlSurface();
        public float AspectRatio = 1.0f;
        public float angle = 0.0f;
        public float angleX = 0.0f;
        public float angleY = 0.0f;
        public bool dragging = false;
        public float zoom = 50;
        public PointF dragPos;

        public OpenTK.GLControl Control;

        public GlSurface()
        {
            GL.MatrixMode(MatrixMode.Projection);
            GL.ClearDepth(1); // clear whole depth buffer?
            GL.Enable(EnableCap.DepthTest);
            GL.DepthMask(true);
        }


        public void SetControl(OpenTK.GLControl control)
        {
            Control = control;
            Control.Resize += new EventHandler(Resize);
            Control.MouseWheel += new MouseEventHandler(Control_MouseWheel);
            Resize(this, null);
        }

        void Control_MouseWheel(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (e.Delta < 0)
            {
                zoom += 5;
            }else
            {
                zoom -= 5;
            }
        }

        /// <summary>
        /// Converts a point in control space (control.width | control.height) into GL space (-1.0 to 1.0)
        /// </summary>
        /// <param name="point"></param>
        /// <param name="control"></param>
        /// <returns></returns>
        public PointF ControlPointToGlPoint(PointF point, Control control)
        {
            //point.X = point.X - (_controlSize.Width/2);
            //point.Y = point.Y + (_controlSize.Height / 2);
            float nx = (-1.0f + ((point.X * 2) / control.Width));
            float ny = ((point.Y * 2) / control.Height);

            return new PointF(nx, ny);
        }

        /// <summary>
        /// Translates a point from Screen to GL (inverted Y) coordinates.
        /// </summary>
        /// <param name="pointF"></param>
        /// <returns></returns>
        public static PointF ScreenPointToGlPoint(PointF pointF)
        {
            return new PointF(pointF.X, 1.0f - pointF.Y);
        }

        public void MouseDown(System.Windows.Forms.MouseEventArgs e)
        {
            if ((GLControl.MouseButtons & MouseButtons.Left) == MouseButtons.Left)
            {
                dragging = true;
                var p = new Point(e.X, e.Y);
                dragPos = ScreenPointToGlPoint(ControlPointToGlPoint(p, Control));
            }
        }

        public void MouseUp(System.Windows.Forms.MouseEventArgs e)
        {
            if ((GLControl.MouseButtons & MouseButtons.Left) == MouseButtons.Left)
            {
                dragging = false;
            }
        }

        public void MouseMove(System.Windows.Forms.MouseEventArgs e)
        {
            if ((GLControl.MouseButtons & MouseButtons.Left) == MouseButtons.Left & dragging)
            { 
                var p = new Point(e.X, e.Y);
                PointF pos = ScreenPointToGlPoint(ControlPointToGlPoint(p, Control));
                float xAmount = dragPos.X - pos.X;
                float yAmount = dragPos.Y - pos.Y;
                angleX += xAmount*2;
                angleY += yAmount*2;
                //GL.Rotate(angleX, 1.0f, 0.0f, 0.0f);
               // GL.Rotate(angleY, 0.0f, 1.0f, 0.0f);
            }
        }

        public void Render()
        {
            Matrix4 lookat = Matrix4.LookAt(0, 0, zoom, 0, 0, 0, 0, 1, 0);
            GL.MatrixMode(MatrixMode.Modelview);
            GL.LoadMatrix(ref lookat);

            //GL.Rotate(angle, 1.0f, 0.0f, 0.0f);
            //angle += 0.1f;

            GL.Rotate(angleY, 1.0f, 0.0f, 0.0f);
            GL.Rotate(-angleX, 0.0f, 1.0f, 0.0f);

            GL.Clear(ClearBufferMask.ColorBufferBit | ClearBufferMask.DepthBufferBit);

            //DrawCube();
            PixelList.CurrentList.Render();

            Control.SwapBuffers();
        }

        public void GluPerspective(double fovy, double aspect, double zNear, double zFar)
        {
            Double xmin, xmax, ymin, ymax;

            ymax = zNear * Math.Tan(fovy * Math.PI / 360.0);
            ymin = -ymax;
            xmin = ymin * aspect;
            xmax = ymax * aspect;


            GL.Frustum(xmin, xmax, ymin, ymax, zNear, zFar);
        }

        public void Resize(object sender, EventArgs e)
        {
            OpenTK.GLControl c = Control;

            if (c.ClientSize.Height == 0)
                c.ClientSize = new System.Drawing.Size(c.ClientSize.Width, 1);

            GL.Viewport(0, 0, c.ClientSize.Width, c.ClientSize.Height);

            float aspect_ratio = Control.Width / (float)Control.Height;
            //Matrix4 perpective = Matrix4.CreatePerspectiveFieldOfView(MathHelper.PiOver4, aspect_ratio, 1, 64);
            GL.MatrixMode(MatrixMode.Projection);
            //GL.LoadMatrix(ref perpective);
            GluPerspective(90.0f, aspect_ratio, 1.0f, 500.0f);
        }
    }
}
