//  ---------------------------------------------------------------------------
//  This file is part of Scenetone, a music player aimed for playing old
//  music tracker modules and C64 tunes.
//
//  Copyright (C) 2006  Jani Vaarala <flame@pygmyprojects.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  --------------------------------------------------------------------------

/***************************************************************************************************************************
 * Version history *
 *******************

    v0.1    few weeks before christmas 2005    

    v0.2    5.1.2006          Multithreaded, multibuffered version. No sound skipping etc.

    v0.3    4.3.2006          MP3 generation through temp .WAV file

    v0.4    4.4.2006	      Plugin based architecture, but linked into one application (carbide issue with
			      linking ESTLIB...)
							  
    v0.5    13.4.2006	      Subsong selection, new UI for wav generation: play tune, change subsong etc.., then
			      "generate wav from current song". Ask for wav generation length in seconds.

  --- Version history under GPL


    v0.6    5.8.2006	      Keep playing song while selecting next song (multitask) + display currently playing song +
			      first try at adding SVG based icon (gradients not working). Also, added small OpenGL ES
			      visualizer (visualizer is not ready yet, missing stuff, but it displays something ->
			      framework is there). Removed MP3 due to licensing issues and moved back to .WAV generation
			      instead. Added GPL info everywhere, this SW is now under GPL. This version is the first
			      "snapshot" version to go to public distribution.

    v0.7   7.8.2006           Allocated an UID from Symbian for this project from the unprotected range and changed the SW to use
                              it. Also, OpenGL ES visualizer is disabled by default (for now, until it is ready and can be
                              either scaled to device 3D performance and/or turned on/off from the UI).

    v0.8   29.8.2006          Implemented settings file, where one can store the default folder to open when "open module"
     			      is selected. Changed file open code to keep the folder from "open module" to another (this
     			      is "per-session" last used directory, which is initialized from the default path stored in
     			      settings file). Also implemented "start prev/next module" functionality. Pressing "4" starts
     			      previous module and "6" the next module in the file list of the last visited directory. Wrapping
     			      is implemented also. Also added more error checking (should do it properly all around in the
     			      SW) to not crash for SID songs that just dont work. Better way would be to fix the issues
     			      with those songs OR to show that initialization failed instead just being silent.

	v0.9	20.12.2006	Better cleanup from sid and mikmod modules -> should not leak memory (as much
				at least anymore). Updated 
						
For next versions:

    - handle following case: press "next song" after WAV generation (crashes for some reason)
	
    - implement begin/end markers for WAV generation (e.g., "*" key marks the beginning and "#" the end or something like that), if markers were
      set during last playback -> use them, if not -> ask for time as before

    - if mikmod supports "end of song" somehow, implement "move to next song" if previous ends (default behaviour? configurable?)
         * for looping mods: you are SOL
         * for SIDS: configurable timeout

    - add support for changing the default .WAV generation path (store in settings file)

    - randomized play

    - recursive dir scan

    - support for 2nd edition phones (UI vs. player thread/server cleanup, 2nd ed: server process and ui
      process instead of server thread and ui thread in 3rd edition... reason: globals...)
		
*/

// TODO: error checking + cleanup + "if not initialized" guards everywhere where necessary
// TODO: cleaner progress bar implementation (global one is not OK)
// TODO: display more info on song (subsong number, seconds played/max time, also display in "idle screen" if possible, like
//       the standard audio player does...)
// TODO: 2-pass AAC/AAC+ encoding support (MMF API allows to compress from file -> file, but not on-the-fly)
// TODO: because of the OpenGL ES visualizer now the "currently playing" is not actually displayed anymore ...
//       -> should do that in ES side as well
// TODO: add ogg/vorbis support
// TODO: S60 2nd version (global stuff is anyway behind client-server comms at the moment.. however, some data is
//       exchanged between threads in incorrect way -> clean/abstract that out first)
// TODO: tytyy requestin uusiminen odottaakkin ("ok") ettei vahingossa issueta useampia komentoja perkkin -> stray

// INCLUDE FILES
#include <avkon.hrh>
#include <aknnotewrappers.h>
#include <stringloader.h>
#include <Scenetone.rsg>
#include <f32file.h>
#include <s32file.h>
#include <utf.h>
#include <e32base.h>

#include <eikon.hrh>
#include <eikprogi.h>
#include <eikinfo.h> 

#include <AknCommonDialogs.h> 
#include "scenetonesound.h"

#include "Scenetone.pan"
#include "ScenetoneAppUi.h"
#include "ScenetoneAppView.h"
#include "Scenetone.hrh"

#include "settingsutil.h"

_LIT(KScenetoneVersion, 			"Scenetone v0.9");
_LIT(KScenetoneSettingsFile, 			"\\private\\A0001186\\settings.bin");
_LIT(KScenetoneSettingKeyDefaultPath, 	        "MODPATH");
_LIT(KScenetoneDefaultPath, 			"e:\\myown\\mods");

// Note: actual writer will append the tail ".wav" or ".mp3" etc..
_LIT(KWavDestination, 				"c:\\data\\sounds\\digital\\scenetone");
_LIT(KNowPlayingFormatString,  			"Module: '%S'");

// ============================ MEMBER FUNCTIONS ===============================


// -----------------------------------------------------------------------------
// CScenetoneAppUi::ConstructL()
// Symbian 2nd phase constructor can leave.
// -----------------------------------------------------------------------------
//

void CScenetoneAppUi::ConstructL()
    {
    // Initialise app UI with standard value.
    BaseConstructL();

    iAppView = CScenetoneAppView::NewL( ClientRect() );
    iAppView->SetMopParent(this);
    AddToStackL(iAppView);

    iSound = CScenetoneSound::NewL();
    iSound->SetVisualizer(iAppView->iScenetone3D);
    iVolume = 50;
    iConverting = EFalse;
    //iSound->SetVolume(iVolume);

    // Load and parse settings file (default file part of SIS file)
    iSettingsUtil = CSettingsUtil::NewL();
    TInt err = iSettingsUtil->ReadSettings(KScenetoneSettingsFile);
    if(!err)
    {
	iSettingsUtil->GetSetting( KScenetoneSettingKeyDefaultPath, iDefaultPath );
    }
    else
    {
	iDefaultPath = 	KScenetoneDefaultPath;
    }
}
// -----------------------------------------------------------------------------
// CScenetoneAppUi::CScenetoneAppUi()
// C++ default constructor can NOT contain any code, that might leave.
// -----------------------------------------------------------------------------
//
CScenetoneAppUi::CScenetoneAppUi()
    {
    // No implementation required
    }

// -----------------------------------------------------------------------------
// CScenetoneAppUi::~CScenetoneAppUi()
// Destructor.
// -----------------------------------------------------------------------------
//
CScenetoneAppUi::~CScenetoneAppUi()
    {
    if ( iAppView )
        {
        delete iAppView;
        iAppView = NULL;
        }

	if(iCurDir) delete iCurDir;

    }

TKeyResponse CScenetoneAppUi::HandleKeyEventL(
    const TKeyEvent& aKeyEvent, TEventCode aType )
{
	if(iConverting) return EKeyWasNotConsumed;

    if( aType == EEventKey)
	{
	    if( aKeyEvent.iCode == EKeyLeftArrow )
	    {
			iVolume -= 10;
			if(iVolume < 0) iVolume = 0;
			iSound->SetVolume(iVolume);
			return EKeyWasConsumed;
	    }
	    else if( aKeyEvent.iCode == EKeyRightArrow )
	    {
			iVolume += 10;
			if(iVolume > 100) iVolume = 100;
			iSound->SetVolume(iVolume);
			return EKeyWasConsumed;
	    }
		else if( aKeyEvent.iCode == EKeyUpArrow )
		{
			iAppView->iScenetone3D->Stop();
			iSound->NextSubsong();
			iAppView->iScenetone3D->Start();
		}
		else if( aKeyEvent.iCode == EKeyDownArrow )
		{
			iAppView->iScenetone3D->Stop();
			iSound->PrevSubsong();
			iAppView->iScenetone3D->Start();
		}
		else if( aKeyEvent.iCode == '4' || aKeyEvent.iCode == '6' )
		{
			if((iCurrentlyPlaying.Length() > 0) && iCurDir)
			{
				// prev/next song in directory
				TInt adder = -1;
				if(aKeyEvent.iCode == '6')		adder = 1;

				TInt idx = 0;
				while(1)
				{
					if((*iCurDir)[idx].iName == iCurrentlyPlaying)
					{
						break;
					}
					idx++;				
				}
			
				idx = idx + adder;
				if(idx < 0) 					idx = iCurDir->Count()-1;
				if(idx > (iCurDir->Count()-1))  idx = 0;

				// now idx is the next song name to play, construct a full filename
				iCurrentlyPlaying = (*iCurDir)[idx].iName;
			
				TBuf<512> fname;
				fname = iDefaultPath;
				fname.Append(iCurrentlyPlaying);
			
				iAppView->iScenetone3D->Stop();
				iSound->StopL();
	    		iSound->SetFileName(fname);
    			iSound->StartL(); 

				TParse parse;
				parse.Set(fname, NULL, NULL);

	    		iAppView->iPlayText.Format (KNowPlayingFormatString,  &parse.Name());
				iAppView->iScenetone3D->Start();
#if !defined(SCENETONE_SUPPORT_VISUALIZER)
				iAppView->DrawNow();
#endif

			}
		}
	}
    return EKeyWasNotConsumed;
}


// open_bit_stream_w -> remove bitstream.c, functions defined here
// close_bit_stream
// putbits

// need wave_get     -> remove wave.c


void CScenetoneAppUi::AskTimeAndGenerate()
{
	TInt itime = 35;
    CAknNumberQueryDialog *dlg = new (ELeave) CAknNumberQueryDialog( itime );
	dlg->PrepareLC( R_SCENETONE_GENERATE_TIME_QUERY );

	if(dlg->RunLD())
	{
		iAppView->iScenetone3D->Stop();
		iSound->GenerateWav(KWavDestination, itime);
	}
}

void CScenetoneAppUi::GetFileAndPlay()
{
    TBuf8<KMaxFileName+1> fname8;
    TFileName fname;
    
	iAppView->iScenetone3D->Stop();
	
    if (AknCommonDialogs::RunSelectDlgLD(fname, iDefaultPath, R_FILE_SELECTION_DIALOG))
    {
		TInt chars = CnvUtfConverter::ConvertFromUnicodeToUtf8(fname8, fname);
		fname8.Append(TChar('\0'));

		iSound->StopL();
		
    	iSound->SetFileName(fname);
    	iSound->StartL(); 

		TParse parse;
		parse.Set(fname, NULL, NULL);

    	iAppView->iPlayText.Format (KNowPlayingFormatString,  &parse.Name());
		iAppView->iScenetone3D->Start();

#if !defined(SCENETONE_SUPPORT_VISUALIZER)
		iAppView->DrawNow();
#endif
	
		if((iDefaultPath != parse.DriveAndPath()) || !iCurDir)
		{
			// folder was changed, store new folder and process the directory tree
			iDefaultPath = parse.DriveAndPath();		

			RFs session;
			session.Connect();

			// delete old one
			if(iCurDir) delete iCurDir;
			iCurDir = NULL;
			session.GetDir(iDefaultPath, KEntryAttNormal, ESortByName, iCurDir);
		}

		iCurrentlyPlaying = parse.NameAndExt();
	}
}


_LIT(KAboutText, "(c) 2006 Jani Vaarala\nLicensed under GPL\nDownload from:\nwww.pygmyprojects.com");
_LIT(KDefaultFolderStored, "Last folder will be stored as default folder at application exit.");

// -----------------------------------------------------------------------------
// CScenetoneAppUi::HandleCommandL()
// Takes care of command handling.
// -----------------------------------------------------------------------------
//
void CScenetoneAppUi::HandleCommandL( TInt aCommand )
{
	if(iConverting) return;

    switch( aCommand )
    {
        case EEikCmdExit:
        case EAknSoftkeyExit:
	    	iSound->Exit();
	    	delete iSound;

			iSettingsUtil->StoreSettings( KScenetoneSettingsFile );
			
            Exit();
            break;

        case EScenetoneCommand1:
        {
			TInt playstate = iSound->State();
			GetFileAndPlay();
            break;
		}

		/* Generate MP3 of the current playing song + subsong */
        case EScenetoneCommand2:
        {
			TInt playstate = iSound->State();
			if(playstate == CScenetoneSound::EPlaying)
			{
				AskTimeAndGenerate();
			}
	    }
        break;
        
        /* Set current path as the default path in the future */
        case EScenetoneTakeCurrentFolderAsDefaultPath:
		{
			iSettingsUtil->SetSetting( KScenetoneSettingKeyDefaultPath, iDefaultPath );
			CEikInfoDialog::RunDlgLD( KScenetoneVersion, KDefaultFolderStored );
			break;
		}        	
    
		/* Display ABOUT box */
		case EScenetoneAbout:
		{
			CEikInfoDialog::RunDlgLD( KScenetoneVersion, KAboutText );
		}
		break;
		
        default:
            Panic( EScenetoneUi );
            break;
        }
    }
    
// -----------------------------------------------------------------------------
//  Called by the framework when the application status pane
//  size is changed.  Passes the new client rectangle to the
//  AppView
// -----------------------------------------------------------------------------
//
void CScenetoneAppUi::HandleStatusPaneSizeChange()
{
	iAppView->SetRect( ClientRect() );
	
} 

// End of File

