/*
 * $Id: iofile.c 154 2007-11-12 23:10:08Z ehaase $
 *
 * Input/Output file stream routines.
 *
 * Tries to load from a RCDATA resource first; if not
 * found, looks in the file system.
 *
 * Copyright (c) 1995-1999 Carlos Hasan
 * Copyright (c) 2007      Enver Haase
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <stdio.h>
#include <limits.h>
#include <malloc.h>
#include <string.h>
#include <windows.h> /*resources*/

#include "../compat.hpp"
#include "iofile.h"

/*
 * Audio I/O file stream routines
 */
static FILE *lpStream = NULL;


static LONG   position = 0;
static UINT8  *module = NULL;
static HGLOBAL _resource = NULL;

unsigned int IOCloseResource(VOID)
{
    if (_resource != NULL)
        FreeResource(_resource);
    return IO_ERROR_NONE;
}

unsigned int IOOpenResource(LPSTR lpszResourceName)
{
    HRSRC   res;
    HGLOBAL loadRes;
    
    position = 0;

    res = FindResourceA (NULL /* our .EXE */, lpszResourceName, RT_RCDATA);
    if (res == NULL)
        return IO_ERROR_FILENOTFOUND;

    loadRes = LoadResource (NULL /* our .EXE */, res);
    if (loadRes == NULL)
        return IO_ERROR_FILENOTFOUND;

    _resource = loadRes;
    module = (UINT8 *) LockResource(_resource);
    if (module == NULL)
    {
        IOCloseResource(); /* Free resource */
        return IO_ERROR_FILENOTFOUND;
    }
    return IO_ERROR_NONE;
}




unsigned int IOOpenFile(LPSTR lpszFileName)
{
    /* First try if we can get it as a resource rather than a real file. */
    if (IOOpenResource(lpszFileName) != IO_ERROR_FILENOTFOUND)
    {
        return IO_ERROR_NONE;
    }

    /* Try opening as a real file. */
    fopen_s(&lpStream, lpszFileName, "rb");
    if (lpStream != NULL)
    {
        return IO_ERROR_NONE;
    }

    return IO_ERROR_FILENOTFOUND;
}

unsigned int IOCloseFile(VOID)
{
    if (_resource != NULL)
    {
        IOCloseResource();
        return IO_ERROR_NONE;
    }
    else
    {
        fclose(lpStream);
        lpStream = NULL;
        return IO_ERROR_NONE;
    }
}





unsigned int IOSeekFile(LONG dwOffset, UINT nWhere)
{
    if (_resource != NULL){
        if (nWhere == SEEK_SET)
            position = dwOffset;
        if (nWhere == SEEK_CUR)
            position += dwOffset;
    }
    else
    {
        fseek(lpStream, dwOffset, nWhere);
    }
    return IO_ERROR_NONE;
}

unsigned int IOReadFile(LPVOID lpData, UINT nSize)
{
    UINT i;
    if (_resource != NULL)
    {
        UINT8 *data = (UINT8 *) lpData;
        for (i=0; i < nSize; i++)
        {
            *data++ = module[position++];
        }
    }
    else
    {
        if (fread(lpData, 1, nSize, lpStream) != nSize)
            memset(lpData, 0, nSize);
    }
    return IO_ERROR_NONE;
}

/* little-endian input routines */
unsigned int IOReadChar(LPBYTE lpData)
{
    if (_resource != NULL)
    {
        IOReadFile(lpData, 1);
    }
    else
    {
        if (fread(lpData, 1, sizeof(BYTE), lpStream) != sizeof(BYTE))
            memset(lpData, 0, sizeof(BYTE));
    }
    return IO_ERROR_NONE;
}

unsigned int IOReadShort(LPWORD lpData)
{
    if (_resource != NULL)
    {
        IOReadFile(lpData, sizeof(WORD));
    }
    else
    {
        if (fread(lpData, 1, sizeof(WORD), lpStream) != sizeof(WORD))
            memset(lpData, 0, sizeof(WORD));
    }

#ifdef __BIGENDIAN__
    *lpData = MAKEWORD(HIBYTE(*lpData), LOBYTE(*lpData));
#endif
    return IO_ERROR_NONE;
}

unsigned int IOReadLong(LPDWORD lpData)
{
    if (_resource != NULL)
    {
        IOReadFile(lpData, sizeof(DWORD));
    }
    else
    {
        if (fread(lpData, 1, sizeof(DWORD), lpStream) != sizeof(DWORD))
            memset(lpData, 0, sizeof(DWORD));
    }
#ifdef __BIGENDIAN__
    *lpData = MAKELONG(
        MAKEWORD(HIBYTE(HIWORD(*lpData)), LOBYTE(HIWORD(*lpData))),
	MAKEWORD(HIBYTE(LOWORD(*lpData)), LOBYTE(LOWORD(*lpData))));
#endif
    return IO_ERROR_NONE;
}

/* big-endian input routines */
unsigned int IOReadCharM(LPBYTE lpData)
{
    return IOReadChar(lpData); /* no endianness for 1 byte */
}

unsigned int IOReadShortM(LPWORD lpData)
{
    if (_resource != NULL)
    {
        IOReadFile(lpData, sizeof(WORD));
    }
    else
    {
        if (fread(lpData, 1, sizeof(WORD), lpStream) != sizeof(WORD))
            memset(lpData, 0, sizeof(WORD));
    }
#ifndef __BIGENDIAN__
    *lpData = MAKEWORD(HIBYTE(*lpData), LOBYTE(*lpData));
#endif
    return IO_ERROR_NONE;
}

unsigned int IOReadLongM(LPDWORD lpData)
{
    if (_resource != NULL)
    {
        IOReadFile(lpData, sizeof(DWORD));
    }
    else
    {
        if (fread(lpData, 1, sizeof(DWORD), lpStream) != sizeof(DWORD))
            memset(lpData, 0, sizeof(DWORD));
    }
#ifndef __BIGENDIAN__
    *lpData = MAKELONG(
        MAKEWORD(HIBYTE(HIWORD(*lpData)), LOBYTE(HIWORD(*lpData))),
        MAKEWORD(HIBYTE(LOWORD(*lpData)), LOBYTE(LOWORD(*lpData))));
#endif
    return IO_ERROR_NONE;
}
