/*-----------------------------------------------------------------------------
 * File: v3d.c
 * Written by: Fredrik Kling, 1997-05-07
 * Description: V3D scene loader
 *
 * Updates:
 * -- Date -- | ----- Name ----- |-- Did what....
 * 1997-08-22 | Fredrik Kling    | Added some functions to make 3d-studio loader work smoothly...
 * 1997-08-01 | Fredrik Kling    | Parent object hantering... (bde i savern och i loadern)
 * 1997-07-27 | Fredrik Kling    | Spara i olika format!
 * 1997-05-07 | Fredrik Kling    | Implementation
 * 1997-05-17 | Mikael Lindkvist | Skrev om till nya parsern....
 * 1997-06-15 | Alexander Boczar | Snygga upp la till/tog bort/ndrade...
 * 1997-06-16 | Fredrik Kling    | Fixade till fel-hanteringen, animations import...
 * 1997-06-24 | Fredrik Kling    | Nullobjects, cameras, nya animationer..
 *
 * Kommentar:
 *
 * Todo:
 *		Lgga till evaluatorn....
 *
 *----------------------------------------------------------------------------*/

#include "system/xstddef.h"
#include "system/xstdio.h"
#include "system/xstdlib.h"
#include "system/xstring.h"
#include "formats/v3d.h"
#include "formats/v3o.h"
#include "objects/object.h"
#include "objects/light.h"
#include "objects/camera.h"
#include "misc/spline.h"
#include "misc/parser.h"

typedef enum
{
	ERR_NOERR,
	ERR_NOFILE,
	ERR_NOMEM,
	ERR_CUSTOM,
	ERR_OBJECT,
} ERROR;

static ERROR error = ERR_NOERR;
static char errstr[256];

static void v3d_anim_save (FILE *f, ANIM *anim)
{
	KEY *key;
	int i,j;

	fprintf (f,"    ANIM\n");
	fprintf (f,"    {\n");
	fprintf (f,"        INFOCH %i\n",anim->values);
	fprintf (f,"        KEYNUM %i\n",anim->keys);

	key = anim->first;
	for (i=0;i<anim->keys;i++)
	{
		fprintf (f,"        KEY ");
		fprintf (f,"%f ",key->frame);
		// Values should always be 9...
		for (j=0;j<anim->values;j++) fprintf (f,"%f ",key->value[j]);
		fprintf (f,"%f ",key->tension);
		fprintf (f,"%f ",key->bias);
		fprintf (f,"%f ",key->contunity);
		fprintf (f,"\n");
		key=key->next;
	}
	fprintf (f,"    }\n");

  return;
}

static void v3d_addcamera (V3D *v3d, CAMERA *cam)
{
  v3d->activecamera = cam;
}

static OBJECT *v3d_getobject (V3D *v3d, int t)
{
	int i;
	OBJECT *obj;
	obj=v3d->obj;
	for (i=0;i<t;i++)
	{
		obj = obj->next;
		if (obj==NULL) return NULL;
	}
	return obj;
}

void v3d_obj2scene (V3D *v3d,OBJECT *obj)
{
	OBJECT *next;
	if (obj->type == T_VLGH)
	{
		if (v3d->lightlist==NULL) v3d->lightlist=obj;
			else {
						 obj->next = v3d->lightlist;
						 v3d->lightlist->prev = obj;
					 v3d->lightlist=obj;
					 }
  } else
			if ((obj->type==T_V3O) || (obj->type==T_NULL))
			{
				if (v3d -> obj==NULL) v3d->obj=obj;
					else
						{
							next = v3d->obj;
							while (next->next!=NULL) next=next->next;
							next->next=obj;
							obj->prev = next;
				 		}
			}
}

static int readanim (PARSE *p, OBJECT *obj)
{
	char tok[256];
  int cmd,first,infoch,keynum,i;
	float xf,yf,zf;
//  KEY key;
  KEY *n;
	parse_this (p,"{");

	first = 0;
	xf = yf = zf = 1.0;
	while (cmd=parse_case (parse_get (p,tok),"infoch keynum key scale"))
	{
		switch (cmd)
		{
			case 1 :	// Infochannels.
				infoch = parse_geti (p);
				obj->anim = spline_createanim (infoch);
				break;
			case 2 : // key num;
				keynum = parse_geti (p);
				break;
			case 3 : // keys!
        n = spline_addkey (obj->anim,parse_getf (p));
        for (i=0;i<infoch;i++) n->value[i] = parse_getf (p);

				n->value[ANIM_POS_X]*=xf;
				n->value[ANIM_POS_Y]*=yf;
				n->value[ANIM_POS_Z]*=zf;

        n -> tension = parse_getf (p);
        n -> bias = parse_getf (p);
        n -> contunity = parse_getf (p);
				break;
			case 4 : // Scale
				xf = parse_getf (p);
				yf = parse_getf (p);
				zf = parse_getf (p);
				break;


		  default : // Error...
				error = ERR_CUSTOM;
				sprintf (errstr, "[ANIM] Unknown token '%s'.",tok);
				return FALSE;
				break;
		}
	}

	if (stricmp(tok,"}"))
	{
		error = ERR_CUSTOM;
		sprintf( errstr, "[PARSE] Expected '}' but found '%s'.", tok);
		return( FALSE);
	}
	return( TRUE);
}
static int docamera (PARSE *p, V3D *v3d)
{
	CAMERA *cam;
	int cmd;
	int t;
	char tok[256];
	OBJECT *dummy;

	parse_this (p,"{");

	cam = camera_create( );
	dummy = object_create ();


	while (cmd=parse_case (parse_get (p,tok),"angleadd upv anim target postarget"))
	{
		switch (cmd)
		{
			case 1: // angleadd
					cam->angleadd[0] = parse_geti (p);
					cam->angleadd[1] = parse_geti (p);
					cam->angleadd[2] = parse_geti (p);
					break;
			case 2: // upv
					cam->upv.x = parse_getf (p);
					cam->upv.y = parse_getf (p);
					cam->upv.z = parse_getf (p);
			 		break;
			case 3: // anim
				 	readanim (p,dummy);
					cam -> anim = dummy -> anim;
					xfree (dummy);
				 	break;
			case 4: // Target
					cam -> num_target = t = parse_geti (p);
          cam -> obj_target = v3d_getobject (v3d,t-1);
					cam -> flags += CAMERAFLAG_OBJTARGET;
					break;
			case 5: // POSTARGET
					cam -> flags += CAMERAFLAG_POSTARGET;
					break;

		default: // if error...
				error = ERR_CUSTOM;
				sprintf( errstr, "[PARSE] Unknown token '%s'.",tok);
				return( FALSE);
		}
	}
	if (!(cam->flags & CAMERAFLAG_OBJTARGET)) cam->flags += CAMERAFLAG_ANGLE;
  v3d_addcamera (v3d,cam);

	if (stricmp(tok,"}"))
	{
		error = ERR_CUSTOM;
		sprintf( errstr, "[PARSE] Expected '}' but found '%s'.", tok);
		return( FALSE);
	}
	return( TRUE);
}


static int dolight (PARSE *p, V3D *v3d)
{
	OBJECT *obj;
	int cmd;
	float f;
	char tok[256];

	parse_this (p,"{");

	obj = object_create( );
	obj->vlgh = vlgh_create( );
	obj->type=T_VLGH;

	while (cmd=parse_case (parse_get (p,tok),"type position color intensity exact falloff anim"))
	{
		switch (cmd)
		{
			case 1: // TYPE
				switch (parse_case (parse_get (p,tok),"spot omni"))
				{
					case 1: obj->vlgh->flags+=VLGHFLAG_SPOT; break;
					case 2: obj->vlgh->flags+=VLGHFLAG_OMNI; break;
				}
				break;
			case 2: // position
				f=parse_getf(p);
				f=parse_getf(p);
				f=parse_getf(p);
				break;

			case 3: // color
				obj->vlgh->color.r=parse_geti(p);
				obj->vlgh->color.g=parse_geti(p);
				obj->vlgh->color.b=parse_geti(p);
				break;
			case 4: // intensity
				obj->vlgh->intens=63*parse_getf(p);		// Multiply by lightlevels...  default: 64... =)
				break;
			case 5: // exact
				obj->vlgh->flags+=VLGHFLAG_EXACT;
				break;
			case 6: // falloff
				obj->vlgh->flags+=VLGHFLAG_DISTANCE;
				obj->vlgh->falloff = parse_getf (p);
				break;
			case 7: // anim.. MUST EXIST...
				 readanim (p,obj);
				 break;
			default: // if error...
				error = ERR_CUSTOM;
				strcpy( errstr, "[PARSE] Unknown directive");
				return( FALSE);
		}
	}
	v3d_obj2scene (v3d,obj);

	if (stricmp(tok,"}"))
	{
		error = ERR_CUSTOM;
		sprintf( errstr, "[PARSE] Expected '}' but found '%s'.", tok);
		return( FALSE);
	}
	return( TRUE);
}


static int doobject(PARSE *p,V3D *v3d, char *objectpath, char *imagepath)
{
	OBJECT *obj;
  int cmd,t;
	float xf,yf,zf,f;
	char tok[256];

	parse_this(p,"{");

	obj = object_create ();
	while (cmd=parse_case(parse_get(p,tok),"type file position scale origo angles angleadd material anim parent vector_rotation"))
	{
		switch (cmd)
		{
			case 1:  // TYPE
				switch (parse_case(parse_get(p,tok),"v3o vio vsnd vcam null"))
				{
					case 1: obj->type=T_V3O; break;
					case 2: obj->type=T_VIO; break;
					case 3: obj->type=T_VSND; break;
					case 4: obj->type=T_VCAM; break;
					case 5: obj->type=T_NULL; break;
					default:
						error = ERR_CUSTOM;
						strcpy(errstr, "[PARSE] Bad or no objecttype");
						return( FALSE);
				}
				break;

			case 2:  // FILE
			{
				char objectname[256];
        if( (objectpath != NULL) && (strlen(objectpath) > 0))
					sprintf( objectname, "%s/%s", objectpath, parse_safeget(p,tok));
				else
					strcpy( objectname, parse_safeget(p,tok));
				if (object_load(obj,objectname,imagepath)==NULL)
				{
					error = ERR_OBJECT;
					return FALSE;
				}
				break;
			}
			case 3:  // POSITION finns inte lngre..  styrs genom animationer...
				f=parse_getf(p);
				f=parse_getf(p);
				f=parse_getf(p);
				break;

			case 4:  // SCALE
				xf=parse_getf(p);
				yf=parse_getf(p);
				zf=parse_getf(p);
				v3o_scale (obj->v3o, xf, yf, zf);
/*
				for (i=0;i<obj->v3o->numvertex;i++)
				{
					obj->v3o->orgvertex[i].x*=xf;
					obj->v3o->orgvertex[i].y*=yf;
					obj->v3o->orgvertex[i].z*=zf;
				}
*/
				break;

			case 5:  // ORIGO
				xf=parse_getf(p);
				yf=parse_getf(p);
				zf=parse_getf(p);
				v3o_translate (obj->v3o,xf,yf,zf);
				break;

			case 6:  // ANGLES finns inte... styrs genom animationer...
				//obj->angle[0]=parse_geti(p);
				//obj->angle[1]=parse_geti(p);
				//obj->angle[2]=parse_geti(p);
				break;

			case 7:  // ANGLEADD
				obj->angleadd[0]=parse_geti(p);
				obj->angleadd[1]=parse_geti(p);
				obj->angleadd[2]=parse_geti(p);
				break;
			case 8:  // MATERIAL finns inte..  r numera inbakat i .v3o filerna...
				//v3d_objmat (p,obj);
				break;
			case 9: // Anim... MSTE finnas fr att allt skall fungera..
				readanim (p,obj);
				break;
			case 10: // Parent!
				obj -> num_parent = t = parse_geti (p);
				obj -> parent = v3d_getobject (v3d,t-1);
				break;
			case 11: // vector_rotation
				obj -> flags += OBJFLAG_VECROT;
			break;
		}
	}
	object_boundingsphere (obj);
	v3d_obj2scene (v3d,obj);
  if (stricmp(tok,"}"))
	{
		error = ERR_CUSTOM;
		sprintf( errstr, "[PARSE] Expected '}' but found '%s'.", tok);
		return( FALSE);
	}
	return( TRUE);
}

// Checks all objects in a list for the one who has a matching name...
static OBJECT *namedobject (OBJECT *obj,char *name)
{
	OBJECT *o;
//	for (o = obj;o!=NULL;o=o->next) if (xstrcmp (o->name,name)) return o;

	return o;
}
/*################################################################
# v3d - objfromname
#
# This routine is currently only used by the 3ds loader
#
#################################################################*/
OBJECT *v3d_objfromname (V3D *v3d,char *name)
{
	OBJECT *obj;
	obj = NULL;

	// Check all object lists in scene for one that matches the name...
/*
	if (v3d->lightlist != NULL) obj = namedobject (v3d->activecamera,name);
	if ((obj != NULL) && (v3d->activecamera != NULL)) obj = namedobject (v3d->activecamer,name);
	if ((obj != NULL) && (v3d->obj!=NULL)) obj = namedobject (v3d->obj, name);
*/
	return obj;
}
/*################################################################
# v3d - create
#################################################################*/

V3D *v3d_create (void)
{
	V3D *v3d;
	if ((v3d = (V3D *)xmalloc (sizeof (V3D)))==NULL)
	{
		error = ERR_NOMEM;
		return NULL;
	}
  v3d -> obj = NULL;
	v3d -> lightlist = NULL;
	v3d -> activecamera = NULL;
	v3d -> background = NULL;
	v3d -> framecounter = 0;
	v3d -> fps = 40;
	v3d -> zmin = 0;
	v3d -> zmax = 1000;
	v3d -> ambient.r = v3d -> ambient.g = v3d->ambient.b = 0;
	v3d -> maxlight.r = v3d -> maxlight.g = v3d -> maxlight.b = 128;
	return v3d;
}

/*################################################################
# v3d - free
#################################################################*/

void v3d_free( V3D *v3d)
{
	if( v3d != NULL)
	{
		/* TODO: Actually deallocate objects */
		xfree( v3d);
	}
}

/*################################################################
# v3d - load
#################################################################*/

V3D *v3d_load( char *filename, char *objectpath, char *imagepath)
{
	V3D *v3d;
	PARSE *p;
	int cmd;
	char *buff;
  DWORD buffsize;
	char tok[256];

	v3d = v3d_create ();
  if( (buff = xfload( filename, &buffsize)) == NULL)
	{
		error = ERR_NOFILE;
		return( NULL);
	}

	/* Add a 'eof' marker */
	buff = realloc( buff, buffsize+1);
	buff[ buffsize] = '\0';

	p = parse_init( buff);

	parse_this(p,"__v3d__");
	while( cmd = parse_case( parse_get( p, tok), "object scene anim morph light camera fps ambient zclip maxlight"))
	{
		switch( cmd)
		{
			case 1: // Object
				if(!doobject(p,v3d,objectpath,imagepath))
				{
					xfree(buff);
					return( NULL);
				}
				break;
			case 5: // Light
				if( !dolight(p,v3d))
				{
					xfree( buff);
					return( NULL);
				}
				break;
			case 6: // Camera
        if (!docamera (p,v3d))
				{
          xfree (buff);
					return NULL;
				}
				break;
			case 7: // fps
				v3d -> fps = parse_getf (p);
				break;
			case 8: // ambient
				v3d -> ambient.r = parse_geti (p);
				v3d -> ambient.g = parse_geti (p);
				v3d -> ambient.b = parse_geti (p);
				break;
			case 9: // zclip
				v3d -> zmin = parse_getf (p);
				v3d -> zmax = parse_getf (p);
				break;
			case 10: // maxlight
				v3d -> maxlight.r = parse_geti (p);
				v3d -> maxlight.g = parse_geti (p);
				v3d -> maxlight.b = parse_geti (p);
				break;
			default:
				error = ERR_CUSTOM;
				strcpy( errstr, "[PARSE] Unknown token.");
				xfree( buff);
				return( NULL);
		}
	}

	if( parse_get( p, tok))
	{
		error = ERR_CUSTOM;
		strcpy( errstr, "[PARSE] Expecting end of file.");
		xfree( buff);
		return( NULL);
	}

	parse_exit( p);
	xfree( buff);

	return( v3d);
}

/*################################################################
	# v3d - save
	#################################################################*/
int v3d_save_normal (V3D *v3d,char *fn)
{
	FILE *f;
	KEY *k;
	OBJECT *obj;
	char obfn[128];
	int i;

	f=fopen (fn,"wt");
	fprintf (f,"__V3D__\n");
	//
	// Save additional scene information...
	//
	printf (" Writing scene information.\n");
	fprintf (f,"AMBIENT	%i %i %i\n",v3d->ambient.r,v3d->ambient.g,v3d->ambient.b);
	fprintf (f,"MAXLIGHT %i %i %i\n",v3d->maxlight.r,v3d->maxlight.g,v3d->maxlight.b);
	fprintf (f,"FPS %6.3f\n",v3d->fps);
  fprintf (f,"ZCLIP %f %f\n",v3d->zmin,v3d->zmax);


	//
	// Save lightlist...
	//
	printf (" Writing lightsources.\n");
  for (i=0,obj=v3d->lightlist;obj!=NULL;obj=obj->next,i++)
	{
		fprintf (f,"LIGHT\n{\n");
		fprintf (f,"    TYPE omni\n");
//    fprintf (f,"    POSITION %9.3f %9.3f %9.3f\n",obj->.x,obj->anim->values[.y,obj->position.z);
    fprintf (f,"    COLOR %i %i %i\n",obj->vlgh->color.r,obj->vlgh->color.g,obj->vlgh->color.b);
		fprintf (f,"		INTENSITY %9.3f\n",obj->vlgh->intens);
		if (obj->vlgh->flags & VLGHFLAG_DISTANCE) fprintf (f,"    FALLOFF %9.3f\n",obj->vlgh->falloff);
    if (obj->anim!=NULL) v3d_anim_save (f,obj->anim);
		fprintf (f,"}\n");
	}

	//
	// Save objects...
	//
	printf (" Writing objects.\n");
  for (i=0,obj=v3d->obj;obj!=NULL;obj=obj->next,i++)
	{
		sprintf (obfn,"obj%i.v3o",i);
		fprintf (f,"OBJECT\n{\n");
		if (obj->type==T_V3O)
		{
			fprintf (f,"    TYPE v3o\n");
			fprintf (f,"    FILE %s\n",obfn);
		} else {
						 fprintf (f,"    TYPE null\n");
					 }
		if (obj->num_parent!=-1)
			fprintf (f,"    PARENT %i\n",obj->num_parent);

		fprintf (f,"    ANGLEADD %i %i %i\n",obj->angleadd[0],obj->angleadd[1],obj->angleadd[2]);
		if (obj->flags & OBJFLAG_VECROT)
			fprintf (f,"    VECTOR_ROTATION\n");

    if (obj->anim!=NULL) v3d_anim_save (f,obj->anim);
		fprintf (f,"}\n");
		if (obj->type==T_V3O) v3o_save (obfn,obj->v3o,V3O_SAVE_NORMAL);
	}
	//
	// Cameras MUST be saved last otherwise loading will fuckup...
	//
	// If none is create we create one...
	//
	if (v3d->activecamera==NULL)
	{
		v3d->activecamera = camera_create ();
		v3d->activecamera->anim = spline_createanim (9);
		v3d->activecamera->flags = CAMERAFLAG_POSTARGET;
		k = spline_addkey (v3d->activecamera->anim,0);
		k->value[ANIM_POS_X] = 0;
		k->value[ANIM_POS_Y] = 0;
		k->value[ANIM_POS_Z] = -1000;
	}
	if (v3d->activecamera!=NULL)
	{
	  printf (" Writing camera.\n");
		fprintf (f,"CAMERA\n{\n");
//	while (cmd=parse_case (parse_get (p,tok),"angleadd upv anim target"))
		fprintf (f,"    UPV %6.3f %6.3f %6.3f\n",v3d->activecamera->upv.x,v3d->activecamera->upv.y,v3d->activecamera->upv.z);

		// Preset...  theese can only be set explicit.
		fprintf (f,"    ANGLEADD %i %i %i\n",0,0,0);
		if (v3d->activecamera->flags & CAMERAFLAG_POSTARGET)
			fprintf (f,"    POSTARGET\n");
//			fprintf (f,"    POSTARGET %f %f %f\n",v3d->activecamera->target.x,v3d->activecamera->target.y,v3d->activecamera->target.z);

	// Special case problem.. we support only pointer stuff..
		// LWS converter stores target in angleadd[0]...

		if (v3d->activecamera->num_target!=-1) fprintf (f,"    TARGET %i\n",v3d->activecamera->num_target);
		if (v3d->activecamera->anim!=NULL) v3d_anim_save (f,v3d->activecamera->anim);
		fprintf (f,"}\n");
	}

	fclose (f);
  return( TRUE);

}
int v3d_save_asm (V3D *v3d, char *fn)
{
	FILE *f;
	OBJECT *obj;
	char obfn[128];
	int i;

	printf (" Generating assembler source..\n");

	f=fopen (fn,"wt");
	fprintf (f,";........----  Source code generated with lws2v3d.exe, version 1.01 ----............\n");
	//
	// Save additional scene information...
	//
	printf (" Writing scene information.\n");
	fprintf (f,"ambient   dd	%i,%i,%i\n",v3d->ambient.r,v3d->ambient.g,v3d->ambient.b);
	fprintf (f,"maxlight  dd	%i,%i,%i\n",v3d->maxlight.r,v3d->maxlight.g,v3d->maxlight.b);
	fprintf (f,"fps       dd	%6.3f\n",v3d->fps);
  fprintf (f,"zclip     dd  %f,%f\n",v3d->zmin,v3d->zmax);
	fprintf (f,"lightlist dd");
	for (i=0,obj=v3d->lightlist;obj!=NULL;obj=obj->next,i++)
	{
		if (obj->next!=NULL)	fprintf (f," ofs light%i,");
			else fprintf (f," ofs light%i");
	}
	fprintf (f,"objectlist dd");
	for (i=0,obj=v3d->obj;obj!=NULL;obj=obj->next,i++)
	{
		if (obj->next!=NULL)	fprintf (f," ofs obj%i,");
			else fprintf (f," ofs obj%i");
	}


	//
	// Save lightlist...
	//
	printf (" Writing lightsources.\n");
  for (i=0,obj=v3d->lightlist;obj!=NULL;obj=obj->next,i++)
	{
		fprintf (f,"light%i  ");
		fprintf (f,"  type dd	LIGHT_OMNI\n");
    fprintf (f,"  color dd %i,%i,%i\n",obj->vlgh->color.r,obj->vlgh->color.g,obj->vlgh->color.b);
		fprintf (f,"  intensity dd %9.3f\n",obj->vlgh->intens);
    if (obj->anim!=NULL) v3d_anim_save (f,obj->anim);
			else fprintf (f,"  anim  dd NULL");
	}

	//
	// Save objects...
	//
	printf (" Writing objects.\n");
  for (i=0,obj=v3d->obj;obj!=NULL;obj=obj->next,i++)
	{
		sprintf (obfn,"obj%i.v3o",i);
		fprintf (f,"OBJECT\n{\n");
		if (obj->type==T_V3O)
		{
			fprintf (f,"    TYPE v3o\n");
			fprintf (f,"    FILE %s\n",obfn);
		} else {
						 fprintf (f,"    TYPE null\n");
					 }
//    fprintf (f,"    ORIGO %9.3f %9.3f %9.3f\n",obj->origo.x,obj->origo.y,obj->origo.z);
//    fprintf (f,"    POSITION %9.3f %9.3f %9.3f\n",obj->position.x,obj->position.y,obj->position.z);
//    fprintf (f,"    ANGLES %i %i %i\n",obj->angle[0],obj->angle[1],obj->angle[2]);
		fprintf (f,"    ANGLEADD %i %i %i\n",obj->angleadd[0],obj->angleadd[1],obj->angleadd[2]);
    if (obj->anim!=NULL) v3d_anim_save (f,obj->anim);
		fprintf (f,"}\n");
		if (obj->type==T_V3O) v3o_save (obfn,obj->v3o,V3O_SAVE_ASM);
	}
	//
	// Cameras MUST be saved last otherwise loading will fuckup...
	if (v3d->activecamera!=NULL)
	{
	  printf (" Writing camera.\n");
		fprintf (f,"CAMERA\n{\n");
//	while (cmd=parse_case (parse_get (p,tok),"angleadd upv anim target"))
		fprintf (f,"    UPV %6.3f %6.3f %6.3f\n",v3d->activecamera->upv.x,v3d->activecamera->upv.y,v3d->activecamera->upv.z);

		// Preset...  theese can only be set explicit.
		fprintf (f,"    ANGLEADD %i %i %i\n",0,0,0);

	// Special case problem.. we support only pointer stuff..
		// LWS converter stores target in angleadd[0]...

		if (v3d->activecamera->num_target!=-1) fprintf (f,"    TARGET %i\n",v3d->activecamera->num_target);
		if (v3d->activecamera->anim!=NULL) v3d_anim_save (f,v3d->activecamera->anim);
		fprintf (f,"}\n");
	} else

	fclose (f);
  return( TRUE);


}

int v3d_save (V3D *v3d,char *fn,int type)
{
	switch (type)
	{
		case V3D_SAVE_NORMAL : if (v3d_save_normal (v3d,fn)==TRUE) return TRUE;
													 break;
		case V3D_SAVE_ASM : if (v3d_save_asm (v3d,fn)==TRUE) return TRUE;
												break;
		case V3D_SAVE_C :
		case V3D_SAVE_PAS :
		default	: return FALSE;
	}
	return FALSE;

}

/*################################################################
	# v3d - dump
	#################################################################*/

void v3d_dump (V3D *v3d)
{
	OBJECT *o;
	int i;
	printf ("\n-- Dumping contents of V3D structure! --\n");
  printf ("Ambient.: %i,%i,%i\n",v3d->ambient.r,v3d->ambient.g,v3d->ambient.b);
  printf ("Maxlight: %i,%i,%i\n",v3d->maxlight.r,v3d->maxlight.g,v3d->maxlight.b);
  printf ("Zclip...: %f,%f\n",v3d->zmin,v3d->zmax);
  printf ("Fps.....: %f\n",v3d->fps);


  if (v3d->obj!=NULL)
	{
		printf ("Global object table found!\n");
    for (i=0,o=v3d->obj;o!=NULL;o=o->next,i++)
		{
			printf ("Object: %i\n",i);
			if (o->type==T_V3O) printf ("    Type....: V3O\n");
//      printf ("   Origo...: %6.3f,%6.3f,%6.3f\n",o->origo.x,o->origo.y,o->origo.z);
//      printf ("   Position: %6.3f,%6.3f,%6.3f\n",o->position.x,o->position.y,o->position.z);
			printf ("   Radius..: %6.3f\n",o->radius);
      if (o->type==T_V3O) v3o_dump (o->v3o,0,0);
		}
	}
	printf ("\n----------------------------------------------\n");
  if (v3d->lightlist!=NULL)
	{
		printf ("Global lighttable found!\n");
    for (i=0,o=v3d->lightlist;o!=NULL;o=o->next,i++)
		{
			printf ("Light: %i\n",i);
			if (o->vlgh->flags & VLGHFLAG_SPOT) printf ("    Type....: Spotlight\n");
				else printf ("    Type....: Omnilight\n");
			printf ("    Color...: %i,%i,%i\n",o->vlgh->color.r,o->vlgh->color.g,o->vlgh->color.b);
			printf ("    Intens..: %6.3f\n",o->vlgh->intens);

			printf ("    Use dist: ");
			if (o->vlgh->flags & VLGHFLAG_DISTANCE) printf ("yes\n");
				else printf ("no\n");
			printf ("    Exact...: ");
			if (o->vlgh->flags & VLGHFLAG_EXACT) printf ("yes\n");
				else printf ("no\n");
		}
	}
	printf ("\n----------------------------------------------\n");
}

/*################################################################
	# v3d - geterror
	#################################################################*/

char *v3d_geterror(void)
{
	switch( error)
	{
		case ERR_NOERR: return( "[V3D] No error");
		case ERR_NOFILE: return( "[V3D] Unable to open file");
		case ERR_NOMEM: return( "[V3D] Unable to allocate memory");
		case ERR_OBJECT: {
											 sprintf (errstr,"[V3D] - %s.",object_geterror ());
											 return errstr;
										 }
		case ERR_CUSTOM:
		{
			char str[256];
			strcpy( str, errstr);
			sprintf( errstr, "[V3D] %s", str);
			return( errstr);
		}
		default: return( "[V3D] Unknown error!");
	}
}

/*################################################################
	# v3d - setup
	#################################################################*/

int v3d_setup( V3D *v3d, RGB *palette, BYTE *lighttab)
{
	OBJECT *object;
	RGBA col;
	RGB co;

	if( palette != NULL)
	{
		RGB *colorlist = NULL;
		VIO **imagelist = NULL;
		int colors = 0;
		int images = 0;
		int i,j;

		object = v3d->obj;
		while( object != NULL)
		{
			if( object->type & T_V3O)
			{
				V3O *v3o = object->v3o;

				for( i=0; i<v3o->nummaterial; i++)
				{
					if( v3o->material[ i].flags & V3OMATERIALFLAG_TEXTURE)
					{
						int duplicate;

						duplicate = FALSE;
						for( j=0; j<images; j++)
							if( imagelist[ j] == v3o->texture[ v3o->material[i].texture].image)
								duplicate = TRUE;
						if( !duplicate)
						{
							if( (imagelist = (VIO **)realloc( imagelist, (images+1) * sizeof( VIO *))) == NULL)
								return( FALSE);
							imagelist[ images++] = v3o->texture[ v3o->material[ i].texture].image;

							if( (colorlist = (RGB *)realloc( colorlist, (colors+256) * sizeof( RGB))) == NULL)
								return( FALSE);
							for( j=0; j<256; j++)
								colorlist[ colors++] = v3o->texture[ v3o->material[ i].texture].image->palette[j];
						}

					}
					else if( v3o->material[ i].flags & (V3OMATERIALFLAG_FLAT | V3OMATERIALFLAG_GOURADE))
					{
						if( (colorlist = (RGB *)realloc( colorlist, (colors+1) * sizeof( RGB))) == NULL)
							return( FALSE);
						col = v3o->material[i].color;
				 		//colorlist[ colors++]= v3o->material[ i].color;
						colorlist[ colors].r = col.r;
						colorlist[ colors].g = col.g;
						colorlist[ colors].b = col.b;
						colors++;
					}

				}
			}
			object = object->next;
		}

		memset( palette, 0, sizeof(RGB) * 256);
		makepalette( palette, colorlist, colors);
    palette[0].r = palette[0].g = palette[0].b = 0;
    palette[255].r = palette[255].g = palette[255].b = 255;

		xfree( colorlist);

		if( lighttab != NULL)
		{
      RGB ambient = v3d->ambient, light = v3d->maxlight;
			inittab();
			addlighttab( lighttab, ambient, light);
			if( !maketab( palette))
			{
				xfree( imagelist);
				return( FALSE);
			}
		}
    palette[0].r = palette[0].g = palette[0].b = 0;
    palette[255].r = palette[255].g = palette[255].b = 255;

		object = v3d->obj;
		while( object != NULL)
		{
			if( object->type & T_V3O)
			{
				V3O *v3o = object->v3o;

				for( i=0; i<v3o->nummaterial; i++)
					if( !(v3o->material[ i].flags & V3OMATERIALFLAG_TEXTURE))
						if( v3o->material[ i].flags & (V3OMATERIALFLAG_FLAT | V3OMATERIALFLAG_GOURADE))
						{
							co.r = v3o->material[i].color.r;
							co.g = v3o->material[i].color.g;
							co.b = v3o->material[i].color.b;

							v3o->material[ i].ci = findcolor( palette, co);
						}
			}
			object = object->next;
		}

		for( i=0; i<images; i++)
			remapdata( palette, imagelist[i]->palette, imagelist[i]->image8, imagelist[i]->width, imagelist[i]->height);

		xfree( imagelist);

	}

	return( TRUE);
}



