//-------------------------------------------------------------------------
//
// File:		DeviceContextC.h
// Desc:		Device context class.
// Author:		memon <memon@inside.org>
//
//-------------------------------------------------------------------------
//	Copyright (c) 2000-2002 Moppi Productions. All Rights Reserved.
//  This file is part of Moppi Demopaja SDK. For conditions of 
//  distribution and use, see the accompanying license.txt file.
//  http://moppi.inside.org/demopaja/
//-------------------------------------------------------------------------

#ifndef __DEMOPAJA_DEVICECONTEXTC_H__
#define __DEMOPAJA_DEVICECONTEXTC_H__

#pragma warning( disable : 4786 )		// long names generated by STL

#include "PajaTypes.h"
#include "DeviceInterfaceI.h"
#include "ClassIdC.h"
#include <vector>


namespace PajaSystem {

	//! Device context class.
	/*!	The purpose of the device context class is to gather all
		the device interfaces available to the effects. The interfaces
		may range from renderin APIs (such as OpenGL, or Direct3D) to
		music system APIs. The interface holds just enough information
		to use the API and some helper methods to help the use of
		the API under the editor. Device context simply holds these
		interfaces. These interfaces can be queried from the context. 

		This class is implemented by the system. 
	*/
	class DeviceContextC
	{
	public:
		//! Default constructor.
		/*!	Initialises the structures used by the device context. Used internally. */
		DeviceContextC();

		//! Default destructor.
		/*! The destructor. Releases all resgitered interfaces. Used internally. */
		virtual ~DeviceContextC();

		//! Query specified device interface.
		/*!	\param rClassId ID of the interface to query. 
			\return Pointer to existing interface. If no interface was found the return value is NULL.

			This is the only method user needs to call from this class. It makes a query to
			the list inside the device context to see if the queried interface
			is available. If it is available pointer to the interface will be
			returned, otherwise the return value is NULL. 

			The class IDs for the interfaces are defined in the header files of the interfaces. 

			Example:
			\code
			void
			TestEffectC::do_frame( DeviceContextC* pContext )
			{

				// Get the OpenGL device.
				OpenGLDeviceC*	pDevice = (OpenGLDeviceC*)pContext->query_interface( CLASS_OPENGL_DEVICEDRIVER );
				if( !pDevice ) {
					// No device available... bail out!
					return;
				}

				// do the effect...
			}
			\endcode
		*/
		virtual DeviceInterfaceI*	query_interface( const PluginClass::ClassIdC& rClassId );

		//! Queries an interface based on super class ID.
		/*!	Example:

			\code
			...
			GraphicsDeviceI*	pGraphDevice = (GraphicsDeviceI*)pContext->query_interface( SUPERCLASS_GRAPHICSDEVICE );
			...
			\endcode
		*/
		
		virtual DeviceInterfaceI*	query_interface( const PluginClass::SuperClassIdC& rSClassId );

		//! Register device interface
		/*! The system (or the Demopaja player) uses this method internally to register device interfaces to the device context. */
		virtual void				register_interface( DeviceInterfaceI* pInterface );

		//! Unregister device interface
		/*! The system (or the Demopaja player) uses this method internally to register device interfaces to the device context. */
		virtual void				unregister_interface( const PluginClass::ClassIdC& rClassId );

		//! Returns number of interfaces in the device context.
		virtual PajaTypes::uint32	get_interface_count() const;

		//! Returns specified interface.
		virtual DeviceInterfaceI*	get_interface( PajaTypes::uint32 ui32Index );

	private:
		std::vector<DeviceInterfaceI*>	m_rInterfaces;
	};

}; // namespace

#endif // __DEMOPAJA_DEVICECONTEXTC_H__