#ifndef __DATABLOCKI_H__
#define __DATABLOCKI_H__


namespace Edit {

	//! Plugin data interface.
	/*! Data block class is base class for every object created in a plugin class in
		a DLL and returned to the Demopaja system. Data block class takes care that
		the memory management of the main application and the plugin won't corrupt.
		This class also hides the constructor and destructor to prevent the misuse
		of the classes derived from this class. 
	*/
	class DataBlockI
	{
	public:
		//! Creates new datablock.
		/*!	The create method of data block class creates new instance of the same class the data block is. 

			Example implementation:
			\code
			DataBlockI*
			TGAImportC::create()
			{
				return new TGAImportC;
			}
			\endcode

			\return Pointer to a new instance of this class. 
		*/
		virtual DataBlockI*			create() = 0;

		//! Deletes data block.
		/*!	This method should delete all the contents of this class.
			The default implementation is shown below. So in practice,
			you can implement the desctructor method to release the
			data as usual. 

			The default implementation of this method is: 

			\code
			void
			DataBlockI::release()
			{
				delete this;
			}
			\endcode
		*/
		virtual void				release();

		//! Deep copy from a data block.
		/*!	\param pBlock Pointer to datablock of same class to data from.
		
			This method should make a deep copy (that is, copy everything)
			of the instance specified by the argument.

			\code
			Example Implementation
			void
			TGAImportC::copy( DataBlockI* pBlock )
			{
				TGAImportC* pFile = (TGAImportC*)pBlock;

				m_i32Width = pFile->m_i32Width;
				m_i32Height = pFile->m_i32Height;
				m_i32Bpp = pFile->m_i32Bpp;
				m_sFileName = pFile->m_sFileName;

				// duplicate data
				delete m_pData;
				uint32 ui32DataSize = m_i32Width * m_i32Height * (m_i32Bpp / 8);
				m_pData = new uint8[ui32DataSize];
				memcpy( m_pData, pFile->m_pData, ui32DataSize );
			}
			\endcode
		*/
		virtual void				copy( DataBlockI* pBlock ) = 0;

	protected:
		//! Protected default constructor.
		DataBlockI();
		//! Protected default desctructor.
		virtual ~DataBlockI();
	};

};	// namespace

#endif // __DATABLOCKI_H__
