//	vectors.cpp

#include <math.h>
#include "vectors.h"

// sin and cos tables
double SinT[NumDegrees], CosT[NumDegrees];

// fast float to int conversion
#ifndef DTOI_MAGIK
#define DTOI_MAGIK ((((65536.0 * 65536.0 * 16) + (65536.0 * 0.5)) * 65536.0))

int dtoi(double n) {
	double temp = DTOI_MAGIK + n;
	return ((*(int *)&temp) - 0x80000000);
}
#endif



// vector class implementation

void TVector::Normalize() {
	double magn = Magnitude();
	if (magn != 0.0) {
		x /= magn;
		y /= magn;
		z /= magn;
	}
}

double TVector::Magnitude() {
	return sqrt(x * x + y * y + z * z);
}


void TVector::Rotate(int ax, int ay, int az) {
	double nx = x;
	double ny = y;
	double nz = z;
	
	if (ax < 0) ax += NumDegrees;
	if (ay < 0) ay += NumDegrees;
	if (az < 0) az += NumDegrees;
	
	if (ax) {
		ny = y * CosT[ax] - z * SinT[ax];
		nz = y * SinT[ax] + z * CosT[ax];
		y = ny;
		z = nz;
	}
	
	if (ay) {
		nx = x * CosT[ay] - z * SinT[ay];
		nz = x * SinT[ay] + z * CosT[ay];
		x = nx;
		z = nz;
	}
	
	if (az) {
		nx = x * CosT[az] - y * SinT[az];
		ny = x * SinT[az] + y * CosT[az];
		x = nx;
		y = ny;
	}
}


// common vector operations

double DotProduct(TVector a, TVector b) {
	return a.x * b.x + a.y * b.y + a.z * b.z;
}


TVector CrossProduct(TVector a, TVector b) {
	TVector r;
	r.x = a.y * b.z - a.z * b.y;
	r.y = b.x * a.z - a.x * b.z;
	r.z = a.x * b.y - a.y * b.x;
	return r;
}

TVector operator *(TVector a, TVector b) {
	TVector r;
	r.x = a.y * b.z - a.z * b.y;
	r.y = b.x * a.z - a.x * b.z;
	r.z = a.x * b.y - a.y * b.x;
	return r;
}


TVector AddVectors(TVector a, TVector b) {
	TVector r;
	r.x = a.x + b.x;
	r.y = a.y + b.y;
	r.z = a.z + b.z;
	return r;
}

TVector operator +(TVector a, TVector b) {
	TVector r;
	r.x = a.x + b.x;
	r.y = a.y + b.y;
	r.z = a.z + b.z;
	return r;
}

TVector operator +=(TVector &a, TVector b) {
	a.x += b.x;
	a.y += b.y;
	a.z += b.z;
	return a;
}


TVector SubVectors(TVector a, TVector b) {
	TVector r;
	r.x = a.x - b.x;
	r.y = a.y - b.y;
	r.z = a.z - b.z;
	return r;
}

TVector operator -(TVector a, TVector b) {
	TVector r;
	r.x = a.x - b.x;
	r.y = a.y - b.y;
	r.z = a.z - b.z;
	return r;
}

TVector operator -=(TVector &a, TVector b) {
	a.x -= b.x;
	a.y -= b.y;
	a.z -= b.z;
	return a;
}


TVector MulVector(TVector v, double k) {
	TVector r;
	r.x = k * v.x;
	r.y = k * v.y;
	r.z = k * v.z;
	return r;
}

TVector operator *(TVector v, double k) {
	TVector r;
	r.x = k * v.x;
	r.y = k * v.y;
	r.z = k * v.z;
	return r;
}

TVector operator *(double k, TVector v) {
	TVector r;
	r.x = k * v.x;
	r.y = k * v.y;
	r.z = k * v.z;
	return r;
}


TVector DivVector(TVector v, double k) {
	TVector r;
	r.x = v.x / k;
	r.y = v.y / k;
	r.z = v.z / k;
	return r;
}

TVector operator /(TVector v, double k) {
	TVector r;
	r.x = v.x / k;
	r.y = v.y / k;
	r.z = v.z / k;
	return r;
}


TVector CopyVector(TVector source) {
	TVector r;
	r.x = source.x;
	r.y = source.y;
	r.z = source.z;
	return r;
}


// TVertex class implementation

TVertex::TVertex() {
	SetP(0, 0, 0);
	normal.SetP(0, 0, 0);
}

TVertex::TVertex(double nx, double ny, double nz) {
	SetP(nx, ny, nz);
	normal.SetP(0, 0, 0);
	u = 0;
	v = 0;
}


void TVertex::CalcEnvUV() {
	u = (normal.x + 1.0) / 2.0;
	v = (normal.y + 1.0) / 2.0;
}


void TVertex::VRotate(int ax, int ay, int az) {
	Rotate(ax, ay, az);
	normal.Rotate(ax, ay, az);
}


void TVertex::FakeNormal() {
	normal.x = x;
	normal.y = y;
	normal.z = z;
	normal.Normalize();
}


// setup trig tables
void CalcTrigTables() {
	for (int ang = 0; ang < NumDegrees; ang++) {
		SinT[ang] = sin(ang * 2.0 * Pi / (double)NumDegrees);
		CosT[ang] = cos(ang * 2.0 * Pi / (double)NumDegrees);
	}
}

