#include "o3d.h"
#include "tables.h"
#include <string.h>
#include <stdlib.h>

extern unsigned int isqrt(unsigned int);

#define MINZ (-MULT/8)

const int NEGATIVE_INFINITY=0x80808080;

VERTEX3D		vertices[MAX_VERTICES];
VECTOR2D		drawvert[MAX_VERTICES];
REFTRI3D		triangles[MAX_TRIANGLES];
VECTOR3D		tnorms[MAX_TRIANGLES];
TRIANGLE3D_DATA	*overloads[MAX_TRIANGLES];
unsigned char	vis[MAX_VERTICES];
int				vlights[MAX_VERTICES];
VECTOR3D		vnorms[MAX_VERTICES];
int				tz[MAX_TRIANGLES];

VERTEX3D		stars[MAX_STARS];

unsigned short	*view3d;
NODE3D			main3d;
VECTOR3D		light3d,tlight3d;
int				proj3d;
COLOR3D			col3d;
int				gdMin;
int				zMin;
int				zMax;
int				nbStars;
int				zStars;
TRIANGLE3D_TYPE	shading3d;

unsigned short	_curv,_curt,_curt2;

int				zbuf[SIZE];

int ly0[WIDTH];
int ly1[WIDTH];
int ly2[WIDTH];

int lz0[WIDTH];
int lz1[WIDTH];
int lz2[WIDTH];

int lr0[WIDTH];
int lr1[WIDTH];
int lg0[WIDTH];
int lg1[WIDTH];
int lb0[WIDTH];
int lb1[WIDTH];

int tempZ[WIDTH];
int tempR[WIDTH];
int tempG[WIDTH];
int tempB[WIDTH];

/**
 * MACROS
 */

#define MIN(a,b) (a<b?a:b)
#define MAX(a,b) (a>b?a:b)

#define GPCOLU(a) ((a)>247?248:((a)&0xf8)+(((a)&4)<<1))
#define GPCOL(r,g,b) ((GPCOLU(r)<<8)|(GPCOLU(g)<<3)|(GPCOLU(b)>>2))
#define GPCOL3D(c) GPCOL((c)->r,(c)->g,(c)->b)

/**
 * Set the frame buffer to fill
 */

NODE3D* o3d_init(SCENE3D *s,int proj)
{
	gm_memset(&main3d,0,sizeof(NODE3D));
	main3d.data.s = s;
	main3d.type = SCENE;
	proj3d = proj;
	col3d.r = 127;
	col3d.g = 127;
	col3d.b = 127;
	light3d.x = -1000;
	light3d.y = 1000;
	light3d.z = 1000;
	gdMin = -1000;
	zMin = -1500;
	zMax = 500;
	nbStars = 0;
	shading3d = FLAT;
	return &main3d;
}

VECTOR3D *o3d_setLightPos(int x,int y,int z)
{
	light3d.x = x;
	light3d.y = y;
	light3d.z = z;
	return &light3d;
}

COLOR3D *o3d_setLightColor(unsigned char r,unsigned char g,unsigned char b)
{
	col3d.r = r;
	col3d.g = g;
	col3d.b = b;
	return &col3d;
}

void o3d_setGDMin(int min)
{
	gdMin = min;
}

void o3d_setDepths(int zmin,int zmax)
{
	zMin = zmin;
	zMax = zmax;
}

void o3d_setShading(TRIANGLE3D_TYPE s)
{
	shading3d = s;
}

NODE3D* o3d_addObject(SCENE3D *s,OBJECT3D* o,int x,int y,int z,unsigned short rx,unsigned short ry,unsigned short rz)
{
	return o3d_addNode(s,o,OBJECT,x,y,z,rx,ry,rz);
}

NODE3D* o3d_addScene(SCENE3D *s,SCENE3D* sc,int x,int y,int z,unsigned short rx,unsigned short ry,unsigned short rz)
{
	return o3d_addNode(s,sc,SCENE,x,y,z,rx,ry,rz);
}

NODE3D* o3d_addSprite(SCENE3D *s,TEXTURE3D* sp,int x,int y,int z)
{
	return o3d_addNode(s,sp,SPRITE,x,y,z,0,0,0);
}

NODE3D* o3d_addNode(SCENE3D *s,void* n,NODE3D_TYPE t,int x,int y,int z,unsigned short rx,unsigned short ry,unsigned short rz)
{
	s->nodes[s->count].type		= t;
	s->nodes[s->count].data.n	= n;
	s->nodes[s->count].pos.x	= x;
	s->nodes[s->count].pos.y	= y;
	s->nodes[s->count].pos.z	= z;
	s->nodes[s->count].rot.rx	= rx;
	s->nodes[s->count].rot.ry	= ry;
	s->nodes[s->count].rot.rz	= rz;
	s->nodes[s->count].overload = NULL;
	return &s->nodes[s->count++];
}

void o3d_computeFrame(unsigned short *v)
{
	unsigned short j;

	view3d = v;

	gm_memset(zbuf,0x80,SIZE*sizeof(int));

	/* compute transformations */
	_curt = 0;
	transformNode(&main3d,NULL);
	multiplyVector(main3d.m,&light3d,&tlight3d);
	_curv  = 0;
	_curt  = 0;
	_curt2 = 0;
	gm_memset(vis,0,MAX_VERTICES);
	transformVertices(&main3d);

	/* drawing */
	for (j=0;j<_curt;j++)
		drawTriangle(j);

	if (nbStars>0)
		showStars();
}

/**
 * Stars initialization
 */

void o3d_initStars(int nb,int r)
{
	int i,r2=r<<1;
	nbStars = nb;
	zStars = r;
	for (i=0;i<nb;i++)
	{
		stars[i].x = (rand()%r2)-r;
		stars[i].y = (rand()%r2)-r;
		stars[i].z = (rand()%r2)-r;
	}
}

/**
 * Matrice initialisation
 */

void initMatrix(MATRIX3D m,MATRIX_TYPE t,short v0,short v1,short v2)
{
	int i;
	gm_memset(m,0,sizeof(int)*16);
	switch(t)
	{
		case VAL:
			for (i=0;i<16;m[i++]=v0);
			break;
		case ROTX:
			m[0]  =  MULT;
			m[5]  =  tcos[v0];
			m[6]  =  tsin[v0];
			m[9]  = -tsin[v0];
			m[10] =  tcos[v0];
			m[15] =  MULT;
			break;
		case ROTY:
			m[0]  =  tcos[v0];
			m[2]  = -tsin[v0];
			m[5]  =  MULT;
			m[8]  =  tsin[v0];
			m[10] =  tcos[v0];
			m[15] =  MULT;
			break;
		case ROTZ:
			m[0]  =  tcos[v0];
			m[1]  =  tsin[v0];
			m[4]  = -tsin[v0];
			m[5]  =  tcos[v0];
			m[10] =  MULT;
			m[15] =  MULT;
			break;
		case ROTXYZ:
			{
				int cx = tcos[v0],cy = tcos[v1],cz = tcos[v2],
					sx = tsin[v0],sy = tsin[v1],sz = tsin[v2];
				m[0]  = (cy*cz)>>SHIFT;
				m[1]  = (cy*sz)>>SHIFT;
				m[2]  = -sy;
				m[4]  = (((sx*sy*cz)>>SHIFT)-(cx*sz))>>SHIFT;
				m[5]  = (((sx*sy*sz)>>SHIFT)+(cx*cz))>>SHIFT;
				m[6]  = (sx*cy)>>SHIFT;
				m[8]  = (((cx*sy*cz)>>SHIFT)+(sx*sz))>>SHIFT;
				m[9]  = (((cx*sy*sz)>>SHIFT)-(sx*cz))>>SHIFT;
				m[10] = (cx*cy)>>SHIFT;
				m[15] = MULT;
			}
			break;
		case SCALE:
			m[0]  = v0;
			m[5]  = v1;
			m[10] = v2;
			m[15] = MULT;
			break;
		case TRANS:
			m[12] = v0<<SHIFT;
			m[13] = v1<<SHIFT;
			m[14] = v2<<SHIFT;
			m[15] = MULT;
			break;
		case IDENT:
			m[0]  = MULT;
			m[5]  = MULT;
			m[10] = MULT;
			m[15] = MULT;
			break;
		case PROJ:
			m[0]  = MULT;
			m[5]  = MULT;
			m[10] = MULT;
			m[12] = v0<<SHIFT;
			m[15] = MULT;
			break;
	}
}

/**
 * Rotation and Trans Matrix initialisation
 */

void initRotTransMatrix(MATRIX3D m,ROTATION3D *rot,VECTOR3D *pos)
{
	initMatrix(m,ROTXYZ,rot->rx,rot->ry,rot->rz);
	m[12] = pos->x<<SHIFT;
	m[13] = pos->y<<SHIFT;
	m[14] = pos->z<<SHIFT;
}

/**
 * Matrice multiplication r = a*b (with a!=r && b!=r)
 */

void multiplyMatrix(MATRIX3D a,MATRIX3D b,MATRIX3D r)
{
	int i,j,k;
	gm_memset(r,0,sizeof(int)*16);
	for(i=0;i<4;i++)
		for(j=0;j<4;j++)
			for(k=0;k<4;k++)
				r[i*4+j] += a[k*4+j]*b[i*4+k];
	for(i=0;i<16;i++)
		r[i]>>=SHIFT;
}

/**
 * Vertex multiplication r = m*v (with v==r || v!=r)
 */
 
void multiplyVector(MATRIX3D m,VECTOR3D *v,VECTOR3D *r)
{
    int x = v->x * m[0] + v->y * m[4] + v->z *m[8]  + m[12];
    int y = v->x * m[1] + v->y * m[5] + v->z *m[9]  + m[13];
    r->z = (v->x * m[2] + v->y * m[6] + v->z *m[10] + m[14])>>SHIFT;
    r->x = x>>SHIFT;
    r->y = y>>SHIFT;
}

/**
 * Length of vector
 */

int length(VECTOR3D *v)
{
	return isqrt((v->x*v->x)+(v->y*v->y)+(v->z*v->z));
}

/**
 * Length of vector to 1 (MULT)
 */

void normalize(VECTOR3D *v)
{
	int l = length(v);
	v->x = (v->x<<SHIFT)/l;
	v->y = (v->y<<SHIFT)/l;
	v->z = (v->z<<SHIFT)/l;
}

/**
 * Compute all 3D scene transformations matrices + triangle normals
 */

_MATRIX3D(tempMat);

void transformNode(NODE3D *node,MATRIX3D m0)
{
	unsigned short i;
	MATRIX3D m = m0==NULL ? node->m : tempMat;
	VERTEX3D v;

	initRotTransMatrix(m,&node->rot,&node->pos);
	if (m0!=NULL)
		multiplyMatrix(m0,m,node->m);
	if (node->type==SCENE)
	{
		SCENE3D *s=node->data.s;
		for (i=0;i<s->count;i++)
			transformNode(&s->nodes[i],node->m);
	}
	else if (node->type==OBJECT)
	{
		OBJECT3D *o = node->data.o;
		VECTOR3D *n = o->tnorm;
		int x,y,mask=0;
		for (i=0;(i<8) && (mask!=15);i++)
		{
			multiplyVector(node->m,&o->bbox[i],&v);
			x = ((v.x*proj3d)/(proj3d-v.z))+WIDTH2;
			y = ((v.y*proj3d)/(proj3d-v.z))+HEIGHT2;
			if (x>=0&&x<WIDTH&&y>=0&&y<HEIGHT)
				break;
			mask |= (x>=0 ? 1 : 0) | (x<WIDTH ? 2 : 0) | (y>=0 ? 4 : 0) | (y<HEIGHT ? 8 : 0);
		}
		if (i==8 && (mask!=15))
		{
			node->vis=0;
			return;
		}
		for (i=0;i<o->nbt;i++)
			multiplyVector(node->m,&n[i],&tnorms[_curt++]);
		node->vis=1;
	}
}

/**
 * Transform all visible vertices
 */

void transformVertices(NODE3D *node)
{
	int i;
	if (node->type==SCENE)
	{
		SCENE3D *s=node->data.s;
		for (i=0;i<s->count;i++)
			transformVertices(&s->nodes[i]);
	}
	else if (node->type==SPRITE)
	{
		VERTEX3D v;
		TEXTURE3D *t = node->data.t;
		int j,k,l,x,y,w,h,fx,fy,lx,ly,dx,tw,th,dk,dl,fl;
		int w2 = (t->w+1)>>1;
		int h2 = (t->h+1)>>1;
		int trans = t->trans-1,pix;

		multiplyVector(node->m,&node->pos,&v);

		j = (((v.x-w2)*proj3d)/(proj3d-v.z))+WIDTH2;
		k = (((v.x+w2)*proj3d)/(proj3d-v.z))+WIDTH2;
		if (j==k)
			return;
		if (j>k)
		{
			x = k;
			w = j-k;
		}
		else
		{
			x = j;
			w = k-j;
		}
		j = (((v.y-h2)*proj3d)/(proj3d-v.z))+HEIGHT2;
		k = (((v.y+h2)*proj3d)/(proj3d-v.z))+HEIGHT2;
		if (j==k)
			return;
		if (j>k)
		{
			y = k;
			h = j-k;
		}
		else
		{
			y = j;
			h = k-j;
		}

		if ((x>=WIDTH)||(x+w<0)||(y>=HEIGHT)||(y+h<0))
			return;

		fx = MAX(x,0);
		lx = x+w+1;
		lx = MIN(lx,WIDTH);
		fy = MAX(y,0);
		ly = y+h+1;
		ly = MIN(ly,HEIGHT);
		dx = fx*HEIGHT;
		tw = t->w+1;
		th = t->h+1;
		dk = (t->w<<8)/w;
		k = (fx-x)*dk;
		dl = (t->h<<8)/h;
		fl = (h-(fy-y))*dl;
		for (i=fx;i<lx;i++)
		{
			l = fl;
			for (j=fy;j<ly;j++)
			{
				if (zbuf[dx+j]<v.z)
				{
					
					pix = t->pix[((l>>8)*tw)+(k>>8)];
					if (trans!=pix)
					{
						view3d[dx+j] = t->gpal[pix];
						zbuf[dx+j] = v.z;
					}
				}
				l-=dl;
			}
			k+=dk;
			dx+=HEIGHT;
		}
	}
	else if (node->vis)
	{
		OBJECT3D   *o = node->data.o;
		VERTEX3D   *v = o->vert;
		TRIANGLE3D *t = o->tri;
		VECTOR3D   *n = o->vnorm;

		for (i=0;i<o->nbt;i++)
		{
			if (tnorms[_curt2].z>=MINZ)
			{
				vis[triangles[_curt].a  = t[i].a+_curv] |= t[i].type;
				vis[triangles[_curt].b  = t[i].b+_curv] |= t[i].type;
				vis[triangles[_curt].c  = t[i].c+_curv] |= t[i].type;
				if (_curt<_curt2)
				{
					tnorms[_curt].x=tnorms[_curt2].x;
					tnorms[_curt].y=tnorms[_curt2].y;
					tnorms[_curt].z=tnorms[_curt2].z;
				}
				overloads[_curt] = node->overload;
				triangles[_curt++].orig = &t[i];
			}
			_curt2+=1;
		}
		for (i=0;i<o->nbv;i++)
		{
			if (vis[_curv])
			{
				multiplyVector(node->m,&v[i],&vertices[_curv]);
				drawvert[_curv].x = WIDTH2-((vertices[_curv].x*proj3d)/(proj3d-vertices[_curv].z));
				drawvert[_curv].y = ((vertices[_curv].y*proj3d)/(proj3d-vertices[_curv].z))+HEIGHT2;
				if (shading3d==GOURAUD)
				{
					multiplyVector(node->m,&n[i],&vnorms[_curv]);
					vlights[_curv] = computeLight(&vertices[_curv],&vnorms[_curv]);
				}
			}
			_curv+=1;
		}
		for (i=0;i<_curt;i++)
			tz[i] = (vertices[triangles[i].a].z+vertices[triangles[i].b].z+vertices[triangles[i].c].z)/3;
	}
}

/**
 * Compute light
 */

VECTOR3D vlight;

int computeLight(VERTEX3D *v,VECTOR3D *n)
{
	vlight.x = tlight3d.x-v->x;
	vlight.y = tlight3d.y-v->y;
	vlight.z = tlight3d.z-v->z;
	normalize(&vlight);
	vlight.x += n->x;
	vlight.y += n->y;
	vlight.z += n->z;
	return length(&vlight);
}

/**
 *
 */

#define LIGHT(a,b,l) (l>MULT? a+((b*(l-MULT))>>SHIFT) : ((a*l)>>SHIFT))

void computeColor(int l,COLOR3D *col,COLOR3D *res)
{
	int c = LIGHT(col->r,col3d.r,l);
	res->r = MIN(c,255);
	c = LIGHT(col->g,col3d.g,l);
	res->g = MIN(c,255);
	c = LIGHT(col->b,col3d.b,l);
	res->b = MIN(c,255);
}

/**
 *
 */

void computeLights(unsigned short a,unsigned short b,int *lr,int *lg,int *lb,COLOR3D *col)
{
	COLOR3D ca,cb;

	computeColor(vlights[a],col,&ca);
	computeColor(vlights[b],col,&cb);

	line(drawvert[a].x,ca.r,drawvert[b].x,cb.r,lr);
	line(drawvert[a].x,ca.g,drawvert[b].x,cb.g,lg);
	line(drawvert[a].x,ca.b,drawvert[b].x,cb.b,lb);
}

/**
 * Draw a "flat light" triangle
 */

void drawFlatTriangle(unsigned short a,unsigned short b,unsigned short c,COLOR3D *col)
{
	int ax = MAX(drawvert[a].x,0);
	int bx = MIN(drawvert[b].x,WIDTH-1);
	int cx = MIN(drawvert[c].x,WIDTH-1);
	int i,j,ymin,ymax;
	int *y0,*y1,*z0,*z1;
	int dx = ax*HEIGHT;
	unsigned short nc = GPCOL3D(col);

	if ((ly0[ax]<ly1[ax]) || (ly0[bx]<ly1[bx]))
	{
		y0 = ly0;
		y1 = ly1;
		z0 = lz0;
		z1 = lz1;
	}
	else
	{
		y0 = ly1;
		y1 = ly0;
		z0 = lz1;
		z1 = lz0;
	}
	for (i=ax;i<=bx;i++)
	{
		ymin = MAX(y0[i],0);
		ymax = MIN(y1[i],HEIGHT-1);
		if (ymin<HEIGHT && ymax>=0)
		{
			line(y0[i],z0[i],y1[i],z1[i],tempZ);
			for (j=ymin;j<=ymax;j++)
			{
				if (tempZ[j]>zbuf[dx+j])
				{
					view3d[dx+j] = nc;
					zbuf[dx+j] = tempZ[j];
				}
			}
		}
		dx+=HEIGHT;
	}
	if (bx<cx)
	{
		if (ly0[i]<ly2[i])
		{
			y0 = ly0;
			y1 = ly2;
			z0 = lz0;
			z1 = lz2;
		}
		else
		{
			y0 = ly2;
			y1 = ly0;
			z0 = lz2;
			z1 = lz0;
		}
		for (i=ax==bx?bx:i;i<=cx;i++)
		{
			ymin = MAX(y0[i],0);
			ymax = MIN(y1[i],HEIGHT-1);
			if (ymin<HEIGHT && ymax>=0)
			{
				line(y0[i],z0[i],y1[i],z1[i],tempZ);
				for (j=ymin;j<=ymax;j++)
				{
					if (tempZ[j]>zbuf[dx+j])
					{
						view3d[dx+j] = nc;
						zbuf[dx+j] = tempZ[j];
					}
				}
			}
			dx+=HEIGHT;
		}
	}
}

/**
 * Draw a "gouraud light" triangle
 */

void drawGouraudTriangle(unsigned short a,unsigned short b,unsigned short c,COLOR3D *col)
{
	int ax = MAX(drawvert[a].x,0);
	int bx = MIN(drawvert[b].x,WIDTH-1);
	int cx = MIN(drawvert[c].x,WIDTH-1);
	int i,j,ymin,ymax;
	int *y0,*y1,*z0,*z1,*r0,*r1,*g0,*g1,*b0,*b1;
	int dx = ax*HEIGHT;

	computeLights(a,c,lr0,lg0,lb0,col);
	computeLights(a,b,lr1,lg1,lb1,col);

	if ((ly0[ax]<ly1[ax]) || (ly0[bx]<ly1[bx]))
	{
		y0 = ly0;
		y1 = ly1;
		z0 = lz0;
		z1 = lz1;
		r0 = lr0;
		r1 = lr1;
		g0 = lg0;
		g1 = lg1;
		b0 = lb0;
		b1 = lb1;
	}
	else
	{
		y0 = ly1;
		y1 = ly0;
		z0 = lz1;
		z1 = lz0;
		r0 = lr1;
		r1 = lr0;
		g0 = lg1;
		g1 = lg0;
		b0 = lb1;
		b1 = lb0;
	}
	for (i=ax;i<=bx;i++)
	{
		ymin = MAX(y0[i],0);
		ymax = MIN(y1[i],HEIGHT-1);
		if (ymin<HEIGHT && ymax>=0)
		{
			line(y0[i],z0[i],y1[i],z1[i],tempZ);
			line(y0[i],r0[i],y1[i],r1[i],tempR);
			line(y0[i],g0[i],y1[i],g1[i],tempG);
			line(y0[i],b0[i],y1[i],b1[i],tempB);
			for (j=ymin;j<=ymax;j++)
			{
				if (tempZ[j]>zbuf[dx+j])
				{
					view3d[dx+j] = GPCOL(tempR[j],tempG[j],tempB[j]);
					zbuf[dx+j] = tempZ[j];
				}
			}
		}
		dx+=HEIGHT;
	}
	if (bx<cx)
	{
		computeLights(b,c,lr1,lg1,lb1,col);
		if (ly0[i]<ly2[i])
		{
			y0 = ly0;
			y1 = ly2;
			z0 = lz0;
			z1 = lz2;
			r0 = lr0;
			r1 = lr1;
			g0 = lg0;
			g1 = lg1;
			b0 = lb0;
			b1 = lb1;
		}
		else
		{
			y0 = ly2;
			y1 = ly0;
			z0 = lz2;
			z1 = lz0;
			r0 = lr1;
			r1 = lr0;
			g0 = lg1;
			g1 = lg0;
			b0 = lb1;
			b1 = lb0;
		}
		for (i=ax==bx?bx:i;i<=cx;i++)
		{
			ymin = MAX(y0[i],0);
			ymax = MIN(y1[i],HEIGHT-1);
			if (ymin<HEIGHT && ymax>=0)
			{
				line(y0[i],z0[i],y1[i],z1[i],tempZ);
				line(y0[i],r0[i],y1[i],r1[i],tempR);
				line(y0[i],g0[i],y1[i],g1[i],tempG);
				line(y0[i],b0[i],y1[i],b1[i],tempB);
				for (j=ymin;j<=ymax;j++)
				{
					if (tempZ[j]>zbuf[dx+j])
					{
						view3d[dx+j] = GPCOL(tempR[j],tempG[j],tempB[j]);
						zbuf[dx+j] = tempZ[j];
					}
				}
			}
			dx+=HEIGHT;
		}
	}
}

/**
 *
 */

int determinate(int *a,int *b,int *c)
{
	return 	 (a[0]*b[1]*c[2]
			+ b[0]*c[1]*a[2]
			+ c[0]*a[1]*b[2]
			- a[2]*b[1]*c[0]
			- b[2]*c[1]*a[0]
			- c[2]*a[1]*b[0]);
}

/**
 * Draw a "flat light" textured triangle
 */

unsigned short tempGPal[256];
int detA[3],detB[3],detC[3] = {1,1,1},detD[3];

void drawFlatTexturedTriangle(	unsigned short a,unsigned short b,unsigned short c,
								unsigned char oa,unsigned char ob,unsigned char oc,
								int l,TEXTURE3D_DATA* td)
{
	TEXTURE3D *t = td->t;
	int ax = MAX(drawvert[a].x,0);
	int bx = MIN(drawvert[b].x,WIDTH-1);
	int cx = MIN(drawvert[c].x,WIDTH-1);
	int i,j,ymin,ymax;
	int *y0,*y1,*z0,*z1;
	int dx = ax*HEIGHT;
	COLOR3D tempCol;
	int	dDet;
	int ux,vx,wx;
	int uy,vy,wy;
	int	dw=t->w+1,dh=t->h+1,x,y,pix,trans=t->trans-1;
	unsigned short *pal;

	if (l!=NEGATIVE_INFINITY)
	{
		for (i=0;i<=t->nbc;i++)
		{
			computeColor(l,&t->pal[i],&tempCol);
			tempGPal[i] = GPCOL3D(&tempCol);
		}
		pal = tempGPal;
	}
	else
		pal = t->gpal;

	detA[0] = drawvert[a].x;
	detA[1] = drawvert[b].x;
	detA[2] = drawvert[c].x;

	detB[0] = drawvert[a].y;
	detB[1] = drawvert[b].y;
	detB[2] = drawvert[c].y;

	detD[0] = td->x[oa];
	detD[1] = td->x[ob];
	detD[2] = td->x[oc];

	dDet = determinate(detA,detB,detC);
	if (dDet==0)
		return;

	ux = determinate(detD,detB,detC);
	vx = determinate(detA,detD,detC);
	wx = determinate(detA,detB,detD);

	detD[0] = td->y[oa];
	detD[1] = td->y[ob];
	detD[2] = td->y[oc];

	uy = determinate(detD,detB,detC);
	vy = determinate(detA,detD,detC);
	wy = determinate(detA,detB,detD);

	if ((ly0[ax]<ly1[ax]) || (ly0[bx]<ly1[bx]))
	{
		y0 = ly0;
		y1 = ly1;
		z0 = lz0;
		z1 = lz1;
	}
	else
	{
		y0 = ly1;
		y1 = ly0;
		z0 = lz1;
		z1 = lz0;
	}
	for (i=ax;i<=bx;i++)
	{
		ymin = MAX(y0[i],0);
		ymax = MIN(y1[i],HEIGHT-1);
		if (ymin<HEIGHT && ymax>=0)
		{
			line(y0[i],z0[i],y1[i],z1[i],tempZ);
			for (j=ymin;j<=ymax;j++)
			{
				if (tempZ[j]>zbuf[dx+j])
				{
					x = ((i*ux+j*vx+wx)/dDet);
					y = ((i*uy+j*vy+wy)/dDet);
					pix = t->pix[(x<0?0:(x<dw?x:0))+(y<0?0:(y<dh?y*dw:0))];
					if (pix!=trans)
					{
						view3d[dx+j] = pal[pix];
						zbuf[dx+j] = tempZ[j];
					}
				}
			}
		}
		dx+=HEIGHT;
	}
	if (bx<cx)
	{
		if (ly0[i]<ly2[i])
		{
			y0 = ly0;
			y1 = ly2;
			z0 = lz0;
			z1 = lz2;
		}
		else
		{
			y0 = ly2;
			y1 = ly0;
			z0 = lz2;
			z1 = lz0;
		}
		for (i=ax==bx?bx:i;i<=cx;i++)
		{
			ymin = MAX(y0[i],0);
			ymax = MIN(y1[i],HEIGHT-1);
			if (ymin<HEIGHT && ymax>=0)
			{
				line(y0[i],z0[i],y1[i],z1[i],tempZ);
				for (j=ymin;j<=ymax;j++)
				{
					if (tempZ[j]>zbuf[dx+j])
					{
						x = ((i*ux+j*vx+wx)/dDet);
						y = ((i*uy+j*vy+wy)/dDet);
						pix = t->pix[(x<0?0:(x<dw?x:0))+(y<0?0:(y<dh?y*dw:0))];
						if (pix!=trans)
						{
							view3d[dx+j] = pal[pix];
							zbuf[dx+j] = tempZ[j];
						}
					}
				}
			}
			dx+=HEIGHT;
		}
	}
}

/**
 * Draw a "gouraud light" textured triangle
 */

void drawGouraudTexturedTriangle(	unsigned short a,unsigned short b,unsigned short c,
									unsigned char oa,unsigned char ob,unsigned char oc,
									TEXTURE3D_DATA* td)
{
	TEXTURE3D *t = td->t;
	int ax = MAX(drawvert[a].x,0);
	int bx = MIN(drawvert[b].x,WIDTH-1);
	int cx = MIN(drawvert[c].x,WIDTH-1);
	int i,j,ymin,ymax;
	int *y0,*y1,*z0,*z1,*l0,*l1;
	int dx = ax*HEIGHT;
	int	dDet;
	int ux,vx,wx;
	int uy,vy,wy;
	int	dw = t->w+1,dh=t->h+1,x,y,pix,trans = t->trans-1;
	COLOR3D col2;

	detA[0] = drawvert[a].x;
	detA[1] = drawvert[b].x;
	detA[2] = drawvert[c].x;

	detB[0] = drawvert[a].y;
	detB[1] = drawvert[b].y;
	detB[2] = drawvert[c].y;

	detD[0] = td->x[oa];
	detD[1] = td->x[ob];
	detD[2] = td->x[oc];

	dDet = determinate(detA,detB,detC);
	if (dDet==0)
		return;

	ux = determinate(detD,detB,detC);
	vx = determinate(detA,detD,detC);
	wx = determinate(detA,detB,detD);

	detD[0] = td->y[oa];
	detD[1] = td->y[ob];
	detD[2] = td->y[oc];

	uy = determinate(detD,detB,detC);
	vy = determinate(detA,detD,detC);
	wy = determinate(detA,detB,detD);

	line(drawvert[a].x,vlights[a],drawvert[c].x,vlights[c],lr0);
	line(drawvert[a].x,vlights[a],drawvert[b].x,vlights[b],lr1);

	if ((ly0[ax]<ly1[ax]) || (ly0[bx]<ly1[bx]))
	{
		y0 = ly0;
		y1 = ly1;
		z0 = lz0;
		z1 = lz1;
		l0 = lr0;
		l1 = lr1;
	}
	else
	{
		y0 = ly1;
		y1 = ly0;
		z0 = lz1;
		z1 = lz0;
		l0 = lr1;
		l1 = lr0;
	}
	for (i=ax;i<=bx;i++)
	{
		ymin = MAX(y0[i],0);
		ymax = MIN(y1[i],HEIGHT-1);
		if (ymin<HEIGHT && ymax>=0)
		{
			line(y0[i],z0[i],y1[i],z1[i],tempZ);
			line(y0[i],l0[i],y1[i],l1[i],tempR);
			for (j=ymin;j<=ymax;j++)
			{
				if (tempZ[j]>zbuf[dx+j])
				{
					x = ((i*ux+j*vx+wx)/dDet);
					y = ((i*uy+j*vy+wy)/dDet);
					pix = t->pix[(x<0?0:(x<dw?x:0))+(y<0?0:(y<dh?y*dw:0))];
					if (pix!=trans)
					{
						computeColor(tempR[j],&t->pal[pix],&col2);
						view3d[dx+j] = GPCOL(col2.r,col2.g,col2.b);
						zbuf[dx+j] = tempZ[j];
					}
				}
			}
		}
		dx+=HEIGHT;
	}
	if (bx<cx)
	{
		line(drawvert[b].x,vlights[b],drawvert[c].x,vlights[c],lr1);
		if (ly0[i]<ly2[i])
		{
			y0 = ly0;
			y1 = ly2;
			z0 = lz0;
			z1 = lz2;
			l0 = lr0;
			l1 = lr1;
		}
		else
		{
			y0 = ly2;
			y1 = ly0;
			z0 = lz2;
			z1 = lz0;
			l0 = lr1;
			l1 = lr0;
		}
		for (i=ax==bx?bx:i;i<=cx;i++)
		{
			ymin = MAX(y0[i],0);
			ymax = MIN(y1[i],HEIGHT-1);
			if (ymin<HEIGHT && ymax>=0)
			{
				line(y0[i],z0[i],y1[i],z1[i],tempZ);
				line(y0[i],l0[i],y1[i],l1[i],tempR);
				for (j=ymin;j<=ymax;j++)
				{
					if (tempZ[j]>zbuf[dx+j])
					{
						x = ((i*ux+j*vx+wx)/dDet);
						y = ((i*uy+j*vy+wy)/dDet);
						pix = t->pix[(x<0?0:(x<dw?x:0))+(y<0?0:(y<dh?y*dw:0))];
						if (pix!=trans)
						{
							computeColor(tempR[j],&t->pal[pix],&col2);
							view3d[dx+j] = GPCOL(col2.r,col2.g,col2.b);
							zbuf[dx+j] = tempZ[j];
						}
					}
				}
			}
			dx+=HEIGHT;
		}
	}
}

/**
 *
 */

#define MINY(a,b,c) MIN(MIN(vertices[a].y,vertices[b].y),vertices[c].y)

void drawTriangle(unsigned short n)
{
	unsigned short a = triangles[n].a;
	unsigned short b = triangles[n].b;
	unsigned short c = triangles[n].c;
	unsigned short aa,bb,cc;
	unsigned char  oa,ob,oc;
	TRIANGLE3D *t = triangles[n].orig;
	VERTEX3D v;
	COLOR3D col;
	TRIANGLE3D_TYPE type = 0;
	TRIANGLE3D_DATA *data = NULL;

	if (MINY(a,b,c)>=HEIGHT || tz[n]<zMin || tz[n]>zMax)
		return;

	if (drawvert[a].x<drawvert[b].x)
	{
		if (drawvert[a].x<drawvert[c].x)
		{
			aa = a;
			oa = 0;
			if (drawvert[b].x<drawvert[c].x)
			{
				bb = b;
				cc = c;
				ob = 1;
				oc = 2;
			}
			else
			{
				bb = c;
				cc = b;
				ob = 2;
				oc = 1;
			}
		}
		else
		{
			aa = c;
			bb = a;
			cc = b;
			oa = 2;
			ob = 0;
			oc = 1;
		}
	}
	else
	{
		if (drawvert[b].x<drawvert[c].x)
		{
			aa = b;
			oa = 1;
			if (drawvert[a].x<drawvert[c].x)
			{
				bb = a;
				cc = c;
				ob = 0;
				oc = 2;
			}
			else
			{
				bb = c;
				cc = a;
				ob = 2;
				oc = 0;
			}
		}
		else
		{
			aa = c;
			bb = b;
			cc = a;
			oa = 2;
			ob = 1;
			oc = 0;
		}
	}

	if (drawvert[cc].x<0 || drawvert[aa].x>=WIDTH || drawvert[cc].x==drawvert[aa].x)
		return;

	line(drawvert[aa].x,drawvert[aa].y,drawvert[cc].x,drawvert[cc].y,ly0);
	line(drawvert[aa].x,vertices[aa].z,drawvert[cc].x,vertices[cc].z,lz0);

	line(drawvert[aa].x,drawvert[aa].y,drawvert[bb].x,drawvert[bb].y,ly1);
	line(drawvert[aa].x,vertices[aa].z,drawvert[bb].x,vertices[bb].z,lz1);

	line(drawvert[bb].x,drawvert[bb].y,drawvert[cc].x,drawvert[cc].y,ly2);
	line(drawvert[bb].x,vertices[bb].z,drawvert[cc].x,vertices[cc].z,lz2);

	switch(shading3d)
	{
		case GOURAUD:
			if (tz[n]>=gdMin)
			{
				if (t->type==COLORED)
					type = GOURAUD;
				else
					type = TEXTURED_G;
				break;
			} /* else FLAT */
		case FLAT:
			if (t->type==COLORED)
				type = FLAT;
			else
				type = TEXTURED_F;
			break;
		default:
			type = t->type;
			break;
	}

	data = overloads[n]?overloads[n]:&t->data;

	switch(type)
	{
		case COLORED:
			drawFlatTriangle(aa,bb,cc,data->c);
			break;
		case FLAT:
			v.x = (vertices[a].x+vertices[b].x+vertices[c].x)/3;
			v.y = (vertices[a].y+vertices[b].y+vertices[c].y)/3;
			v.z = tz[n];
			computeColor(computeLight(&v,&tnorms[n]),data->c,&col);
			drawFlatTriangle(aa,bb,cc,&col);
			break;
		case GOURAUD:
			drawGouraudTriangle(aa,bb,cc,data->c);
			break;
		case TEXTURED:
			drawFlatTexturedTriangle(aa,bb,cc,oa,ob,oc,NEGATIVE_INFINITY,data->td);
			break;
		case TEXTURED_F:
			v.x = (vertices[a].x+vertices[b].x+vertices[c].x)/3;
			v.y = (vertices[a].y+vertices[b].y+vertices[c].y)/3;
			v.z = tz[n];
			drawFlatTexturedTriangle(aa,bb,cc,oa,ob,oc,computeLight(&v,&tnorms[n]),data->td);
			break;
		case TEXTURED_G:
			drawGouraudTexturedTriangle(aa,bb,cc,oa,ob,oc,data->td);
			break;
	}
}

/**
 * Bresenham
 */

#define SETPIXEL(x,y) if (x>=0&&x<WIDTH) l[x] = y;

void line(int x,int y,int x1,int y1,int* l)
{
	int	dx		= x1-x,
		dy		= (y<y1?y1-y:y-y1);

	if (dy>dx) /* Invert */
	{
		int incrE	= 2*dx,
			incrNE	= 2*(dx-dy),
			d		= 2*dx-dy,
			yy		= (y<y1?1:-1);

		for (y=y+yy;y!=y1+yy;y+=yy)
		{
			if (d<=0)
				d+=incrE;
			else
			{
				d+=incrNE;
				SETPIXEL(x,y-yy);
				x+=1;
			}
		}
		SETPIXEL(x,y-yy);
	}
	else /* Normal */
	{
		int incrE	= 2*dy,
			incrNE	= 2*(dy-dx),
			d		= 2*dy-dx,
			yy		= (y<y1?1:-1);

		SETPIXEL(x,y);
		for (x=x+1;x<=x1 && x<WIDTH;x++)
		{
			if (d<=0)
				d+=incrE;
			else
			{
				d+=incrNE;
				y+=yy;
			}
			SETPIXEL(x,y);
		}
	}
}

/**
 * Show stars in view
 */

void showStars()
{
	int i,x,y,z=(zMax+zStars)/31;
	VERTEX3D v;
	unsigned char c;
	for (i=0;i<nbStars;i++)
	{
		multiplyVector(main3d.m,&stars[i],&v);
		if (v.z>zMax)
			continue;

		x = WIDTH2-((v.x*proj3d)/(proj3d-v.z));
		if (x<0||x>=WIDTH)
			continue;

		y = ((v.y*proj3d)/(proj3d-v.z))+HEIGHT2;
		if (y<0||y>=HEIGHT||zbuf[y*HEIGHT+x]>v.z)
			continue;

		c = 31-((zMax-v.z)/z);
		view3d[x*HEIGHT+y] = (c<<11)|(c<<6)|(c<<1);
	}
}
