//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#include "stdafx.h"
#include "nulstein.h"
#include "ParallelFor.h"
#include "Gfx/Gfx.h"
#include "nulsteinWindow.h"
#include <mmSystem.h>

//________________________________________________________________________________
CFrame::CFrame()
{
	m_pGfx = NULL;
	
	m_t0 = 0xffffffff;
	
	m_EntityCount	= 0;
	m_LightCount	= 0;
	m_DisplayList.m_View = m_View;
}

bool CFrame::Open(CGfx*	pGfx, CSono *pSono, CTaskPool* pTaskPool, IGame* pGame, void* pGameAsVoid)
{
	bool bOk;
	
	/* basics */ 
	m_pGfx			= pGfx;
	m_pSono			= pSono;
	m_pTaskPool		= pTaskPool;
	m_pGame			= pGame;
	m_pGameAsVoid	= pGameAsVoid;
	
	/* create lights */ 
	m_Light[0].Pos.Set(-1.0f, -2.0f, 1.0f);
	m_Light[0].Col.Set( 1.0f,  1.0f, 1.0f);
	
	m_LightCount = 1;
	
	/* init game */ 
	bOk = pGame->Init(this);
	IF_FAILED_RETURN(bOk);

	/* init Time (t is taken from Sono, 0 = start of buffer) */ 
	bOk = m_pSono->StartPlaying();
	m_t0 = 0;
	
	return true;
}

void	CFrame::Close()
{
}

CEntitySlot*	CFrame::AllocEntity()
{	/* placeholder */ 
	return &m_Entity[m_EntityCount++];
}

//________________________________________________________________________________
#define GFXUPDATESTREAMSIZE (128*1024)

class CFrameDrawer : public IForTask
{
public:
	void Init(CFrame*	pFrame)
	{
		m_pFrame = pFrame; 
		
		for(int i=0; i<MAX_THREADS; i++)
		{
			m_DisplayListAdder[i].m_pDisplayList = &m_pFrame->m_DisplayList;
		}
	}
	
	virtual bool DoRange(CWorkerThread* pThread, const CRange &Range) const
	{
		bool				bOk;
		CDisplayListAdder*	pDL = &m_DisplayListAdder[pThread->GetWorkerIndex()];
		int					InitialAddCount;

		for(int iEntity= Range.begin; iEntity < Range.end; iEntity++)
		{
			CEntitySlot*	pSlot;
			CEntity*		pEntity;
			uint32_t		EntityKey;
			
			/* check entity wants to draw */ 
			pSlot = &m_pFrame->m_Entity[iEntity];
			if (pSlot->m_Flags & ENTITY_NODRAW)
				continue;
			
			/* draw */ 
			EntityKey		= iEntity << 16;
			pEntity			= pSlot->m_pEntity;
			InitialAddCount = pDL->m_AddCount;
			for(int iView=0; iView<m_pFrame->m_ViewCount; iView++)
			{	
				bOk = pEntity->Draw(pDL, EntityKey, iView);
				ASSERT( SUCCEEDED(bOk) );
			}
		}
		
		return true;
	}
	
public:
	CFrame*				m_pFrame;
	mutable CDisplayListAdder	m_DisplayListAdder[MAX_THREADS];
};

bool CFrame::Draw()
{
	PROFILE_SCOPE("CFrame::Draw")
	
	bool			bOk;
	CFrameDrawer	FrameDrawer;
	
	/* reset display-list(s) */ 
	m_DisplayList.Begin();
	
	/* parallel_for draw everybody */ 
	FrameDrawer.m_pFrame		= this;
	FrameDrawer.Init(this);
	
	bOk = ParallelFor(&FrameDrawer, CRange(0, m_EntityCount, 16));
	IF_FAILED_RETURN(bOk);
	
	return true;
}

//________________________________________________________________________________
bool CFrame::Render()
{
	bool			bOk;
	int				iItem;
	sDisplayItem*	pItem;
	CDisplayItemKey LastKey;
	
	{
		PROFILE_SCOPE("CFrame::Render")
		
		// m_pGfx->m_spDevice->SetRenderState(D3DRS_FILLMODE, D3DFILL_WIREFRAME);
		
		/* Set in out of range state to detect when any value changes */ 
		LastKey.MakeOutOfRange();
		
		/* run list */ 
		for(iItem=0; iItem<m_DisplayList.m_ItemCount; iItem++)
		{
			CDisplayItemKey Key;
			CEntity*		pEntity;
			
			pItem = &m_DisplayList.m_Item[iItem];
			
			if (0== pItem->m_Key+1) 
				break;
			
			Key.Decode( pItem->m_Key );
			
			bOk = ProcessRenderContextChanges(&LastKey, Key);
			ASSERT( SUCCEEDED(bOk) );
			
			pEntity = m_Entity[ pItem->m_EntityKey>>16 ].m_pEntity;
			
			bOk = pEntity->Render(m_pGfx, Key, pItem->m_EntityKey, pItem->m_Param);
			ASSERT( SUCCEEDED(bOk) );
		}

		return true;
	}
}


//________________________________________________________________________________
class CFramePreUpdater : public IForTask
{
public:
	virtual bool DoRange(CWorkerThread* pThread, const CRange &Range) const
	{
		bool bOk;
		int		iEntity;
		
		for(iEntity = Range.begin; iEntity < Range.end; iEntity++)
		{
			bOk = m_pFrame->m_Entity[iEntity].PreUpdate(m_pFrame, m_DeltaTime);
			ASSERT( SUCCEEDED(bOk) );
		}
		
		return true;
	}
	
public:
	CFrame*	m_pFrame;
	float	m_DeltaTime;
};

class CFrameUpdater : public IForTask
{
public:
	virtual bool DoRange(CWorkerThread* pThread, const CRange &Range) const
	{
		bool bOk;
		int		iEntity;
		
		for(iEntity = Range.begin; iEntity < Range.end; iEntity++)
		{
			bOk = m_pFrame->m_Entity[iEntity].Update(m_pFrame, m_DeltaTime);
			ASSERT( SUCCEEDED(bOk) );
		}
		
		return true;
	}
	
public:
	CFrame*	m_pFrame;
	float	m_DeltaTime;
};

//________________________________________________________________________________
bool CFrame::Update(int TickCount)
{
	float		dt;
	
	CWorkerThread*		pWorkerThread;
	CFramePreUpdater	PreUpdater;
	CFrameUpdater		Updater;
	
	pWorkerThread = CWorkerThread::GetCurrent();
	
	/* convert ticks to seconds (handle slomo here) */ 
	dt = TickCount / 1000.0f;
	
	/* parallel_for pre-update everybody */ 
	{	PROFILE_SCOPE("CFrame::Update - preUpdate")
		
		PreUpdater.m_pFrame		= this;
		PreUpdater.m_DeltaTime	= dt;
		ParallelFor(pWorkerThread, &PreUpdater, CRange(0, m_EntityCount, 8));
	}
	
	/* parallel_for update everybody */ 
	{	PROFILE_SCOPE("CFrame::Update - Update")
		Updater.m_pFrame	= this;
		Updater.m_DeltaTime	= dt;
		ParallelFor(pWorkerThread, &Updater, CRange(0, m_EntityCount, 8));
	}
	
	return true;
}

