//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#include "StdAfx.h"
#include "Gfx.h"

#pragma intrinsic(memset)


// #define ENABLE_PERFHUD

CGfx::CGfx()
{
	m_hWnd			= NULL;
	m_bFullscreen	= false;
	m_BehaviorFlags	= 0;
}

bool CGfx::Open(HWND hWnd)
{
	bool bOk;
	
	m_hWnd = hWnd;
	
	/* D3D */ 
	m_spD3D = Direct3DCreate9(D3D_SDK_VERSION);
	if (!m_spD3D) 
		return false;

	/* Device */ 
	bOk = CreateDevice();
	IF_FAILED_RETURN(bOk);
	
	bOk = OnResetDevice();
	IF_FAILED_RETURN(bOk);

	/* Vertex Declarations */ 
	bOk = DeclareShaders();
	IF_FAILED_RETURN(bOk);
	
	return true;
}

void CGfx::Close()
{
	bool bOk;
	
	/* D3D */ 
	bOk = OnLostDevice();
	
	m_spDevice	= NULL;
	m_spD3D		= NULL;
}

int CGfx::TryMultisample(D3DPRESENT_PARAMETERS* pD3DPP, UINT Adapter, D3DDEVTYPE DeviceType, D3DMULTISAMPLE_TYPE MultiSampleType)
{
	HRESULT hr;
	
	hr = m_spD3D->CheckDeviceMultiSampleType(	Adapter, 
												DeviceType , 
												pD3DPP->BackBufferFormat, 
												pD3DPP->Windowed, 
												MultiSampleType, 
												NULL );
	if (D3DERR_NOTAVAILABLE==hr) return 0;
	if (FAILED(hr)) return -1;

	hr = m_spD3D->CheckDeviceMultiSampleType(	Adapter, 
												DeviceType , 
												pD3DPP->AutoDepthStencilFormat, 
												pD3DPP->Windowed, 
												MultiSampleType, 
												NULL );
	if (D3DERR_NOTAVAILABLE==hr) return 0;
	if (FAILED(hr)) return -1;
		
	pD3DPP->MultiSampleType = MultiSampleType;
	return 1;
}

//________________________________________________________________________________
bool CGfx::CreateDevice()
{
	HRESULT				hr;
	D3DDISPLAYMODE		DisplayMode;
	ULONG				BehaviorFlags	= 0;
	D3DDEVTYPE			DeviceType		= D3DDEVTYPE_HAL;
	UINT				AdapterToUse	= D3DADAPTER_DEFAULT;
	UINT				PresInterval	= 0; // D3DPRESENT_INTERVAL_IMMEDIATE;
	int					res;

	memset(&m_D3DPP, 0, sizeof(m_D3DPP) );

	// Get current display mode
	hr = m_spD3D->GetAdapterDisplayMode(AdapterToUse, &DisplayMode);
	IF_FAILED_ASSERT(hr);

	//____________________________________________________________________________
	if (m_bFullscreen)
	{
		m_D3DPP.Windowed               = FALSE;
		m_D3DPP.SwapEffect             = D3DSWAPEFFECT_DISCARD;
		m_D3DPP.BackBufferWidth        = GetSystemMetrics(SM_CXSCREEN);
		m_D3DPP.BackBufferHeight       = GetSystemMetrics(SM_CYSCREEN);
		m_D3DPP.BackBufferFormat       = D3DFMT_X8R8G8B8;
		m_D3DPP.PresentationInterval   = PresInterval;
		m_D3DPP.EnableAutoDepthStencil = TRUE;
		m_D3DPP.AutoDepthStencilFormat = D3DFMT_D24S8;
	}
	else
	{
		m_D3DPP.Windowed				= TRUE;
		m_D3DPP.SwapEffect				= D3DSWAPEFFECT_DISCARD;
		m_D3DPP.BackBufferFormat		= DisplayMode.Format;
		m_D3DPP.PresentationInterval    = PresInterval;
		m_D3DPP.EnableAutoDepthStencil	= TRUE;
		m_D3DPP.AutoDepthStencilFormat	= D3DFMT_D24S8;
	}
	
	// Check multisampling capabilities
	res = TryMultisample(&m_D3DPP, AdapterToUse, DeviceType, D3DMULTISAMPLE_2_SAMPLES);
	if (res<0) return false;

	res = TryMultisample(&m_D3DPP, AdapterToUse, DeviceType, D3DMULTISAMPLE_4_SAMPLES);
	if (res<0) return false;
	
	// Default settings
#if defined(DEBUG_VS)||defined(DEBUG_PS)
	#ifdef DEBUG_PS
	// Debug pixel shaders
	m_D3DPP.MultiSampleType = 0;
	m_BehaviorFlags = BehaviorFlags | D3DCREATE_SOFTWARE_VERTEXPROCESSING;
	err = HR2RV( m_spD3D->CreateDevice(	AdapterToUse, 
											D3DDEVTYPE_REF, 
											m_hWnd, // ?! If you specify NULL, you also will need to specify a non-NULL handle when calling IDirect3DDevice9::Present. 
											m_BehaviorFlags,
											&m_D3DPP, 
											&m_spDevice 
										) );
	if (err.ok()) 
	{
		m_CreationLog += "Created DEBUGPS REF device, with software vertex processing";
		return true;
	}
	#else
	// Debug vertex shaders
	m_BehaviorFlags = BehaviorFlags | D3DCREATE_SOFTWARE_VERTEXPROCESSING;
	err = HR2RV( m_spD3D->CreateDevice(	AdapterToUse, 
											DeviceType, 
											m_hWnd, // ?! If you specify NULL, you also will need to specify a non-NULL handle when calling IDirect3DDevice9::Present. 
											m_BehaviorFlags,
											&m_D3DPP, 
											&m_spDevice 
										) );
	if (err.ok()) 
	{
		m_CreationLog += "Created DEBUGVS HAL device, with software vertex processing";
		return true;
	}
	#endif
#else
	
	#ifdef ENABLE_PERFHUD
	// Look for 'NVIDIA PerfHUD' adapter
	for (UINT iAdapter=0; iAdapter<m_spD3D->GetAdapterCount(); iAdapter++)
	{
		D3DADAPTER_IDENTIFIER9	Identifier;
		bool					bOk;
		
		bOk = m_spD3D->GetAdapterIdentifier(iAdapter, 0, &Identifier);
		MessageBoxA(m_hWnd, Identifier.Description, Identifier.Description, MB_OK);
		if (strstr(Identifier.Description,"PerfHUD") != 0)
		{
			AdapterToUse	= iAdapter;
			DeviceType		= D3DDEVTYPE_REF;
			m_bUnderPerfHud = true;
			break;
		}
	}
	#endif

	// Try hardware vertex processing
	m_BehaviorFlags = BehaviorFlags | D3DCREATE_HARDWARE_VERTEXPROCESSING;
	hr = m_spD3D->CreateDevice(	AdapterToUse, 
									DeviceType, 
									GetDesktopWindow(), // ?! If you specify NULL, you also will need to specify a non-NULL handle when calling IDirect3DDevice9::Present. This also creates biggest backbuffer we might need...
									m_BehaviorFlags,
									&m_D3DPP, 
									&m_spDevice 
								  );
	if (SUCCEEDED(hr)) return true;
	

	// Try software vertex processing
	m_BehaviorFlags = BehaviorFlags | D3DCREATE_SOFTWARE_VERTEXPROCESSING;
	hr = m_spD3D->CreateDevice(	AdapterToUse, 
								DeviceType, 
								GetDesktopWindow(),
								m_BehaviorFlags,
								&m_D3DPP, 
								&m_spDevice 
							   );
	if (SUCCEEDED(hr)) return true;
#endif

	m_BehaviorFlags = 0;
	return false;
}

bool CGfx::DeclareShaders()
{
	static const D3DVERTEXELEMENT9 declPN[] =
	{
		{ 0, 0, D3DDECLTYPE_FLOAT3, D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_POSITION, 0 },
		{ 0,12, D3DDECLTYPE_FLOAT3, D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_NORMAL,   0 },
		D3DDECL_END()
	};
	
	HRESULT hr;

	hr = m_spDevice->CreateVertexDeclaration(declPN, &m_spVtxDclPN);
	IF_FAILED_RETURN(hr);
	
	hr = m_VSHBasic.Create(this, m_spVtxDclPN);
	IF_FAILED_RETURN(hr);
	
	hr = m_PSHBasic.Create(this);
	IF_FAILED_RETURN(hr);
	
	return true;
}

//________________________________________________________________________________
bool CGfx::OnLostDevice()
{
	
	return true;
}

bool CGfx::OnResetDevice()
{
	
	return true;
}

//________________________________________________________________________________
bool CGfx::BeginFrame()
{
	bool			bOk;
	HRESULT			hr;
	D3DVIEWPORT9	viewData;

	/* check if we need to restore the device */ 
	hr = m_spDevice->TestCooperativeLevel();
	switch(hr)
	{
		case D3DERR_DEVICELOST:		
			m_bNeedsReset = true;
			return false;
			
		case D3DERR_DEVICENOTRESET:
		{
			m_bNeedsReset = true;
			
			bOk = Reset();
			IF_FAILED_RETURN(bOk);
		}	break;
	}

	/* begin scene */ 
	hr = m_spDevice->BeginScene();
	IF_FAILED_RETURN(hr);
	
	/* setup viewport */ 
	if (m_bFullscreen)
	{
		m_ClientRect.left	= 0;
		m_ClientRect.top	= 0;
		m_ClientRect.right	= m_D3DPP.BackBufferWidth;
		m_ClientRect.bottom	= m_D3DPP.BackBufferHeight;
	}
	else
	{
		GetClientRect(m_hWnd, &m_ClientRect);
	}
	
	viewData.X		= 0;
	viewData.Y		= 0;
	viewData.Width	= m_ClientRect.right  - m_ClientRect.left;
	viewData.Height = m_ClientRect.bottom - m_ClientRect.top;
	viewData.MinZ	= 0.0f;
	viewData.MaxZ	= 1.0f;

	hr = m_spDevice->SetViewport(&viewData);
	IF_FAILED_RETURN(hr);
	
	/* Clear */ 
	hr = m_spDevice->Clear(0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER|D3DCLEAR_STENCIL, 0, 1.0f, 0);
	IF_FAILED_ASSERT(hr);
	
	return true;
}

bool CGfx::EndFrame()
{
	HRESULT hr;
	
	hr = m_spDevice->EndScene();
	IF_FAILED_RETURN(hr);

	/* put on screen */ 
	if (m_bFullscreen)
	{	
		hr = m_spDevice->Present(NULL, NULL, m_hWnd, NULL);
		IF_FAILED_RETURN(hr);
	}
	else
	{
		hr = m_spDevice->Present(&m_ClientRect, &m_ClientRect, m_hWnd, NULL);
		IF_FAILED_RETURN(hr);
	}
	
	return true;
}

bool CGfx::Reset()
{
	bool	bOk;
	HRESULT hr;
	
	bOk = OnLostDevice();
	IF_FAILED_RETURN(bOk);
	
	hr = m_spDevice->Reset(&m_D3DPP);
	IF_FAILED_RETURN(hr);
	
	hr = OnResetDevice();
	IF_FAILED_RETURN(hr);
	
	return true;
}

//________________________________________________________________________________
bool CGfx::SetLights(const CLight* Lights, unsigned Count)
{
	m_Light      = Lights;
	m_LightCount = Count;

	return true;
}

bool CGfx::SelectAlpha(bool bAlphaEnable)
{
	HRESULT hr;
	
	hr = m_spDevice->SetRenderState(D3DRS_ALPHABLENDENABLE, bAlphaEnable);
	IF_FAILED_RETURN(hr);
	
	if (bAlphaEnable)
	{
		hr = m_spDevice->SetRenderState(D3DRS_SRCBLEND,		D3DBLEND_SRCALPHA);
		hr = m_spDevice->SetRenderState(D3DRS_DESTBLEND,	D3DBLEND_INVSRCALPHA);
	}

	hr = m_spDevice->SetRenderState(D3DRS_ZWRITEENABLE,	!bAlphaEnable);
	
	return true;
}

